export default {
    name: 'RoomImageCarousel',
    props: {
        images: {
            type: Array,
            default: () => []
        },
        roomName: {
            type: String,
            default: 'Room'
        },
        height: {
            type: [String, Number],
            default: 200
        },
        showFullscreen: {
            type: Boolean,
            default: true
        },
        autoPlay: {
            type: Boolean,
            default: false
        },
        interval: {
            type: Number,
            default: 5000
        }
    },

    emits: ['image-change', 'fullscreen-open', 'fullscreen-close'],

    data() {
        return {
            currentIndex: 0,
            isFullscreen: false,
            isLoading: false,
            touchStartX: 0,
            touchEndX: 0,
            isDragging: false,
            showControls: false,
            autoPlayTimer: null,
            isTransitioning: false,

            // Configuration
            swipeConfig: {
                minSwipeDistance: 50,
                maxSwipeTime: 300,
                touchThreshold: 10
            }
        };
    },

    computed: {
        processedImages() {
            if (!this.images || !Array.isArray(this.images)) {
                return [];
            }

            return this.images.map((url, index) => ({
                url: url,
                loaded: false,
                error: false,
                index: index
            }));
        },

        carouselClasses() {
            return {
                'single-image': this.processedImages.length <= 1,
                'multiple-images': this.processedImages.length > 1,
                'mobile': this.isMobile
            };
        },

        showArrows() {
            return this.processedImages.length > 1;
        },

        isMobile() {
            return window.innerWidth <= 768;
        }
    },

    watch: {
        currentIndex(newIndex) {
            this.$emit('image-change', newIndex);
        },

        isFullscreen(isOpen) {
            if (isOpen) {
                this.$emit('fullscreen-open', this.currentIndex);
                document.addEventListener('keydown', this.handleKeydown);
            } else {
                this.$emit('fullscreen-close');
                document.removeEventListener('keydown', this.handleKeydown);
            }
        },

        autoPlay: {
            handler(enabled) {
                if (enabled && this.processedImages.length > 1) {
                    this.startAutoPlay();
                } else {
                    this.stopAutoPlay();
                }
            },
            immediate: true
        }
    },

    mounted() {
        // Add keyboard navigation
        document.addEventListener('keydown', this.handleKeydown);

        // Start autoplay if enabled
        if (this.autoPlay && this.processedImages.length > 1) {
            this.startAutoPlay();
        }
    },

    beforeUnmount() {
        document.removeEventListener('keydown', this.handleKeydown);
        this.stopAutoPlay();
    },

    methods: {
        nextImage() {
            if (this.processedImages.length <= 1 || this.isTransitioning) return;

            this.isTransitioning = true;
            this.currentIndex = (this.currentIndex + 1) % this.processedImages.length;
            this.resetAutoPlay();

            // Reset transition flag after animation completes
            setTimeout(() => {
                this.isTransitioning = false;
            }, 500);
        },

        previousImage() {
            if (this.processedImages.length <= 1 || this.isTransitioning) return;

            this.isTransitioning = true;
            this.currentIndex = this.currentIndex === 0
                ? this.processedImages.length - 1
                : this.currentIndex - 1;
            this.resetAutoPlay();

            // Reset transition flag after animation completes
            setTimeout(() => {
                this.isTransitioning = false;
            }, 500);
        },

        goToImage(index) {
            if (index >= 0 && index < this.processedImages.length) {
                this.currentIndex = index;
                this.resetAutoPlay();
            }
        },

        handleTouchStart(event) {
            if (!this.isMobile || this.processedImages.length <= 1) return;

            this.touchStartX = event.touches[0].clientX;
            this.isDragging = true;

            // Prevent page scroll during swipe
            event.preventDefault();
        },

        handleTouchEnd(event) {
            if (!this.isDragging || this.processedImages.length <= 1) return;

            this.touchEndX = event.changedTouches[0].clientX;
            this.handleSwipe();
            this.isDragging = false;
        },

        handleSwipe() {
            const swipeDistance = this.touchStartX - this.touchEndX;
            const absSwipeDistance = Math.abs(swipeDistance);

            if (absSwipeDistance < this.swipeConfig.minSwipeDistance) return;

            if (swipeDistance > 0) {
                // Swiped left - next image
                this.nextImage();
            } else {
                // Swiped right - previous image
                this.previousImage();
            }

            // Add haptic feedback if supported
            if (navigator.vibrate) {
                navigator.vibrate(50);
            }
        },

        handleKeydown(event) {
            if (!this.isFullscreen) return;

            switch (event.key) {
                case 'ArrowLeft':
                    event.preventDefault();
                    this.previousImage();
                    break;
                case 'ArrowRight':
                    event.preventDefault();
                    this.nextImage();
                    break;
                case 'Escape':
                    event.preventDefault();
                    this.closeFullscreen();
                    break;
            }
        },

        handleImageLoad(index) {
            if (this.processedImages[index]) {
                this.processedImages[index].loaded = true;
            }
        },

        handleImageError(index) {
            if (this.processedImages[index]) {
                this.processedImages[index].error = true;
                // Set fallback image
                this.processedImages[index].url = 'https://via.placeholder.com/300x200?text=Room+Image';
            }
        },

        openFullscreen(index = null) {
            if (!this.showFullscreen) return;

            if (index !== null) {
                this.currentIndex = index;
            }
            this.isFullscreen = true;
        },

        closeFullscreen() {
            this.isFullscreen = false;
        },

        startAutoPlay() {
            if (this.autoPlayTimer) return;

            this.autoPlayTimer = setInterval(() => {
                this.nextImage();
            }, this.interval);
        },

        stopAutoPlay() {
            if (this.autoPlayTimer) {
                clearInterval(this.autoPlayTimer);
                this.autoPlayTimer = null;
            }
        },

        resetAutoPlay() {
            if (this.autoPlay) {
                this.stopAutoPlay();
                this.startAutoPlay();
            }
        }
    },

    template: `
    <div class="room-image-carousel" :class="carouselClasses">
      <!-- Main carousel container -->
      <div 
        class="carousel-container" 
        @touchstart="handleTouchStart" 
        @touchend="handleTouchEnd"
        @mouseenter="showControls = true"
        @mouseleave="showControls = false"
      >
        <!-- Image display area -->
        <div class="image-container">
          <transition-group name="slide" tag="div" class="image-wrapper">
            <v-img
              v-for="(image, index) in processedImages"
              :key="\`image-\${index}\`"
              v-show="index === currentIndex"
              :src="image.url"
              :height="height"
              :alt="\`\${roomName} - Image \${index + 1}\`"
              class="carousel-image"
              cover
              @load="handleImageLoad(index)"
              @error="handleImageError(index)"
              @click="openFullscreen(index)"
            >
              <template v-slot:placeholder>
                <div class="image-loading">
                  <v-progress-circular indeterminate color="primary" />
                </div>
              </template>
            </v-img>
          </transition-group>
        </div>
      </div>

      <!-- Indicators -->
      <div v-if="processedImages.length > 1" class="carousel-indicators">
        <!-- Dot indicators for <= 5 images -->
        <div v-if="processedImages.length <= 5" class="dot-indicators">
          <button
            v-for="(image, index) in processedImages"
            :key="\`dot-\${index}\`"
            class="dot-indicator"
            :class="{ active: index === currentIndex }"
            @click="goToImage(index)"
            :aria-label="\`Go to image \${index + 1}\`"
          />
        </div>
        
        <!-- Counter for > 5 images -->
        <div v-else class="counter-indicator">
          {{ currentIndex + 1 }} of {{ processedImages.length }}
        </div>
      </div>

      <!-- Fullscreen modal -->
      <v-dialog v-model="isFullscreen" fullscreen>
        <v-card class="fullscreen-carousel">
          <v-toolbar dark color="black">
            <v-toolbar-title>{{ roomName }} - Images</v-toolbar-title>
            <v-spacer />
            <v-btn icon @click="closeFullscreen">
              <v-icon>mdi-close</v-icon>
            </v-btn>
          </v-toolbar>
          
          <!-- Fullscreen carousel content -->
          <div class="fullscreen-content">
            <div class="fullscreen-image-container">
              <v-img
                :src="processedImages[currentIndex]?.url"
                :alt="\`\${roomName} - Image \${currentIndex + 1}\`"
                contain
                class="fullscreen-image"
              />
              
              <!-- Fullscreen navigation -->
              <div v-if="processedImages.length > 1" class="fullscreen-arrows">
                <v-btn
                  icon
                  size="large"
                  class="fullscreen-arrow-btn"
                  @click="previousImage"
                >
                  <v-icon>mdi-chevron-left</v-icon>
                </v-btn>
                <v-btn
                  icon
                  size="large"
                  class="fullscreen-arrow-btn"
                  @click="nextImage"
                >
                  <v-icon>mdi-chevron-right</v-icon>
                </v-btn>
              </div>
            </div>
            
            <!-- Fullscreen indicators -->
            <div v-if="processedImages.length > 1" class="fullscreen-indicators">
              <div v-if="processedImages.length <= 10" class="fullscreen-dots">
                <button
                  v-for="(image, index) in processedImages"
                  :key="\`fs-dot-\${index}\`"
                  class="fullscreen-dot"
                  :class="{ active: index === currentIndex }"
                  @click="goToImage(index)"
                />
              </div>
              <div v-else class="fullscreen-counter">
                {{ currentIndex + 1 }} / {{ processedImages.length }}
              </div>
            </div>
          </div>
        </v-card>
      </v-dialog>
    </div>
  `,

    style: `
    <style scoped>
    .room-image-carousel {
      position: relative;
      border-radius: 8px;
      overflow: hidden;
      background: #f5f5f5;
      width: 100%;
    }

    .carousel-container {
      position: relative;
      cursor: pointer;
      width: 100%;
      height: 100%;
    }

    .image-container {
      position: relative;
      overflow: hidden;
      width: 100%;
      height: 100%;
    }

    .image-wrapper {
      position: relative;
      width: 100%;
      height: 100%;
    }

    .carousel-image {
      border-radius: 0;
      width: 100% !important;
      height: 100% !important;
      object-fit: cover !important;
    }

    .carousel-image .v-img__img {
      width: 100% !important;
      height: 100% !important;
      object-fit: cover !important;
      object-position: center !important;
    }

    .image-loading {
      display: flex;
      align-items: center;
      justify-content: center;
      height: 100%;
      background: rgba(0, 0, 0, 0.1);
    }

    /* Navigation arrows */
    .carousel-arrows {
      position: absolute;
      top: 12px;
      left: 12px;
      right: 12px;
      height: 40px;
      display: flex;
      justify-content: space-between;
      align-items: center;
      pointer-events: none;
      z-index: 3;
    }

    .arrow-btn {
      pointer-events: all;
      background: rgba(255, 255, 255, 0.95) !important;
      backdrop-filter: blur(8px);
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
      border: 1px solid rgba(255, 255, 255, 0.2);
      width: 36px !important;
      height: 36px !important;
      min-width: 36px !important;
    }

    .arrow-btn:hover {
      background: rgba(255, 255, 255, 1) !important;
      transform: scale(1.1);
      box-shadow: 0 4px 16px rgba(0, 0, 0, 0.2);
    }

    .arrow-btn:active {
      transform: scale(0.95);
    }

    .show-on-hover .arrow-btn {
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    /* Indicators */
    .carousel-indicators {
      position: absolute;
      bottom: 12px;
      left: 50%;
      transform: translateX(-50%);
      z-index: 2;
    }

    .dot-indicators {
      display: flex;
      gap: 8px;
    }

    .dot-indicator {
      width: 8px;
      height: 8px;
      border-radius: 50%;
      border: none;
      background: rgba(255, 255, 255, 0.5);
      cursor: pointer;
      transition: all 0.2s ease;
    }

    .dot-indicator.active {
      background: white;
      transform: scale(1.2);
    }

    .dot-indicator:hover {
      background: rgba(255, 255, 255, 0.8);
    }

    .counter-indicator {
      background: rgba(0, 0, 0, 0.7);
      color: white;
      padding: 4px 8px;
      border-radius: 12px;
      font-size: 0.8rem;
      font-weight: 500;
    }

    /* Fullscreen styles */
    .fullscreen-carousel {
      background: black;
    }

    .fullscreen-content {
      position: relative;
      height: calc(100vh - 64px);
      display: flex;
      flex-direction: column;
    }

    .fullscreen-image-container {
      position: relative;
      flex: 1;
      display: flex;
      align-items: center;
      justify-content: center;
    }

    .fullscreen-image {
      max-height: 100%;
      max-width: 100%;
    }

    .fullscreen-arrows {
      position: absolute;
      top: 50%;
      left: 0;
      right: 0;
      transform: translateY(-50%);
      display: flex;
      justify-content: space-between;
      padding: 0 20px;
      pointer-events: none;
    }

    .fullscreen-arrow-btn {
      pointer-events: all;
      background: rgba(255, 255, 255, 0.2) !important;
      color: white !important;
    }

    .fullscreen-arrow-btn:hover {
      background: rgba(255, 255, 255, 0.3) !important;
    }

    .fullscreen-indicators {
      padding: 20px;
      display: flex;
      justify-content: center;
    }

    .fullscreen-dots {
      display: flex;
      gap: 12px;
    }

    .fullscreen-dot {
      width: 12px;
      height: 12px;
      border-radius: 50%;
      border: 2px solid rgba(255, 255, 255, 0.5);
      background: transparent;
      cursor: pointer;
      transition: all 0.2s ease;
    }

    .fullscreen-dot.active {
      background: white;
      border-color: white;
    }

    .fullscreen-counter {
      color: white;
      font-size: 1rem;
      font-weight: 500;
    }

    /* Slide transitions */
    .slide-enter-active {
      transition: all 0.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    }

    .slide-leave-active {
      transition: all 0.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
    }

    .slide-enter-from {
      opacity: 0;
      transform: translateX(100%);
    }

    .slide-leave-to {
      opacity: 0;
      transform: translateX(-100%);
    }

    .slide-enter-to,
    .slide-leave-from {
      opacity: 1;
      transform: translateX(0);
    }

    /* Mobile optimizations */
    @media (max-width: 768px) {
      .carousel-arrows {
        top: 8px;
        left: 8px;
        right: 8px;
        height: 32px;
      }
      
      .arrow-btn {
        width: 32px !important;
        height: 32px !important;
        min-width: 32px !important;
        background: rgba(255, 255, 255, 0.9) !important;
      }
      
      .carousel-indicators {
        bottom: 8px;
      }
      
      .dot-indicator {
        width: 6px;
        height: 6px;
      }
      
      .counter-indicator {
        font-size: 0.7rem;
        padding: 3px 6px;
      }
      
      .fullscreen-arrows {
        padding: 0 10px;
      }
    }

    /* Prevent text selection during swipe */
    .room-image-carousel * {
      -webkit-user-select: none;
      -moz-user-select: none;
      -ms-user-select: none;
      user-select: none;
    }
    </style>
  `
};