// BentoButton component
const BentoButton = {
    props: {
      item: {
        type: Object,
        required: true
      },
      isHovered: {
        type: Boolean,
        default: false
      },
      isVisible: {
        type: Boolean,
        default: false
      },
      animDirection: {
        type: String,
        default: 'none'
      }
    },
    computed: {
      // Get animation class based on visibility and direction
      getAnimationClass() {
        if (!this.isVisible) {
          return 'opacity-0 scale-50';
        }
        
        switch (this.animDirection) {
          case 'bubble':
            return 'opacity-100 scale-100';
          case 'left':
            return 'opacity-100 translate-x-0';
          case 'right':
            return 'opacity-100 translate-x-0';
          default:
            return 'opacity-100 translate-y-0';
        }
      },
      
      // Generate the inline style object for the button
      buttonStyle() {
        let transform = '';
        
        if (!this.isVisible) {
          switch (this.animDirection) {
            case 'bubble':
              transform = 'scale(0.5)';
              break;
            case 'left':
              transform = 'translateX(-30px)';
              break;
            case 'right':
              transform = 'translateX(30px)';
              break;
          }
        }
        
        return {
          backgroundColor: this.item.bgColor,
          gridArea: this.item.area,
          height: '100%',
          transitionDelay: this.isVisible ? `${this.item.animDelay || 0}ms` : '0ms',
          transformOrigin: this.getTransformOrigin(),
          transform: transform
        };
      },
      
      // Classes for hover effects
      hoverClasses() {
        return this.isHovered ? 'shadow-lg shadow-black/20' : 'shadow-md';
      },
      
      // Classes for text color based on light/dark background
      textClasses() {
        return this.item.textLight ? 'text-white' : 'text-gray-800';
      },
      
      // Classes for content text
      contentClasses() {
        return `mt-2 transition-all duration-300 ease-in-out ${this.item.textLight ? 'text-white/90' : 'text-gray-700'} 
          ${this.isHovered ? 'opacity-100' : 'opacity-80'} ${this.item.large ? 'text-sm' : 'text-xs'}`;
      },
      
      // Classes for action button
      actionButtonClasses() {
        return `mt-3 px-3 py-1.5 rounded-lg text-sm font-medium transition-all duration-300 transform 
          ${this.isHovered ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-4'} 
          ${this.item.textLight ? 'bg-white/20 hover:bg-white/30 text-white' : 'bg-gray-800/10 hover:bg-gray-800/20 text-gray-800'}`;
      },
      
      // Classes for decorative circle
      decorativeCircleClasses() {
        return `absolute right-0 bottom-0 w-32 h-32 rounded-full transition-all duration-500 ease-in-out 
          ${this.item.textLight ? 'bg-white/10' : 'bg-black/5'} 
          ${this.isHovered ? 'opacity-100 scale-100' : 'opacity-0 scale-0'} transform translate-x-1/4 translate-y-1/4`;
      },
      
      // Classes for hover border
      hoverBorderClasses() {
        return `absolute inset-0 border-2 rounded-xl transition-opacity duration-300 ease-in-out pointer-events-none 
          ${this.isHovered ? 'opacity-100' : 'opacity-0'} 
          ${this.item.textLight ? 'border-white/20' : 'border-black/10'}`;
      }
    },
    methods: {
      // Calculate transform origin based on item position
      getTransformOrigin() {
        if (this.animDirection === 'bubble') {
          const index = parseInt(this.item.area.substring(1)) % 3;
          const origins = ['center top', 'center center', 'center bottom'];
          return origins[index];
        }
        return 'center';
      },
      
      // Event handlers
      onHover() {
        this.$emit('item-hover', this.item.id);
      },
      
      onLeave() {
        this.$emit('item-hover', null);
      },
      
      onClick() {
        this.$emit('item-click', this.item.id);
      }
    },
    template: `
      <div
        :class="'relative overflow-hidden rounded-xl transform transition-all duration-300 ease-out cursor-pointer ' + hoverClasses + ' ' + getAnimationClass"
        :style="buttonStyle"
        @mouseenter="onHover"
        @mouseleave="onLeave"
        @click="onClick"
      >
        <div class="relative z-10 h-full p-5 flex flex-col">
          <!-- Icon & Title Section -->
          <div class="flex justify-between items-start">
            <h3 :class="'font-bold ' + textClasses" 
                :style="{ fontSize: item.large ? '1.25rem' : '1.1rem' }">
              {{ item.title }}
            </h3>
            <div :class="'text-3xl transition-transform duration-300 ' + (isHovered ? 'scale-110' : 'scale-100')">
              {{ item.icon }}
            </div>
          </div>
          
          <!-- Content Section -->
          <div class="mt-auto">
            <div v-if="item.content" :class="contentClasses">
              {{ item.content }}
            </div>
            
            <!-- Action Button - only shows on hover -->
            <button v-if="item.action" :class="actionButtonClasses">
              {{ item.action }}
            </button>
          </div>
        </div>
  
        <!-- Decorative Elements -->
        <div :class="decorativeCircleClasses"></div>
        
        <div v-if="item.hasPattern" class="absolute inset-0 opacity-10">
          <div class="absolute inset-0" 
               :style="{
                 backgroundImage: 'radial-gradient(circle, rgba(255,255,255,0.8) 1px, transparent 1px)', 
                 backgroundSize: '16px 16px'
               }" 
          />
        </div>
        
        <!-- Hover border effect -->
        <div :class="hoverBorderClasses"></div>
      </div>
    `
  };