// DynamicCategoryBento component (main component)
const DynamicCategoryBento = {
    components: {
      'category-button': CategoryButton,
      'bento-button': BentoButton
    },
    data() {
      return {
        activeCategory: 'sales',
        hoveredItem: null,
        visibleItems: [],
        isTransitioning: false,
        animationDirection: 'none',
        
        // Category definitions
        categories: [
          { id: 'sales', name: 'Sales', icon: '💼' },
          { id: 'inventory', name: 'Inventory', icon: '📦' },
          { id: 'setup', name: 'Setup', icon: '⚙️' },
          { id: 'customer', name: 'Customer', icon: '👥' }
        ],
        
        // All bento items organized by category
        bentoItemsByCategory: {
          sales: [
            {
              id: 's1',
              title: "Sales Order",
              icon: "📝",
              content: "Create and manage sales orders",
              action: "Create",
              bgColor: "#FF3A29",
              textLight: true,
              area: "a1",
              animDelay: 0
            },
            {
              id: 's2',
              title: "Event Order",
              icon: "🎪",
              content: "Manage event-based sales",
              action: "View",
              bgColor: "#4285F4",
              textLight: true,
              area: "a2",
              animDelay: 60
            },
            {
              id: 's3',
              title: "Delivery Order",
              icon: "🚚",
              content: "Track shipping and delivery",
              action: "Create",
              bgColor: "#E8EAED",
              textLight: false,
              area: "a3",
              animDelay: 90
            },
            {
              id: 's4',
              title: "Invoice",
              icon: "📄",
              content: "Generate and manage invoices",
              action: "Generate",
              bgColor: "#FBBC04",
              textLight: false,
              area: "b1",
              animDelay: 30
            },
            {
              id: 's5',
              title: "Receipt",
              icon: "🧾",
              content: "Create and print receipts",
              action: "Print",
              bgColor: "#34A853",
              textLight: true,
              area: "b2",
              animDelay: 120
            },
            {
              id: 's6',
              title: "Credit Sales",
              icon: "💳",
              content: "Manage customer credit",
              action: "View",
              bgColor: "#4A4A4A",
              textLight: true,
              area: "c1",
              animDelay: 75
            },
            {
              id: 's7',
              title: "Return Order",
              icon: "↩️",
              content: "Process returned merchandise",
              action: "Process",
              bgColor: "#9C27B0",
              textLight: true,
              area: "c2",
              animDelay: 105
            },
            {
              id: 's8',
              title: "Quotation",
              icon: "📊",
              content: "Create and track quotations",
              action: "Create",
              bgColor: "#F9E3BE",
              textLight: false,
              area: "c3",
              animDelay: 45
            }
          ],
          
          inventory: [
            {
              id: 'i1',
              title: "Stock In",
              icon: "📥",
              content: "Record incoming stock",
              action: "Add Stock",
              bgColor: "#4285F4",
              textLight: true,
              area: "a1",
              animDelay: 0
            },
            {
              id: 'i2',
              title: "Stock Out",
              icon: "📤",
              content: "Record outgoing stock",
              action: "Release",
              bgColor: "#FF3A29",
              textLight: true,
              area: "a2",
              animDelay: 60
            },
            {
              id: 'i3',
              title: "Stock Transfer",
              icon: "↔️",
              content: "Move items between locations",
              action: "Transfer",
              bgColor: "#E8EAED",
              textLight: false,
              area: "a3",
              animDelay: 90
            },
            {
              id: 'i4',
              title: "Stock Adjustment",
              icon: "⚖️",
              content: "Adjust inventory quantities",
              action: "Adjust",
              bgColor: "#FBBC04",
              textLight: false,
              area: "b1",
              animDelay: 30
            },
            {
              id: 'i5',
              title: "Stock Count",
              icon: "🔢",
              content: "Physical inventory counting",
              action: "Start Count",
              bgColor: "#34A853",
              textLight: true,
              area: "b2",
              animDelay: 120
            },
            {
              id: 'i6',
              title: "Stock Expiry Management",
              icon: "📅",
              content: "Track and manage expiring items",
              action: "View",
              bgColor: "#4A4A4A",
              textLight: true,
              area: "c1",
              animDelay: 75
            },
            {
              id: 'i7',
              title: "Stock Valuation",
              icon: "💰",
              content: "Assess inventory value",
              action: "Calculate",
              bgColor: "#9C27B0",
              textLight: true,
              area: "c2",
              animDelay: 105
            },
            {
              id: 'i8',
              title: "Stock Movement",
              icon: "📊",
              content: "Track item movement history",
              action: "View",
              bgColor: "#F9E3BE",
              textLight: false,
              area: "c3",
              animDelay: 45
            },
            {
              id: 'i9',
              title: "Stock Return",
              icon: "↩️",
              content: "Process returns to vendors",
              action: "Return",
              bgColor: "#3F51B5",
              textLight: true,
              area: "d1",
              animDelay: 15
            },
            {
              id: 'i10',
              title: "Stock Reconciliation",
              icon: "🔍",
              content: "Reconcile system vs physical stock",
              action: "Reconcile",
              bgColor: "#607D8B",
              textLight: true,
              area: "d2",
              animDelay: 85
            }
          ],
          
          setup: [
            {
              id: 'su1',
              title: "Customer Profile",
              icon: "👤",
              content: "Manage customer information",
              action: "Manage",
              bgColor: "#4285F4",
              textLight: true,
              area: "a1",
              animDelay: 0
            },
            {
              id: 'su2',
              title: "Stock Profile",
              icon: "📦",
              content: "Set up product information",
              action: "Configure",
              bgColor: "#FF3A29",
              textLight: true,
              area: "a2",
              animDelay: 60
            },
            {
              id: 'su3',
              title: "Warehouse Profile",
              icon: "🏭",
              content: "Configure storage locations",
              action: "Setup",
              bgColor: "#E8EAED",
              textLight: false,
              area: "a3",
              animDelay: 90
            },
            {
              id: 'su4',
              title: "Vendor Profile",
              icon: "🤝",
              content: "Manage supplier information",
              action: "Add",
              bgColor: "#FBBC04",
              textLight: false,
              area: "b1",
              animDelay: 30
            },
            {
              id: 'su5',
              title: "Price Group",
              icon: "💲",
              content: "Configure pricing structures",
              action: "Edit",
              bgColor: "#34A853",
              textLight: true,
              area: "b2",
              animDelay: 120
            },
            {
              id: 'su6',
              title: "Customer Group",
              icon: "👥",
              content: "Organize customers by segment",
              action: "Manage",
              bgColor: "#4A4A4A",
              textLight: true,
              area: "c1",
              animDelay: 75
            },
            {
              id: 'su7',
              title: "UOM",
              icon: "📏",
              content: "Units of measurement setup",
              action: "Configure",
              bgColor: "#9C27B0",
              textLight: true,
              area: "c2",
              animDelay: 105
            },
            {
              id: 'su8',
              title: "Discount & Promotions",
              icon: "🏷️",
              content: "Set up special offers",
              action: "Create",
              bgColor: "#F9E3BE",
              textLight: false,
              area: "c3",
              animDelay: 45
            },
            {
              id: 'su9',
              title: "Tax Setup",
              icon: "📜",
              content: "Configure tax rates and rules",
              action: "Setup",
              bgColor: "#3F51B5",
              textLight: true,
              area: "d1",
              animDelay: 15
            }
          ],
          
          customer: [
            {
              id: 'c1',
              title: "Registration",
              icon: "📝",
              content: "New customer sign-up",
              action: "Register",
              bgColor: "#4285F4",
              textLight: true,
              area: "a1",
              animDelay: 0
            },
            {
              id: 'c2',
              title: "Order History",
              icon: "📚",
              content: "View customer purchase history",
              action: "View",
              bgColor: "#FF3A29",
              textLight: true,
              area: "a2",
              animDelay: 60
            },
            {
              id: 'c3',
              title: "Payment History",
              icon: "💰",
              content: "Track customer payments",
              action: "View",
              bgColor: "#E8EAED",
              textLight: false,
              area: "a3",
              animDelay: 90
            },
            {
              id: 'c4',
              title: "Credit Limit",
              icon: "⚠️",
              content: "Manage customer credit limits",
              action: "Adjust",
              bgColor: "#FBBC04",
              textLight: false,
              area: "b1",
              animDelay: 30
            },
            {
              id: 'c5',
              title: "Loyalty Program",
              icon: "🏆",
              content: "Reward program management",
              action: "Manage",
              bgColor: "#34A853",
              textLight: true,
              area: "b2",
              animDelay: 120
            },
            {
              id: 'c6',
              title: "Complaints & Feedback",
              icon: "📣",
              content: "Handle customer issues",
              action: "Respond",
              bgColor: "#4A4A4A",
              textLight: true,
              area: "c1",
              animDelay: 75
            },
            {
              id: 'c7',
              title: "Price Tiering",
              icon: "📊",
              content: "Customer-specific pricing",
              action: "Configure",
              bgColor: "#9C27B0",
              textLight: true,
              area: "c2",
              animDelay: 105
            },
            {
              id: 'c8',
              title: "Statement",
              icon: "📄",
              content: "Generate customer statements",
              action: "Generate",
              bgColor: "#F9E3BE",
              textLight: false,
              area: "c3",
              animDelay: 45
            }
          ]
        }
      };
    },
    computed: {
      // Get the current items for the active category
      currentItems() {
        return this.bentoItemsByCategory[this.activeCategory] || [];
      }
    },
    methods: {
      // Handle item click
      handleItemClick(id) {
        console.log(`Clicked item ${id}`);
        // Add your click handler logic here
      },
      
      // Set the hovered item
      setHoveredItem(id) {
        this.hoveredItem = id;
      },
      
      // Handle category change
      handleCategoryChange(categoryId) {
        // Prevent changing to the same category or during transition
        if (categoryId === this.activeCategory || this.isTransitioning) return;
        
        this.isTransitioning = true;
        this.animationDirection = 'bubble';
        
        // Immediately hide previous items without animation
        this.visibleItems = [];
        
        // Immediately change to new category without waiting for fade out
        this.activeCategory = categoryId;
        
        // Add a small delay before showing new items to ensure DOM has updated
        setTimeout(() => {
          this.visibleItems = this.bentoItemsByCategory[categoryId].map(item => item.id);
          this.isTransitioning = false;
        }, 50);
      }
    },
    mounted() {
      // Initialize visible items on first render
      this.animationDirection = 'bubble';
      setTimeout(() => {
        this.visibleItems = this.bentoItemsByCategory[this.activeCategory].map(item => item.id);
      }, 100);
    },
    template: `
      <div class="w-full max-w-6xl mx-auto p-4">
        <!-- Dynamic Island Style Category Buttons -->
        <div class="flex justify-center gap-3 mb-6 overflow-x-auto py-2 px-4">
          <category-button
            v-for="category in categories"
            :key="category.id"
            :category="category"
            :activeCategory="activeCategory"
            @category-click="handleCategoryChange"
          ></category-button>
        </div>
        
        <!-- Bento Grid -->
        <div class="relative overflow-hidden">
          <div class="grid gap-4 auto-rows-auto grid-areas-bento min-h-[600px] transition-[height] duration-200 ease-in-out">
            <bento-button
              v-for="item in currentItems"
              :key="item.id"
              :item="item"
              :isHovered="hoveredItem === item.id"
              :isVisible="visibleItems.includes(item.id)"
              :animDirection="animationDirection"
              @item-hover="setHoveredItem"
              @item-click="handleItemClick"
            ></bento-button>
          </div>
        </div>
      </div>
    `
  };