export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C'
      },

      // Dynamic UI States
      scrollPosition: 0,
      currentTime: new Date(),
      timeInterval: null,
      isLoading: false,
      weatherData: { temp: 28, condition: 'Partly Cloudy', humidity: 72 },

      // Animation States
      cardHover: null,
      showNotification: false,
      notificationMessage: '',

      // Premium Content
      userData: {
        name: "",
        role: "platinum",
        membershipId: "",
        avatar: null, // Will use v-avatar with initials if image not available
        balance: 0,
        rewards: 0
      },

      // QR Code data
      memberQRCode: null,
      largeQRCode: null,
      showQRModal: false,

      // Activity Data
      recentActivities: [],
      highlights: [],
      clubEvents: [],
      quickActions: []
    }
  },
  computed: {
    // Dynamic greeting based on time
    greeting() {
      const hour = this.currentTime.getHours()
      return hour < 12 ? 'Good Morning' :
        hour < 18 ? 'Good Afternoon' : 'Good Evening'
    },

    // Parallax effects for visual depth
    headerStyle() {
      return {
        transform: `translateY(${this.scrollPosition * 0.3}px)`
      }
    },

    cloudStyle() {
      return {
        transform: `translateX(${this.scrollPosition * -0.2}px)`,
        opacity: Math.max(0, 1 - this.scrollPosition * 0.003)
      }
    },

    // Membership badge styling
    membershipBadge() {
      return {
        platinum: { color: '#E5E4E2', textColor: '#1A3C28', text: 'Platinum Member', icon: 'mdi-crown' },
        gold: { color: '#FFD700', textColor: '#1A3C28', text: 'Gold Member', icon: 'mdi-medal' },
        silver: { color: '#C0C0C0', textColor: '#1A3C28', text: 'Silver Member', icon: 'mdi-medal-outline' },
        junior: { color: '#A3BA95', textColor: '#1A3C28', text: 'Junior Account', icon: 'mdi-account-child' }
      }[this.userData.role] || { color: '#A3BA95', textColor: '#1A3C28', text: 'Member', icon: 'mdi-account' }
    },

    // Format currency consistently
    formattedBalance() {
      return this.formatCredit(this.userData.balance)
    },

    // Get user initials for avatar fallback
    userInitials() {
      return this.userData.name.split(' ').map(name => name[0]).join('')
    },

    // Current date formatted nicely
    formattedDate() {
      const options = { weekday: 'long', month: 'long', day: 'numeric' }
      return this.currentTime.toLocaleDateString('en-US', options)
    }
  },
  methods: {
    handleScroll() {
      this.scrollPosition = window.scrollY || window.pageYOffset
    },

    formatCurrency(amount) {
      return "RM " + parseFloat(amount).toFixed(2)
    },

    formatCredit(amount) {
      return `<i class="mdi mdi-circle-multiple-outline"></i> ${parseFloat(amount).toFixed(2)}`
    },

    // Interactive methods
    handleCardHover(index) {
      this.cardHover = index
    },

    resetCardHover() {
      this.cardHover = null
    },

    // Refresh data from API
    async refreshData() {
      /*await this.fetchUserDataAndBalance();

      // Update current time and weather
      this.currentTime = new Date();
      this.weatherData = {
        temp: 27 + Math.floor(Math.random() * 5),
        condition: ['Sunny', 'Partly Cloudy', 'Light Rain', 'Humid'][Math.floor(Math.random() * 4)],
        humidity: 65 + Math.floor(Math.random() * 20)
      };

      this.showNotification = true;
      this.notificationMessage = 'Reloading dashboard...';
      setTimeout(() => { 
        window.location.reload();
      }, 1000);*/

      window.location.reload();
    },

    // Fetch home data including wallet balance, events, highlights, and activities
    async fetchUserDataAndBalance() {
      this.isLoading = true;
      try {
        // Call the wallet API to get home data
        const response = await this.$root.callApi('wallet', 'get_home', {
          user_id: sessionStorage.getItem('id')
        });

        if (response.session.user_data) {
          this.userData.name = response.session.user_data.fullname || this.userData.name;
          this.userData.membershipId = response.session.user_data.member_no || this.userData.membershipId;
          this.userData.role = response.session.user_data.member_type || this.userData.role;
          
          // No need to generate small QR code anymore since we're using an icon
        }

        if (response.data && response.data.status === "YES") {
          // Update wallet balance
          this.userData.balance = parseFloat(response.data.wallet.balance);
          this.userData.rewards = parseFloat(response.data.wallet.rewards);

          // Update club events
          this.clubEvents = response.data.events || [];

          // Update highlights with action methods
          this.highlights = (response.data.highlights || []).map(highlight => {
            return {
              ...highlight,
              action: () => {
                this.showNotification = true;
                this.notificationMessage = 'Opening details...';

                // All highlights are now events, so always load the event detail screen
                let targetScreen = 'modules/events/801-event-detail.js';
                
                setTimeout(() => this.$root.loadComponent(targetScreen), 800);
              }
            };
          });

          // Update recent activities
          this.recentActivities = response.data.activities || [];

          // Update quick actions
          this.quickActions = response.data.quickActions || [
            { icon: 'mdi-bed', label: 'Check-In', screen: 'screen_010_check_in.js', color: '#DDA333' },
            { icon: 'mdi-food-fork-drink', label: 'Order F&B', screen: 'screen_005_fnb.js', color: '#DDA333' },
            { icon: 'mdi-calendar-check', label: 'Book Facility', screen: 'screen_006_booking.js', color: '#DDA333' },
            { icon: 'mdi-dumbbell', label: 'Use Gym', screen: 'screen_011_gym.js', color: '#DDA333' },
            { icon: 'mdi-calendar-month', label: 'Find Events', screen: 'screen_012_events.js', color: '#DDA333' },
            { icon: 'mdi-parking', label: 'Parking', screen: 'screen_013_parking.js', color: '#DDA333' },
            { icon: 'mdi-qrcode-scan', label: 'QR Transfer', screen: 'screen_011_qr_transfer.js', color: '#DDA333' },
            { icon: 'mdi-bell', label: 'Notifications', screen: 'screen_014_notification.js', color: '#889FAC' }
          ];
        } else {
          console.error('Failed to fetch home data:', response.data.message);
        }
      } catch (error) {
        console.error('Error fetching home data:', error);
      } finally {
        this.isLoading = false;
      }
    },

    // Generate QR code for member
    async generateMemberQRCode() {
      if (this.userData.membershipId) {
        try {
          // Wait for QRCode library to be available
          const maxRetries = 20;
          let retries = 0;
          
          while (typeof window.QRCode === 'undefined' && retries < maxRetries) {
            await new Promise(resolve => setTimeout(resolve, 200));
            retries++;
          }
          
          if (typeof window.QRCode === 'undefined') {
            console.error('QRCode library not loaded, using fallback');
            this.generateFallbackQRCode();
            return;
          }
          
          // Create QR code data with just member number
          const qrData = this.userData.membershipId;
          
          // Create a temporary div to generate QR code
          const tempDiv = document.createElement('div');
          tempDiv.style.width = '120px';
          tempDiv.style.height = '120px';
          document.body.appendChild(tempDiv);
          
          // Generate QR code using davidshimjs library
          const qrcode = new window.QRCode(tempDiv, {
            text: qrData,
            width: 120,
            height: 120,
            colorDark: '#1A3C28',  // Dark green to match theme
            colorLight: '#FFFFFF',
            correctLevel: window.QRCode.CorrectLevel.M
          });
          
          // Wait for QR code to be generated and extract canvas
          setTimeout(() => {
            const canvas = tempDiv.querySelector('canvas');
            if (canvas) {
              this.memberQRCode = canvas.toDataURL();
            } else {
              console.log('Canvas not found, using fallback');
              this.generateFallbackQRCode();
            }
            // Remove temporary div
            document.body.removeChild(tempDiv);
          }, 100);
          
        } catch (error) {
          console.error('Error generating QR code:', error);
          this.generateFallbackQRCode();
        }
      }
    },

    // Fallback QR code generation using canvas
    generateFallbackQRCode() {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      canvas.width = 120;
      canvas.height = 120;
      
      // Fill background
      ctx.fillStyle = '#FFFFFF';
      ctx.fillRect(0, 0, 120, 120);
      
      // Draw simple pattern representing QR code
      ctx.fillStyle = '#1A3C28';
      const blockSize = 6;
      const memberNo = this.userData.membershipId;
      
      // Create a simple hash-based pattern from member number
      for (let i = 0; i < memberNo.length; i++) {
        const charCode = memberNo.charCodeAt(i);
        const x = (charCode * 7 + i * 3) % 16;
        const y = (charCode * 11 + i * 5) % 16;
        ctx.fillRect(x * blockSize + 12, y * blockSize + 12, blockSize - 1, blockSize - 1);
      }
      
      // Add corner markers
      ctx.fillRect(12, 12, blockSize * 3, blockSize * 3);
      ctx.fillRect(84, 12, blockSize * 3, blockSize * 3);
      ctx.fillRect(12, 84, blockSize * 3, blockSize * 3);
      
      // Add member number text
      ctx.fillStyle = '#1A3C28';
      ctx.font = '8px monospace';
      ctx.textAlign = 'center';
      ctx.fillText(this.userData.membershipId.slice(-8), 60, 110);
      
      this.memberQRCode = canvas.toDataURL();
    },

    // Generate large QR code for popup
    async generateLargeQRCode() {
      if (this.userData.membershipId) {
        try {
          // Wait for QRCode library to be available
          const maxRetries = 20;
          let retries = 0;
          
          while (typeof window.QRCode === 'undefined' && retries < maxRetries) {
            await new Promise(resolve => setTimeout(resolve, 200));
            retries++;
          }
          
          if (typeof window.QRCode === 'undefined') {
            console.error('QRCode library not loaded for large QR');
            return;
          }
          
          // Create a temporary div to generate large QR code
          const tempDiv = document.createElement('div');
          tempDiv.style.width = '280px';
          tempDiv.style.height = '280px';
          document.body.appendChild(tempDiv);
          
          // Generate large QR code using davidshimjs library
          const qrcode = new window.QRCode(tempDiv, {
            text: this.userData.membershipId,
            width: 280,
            height: 280,
            colorDark: '#1A3C28',  // Dark green to match theme
            colorLight: '#FFFFFF',
            correctLevel: window.QRCode.CorrectLevel.M
          });
          
          // Wait for QR code to be generated and extract canvas
          setTimeout(() => {
            const canvas = tempDiv.querySelector('canvas');
            if (canvas) {
              this.largeQRCode = canvas.toDataURL();
            }
            // Remove temporary div
            document.body.removeChild(tempDiv);
          }, 100);
          
        } catch (error) {
          console.error('Error generating large QR code:', error);
        }
      }
    },

    // Show QR code popup
    async showQRCodePopup() {
      this.showQRModal = true;
      if (!this.largeQRCode) {
        await this.generateLargeQRCode();
      }
    },

    // Hide QR code popup
    hideQRCodePopup() {
      this.showQRModal = false;
      // Refresh data when QR popup is closed
      this.fetchUserDataAndBalance();
    },

    // Clean up attached styles and reload home styles
    cleanupAndReloadStyles() {
      // Remove all dynamically added style elements (from other pages)
      const existingStyles = document.querySelectorAll('style[data-module-style]');
      existingStyles.forEach(style => style.remove());
      
      // Also remove any style elements that don't have specific attributes (catch-all)
      const allStyles = document.querySelectorAll('head > style');
      allStyles.forEach(style => {
        // Keep only essential styles (don't remove Vuetify or main app styles)
        if (!style.id && !style.getAttribute('data-vite-dev-id') && !style.textContent.includes('v-application')) {
          style.remove();
        }
      });
      
      // Force reload home page styles
      this.loadHomeStyles();
    },
    
    // Load home page styles
    loadHomeStyles() {
      // Remove existing home styles if any
      const existingHomeStyle = document.querySelector('style[data-home-style]');
      if (existingHomeStyle) {
        existingHomeStyle.remove();
      }
      
      // Add home page styles
      const styleElement = document.createElement('style');
      styleElement.setAttribute('data-home-style', 'true');
      styleElement.setAttribute('data-module-style', 'home');
      styleElement.textContent = this.getHomeStyles();
      document.head.appendChild(styleElement);
    },
    
    // Get home page CSS styles
    getHomeStyles() {
      return `
        /* Base Styles */
        .tropical-dashboard {
          --primary: #1A3C28;
          --secondary: #DDA333;
          --accent: #A3BA95;
          --cream: #EADDCB;
          --muted: #889FAC;
          --text-dark: #1E1E1E;
          --text-light: #FFFFFF;
          
          font-family: 'Arial', sans-serif;
          background-color: var(--cream);
          min-height: 100vh;
          position: relative;
          overflow-x: hidden;
        }
        
        /* Loading Overlay */
        .loading-overlay {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: rgba(26, 60, 40, 0.8);
          display: flex;
          justify-content: center;
          align-items: center;
          z-index: 1000;
        }
        
        .loading-spinner {
          display: flex;
          flex-direction: column;
          align-items: center;
        }
        
        .loading-text {
          margin-top: 16px;
          color: white;
          font-weight: 500;
        }
        
        /* Notification Toast */
        .notification-toast {
          position: fixed;
          bottom: 80px;
          left: 50%;
          transform: translateX(-50%);
          background-color: var(--secondary);
          color: var(--primary);
          padding: 12px 24px;
          border-radius: 20px;
          font-weight: 600;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          z-index: 900;
        }
        
        .fade-enter-active,
        .fade-leave-active {
          transition: opacity 0.3s, transform 0.3s;
        }
        
        .fade-enter-from,
        .fade-leave-to {
          opacity: 0;
          transform: translateX(-50%) translateY(20px);
        }
        
        /* Background Decorations */
        .background-decoration {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: -1;
          overflow: hidden;
        }
        
        .bg-leaf {
          position: absolute;
          border-radius: 50% 50% 0 50%;
          opacity: 0.07;
          background-color: var(--accent);
          transform: rotate(45deg);
        }
        
        .bg-leaf-1 {
          width: 400px;
          height: 400px;
          top: -200px;
          left: -100px;
        }
        
        .bg-leaf-2 {
          width: 300px;
          height: 300px;
          bottom: 10%;
          right: -150px;
          transform: rotate(130deg);
        }
        
        .bg-leaf-3 {
          width: 200px;
          height: 200px;
          top: 30%;
          left: -100px;
          transform: rotate(80deg);
        }
        
        .forest-silhouette {
          position: absolute;
          bottom: 0;
          left: 0;
          width: 100%;
          height: 100px;
          background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1440 320"><path fill="%231A3C28" fill-opacity="0.05" d="M0,224L48,213.3C96,203,192,181,288,181.3C384,181,480,203,576,197.3C672,192,768,160,864,165.3C960,171,1056,213,1152,208C1248,203,1344,149,1392,122.7L1440,96L1440,320L1392,320C1344,320,1248,320,1152,320C1056,320,960,320,864,320C768,320,672,320,576,320C480,320,384,320,288,320C192,320,96,320,48,320L0,320Z"></path></svg>');
          background-size: cover;
          opacity: 1;
        }
        
        .gold-shine {
          position: absolute;
          top: -50%;
          right: -50%;
          width: 150%;
          height: 150%;
          background: radial-gradient(circle, rgba(221, 163, 51, 0.15) 0%, transparent 70%);
          transition: transform 0.5s ease-out;
        }
        
        /* Hero Section */
        .hero-section {
          background-color: var(--primary);
          height: 38vh;
          min-height: 280px;
          border-radius: 0 0 30px 30px;
          box-shadow: 0 8px 20px rgba(0,0,0,0.15);
          padding: 20px;
          position: relative;
          overflow: hidden;
          transition: transform 0.5s ease-out;
        }
        
        .leaf-decoration {
          position: absolute;
          width: 150px;
          height: 150px;
          background-color: rgba(255,255,255,0.05);
          border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
          z-index: 0;
        }
        
        .leaf-decoration.left {
          top: -30px;
          left: -50px;
          transform: rotate(35deg);
        }
        
        .leaf-decoration.right {
          bottom: -60px;
          right: -30px;
          transform: rotate(-15deg);
        }
        
        /* App Bar */
        .premium-app-bar {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 10px 0;
          position: relative;
          z-index: 2;
        }
        
        .app-bar-date {
          font-size: 14px;
          color: rgba(255,255,255,0.7);
          font-weight: 500;
        }
        
        .membership-chip {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 4px 12px;
          border-radius: 20px;
          font-size: 12px;
          font-weight: 600;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        /* Hero Content */
        .hero-content {
          position: relative;
          z-index: 1;
          margin-top: 20px;
        }
        
        .greeting {
          font-size: 20px;
          font-weight: 400;
          color: rgba(255,255,255,0.8);
          margin-bottom: 4px;
        }
        
        .user-name {
          font-size: 36px;
          font-weight: 700;
          color: white;
          margin-bottom: 16px;
          background: linear-gradient(90deg, #FFFFFF, #DDA333);
          -webkit-background-clip: text;
          background-clip: text;
          -webkit-text-fill-color: transparent;
          display: inline-block;
        }
        
        .hero-extras {
          display: flex;
          justify-content: space-between;
          align-items: center;
        }
        
        .membership-id {
          font-family: 'Courier New', monospace;
          color: rgba(255,255,255,0.6);
          font-size: 14px;
        }
        
        .weather-widget {
          display: flex;
          align-items: center;
          gap: 8px;
          color: white;
          font-size: 14px;
          background-color: rgba(255,255,255,0.1);
          padding: 4px 12px;
          border-radius: 16px;
        }
        
        /* Content Area */
        .content-area {
        padding: 0 16px 80px 16px; /* Bottom padding for navigation */
          margin-top: -50px;
          position: relative;
          z-index: 2;
        }
        
        /* Wallet Card */
        .premium-card {
          background-color: var(--primary);
          border-radius: 20px;
          overflow: hidden;
          position: relative;
          margin-bottom: 24px;
          box-shadow: 0 8px 20px rgba(0,0,0,0.15);
        }
        
        .card-decoration {
          position: absolute;
          top: -20px;
          right: -20px;
          width: 150px;
          height: 150px;
          background-color: var(--secondary);
          opacity: 0.1;
          border-radius: 50%;
        }
        
        .wallet-content {
          padding: 24px;
          color: white;
          position: relative;
          z-index: 1;
        }
        
        .wallet-main-row {
          display: flex;
          justify-content: space-between;
          align-items: flex-start;
          margin-bottom: 16px;
        }
        
        .wallet-info {
          flex: 1;
        }
        
        .member-qr-section {
          flex-shrink: 0;
          margin-left: 16px;
        }
        
        .qr-code-container {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          background-color: rgba(255,255,255,0.1);
          border-radius: 12px;
          width: 72px;
          height: 72px;
          padding: 0;
          backdrop-filter: blur(10px);
          cursor: pointer;
          transition: all 0.3s ease;
        }
        
        .qr-code-container:hover {
          background-color: rgba(255,255,255,0.2);
          transform: scale(1.05);
        }
        
        .member-qr-code {
          width: 80px;
          height: 80px;
          border-radius: 8px;
          border: 2px solid rgba(255,255,255,0.2);
          background-color: white;
          padding: 4px;
        }
        
        .qr-code-icon {
          background: linear-gradient(90deg, #FFFFFF, #DDA333);
          -webkit-background-clip: text;
          background-clip: text;
          -webkit-text-fill-color: transparent;
          display: inline-block;
          transition: all 0.3s ease;
        }
        
        .qr-code-container:hover .qr-code-icon {
          background: linear-gradient(90deg, #DDA333, #FFFFFF);
          -webkit-background-clip: text;
          background-clip: text;
          -webkit-text-fill-color: transparent;
        }
        
        .qr-label {
          font-size: 11px;
          color: rgba(255,255,255,0.8);
          margin-top: 6px;
          font-weight: 500;
          text-align: center;
        }
        
        /* QR Code Popup Modal */
        .qr-modal-overlay {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: rgba(0, 0, 0, 0.8);
          display: flex;
          justify-content: center;
          align-items: center;
          z-index: 9999;
          backdrop-filter: blur(5px);
        }
        
        .qr-modal-content {
          background-color: white;
          border-radius: 20px;
          padding: 32px;
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
          display: flex;
          flex-direction: column;
          align-items: center;
          max-width: 90vw;
          max-height: 90vh;
          position: relative;
        }
        
        .qr-modal-header {
          margin-bottom: 24px;
          text-align: center;
        }
        
        .qr-modal-title {
          font-size: 24px;
          font-weight: 700;
          color: var(--primary);
          margin-bottom: 8px;
        }
        
        .qr-modal-subtitle {
          font-size: 16px;
          color: var(--muted);
          font-weight: 500;
        }
        
        .large-qr-code {
          width: 280px;
          height: 280px;
          border-radius: 16px;
          box-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
          margin-bottom: 24px;
        }
        
        .qr-modal-close {
          position: absolute;
          top: 16px;
          right: 16px;
          background: none;
          border: none;
          font-size: 24px;
          cursor: pointer;
          color: var(--muted);
          width: 32px;
          height: 32px;
          display: flex;
          align-items: center;
          justify-content: center;
          border-radius: 50%;
          transition: all 0.3s ease;
        }
        
        .qr-modal-close:hover {
          background-color: var(--accent);
          color: white;
        }
        
        .qr-modal-instructions {
          text-align: center;
          color: var(--muted);
          font-size: 14px;
          max-width: 300px;
          line-height: 1.4;
        }
        
        .wallet-header-one {
          display: flex;
          align-items: center;
          gap: 12px;
        }
        
        .wallet-title {
          font-weight: 600;
          font-size: 16px;
        }
        
        .wallet-balance {
          font-size: 36px;
          font-weight: 700;
          margin: 12px 0;
          background: linear-gradient(90deg, #FFFFFF, #DDA333);
          -webkit-background-clip: text;
          background-clip: text;
          -webkit-text-fill-color: transparent;
          display: inline-block;
        }
        
        .wallet-actions {
          display: flex;
          gap: 12px;
          margin-bottom: 16px;
        }
        
        .wallet-action-btn {
          text-transform: none !important;
          font-weight: 600 !important;
        }
        
        .rewards-indicator {
          background-color: rgba(255,255,255,0.1);
          border-radius: 12px;
          padding: 12px;
          display: flex;
          justify-content: space-between;
          align-items: center;
        }
        
        .rewards-label {
          font-size: 14px;
          opacity: 0.8;
        }
        
        .rewards-value {
          font-weight: 700;
          color: var(--secondary);
        }
        
        /* Quick Actions */
        .quick-actions-grid {
          display: grid;
          grid-template-columns: repeat(4, 1fr);
          gap: 16px;
          margin-bottom: 32px;
        }
        
        .quick-action-btn {
          display: flex;
          flex-direction: column;
          align-items: center;
          cursor: pointer;
          transition: transform 0.3s;
        }
        
        .quick-action-btn:hover {
          transform: translateY(-5px);
        }
        
        .action-icon-wrapper {
          width: 56px;
          height: 56px;
          border-radius: 16px;
          background-color: white;
          display: flex;
          align-items: center;
          justify-content: center;
          margin-bottom: 8px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.1);
          position: relative;
          overflow: hidden;
        }
        
        .action-icon-wrapper::after {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: var(--action-color);
          opacity: 0;
          transition: opacity 0.3s;
        }
        
        .quick-action-btn:hover .action-icon-wrapper::after {
          opacity: 0.1;
        }
        
        .action-label {
          font-size: 12px;
          font-weight: 600;
          color: var(--text-dark);
        }
        
        /* Section Containers */
        .section-container {
          margin-bottom: 32px;
        }
        
        .section-header-fnb {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 16px;
        }
        
        .section-title {
          font-size: 20px;
          font-weight: 700;
          color: var(--primary);
          margin: 0;
        }
        
        .section-subtitle {
          font-size: 14px;
          color: var(--muted);
          margin-top: 4px;
        }
        
        /* Highlights Grid */
        .highlights-grid {
          display: grid;
          grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
          gap: 16px;
        }
        
        .highlight-card {
          background-color: white;
          border-radius: 16px;
          padding: 16px;
          display: flex;
          align-items: center;
          gap: 16px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          cursor: pointer;
          transition: all 0.3s;
          position: relative;
          overflow: hidden;
        }
        
        .highlight-card::after {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          width: 5px;
          height: 100%;
          background-color: var(--accent);
          opacity: 0;
          transition: opacity 0.3s;
        }
        
        .highlight-card:hover::after {
          opacity: 1;
        }
        
        .card-hovered {
          transform: translateY(-5px) !important;
          box-shadow: 0 8px 20px rgba(0,0,0,0.1) !important;
        }
        
        .highlight-icon {
          width: 44px;
          height: 44px;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          flex-shrink: 0;
        }
        
        .highlight-content {
          flex: 1;
        }
        
        .highlight-title {
          font-weight: 600;
          font-size: 16px;
          color: var(--text-dark);
          margin-bottom: 4px;
        }
        
        .highlight-date {
          font-size: 13px;
          color: var(--muted);
          margin-bottom: 8px;
        }
        
        .highlight-description {
          font-size: 13px;
          color: var(--text-dark);
          opacity: 0.8;
        }
        
        .card-arrow {
          color: var(--muted);
          transition: transform 0.3s;
        }
        
        .highlight-card:hover .card-arrow {
          transform: translateX(5px);
          color: var(--secondary);
        }
        
        /* Activity List */
        .activity-list {
          background-color: white;
          border-radius: 16px;
          overflow: hidden;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
        }
        
        .activity-item {
          display: flex;
          align-items: center;
          padding: 16px;
          gap: 16px;
          border-bottom: 1px solid rgba(0,0,0,0.05);
          transition: all 0.3s;
          cursor: pointer;
        }
        
        .activity-item:last-child {
          border-bottom: none;
        }
        
        .activity-icon {
          width: 40px;
          height: 40px;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          flex-shrink: 0;
        }
        
        .activity-content {
          flex: 1;
        }
        
        .activity-title {
          font-weight: 600;
          font-size: 15px;
          color: var(--text-dark);
        }
        
        .activity-date {
          font-size: 13px;
          color: var(--muted);
        }
        
        .activity-amount {
          font-weight: 600;
          color: var(--primary);
          font-size: 15px;
        }
        
        /* Events Carousel */
        .events-carousel {
          display: flex;
          gap: 16px;
          overflow-x: auto;
          padding: 8px 4px 20px 4px;
        scrollbar-width: none; /* For Firefox */
        -ms-overflow-style: none;  /* For Internet Explorer and Edge */
        }
        
        .events-carousel::-webkit-scrollbar {
        display: none; /* For Chrome, Safari and Opera */
        }
        
        .event-card {
          flex: 0 0 280px;
          background-color: white;
          border-radius: 16px;
          overflow: hidden;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          cursor: pointer;
          transition: all 0.3s;
        }
        
        .event-image {
          height: 140px;
          background-color: var(--primary);
          position: relative;
          display: flex;
          align-items: flex-end;
          padding: 16px;
          background-size: cover;
          background-position: center;
        }
        
        .event-image-overlay {
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background-image: linear-gradient(to bottom, transparent 50%, rgba(0,0,0,0.7) 100%);
          z-index: 1;
        }
        
        .event-category-social { background-color: #7D9D6B; }
        .event-category-sport { background-color: #6082B6; }
        .event-category-family { background-color: #DDA333; }
        
        .event-category-tag {
          position: absolute;
          top: 16px;
          right: 16px;
          background-color: rgba(255,255,255,0.9);
          color: var(--primary);
          font-weight: 600;
          font-size: 12px;
          padding: 4px 8px;
          border-radius: 4px;
          z-index: 2;
        }
        
        .event-content {
          padding: 16px;
          position: relative;
          z-index: 2;
        }
        
        .event-title {
          font-weight: 600;
          font-size: 16px;
          color: var(--text-dark);
          margin-bottom: 4px;
        }
        
        .event-date {
          font-size: 13px;
          color: var(--secondary);
          font-weight: 500;
          margin-bottom: 8px;
        }
        
        .event-description {
          font-size: 13px;
          color: var(--muted);
          display: -webkit-box;
          -webkit-line-clamp: 2;
          -webkit-box-orient: vertical;
          overflow: hidden;
        }
        
        /* Refresh Action */
        .refresh-action {
          display: flex;
          justify-content: center;
          margin-top: 40px;
        }
        
        .refresh-btn {
          min-width: 200px;
          box-shadow: 0 4px 10px rgba(26, 60, 40, 0.2) !important;
          font-weight: 600 !important;
          letter-spacing: 0.5px;
          text-transform: none !important;
          transition: all 0.3s !important;
        }
        
        .refresh-btn:hover {
          transform: translateY(-3px);
          box-shadow: 0 6px 15px rgba(26, 60, 40, 0.3) !important;
        }
        
        /* Responsive Adjustments */
        @media (max-width: 768px) {
          .hero-section {
            height: 32vh;
          }
          
          .user-name {
            font-size: 28px;
          }
          
          .quick-actions-grid {
            grid-template-columns: repeat(4, 1fr);
          }
          
          .highlights-grid {
            grid-template-columns: 1fr;
          }
          
          .wallet-main-row {
            flex-direction: row;
            justify-content: space-between;
            align-items: flex-start;
          }
          
          .member-qr-section {
            margin-left: 12px;
            margin-top: 0;
          }
          
          .member-qr-code {
            width: 60px;
            height: 60px;
          }
          
          .qr-code-container {
            width: 60px;
            height: 60px;
            padding: 0;
          }
          
          .qr-code-icon {
            font-size: 36px !important;
            height: 43px !important;
          }
        }
        
        /* Empty State Styling */
        .empty-state-container {
          background-color: white;
          border-radius: 16px;
          padding: 32px 16px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
        }
        
        .empty-state-message {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          text-align: center;
        }
        
        .empty-state-text {
          margin-top: 16px;
          font-size: 16px;
          color: var(--muted);
          font-weight: 500;
        }
      `;
    },

    // Navigation methods with visual feedback
    navigateTo(screen) {
      this.showNotification = true
      this.notificationMessage = 'Loading...'
      setTimeout(() => {
        this.$root.loadComponent('modules/' + screen)
      }, 600)
    }
  },
  mounted() {
    // Clean up and reload styles when home page appears
    this.cleanupAndReloadStyles();
    
    // Update time every minute
    this.timeInterval = setInterval(() => {
      this.currentTime = new Date()
    }, 60000)

    // Setup scroll listener for parallax effects
    window.addEventListener('scroll', this.handleScroll)

    // Fetch user data and balance (this refreshes data when page appears)
    this.fetchUserDataAndBalance()

    // Simulate initial data loading
    this.isLoading = true
    setTimeout(() => { this.isLoading = false }, 800)


    
  },
  beforeUnmount() {
    clearInterval(this.timeInterval)
    window.removeEventListener('scroll', this.handleScroll)
  },
  template: `
  <div class="tropical-dashboard">
    <!-- Loading overlay -->
    <div class="loading-overlay" v-if="isLoading">
      <div class="loading-spinner">
        <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
        <div class="loading-text">Loading your premium experience...</div>
      </div>
    </div>
    
    <!-- Notification toast -->
    <transition name="fade">
      <div class="notification-toast" v-if="showNotification">
        {{ notificationMessage }}
      </div>
    </transition>
    
    <!-- Animated Background Decoration -->
    <div class="background-decoration">
      <div class="bg-leaf bg-leaf-1"></div>
      <div class="bg-leaf bg-leaf-2"></div>
      <div class="bg-leaf bg-leaf-3"></div>
      <div class="forest-silhouette"></div>
      <div class="gold-shine" :style="cloudStyle"></div>
    </div>

    <!-- Main Hero Header with Parallax -->
    <div class="hero-section" :style="headerStyle">
      <div class="leaf-decoration left"></div>
      <div class="leaf-decoration right"></div>
      
      <!-- App Bar -->
      <div class="premium-app-bar">
        <div class="user-avatar">
          <v-avatar size="42" :color="theme.accent">
            <v-img v-if="userData.avatar" :src="userData.avatar"></v-img>
            <span v-else class="text-h6" :style="{color: theme.primary}">{{ userInitials }}</span>
          </v-avatar>
        </div>
        
        <div class="app-bar-date">
          {{ formattedDate }}
        </div>
        
        <div class="membership-chip" :style="{backgroundColor: membershipBadge.color}">
          <v-icon size="18" :color="membershipBadge.textColor">{{ membershipBadge.icon }}</v-icon>
          <span :style="{color: membershipBadge.textColor}">{{ membershipBadge.text }}</span>
        </div>
      </div>
      
      <!-- Hero Content -->
      <div class="hero-content">
        <h1 class="greeting">{{ greeting }},</h1>
        <h2 class="user-name">{{ userData.name }}</h2>
        
        <div class="hero-extras">
          <div class="membership-id">{{ userData.membershipId }}</div>
          
          <div class="weather-widget">
            <v-icon color="white" size="18">{{ weatherData.condition.includes('Rain') ? 'mdi-weather-rainy' : 
              weatherData.condition.includes('Cloud') ? 'mdi-weather-partly-cloudy' : 'mdi-weather-sunny' }}</v-icon>
            <span>{{ weatherData.temp }}°C</span>
          </div>
        </div>
      </div>
    </div>

    <!-- Main Content Area -->
    <div class="content-area">
      <!-- Wallet Card with Interactive Elements -->
      <div class="premium-card wallet-card">
        <div class="card-decoration"></div>
        
        <div class="wallet-content">
          <div class="wallet-main-row">
            <div class="wallet-info">
              <div class="wallet-header-one">
                <v-icon color="white" size="24">mdi-wallet</v-icon>
                <div class="wallet-title">My Credit Balance</div>
              </div>
              
              <div class="wallet-balance" v-html="formattedBalance"></div>
            </div>
            
            <!-- Member QR Code Icon -->
            <div class="member-qr-section" v-if="userData.membershipId">
              <div class="qr-code-container" @click="showQRCodePopup">
                <v-icon class="qr-code-icon" size="48">mdi-qrcode</v-icon>
              </div>
            </div>
          </div>
          
          <div class="wallet-actions">
            <v-btn 
              variant="outlined" 
              color="white"
              size="small" 
              class="wallet-action-btn"
              @click="navigateTo('screen_008_wallet_detail.js')"
            >
              View Details
            </v-btn>
            
            <v-btn 
              variant="outlined" 
              color="white"
              size="small"
              class="wallet-action-btn"
              @click="navigateTo('screen_009_wallet_topup.js')"
            >
              Top Up
            </v-btn>

            <v-btn 
              variant="outlined" 
              color="white"
              size="small"
              class="wallet-action-btn"
              @click="navigateTo('screen_010_wallet_transfer.js')"
            >
              Transfer
            </v-btn>
          </div>
          
          <!--<div class="rewards-indicator">
            <div class="rewards-label">Reward Points</div>
            <div class="rewards-value">{{ userData.rewards }}</div>
          </div>-->
        </div>
      </div>
      
      <!-- Quick Actions Grid -->
      <div class="quick-actions-grid">
        <div 
          v-for="(action, i) in quickActions" 
          :key="i"
          class="quick-action-btn"
          @click="navigateTo(action.screen)"
          :style="{ '--action-color': action.color }"
        >
          <div class="action-icon-wrapper">
            <v-icon :color="action.color">{{ action.icon }}</v-icon>
          </div>
          <div class="action-label">{{ action.label }}</div>
        </div>
      </div>


      <!-- Events Carousel -->
      <div class="section-container">
        <div class="section-header-fnb">
          <h3 class="section-title">Upcoming Club Events</h3>
          <v-btn 
            variant="outlined" 
            color="secondary"
            size="small"
            @click="navigateTo('screen_012_events.js')"
          >
            View All
          </v-btn>
        </div>
        
        <div v-if="clubEvents.length > 0" class="events-carousel">
          <div 
            v-for="(event, i) in clubEvents" 
            :key="i"
            class="event-card"
            :class="{ 'card-hovered': cardHover === 'event-'+i }"
            @mouseenter="handleCardHover('event-'+i)"
            @mouseleave="resetCardHover"
            @click="navigateTo('screen_012_events.js')"
          >
            <div class="event-image" :style="{ backgroundImage: 'url(' + event.image + ')', backgroundSize: 'cover', backgroundPosition: 'center' }">
              <div class="event-image-overlay"></div>
              <div class="event-category-tag">{{ event.category }}</div>
            </div>
            <div class="event-content">
              <div class="event-title">{{ event.title }}</div>
              <div class="event-date">{{ event.date }}</div>
              <div class="event-description">{{ event.description }}</div>
            </div>
          </div>
        </div>
        <div v-else class="empty-state-container">
          <div class="empty-state-message">
            <v-icon size="48" color="accent">mdi-calendar-blank</v-icon>
            <div class="empty-state-text">No upcoming events at this time</div>
            <v-btn 
              variant="outlined" 
              color="secondary"
              size="small"
              class="mt-3"
              @click="navigateTo('screen_012_events.js')"
            >
              Browse Events
            </v-btn>
          </div>
        </div>
      </div>
      
      <!-- Highlights Section with Interactive Cards -->
      <div class="section-container">
        <div class="section-header-fnb">
          <h3 class="section-title">Event Highlights</h3>
          <div class="section-subtitle">Featured for you</div>
        </div>
        
        <div v-if="highlights.length > 0" class="highlights-grid">
          <div 
            v-for="(item, i) in highlights" 
            :key="i"
            class="highlight-card"
            :class="{ 'card-hovered': cardHover === 'highlight-'+i }"
            @mouseenter="handleCardHover('highlight-'+i)"
            @mouseleave="resetCardHover"
            @click="item.action()"
          >
            <div class="highlight-icon" :style="{ backgroundColor: item.color }">
              <v-icon color="white">{{ item.icon }}</v-icon>
            </div>
            
            <div class="highlight-content">
              <div class="highlight-title">{{ item.title }}</div>
              <div class="highlight-date">{{ item.date }}</div>
              <div class="highlight-description">{{ item.description }}</div>
            </div>
            
            <div class="card-arrow">
              <v-icon>mdi-chevron-right</v-icon>
            </div>
          </div>
        </div>
        <div v-else class="empty-state-container">
          <div class="empty-state-message">
            <v-icon size="48" color="accent">mdi-star-outline</v-icon>
            <div class="empty-state-text">No event highlights available</div>
          </div>
        </div>
      </div>
      
      <!-- Recent Activity Section -->
      <div class="section-container">
        <div class="section-header-fnb">
          <h3 class="section-title">Recent Activity</h3>
          <v-btn 
            variant="outlined" 
            color="secondary"
            size="small"
            @click="navigateTo('screen_006_booking.js')"
          >
            View All
          </v-btn>
        </div>
        
        <div v-if="recentActivities.length > 0" class="activity-list">
          <div 
            v-for="(activity, i) in recentActivities" 
            :key="i"
            class="activity-item"
            :class="{ 'card-hovered': cardHover === 'activity-'+i }"
            @mouseenter="handleCardHover('activity-'+i)"
            @mouseleave="resetCardHover"
          >
            <div class="activity-icon" :style="{ backgroundColor: activity.color }">
              <v-icon color="white" size="20">{{ activity.icon }}</v-icon>
            </div>
            
            <div class="activity-content">
              <div class="activity-title">{{ activity.title }}</div>
              <div class="activity-date">{{ activity.date }}</div>
            </div>
            
            <div class="activity-amount">{{ formatCurrency(activity.amount) }}</div>
          </div>
        </div>
        <div v-else class="empty-state-container">
          <div class="empty-state-message">
            <v-icon size="48" color="accent">mdi-history</v-icon>
            <div class="empty-state-text">No recent activities to display</div>
            <v-btn 
              variant="outlined" 
              color="secondary"
              size="small"
              class="mt-3"
              @click="navigateTo('screen_006_booking.js')"
            >
              Book Activities
            </v-btn>
          </div>
        </div>
      </div>
      
      
      <!-- Refresh Action -->
      <div class="refresh-action">
        <v-btn 
          rounded="pill"
          color="primary"
          size="large"
          @click="refreshData"
          :loading="isLoading"
          class="refresh-btn"
        >
          <v-icon start>mdi-refresh</v-icon>
          Refresh Dashboard
        </v-btn>
      </div>
    </div>
    
    <!-- QR Code Popup Modal -->
    <div v-if="showQRModal" class="qr-modal-overlay" @click="hideQRCodePopup">
      <div class="qr-modal-content" @click.stop>
        <button class="qr-modal-close" @click="hideQRCodePopup">×</button>
        
        <div class="qr-modal-header">
          <div class="qr-modal-title">Member QR Code</div>
          <div class="qr-modal-subtitle">{{ userData.membershipId }}</div>
        </div>
        
        <div v-if="largeQRCode">
          <img :src="largeQRCode" alt="Large Member QR Code" class="large-qr-code" />
        </div>
        <div v-else>
          <v-progress-circular indeterminate color="primary" size="64"></v-progress-circular>
          <div style="margin-top: 16px; color: var(--muted);">Generating QR Code...</div>
        </div>
        
        <div class="qr-modal-instructions">
          Present this QR code to staff for scanning. It contains your member ID for quick identification.
        </div>
      </div>
    </div>
  </div>
  `
}
