export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
          primary: '#1A3C28',       // Rich rainforest green
          secondary: '#DDA333',     // Golden amber
          accent: '#A3BA95',        // Sage green
          lightBg: '#F8F5F0',       // Luxury paper (for input fields, cards)
          creamBg: '#EADDCB',       // Cream (for page background - will be overlay tint)
          darkText: '#1E1E1E',
          lightText: '#FFFFFF',
          error: '#B71C1C',
          muted: '#78909C',
      },
      // ... rest of your data remains the same
      user_name:"",
      user_password:"",
      device: 'web',
      device_uuid: 'web',
      fcm_token:'',
      loading: false,
      showPassword: false,
      errorMessage: "",
      rules: {
        required: v => !!v || "Field is required",
      },
      apiCall:"login/"
    };
  },
  methods: { // Methods remain the same
    async login() {
      if (!this.$refs.loginForm || !await this.$refs.loginForm.validate()) {
          this.errorMessage = "Please fill in all required fields.";
          return;
      }
      this.errorMessage = "";
      this.loading = true;
      const hashedPassword = this.$root.md5data(this.user_password).toString();
      const formData = {
        username: this.user_name,
        password: hashedPassword,
        device: this.device,
        device_uuid: this.device_uuid,
        fcm_token: this.fcm_token
      };
      const apiUrl = `${this.$root.server_domain}${this.apiCall}`;
      const postData = JSON.stringify(formData);
      console.log(`--> POST ${apiUrl}`, postData);
      try {
          const response = await fetch(apiUrl, {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: postData,
          });
          const textResponse = await response.text();
          console.log(`<-- ${response.status} ${response.statusText}`, textResponse);
          let responseBody;
          try {
              responseBody = JSON.parse(textResponse);
          } catch (e) {
              console.error("Failed to parse JSON response:", e);
              this.errorMessage = "Invalid server response.";
              this.loading = false;
              return;
          }
          console.log("Parsed JSON Response:", responseBody);
          if (responseBody && responseBody.status === "YES") {
            if (this.$root.showSnackbar) this.$root.showSnackbar('Login successful! Welcome back.', 'success');
            this.$root.showFloatingButtons = true;
            this.$root.saveToSession(responseBody.data.id, responseBody.data.session_secret);
            this.$root.loadComponent("modules/home.js");
            this.$root.bottomNavDisplay = true;
          } else {
            this.errorMessage = responseBody.message || "Login failed. Check credentials.";
            if (this.$root.showSnackbar) this.$root.showSnackbar(this.errorMessage, 'error');
          }
      } catch (error) {
          console.error(`Request failed: ${error.message || error}`);
          this.errorMessage = "Login error. Try again later.";
          if (this.$root.showSnackbar) this.$root.showSnackbar(this.errorMessage, 'error');
      } finally {
          this.loading = false;
      }
    },
    
    signUp() {
      // Add your sign up logic here
      console.log("Sign up clicked");
      // Example: this.$root.loadComponent("modules/signup.js");
    }
  },
  template: `
  <v-container fluid class="tropical-login-page fill-height pa-0">
      <!-- Decorative leaf animations (optional, can be removed if too busy with image) -->
      <div class="rainforest-bg-login-animated-leaves">
          <div class="bg-leaf-login leaf-1"></div>
          <div class="bg-leaf-login leaf-2"></div>
      </div>

      <v-row class="fill-height ma-0" align="center" justify="center">
          <v-col cols="12" sm="10" md="7" lg="5" xl="4" class="pa-sm-4 pa-2 d-flex align-center justify-center">
              <v-card class="login-card elevation-12">
                  <div class="login-card-header text-center pa-8">
                      <h1 class="club-title mb-2" :style="{ color: theme.darkText }">THE</h1>
                      <h1 class="club-name mb-2" :style="{ color: theme.darkText }">TAPANG</h1>
                      <h2 class="club-subtitle mb-6" :style="{ color: theme.darkText }">CLUB</h2>
                      <p class="member-login-text" :style="{ color: theme.darkText }">Member Login</p>
                  </div>

                  <v-card-text class="pa-8">
                      <v-form @submit.prevent="login" ref="loginForm">
                          <v-text-field
                              v-model="user_name"
                              label="Username"
                              variant="outlined"
                              :color="theme.primary"
                              :rules="[rules.required]"
                              class="mb-4 tapang-text-field"
                              required
                              density="comfortable"
                              hide-details="auto"
                          ></v-text-field>

                          <v-text-field
                              v-model="user_password"
                              label="Password"
                              :type="showPassword ? 'text' : 'password'"
                              :append-inner-icon="showPassword ? 'mdi-eye-off' : 'mdi-eye'"
                              @click:append-inner="showPassword = !showPassword"
                              variant="outlined"
                              :color="theme.primary"
                              :rules="[rules.required]"
                              class="mb-6 tapang-text-field"
                              required
                              density="comfortable"
                              hide-details="auto"
                          ></v-text-field>

                          <v-alert
                              v-if="errorMessage"
                              type="error"
                              density="compact"
                              class="mb-4"
                              rounded="lg"
                          >
                              {{ errorMessage }}
                          </v-alert>
                      
                          <v-btn
                              :color="theme.primary"
                              size="large"
                              block
                              class="tapang-login-button mb-6"
                              :loading="loading"
                              :disabled="loading || !user_name || !user_password"
                              type="submit"
                              rounded="lg"
                          >
                              LOGIN
                          </v-btn>

                          <div class="text-center mb-4">
                              <span class="register-text" :style="{ color: theme.darkText }">or Register New Account</span>
                          </div>

                          <v-btn
                              :color="theme.primary"
                              size="large"
                              block
                              class="tapang-signup-button"
                              variant="outlined"
                              rounded="lg"
                              @click="signUp"
                          >
                              SIGN UP
                          </v-btn>
                      </v-form>
                  </v-card-text>
                  <!-- Remove footer for cleaner Tapang Club look -->
              </v-card>
          </v-col>
      </v-row>
  </v-container>
  `,
  async mounted(){ // mounted hook remains the same
    console.log("Login page mounted - Tropical Theme with Image BG");
    if (this.$root) {
      this.$root.showFloatingButtons = false;
      this.$root.bottomNavDisplay = false;
    }

    const styleElement = document.createElement('style');
    styleElement.textContent = `
      /* Ensure html and body take full height */
      html, body {
        height: 100% !important;
        margin: 0 !important;
        padding: 0 !important;
        overflow-x: hidden; /* Prevent horizontal scroll */
      }

      /* Ensure Vue app container takes full height */
      #app {
        height: 100vh !important;
        min-height: 100vh !important;
      }

      .tropical-login-page {
        background-image: url('https://pcloudsb.tplinkdns.com/thetapangclub/app/modules/bgimage.png');
        background-size: cover;
        background-position: center center;
        background-repeat: no-repeat;
        background-attachment: fixed; /* Parallax effect */
        position: relative; /* For pseudo-element overlay */
        height: 100vh !important; /* Full viewport height */
        min-height: 100vh !important; /* Minimum full viewport height */
        width: 100vw !important; /* Full viewport width */
        overflow-y: auto; /* Allow vertical scrolling if needed */
        overflow-x: hidden; /* Prevent horizontal scrolling */
      }

      /* Overlay for readability - darker overlay similar to Tapang Club */
      .tropical-login-page::before {
          content: "";
          position: fixed; /* Changed from absolute to fixed for better coverage */
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: linear-gradient(135deg, rgba(0,0,0,0.7) 0%, rgba(26,60,40,0.8) 100%); /* Dark gradient overlay */
          z-index: 0; /* Ensure it's behind the content */
      }

      .rainforest-bg-login-animated-leaves { /* Container for animated leaves */
          position: fixed; /* Changed from absolute to fixed */
          top: 0; 
          left: 0; 
          width: 100%; 
          height: 100%;
          pointer-events: none; 
          z-index: 1; /* Above overlay, below card */
          overflow: hidden;
          opacity: 0.3; /* Make leaves very subtle over image */
      }
      .bg-leaf-login {
          position: absolute;
          background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 150'%3E%3Cpath d='M50 0 C20 50 20 100 50 150 C80 100 80 50 50 0 Z' fill='${this.theme.accent.replace('#', '%23')}' opacity='0.3'/%3E%3C/svg%3E");
          background-repeat: no-repeat;
          background-size: contain;
          animation: float-login-gentle 20s ease-in-out infinite alternate;
      }
      .bg-leaf-login.leaf-1 { width: 250px; height: 375px; top: 0%; left: -50px; animation-duration: 28s; }
      .bg-leaf-login.leaf-2 { width: 200px; height: 300px; bottom: 0%; right: -40px; transform: rotate(50deg) scaleX(-1); animation-duration: 35s; animation-delay: -8s; }

      @keyframes float-login-gentle {
          0% { transform: translateY(0px) translateX(0px) rotate(-1deg); opacity: 0.5; }
          50% { transform: translateY(-20px) translateX(10px) rotate(1deg); opacity: 1; }
          100% { transform: translateY(0px) translateX(0px) rotate(-1deg); opacity: 0.5; }
      }

      /* Ensure v-row takes full height */
      .tropical-login-page .v-row.fill-height {
        height: 100vh !important;
        min-height: 100vh !important;
      }

      /* Ensure v-col centers properly */
      .tropical-login-page .v-col {
        height: 100vh !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .login-card {
          border-radius: 24px !important;
          overflow: hidden;
          max-width: 380px; /* Narrower card like Tapang Club */
          width: 100%; /* Take width from v-col */
          position: relative;
          z-index: 2; /* Above overlay and animated leaves */
          box-shadow: 0 20px 40px rgba(0,0,0,0.3) !important;
          background: rgba(255, 186, 47, 0.95) !important; /* Golden semi-transparent background */
          backdrop-filter: blur(10px); /* Glass effect */
          border: 1px solid rgba(255,255,255,0.2); /* Subtle border */
          opacity: 0.6; /* Set opacity to 80% */
      }
      .login-card-header {
          background: transparent !important;
          border-bottom: none !important;
      }

      /* Tapang Club Typography */
      .club-title {
          font-size: 1.8rem !important;
          font-weight: 400 !important;
          letter-spacing: 3px !important;
          margin: 0 !important;
      }
      .club-name {
          font-size: 2.5rem !important;
          font-weight: 700 !important;
          letter-spacing: 4px !important;
          margin: 0 !important;
          line-height: 1 !important;
      }
      .club-subtitle {
          font-size: 1.2rem !important;
          font-weight: 300 !important;
          letter-spacing: 4px !important;
          margin: 0 !important;
      }
      .member-login-text {
          font-size: 1rem !important;
          font-weight: 400 !important;
          margin: 0 !important;
          opacity: 0.8;
      }
      .register-text {
          font-size: 0.9rem !important;
          font-weight: 400 !important;
          opacity: 0.8;
      }

      /* Tapang Club Input Fields */
      .tapang-text-field .v-field {
          background-color: rgba(255,255,255,0.9) !important;
          border-radius: 8px !important;
          border: 1px solid rgba(26,60,40,0.3) !important;
      }
      .tapang-text-field .v-field--focused {
          /*border: 2px solid ${this.theme.primary} !important;*/
          box-shadow: 0 0 0 3px rgba(26,60,40,0.1) !important;
      }
      .tapang-text-field .v-label {
          color: ${this.theme.darkText} !important;
          opacity: 0.7;
      }
      .tapang-text-field .v-input__details {
          display: none; /* Hide validation messages for cleaner look */
      }

      /* Tapang Club Buttons */
      .tapang-login-button {
          background-color: ${this.theme.primary} !important;
          color: white !important;
          font-weight: 600 !important;
          letter-spacing: 1px !important;
          text-transform: uppercase !important;
          border-radius: 12px !important;
          height: 50px !important;
          box-shadow: 0 4px 15px rgba(26,60,40,0.4) !important;
          transition: all 0.3s ease !important;
      }
      .tapang-login-button:hover {
          background-color: ${this.theme.darkText} !important;
          transform: translateY(-2px);
          box-shadow: 0 6px 20px rgba(26,60,40,0.6) !important;
      }

      .tapang-signup-button {
          color: ${this.theme.primary} !important;
          border: 2px solid ${this.theme.primary} !important;
          font-weight: 600 !important;
          letter-spacing: 1px !important;
          text-transform: uppercase !important;
          height: 50px !important;
          background-color: transparent !important;
          transition: all 0.3s ease !important;
      }
      .tapang-signup-button:hover {
          background-color: ${this.theme.primary} !important;
          color: white !important;
          transform: translateY(-2px);
      }

      @media (max-width: 600px) {
          .login-card {
              margin: 20px; /* Add some margin on mobile for breathing room */
              border-radius: 12px !important; /* Slightly smaller radius on mobile */
              max-width: calc(100vw - 40px); /* Ensure it doesn't exceed viewport */
          }
          .login-card-header h1 {
              font-size: 1.5rem !important;
          }
           .login-card-header .v-icon {
              font-size: 56px !important; /* Slightly smaller icon on mobile */
          }
          .login-card-header p {
              font-size: 0.9rem !important;
          }
           .login-button {
              font-size: 0.95rem !important;
          }
          
          /* Adjust padding on mobile for better space usage */
          .tropical-login-page .v-col {
            padding: 10px !important;
          }
      }

      /* Handle very short screens */
      @media (max-height: 600px) {
        .tropical-login-page .v-col {
          align-items: flex-start !important;
          padding-top: 20px !important;
        }
        .login-card {
          margin-top: 20px;
          margin-bottom: 20px;
        }
      }
    `;
    document.head.appendChild(styleElement);
  }
};