export default {
  props: ["jsonData"],
  data() {
    return {
      loading: false,
      errorMessage: "",
      user: {
        name: "Loading...",
        email: "loading@example.com",
        avatarColor: "indigo",
        initials: "..."
      },

      items: [
        { text: 'Profile', icon: 'mdi-account-outline' },
        { text: 'Security', icon: 'mdi-lock-outline' },
        { text: 'Notifications', icon: 'mdi-bell-outline' },
        { text: 'Preference', icon: 'mdi-cog-outline' },
      ],

      userControls: [
        { title: 'Content filtering', subtitle: 'Set the content filtering level to restrict appts that can be downloaded' },
        { title: 'Task', subtitle: 'Some Setting for task management' },
      ],
       
      settingsItems: [
        { value: 'notifications', title: 'Notifications', subtitle: 'Notify me about updates to apps or games that I downloaded' },
        { value: 'sound', title: 'Sound', subtitle: 'Auto-update apps at any time. Data charges may apply' },
        { value: 'widgets', title: 'Auto-add widgets', subtitle: 'Automatically add home screen widgets when downloads complete' },
      ],

      settingsSelection: [],
    };
  },
  
  methods: {
    async fetchProfileData() {
      this.loading = true;
      this.errorMessage = "";
      
      try {
        // Call the profile API using the root component's callApi method
        // This already uses POST request as defined in the callApi method
        const response = await this.$root.callApi('profile', 'get', {});
        
        if (response && response.status === "YES" && response.data) {
          // Update user data with response from API
          const userData = response.data;
          
          // Update user object with data from API
          this.user = {
            name: userData.name || "Unknown User",
            email: userData.email || "No email provided",
            avatarColor: userData.avatar_color || "indigo",
            // Generate initials from name
            initials: this.getInitials(userData.name || "Unknown User")
          };
          
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('Profile loaded successfully', 'success');
          }
        } else {
          this.errorMessage = response.message || "Failed to load profile data";
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar(this.errorMessage, 'error');
          }
        }
      } catch (error) {
        console.error(`Profile data fetch failed: ${error.message || error}`);
        this.errorMessage = "Error loading profile. Please try again later.";
        if (this.$root.showSnackbar) {
          this.$root.showSnackbar(this.errorMessage, 'error');
        }
      } finally {
        this.loading = false;
      }
    },
    
    getInitials(name) {
      if (!name) return "??";
      
      return name
        .split(' ')
        .map(part => part.charAt(0))
        .join('')
        .toUpperCase()
        .substring(0, 2);
    },

    openSetting(item) {
      alert(`You tapped: ${item.title}`);
      // Replace this with your component loader or router later
    },
    
    logout() {
      this.$root.removeSession();
      this.$root.bottomNavDisplay = false;
      this.$root.loadComponent("modules/login.js");
    }
  },
  
  template: `
    
 <div>
      <!-- Toolbar -->
      <v-toolbar color="black" dark flat>
        <v-toolbar-title>Profile</v-toolbar-title>
      </v-toolbar>

      <v-container class="mt-6">
        <!-- Loading indicator -->
        <v-row v-if="loading" justify="center" class="mb-4">
          <v-progress-circular indeterminate color="primary"></v-progress-circular>
        </v-row>
        
        <!-- Error message -->
        <v-row v-if="errorMessage" justify="center" class="mb-4">
          <v-alert type="error" density="compact" class="mb-4" rounded="lg">
            {{ errorMessage }}
          </v-alert>
        </v-row>
        
        <!-- Avatar & Info -->
        <v-row justify="center" class="mb-4">
          <v-col cols="12" class="text-center">
            <v-avatar :color="user.avatarColor" size="96" class="mx-auto mb-2">
              <span class="white--text text-h4">{{ user.initials }}</span>
            </v-avatar>
            <div class="text-h6 mt-2">{{ user.name }}</div>
            <div class="text-caption grey--text">{{ user.email }}</div>
          </v-col>
        </v-row>

        <!-- Account Section -->
        <v-list>
          <v-list-subheader>Account</v-list-subheader>
          <v-list-item
            v-for="(item, i) in items"
            :key="i"
            @click="openSetting(item)"
          >
            <template v-slot:prepend>
              <v-icon>{{ item.icon }}</v-icon>
            </template>
            <v-list-item-title>{{ item.text }}</v-list-item-title>
            <template v-slot:append>
              <v-icon>mdi-chevron-right</v-icon>
            </template>
          </v-list-item>
        </v-list>

        <!-- User Controls -->
        <v-list lines="three">
          <v-list-subheader>User Controls</v-list-subheader>
          <v-list-item
            v-for="(item, i) in userControls"
            :key="i"
            :title="item.title"
            :subtitle="item.subtitle"
          />
        </v-list>

        <!-- General Settings with Checkbox -->
        <v-divider class="my-4"></v-divider>
        <v-list
          v-model:selected="settingsSelection"
          lines="three"
          select-strategy="leaf"
        >
          <v-list-subheader>General</v-list-subheader>
          <v-list-item
            v-for="item in settingsItems"
            :key="item.value"
            :value="item.value"
            :title="item.title"
            :subtitle="item.subtitle"
          >
            <template v-slot:prepend="{ isSelected, select }">
              <v-list-item-action start>
                <v-checkbox-btn :model-value="isSelected" @update:model-value="select" />
              </v-list-item-action>
            </template>
          </v-list-item>
        </v-list>

        <!-- Logout -->
        <v-row justify="center" class="mt-6 mb-7">
          <v-btn color="red" variant="tonal" @click="logout">
            <v-icon start>mdi-logout</v-icon>
            Logout
          </v-btn>
        </v-row>
      </v-container>
    </div>
  `,
  async mounted(){
    // Fetch profile data when component is mounted
    await this.fetchProfileData();
  }
};