export default {
    props: ["jsonData"],
    data() {
      return {
        // Tropical Rainforest Theme
        theme: {
          primary: '#1A3C28',       // Rich rainforest green
          secondary: '#DDA333',     // Golden amber
          accent: '#A3BA95',        // Sage green
          lightBg: '#F8F5F0',       // Luxury paper
          darkText: '#1E1E1E',
          cream: '#EADDCB',         // Cream
          error: '#B71C1C',
          success: '#2E7D32'
        },
        
        // Payment Flow States
        currentStep: 'scan', // scan, details, pin, confirm, processing, success, error
        
        // User Data
        userData: {
          name: "Jason Tan",
          role: "platinum",
          walletBalance: 2480.50,
          recentTransactions: [/* ... */]
        },
        
        // Camera/Scanning
        cameraActive: false,
        scanSuccess: false,
        scannerAnimationActive: true,
        cameraLoading: true,
        
        // Payment Data
        paymentData: {
          merchantId: null,
          merchantName: null,
          amount: null,
          reference: null,
          timestamp: null,
        },
        
        // PIN Authentication
        pin: '',
        pinError: false,
        pinAttempts: 0,
        
        // Manual Entry
        manualEntryDialog: false,
        manualMerchantName: '',
        manualAmount: '',
        
        // UI States
        loading: false,
        error: null,
        showSnackbar: false,
        snackbarText: '',
        snackbarColor: '',
        paymentAnimationCompleted: false,
        showConfetti: false,
        
        // Security timeout
        securityTimeout: null,
        timeLeft: 60, // Seconds for timeout
        
        // Transaction receipt
        receiptDetails: {
          transactionId: '',
          date: '',
          status: '',
        },
        
        // For demo purposes (simulated scanned data)
        demoQrData: {
          merchantId: 'MER12345',
          merchantName: 'Poolside Café',
          amount: 65.50,
          reference: 'INV-89721',
          timestamp: new Date().toISOString(),
        },
        
        // Animation states
        floatingLeaf: true,
      };
    },
    
    computed: {
      // Format currency consistently
      formattedAmount() {
        return this.paymentData.amount ? this.formatCurrency(this.paymentData.amount) : 'RM 0.00';
      },
      
      formattedBalance() {
        return this.formatCurrency(this.userData.walletBalance);
      },
      
      // Balance after payment
      balanceAfterPayment() {
        if (!this.paymentData.amount) return this.formatCurrency(this.userData.walletBalance);
        return this.formatCurrency(this.userData.walletBalance - this.paymentData.amount);
      },
      
      // Is payment valid?
      isPaymentValid() {
        return this.paymentData.merchantId && 
               this.paymentData.amount && 
               this.paymentData.amount > 0 && 
               this.paymentData.amount <= this.userData.walletBalance;
      },
      
      // Is PIN valid (for demo, let's say PIN is '1234')
      isPinValid() {
        return this.pin === '1234';
      },
      
      // Format timestamp nicely
      formattedTimestamp() {
        if (!this.paymentData.timestamp) return '';
        const date = new Date(this.paymentData.timestamp);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
      },
      
      // Formatted transaction date
      formattedTransactionDate() {
        if (!this.receiptDetails.date) return '';
        const date = new Date(this.receiptDetails.date);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
      },
      
      // Membership badge styling
      membershipBadge() {
        return {
          platinum: { color: '#E5E4E2', textColor: '#1A3C28', text: 'Platinum Member', icon: 'mdi-crown' },
          gold: { color: '#FFD700', textColor: '#1A3C28', text: 'Gold Member', icon: 'mdi-medal' },
          silver: { color: '#C0C0C0', textColor: '#1A3C28', text: 'Silver Member', icon: 'mdi-medal-outline' },
          junior: { color: '#A3BA95', textColor: '#1A3C28', text: 'Junior Account', icon: 'mdi-account-child' }
        }[this.userData.role] || { color: '#A3BA95', textColor: '#1A3C28', text: 'Member', icon: 'mdi-account' }
      }
    },
    
    methods: {
      // Format currency consistently
      formatCurrency(amount) {
        return "RM " + parseFloat(amount).toFixed(2);
      },
      
      // Initialize camera and QR scanning
      initCamera() {
        this.cameraActive = true;
        this.cameraLoading = true;
        this.scannerAnimationActive = true;
        this.showSnackbarMessage('Camera activated', 'info');
        
        // Simulate camera initialization
        setTimeout(() => {
          this.cameraLoading = false;
        }, 1200);
        
        // For demo purposes, set a timeout to simulate a successful scan
        setTimeout(() => {
          this.handleQrScanned(this.demoQrData);
        }, 4000);
      },
      
      // Stop camera
      stopCamera() {
        this.cameraActive = false;
        this.scannerAnimationActive = false;
      },
      
      // Handle successful QR scan
      handleQrScanned(data) {
        this.scanSuccess = true;
        this.paymentData = { ...data };
        this.showSnackbarMessage('QR code scanned successfully!', 'success');
        
        // Transition to payment details screen
        setTimeout(() => {
          this.currentStep = 'details';
          this.stopCamera();
        }, 1200);
      },
      
      // Delete last PIN digit
      deletePinDigit() {
        this.pin = this.pin.slice(0, -1);
      },
      
      // Clear PIN
      clearPin() {
        this.pin = '';
      },
      
      // Validate PIN
      validatePin() {
        this.loading = true;
        
        // Simulate validation delay
        setTimeout(() => {
          if (this.isPinValid) {
            this.pinError = false;
            this.currentStep = 'confirm';
            this.showSnackbarMessage('PIN verified successfully', 'success');
          } else {
            this.pinError = true;
            this.pinAttempts++;
            this.clearPin();
            this.showSnackbarMessage('Incorrect PIN', 'error');
            
            // If too many failed attempts, lock the account
            if (this.pinAttempts >= 3) {
              this.currentStep = 'error';
              this.error = 'Too many failed attempts. Your account has been temporarily locked.';
              this.showSnackbarMessage('Account temporarily locked', 'error');
            }
          }
          
          this.loading = false;
        }, 1000);
      },
      
      // Confirm payment and process
      confirmPayment() {
        this.currentStep = 'processing';
        this.startSecurityTimeout();
        
        // Generate a transaction ID
        this.receiptDetails.transactionId = 'TXN' + Math.floor(100000 + Math.random() * 900000);
        this.receiptDetails.date = new Date().toISOString();
        
        // Simulate processing delay
        setTimeout(() => {
          // 90% chance of success for demo
          if (Math.random() < 0.9) {
            this.currentStep = 'success';
            this.userData.walletBalance -= this.paymentData.amount;
            this.receiptDetails.status = 'Success';
            
            // Show confetti after a short delay
            setTimeout(() => {
              this.showConfetti = true;
              setTimeout(() => this.showConfetti = false, 5000);
            }, 500);
            
            this.showSnackbarMessage('Payment successful!', 'success');
          } else {
            this.currentStep = 'error';
            this.error = 'Transaction failed. Please try again later.';
            this.receiptDetails.status = 'Failed';
            this.showSnackbarMessage('Payment failed', 'error');
          }
          
          this.clearSecurityTimeout();
          this.paymentAnimationCompleted = true;
        }, 2500);
      },
      
      // Cancel payment
      cancelPayment() {
        // Go back to previous step or to scan
        if (this.currentStep === 'confirm') {
          this.currentStep = 'pin';
          this.clearPin();
          this.showSnackbarMessage('Returned to PIN entry', 'info');
        } else if (this.currentStep === 'pin') {
          this.currentStep = 'details';
          this.showSnackbarMessage('Returned to payment details', 'info');
        } else if (this.currentStep === 'details') {
          this.resetPayment();
          this.showSnackbarMessage('Payment cancelled', 'info');
        }
      },
      
      // Reset entire payment flow
      resetPayment() {
        this.currentStep = 'scan';
        this.scanSuccess = false;
        this.paymentData = {
          merchantId: null,
          merchantName: null,
          amount: null,
          reference: null,
          timestamp: null,
        };
        this.clearPin();
        this.pinError = false;
        this.pinAttempts = 0;
        this.error = null;
        this.paymentAnimationCompleted = false;
        
        this.clearSecurityTimeout();
      },
      
      // Start security timeout for processing
      startSecurityTimeout() {
        this.timeLeft = 60;
        this.securityTimeout = setInterval(() => {
          this.timeLeft--;
          if (this.timeLeft <= 0) {
            this.clearSecurityTimeout();
            this.currentStep = 'error';
            this.error = 'Transaction timed out. Please try again.';
            this.showSnackbarMessage('Transaction timed out', 'error');
          }
        }, 1000);
      },
      
      // Clear security timeout
      clearSecurityTimeout() {
        if (this.securityTimeout) {
          clearInterval(this.securityTimeout);
          this.securityTimeout = null;
        }
      },
      
      // Go back to home
      goToHome() {
        this.$root.loadComponent('modules/home.js');
      },
      
      // Go to transactions
      goToTransactions() {
        this.$root.loadComponent('modules/screen_008_wallet_detail.js');
      },
      
      // Handle manual entry (alternative to scanning)
      openManualEntry() {
        this.manualEntryDialog = true;
      },
      
      submitManualEntry() {
        if (this.manualMerchantName && this.manualAmount > 0) {
          const manualData = {
            merchantId: 'MAN' + Math.floor(10000 + Math.random() * 90000),
            merchantName: this.manualMerchantName,
            amount: parseFloat(this.manualAmount),
            reference: 'MAN-' + Math.floor(10000 + Math.random() * 90000),
            timestamp: new Date().toISOString(),
          };
          
          this.manualEntryDialog = false;
          this.handleQrScanned(manualData);
          this.showSnackbarMessage('Payment details entered manually', 'success');
        }
      },
      
      // Show snackbar message
      showSnackbarMessage(text, color = 'info') {
        this.snackbarText = text;
        this.snackbarColor = color;
        this.showSnackbar = true;
        
        // Auto hide after 3 seconds
        setTimeout(() => {
          this.showSnackbar = false;
        }, 3000);
      },
      
      // Generate confetti elements
      generateConfettiElements() {
        const confetti = [];
        const colors = ['#DDA333', '#A3BA95', '#1A3C28', '#EADDCB'];
        
        for (let i = 0; i < 100; i++) {
          confetti.push({
            left: Math.random() * 100 + '%',
            top: -20 + Math.random() * 100 + 'px',
            width: 5 + Math.random() * 10 + 'px',
            height: 5 + Math.random() * 10 + 'px',
            backgroundColor: colors[Math.floor(Math.random() * colors.length)],
            animationDelay: Math.random() * 3 + 's',
            animationDuration: 3 + Math.random() * 2 + 's'
          });
        }
        
        return confetti;
      }
    },
    
    mounted() {
      // Reset any previous state
      this.resetPayment();
      
      // Start camera for scanning after a short delay
      setTimeout(() => {
        this.initCamera();
      }, 1000);
      
      // Add custom CSS styles
      const styleElement = document.createElement('style');
      styleElement.textContent = `
        /* Base Styles */
        .scan-pay-screen {
          background-color: ${this.theme.cream};
          min-height: 100vh;
          position: relative;
          overflow-x: hidden;
          padding-bottom: 24px;
        }
        
        /* Background Decorations */
        .background-decoration {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          pointer-events: none;
          z-index: 0;
        }
        
        .bg-leaf {
          position: absolute;
          border-radius: 50% 50% 0 50%;
          opacity: 0.07;
          background-color: ${this.theme.accent};
          transform: rotate(45deg);
        }
        
        .bg-leaf-1 {
          width: 400px;
          height: 400px;
          top: -200px;
          left: -100px;
        }
        
        .bg-leaf-2 {
          width: 300px;
          height: 300px;
          bottom: 10%;
          right: -150px;
          transform: rotate(130deg);
        }
        
        .bg-leaf-3 {
          width: 200px;
          height: 200px;
          top: 30%;
          left: -100px;
          transform: rotate(80deg);
        }
        
        /* Animated Floating Leaf */
        .floating-leaf {
          position: absolute;
          width: 60px;
          height: 90px;
          background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 150'%3E%3Cpath d='M50 0 C20 50 20 100 50 150 C80 100 80 50 50 0 Z' fill='%23A3BA95' opacity='0.2'/%3E%3C/svg%3E");
          background-repeat: no-repeat;
          background-size: contain;
          pointer-events: none;
          z-index: 1;
          animation: float-leaf 25s linear infinite;
        }
        
        @keyframes float-leaf {
          0% {
            top: -10%;
            left: 20%;
            transform: rotate(0deg);
            opacity: 0;
          }
          10% {
            opacity: 0.2;
          }
          90% {
            opacity: 0.2;
          }
          100% {
            top: 110%;
            left: 70%;
            transform: rotate(360deg);
            opacity: 0;
          }
        }
        
        /* Sunlight Effect */
        .sunlight-effect {
          position: absolute;
          top: -50%;
          right: -50%;
          width: 200%;
          height: 200%;
          background: radial-gradient(ellipse at top right, rgba(221, 163, 51, 0.1) 0%, transparent 70%);
          pointer-events: none;
          z-index: 0;
        }
        
        /* Header */
        .scan-pay-header {
          background-color: ${this.theme.primary};
          color: white;
          border-radius: 0 0 24px 24px;
          position: relative;
          overflow: hidden;
          padding-bottom: 60px;
          margin-bottom: 24px;
          box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }
        
        .step-indicator {
          display: flex;
          justify-content: center;
        }
        
        .step-dot {
          width: 8px;
          height: 8px;
          border-radius: 50%;
          background-color: rgba(255,255,255,0.3);
          margin: 0 4px;
          transition: all 0.3s ease;
        }
        
        .step-dot.active {
          background-color: white;
          transform: scale(1.3);
        }
        
        .header-decoration {
          position: absolute;
          background-color: rgba(255, 255, 255, 0.05);
          border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
          z-index: 0;
        }
        
        .header-decoration.left {
          width: 150px;
          height: 150px;
          top: -50px;
          left: -50px;
          transform: rotate(35deg);
        }
        
        .header-decoration.right {
          width: 150px;
          height: 150px;
          bottom: -50px;
          right: -30px;
          transform: rotate(-15deg);
        }
        
        /* Transitions */
        .fade-transition-enter-active,
        .fade-transition-leave-active {
          transition: all 0.6s cubic-bezier(0.23, 1, 0.32, 1);
        }
        
        .fade-transition-enter-from {
          opacity: 0;
          transform: translateY(20px);
        }
        
        .fade-transition-leave-to {
          opacity: 0;
          transform: translateY(-20px);
        }
        
        /* Card Container styling */
        .card-container {
          background-color: white;
          border-radius: 16px;
          overflow: hidden;
          margin-bottom: 20px;
          box-shadow: 0 4px 25px rgba(0, 0, 0, 0.08);
          position: relative;
          border: 1px solid rgba(163, 186, 149, 0.2);
        }
        
        .card-wrapper {
          transition: transform 0.3s ease;
        }
        
        .card-wrapper:active {
          transform: scale(0.98);
        }
        
        /* Card decorations */
        .card-decoration {
          position: absolute;
          width: 120px;
          height: 120px;
          background-color: ${this.theme.accent};
          opacity: 0.05;
          border-radius: 60% 40% 50% 50%; 
          z-index: 0;
        }
        
        .card-decoration.top-right {
          top: -60px;
          right: -60px;
        }
        
        .card-decoration.bottom-left {
          bottom: -60px;
          left: -60px;
          transform: rotate(45deg);
        }
  
        /* Scanner Animation */
        .camera-container {
          position: relative;
          background: linear-gradient(180deg, #1e2a37 0%, #0c141e 100%);
          overflow: hidden;
        }
        
        .camera-container::before {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          height: 1px;
          background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
        }
        
        .camera-container::after {
          content: "";
          position: absolute;
          bottom: 0;
          left: 0;
          right: 0;
          height: 1px;
          background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
        }
        
        .scanning-animation {
          width: 180px;
          height: 180px;
          border: 2px solid ${this.theme.secondary};
          border-radius: 12px;
          position: relative;
          box-shadow: 0 0 20px rgba(221, 163, 51, 0.3);
        }
        
        .camera-loading-overlay {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: rgba(0, 0, 0, 0.7);
          display: flex;
          justify-content: center;
          align-items: center;
          z-index: 2;
        }
        
        .scan-line {
          position: absolute;
          top: 10%;
          width: 100%;
          height: 2px;
          background: linear-gradient(90deg, transparent, ${this.theme.secondary}, transparent);
          box-shadow: 0 0 8px ${this.theme.secondary};
          animation: scan-animation 2s infinite ease-in-out;
        }
        
        @keyframes scan-animation {
          0% { top: 10%; }
          50% { top: 90%; }
          100% { top: 10%; }
        }
        
        .scan-corner {
          position: absolute;
          width: 20px;
          height: 20px;
          border-color: ${this.theme.secondary};
          border-style: solid;
          border-width: 0;
        }
        
        .corner-top-left {
          top: 0;
          left: 0;
          border-top-width: 2px;
          border-left-width: 2px;
        }
        
        .corner-top-right {
          top: 0;
          right: 0;
          border-top-width: 2px;
          border-right-width: 2px;
        }
        
        .corner-bottom-left {
          bottom: 0;
          left: 0;
          border-bottom-width: 2px;
          border-left-width: 2px;
        }
        
        .corner-bottom-right {
          bottom: 0;
          right: 0;
          border-bottom-width: 2px;
          border-right-width: 2px;
        }
        
        .scan-success-animation {
          background-color: rgba(46, 125, 50, 0.8);
          backdrop-filter: blur(4px);
          border-radius: 12px;
          display: flex;
          align-items: center;
          justify-content: center;
          animation: pulse-success 1s ease-in-out;
        }
        
        @keyframes pulse-success {
          0% { transform: scale(0.8); opacity: 0; }
          50% { transform: scale(1.05); opacity: 1; }
          100% { transform: scale(1); opacity: 1; }
        }
        
        /* PIN Display */
        .pin-dots {
          display: flex;
          justify-content: center;
          gap: 16px;
          margin-bottom: 24px;
        }
        
        .pin-dot {
          width: 12px;
          height: 12px;
          border-radius: 50%;
          border: 2px solid ${this.theme.primary};
          transition: all 0.2s ease;
        }
        
        .pin-dot.filled {
          background-color: ${this.theme.primary};
        }
        
        /* Wallet Info styling */
        .wallet-info {
          background-color: ${this.theme.cream};
          border-radius: 12px;
          padding: 16px;
          position: relative;
          overflow: hidden;
        }
        
        .wallet-info::before {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 5px;
          background: linear-gradient(90deg, ${this.theme.accent}88, transparent);
        }
        
        /* Processing Animation */
        .processing-icon {
          animation: pulse-processing 1.5s infinite ease-in-out;
        }
        
        @keyframes pulse-processing {
          0% { transform: scale(1); opacity: 1; }
          50% { transform: scale(1.1); opacity: 0.8; }
          100% { transform: scale(1); opacity: 1; }
        }
        
        /* Success Animation */
        .success-icon {
          animation: success-animation 0.7s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        }
        
        @keyframes success-animation {
          0% { transform: scale(0); }
          60% { transform: scale(1.2); }
          100% { transform: scale(1); }
        }
        
        /* Error Animation */
        .error-icon {
          animation: error-animation 0.5s cubic-bezier(0.36, 0.07, 0.19, 0.97) both;
        }
        
        @keyframes error-animation {
          0%, 100% { transform: translateX(0); }
          20%, 60% { transform: translateX(-8px); }
          40%, 80% { transform: translateX(8px); }
        }
        
        /* Custom Styling for OTP Input */
        .v-otp-input {
          justify-content: center;
        }
        
        .v-otp-input .v-field {
          box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        }
        
        /* Receipt Card */
        .receipt-card {
          position: relative;
          background-color: ${this.theme.lightBg};
          border-radius: 12px;
          overflow: hidden;
        }
        
        .receipt-card::before {
          content: '';
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 5px;
          background: linear-gradient(90deg, ${this.theme.success}, ${this.theme.accent});
        }
        
        .receipt-header {
          position: relative;
          text-align: center;
          padding: 16px;
          border-bottom: 1px dashed rgba(0, 0, 0, 0.1);
        }
        
        .receipt-watermark {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          font-size: 80px;
          color: rgba(46, 125, 50, 0.03);
          pointer-events: none;
          z-index: 0;
        }
        
        /* Confetti */
        .confetti-container {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          pointer-events: none;
          z-index: 5;
        }
        
        .confetti-piece {
          position: absolute;
          animation: confetti-fall 3s linear forwards;
        }
        
        @keyframes confetti-fall {
          0% { 
            transform: translateY(-20px) rotate(0deg);
            opacity: 0;
          }
          10% {
            opacity: 1;
          }
          100% { 
            transform: translateY(100vh) rotate(360deg);
            opacity: 0;
          }
        }
        
        /* Success Badge */
        .success-badge {
          display: inline-flex;
          align-items: center;
          padding: 4px 12px;
          border-radius: 20px;
          background-color: rgba(46, 125, 50, 0.1);
          color: ${this.theme.success};
          font-weight: 600;
        }
        
        /* Merchant Avatar */
        .merchant-avatar {
          border: 3px solid ${this.theme.lightBg};
          box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        /* Transaction ID display */
        .transaction-id {
          font-family: 'Courier New', monospace;
          letter-spacing: 0.05em;
        }
        
        /* Slider button styles */
        .slide-button {
          position: relative;
          overflow: hidden;
          transition: all 0.3s ease;
        }
        
        .slide-button::after {
          content: '';
          position: absolute;
          top: 0;
          left: -100%;
          width: 100%;
          height: 100%;
          background: rgba(255, 255, 255, 0.2);
          transition: transform 0.6s ease;
        }
        
        .slide-button:hover::after {
          transform: translateX(100%);
        }
        
        /* Details Table */
        .details-table .v-list-item {
          border-bottom: 1px solid rgba(0,0,0,0.05);
          padding: 12px 16px;
        }
        
        .details-table .v-list-item:last-child {
          border-bottom: none;
        }
        
        /* Fixes for mobile */
        @media (max-width: 600px) {
          .scanning-animation {
            width: 150px;
            height: 150px;
          }
          
          /*.v-container {
            padding: 12px !important;
          }*/
        }
      `;
      document.head.appendChild(styleElement);
    },
    
    beforeUnmount() {
      this.clearSecurityTimeout();
      this.stopCamera();
    },
    
    template: `
      <div class="scan-pay-screen">
        <!-- Background Decorations -->
        <div class="background-decoration">
          <div class="bg-leaf bg-leaf-1"></div>
          <div class="bg-leaf bg-leaf-2"></div>
          <div class="bg-leaf bg-leaf-3"></div>
          <div class="sunlight-effect"></div>
          
          <!-- Animated floating leaf -->
          <div v-if="floatingLeaf" class="floating-leaf"></div>
        </div>
        
        <!-- Confetti Animation for Success -->
        <div v-if="showConfetti" class="confetti-container">
          <div 
            v-for="(confetti, index) in generateConfettiElements()"
            :key="index"
            class="confetti-piece"
            :style="confetti"
          ></div>
        </div>
        
        <!-- Header -->
        <div class="scan-pay-header">
          <v-app-bar flat color="transparent" height="64" :absolute="true">
            <template v-slot:prepend>
              <v-btn icon @click="goToHome" variant="text" color="white" class="ml-1">
                <v-icon>mdi-arrow-left</v-icon>
              </v-btn>
            </template>
            
            <v-app-bar-title>
              <div class="text-center" style="position: absolute; width: 100%; left: 0;">
                <div class="text-h6 text-white font-weight-bold mb-0">Family Member</div>
              </div>
            </v-app-bar-title>
            
            <!-- Add an empty append slot to balance the layout -->
            <template v-slot:append>
              <div style="width: 40px;"></div>
            </template>
          </v-app-bar>
          
          <!-- Step Indicator -->
          <div class="step-indicator mt-4">
            <div class="step-dot" :class="{ active: currentStep === 'scan' }"></div>
            <div class="step-dot" :class="{ active: currentStep === 'details' }"></div>
            <div class="step-dot" :class="{ active: currentStep === 'pin' }"></div>
            <div class="step-dot" :class="{ active: currentStep === 'confirm' }"></div>
            <div class="step-dot" :class="{ active: ['processing', 'success', 'error'].includes(currentStep) }"></div>
          </div>
          
          <div class="header-decoration left"></div>
          <div class="header-decoration right"></div>
        </div>
        
        <!-- Main Container -->
        <v-container class="px-4">
          <transition-group name="fade-transition" mode="out-in">
            <!-- SCAN QR STEP -->
            <div v-if="currentStep === 'scan'" key="scan">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="text-center pa-6">
                    <h2 class="text-h6 font-weight-bold mb-2" :style="{color: theme.primary}">Scan QR Code</h2>
                    <p class="text-subtitle-2 text-medium-emphasis mb-6">Point your camera at the merchant's QR code</p>
                    
                    <v-sheet class="camera-container mx-auto mb-6" rounded="lg" color="grey-darken-4" width="100%" height="260">
                      <!-- Camera loading overlay -->
                      <div v-if="cameraLoading" class="camera-loading-overlay">
                        <v-progress-circular indeterminate color="secondary" size="48"></v-progress-circular>
                      </div>
                      
                      <!-- Camera view -->
                      <div class="d-flex justify-center align-center h-100">
                        <div v-if="!scanSuccess" class="scanning-animation">
                          <div class="scan-line"></div>
                          <div class="scan-corners">
                            <div class="scan-corner corner-top-left"></div>
                            <div class="scan-corner corner-top-right"></div>
                            <div class="scan-corner corner-bottom-left"></div>
                            <div class="scan-corner corner-bottom-right"></div>
                          </div>
                        </div>
                        
                        <div v-else class="scan-success-animation" style="width: 180px; height: 180px;">
                          <v-icon icon="mdi-check-circle" size="x-large" color="white"></v-icon>
                        </div>
                      </div>
                    </v-sheet>
                    
                    <v-btn 
                      v-if="!cameraActive" 
                      color="primary" 
                      block 
                      prepend-icon="mdi-camera"
                      class="mb-3 slide-button"
                      @click="initCamera"
                      rounded="pill"
                      elevation="2"
                      height="48"
                    >
                      Start Camera
                    </v-btn>
                    
                    <v-btn 
                      variant="text" 
                      color="secondary"
                      @click="openManualEntry"
                      class="font-weight-medium"
                    >
                      <v-icon start size="small">mdi-keyboard</v-icon>
                      Enter payment details manually
                    </v-btn>
                  </v-card-text>
                </v-card>
              </div>
            </div>
            
            <!-- PAYMENT DETAILS STEP -->
            <div v-else-if="currentStep === 'details'" key="details">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="pa-6">
                    <div class="d-flex align-center mb-6">
                      <v-avatar class="merchant-avatar mr-4" size="56" color="accent">
                        <v-icon icon="mdi-store" color="white" size="24"></v-icon>
                      </v-avatar>
                      
                      <div>
                        <h3 class="text-h6 font-weight-bold" :style="{color: theme.primary}">{{ paymentData.merchantName }}</h3>
                        <p class="text-caption text-medium-emphasis">Ref: {{ paymentData.reference }}</p>
                      </div>
                    </div>
                    
                    <div class="text-center mb-6">
                      <div class="text-subtitle-2 text-medium-emphasis mb-1">Payment Amount</div>
                      <div class="text-h4 font-weight-bold" :style="{color: theme.primary}">{{ formattedAmount }}</div>
                    </div>
                    
                    <div class="wallet-info mb-6">
                      <div class="d-flex justify-space-between align-center mb-2">
                        <span class="text-subtitle-2">Current Balance</span>
                        <span class="text-subtitle-2 font-weight-bold">{{ formattedBalance }}</span>
                      </div>
                      
                      <v-divider class="mb-2"></v-divider>
                      
                      <div class="d-flex justify-space-between align-center">
                        <span class="text-subtitle-2">Balance After Payment</span>
                        <span class="text-subtitle-2 font-weight-bold" :style="{color: theme.success}">{{ balanceAfterPayment }}</span>
                      </div>
                    </div>
                    
                    <v-row>
                      <v-col cols="6">
                        <v-btn 
                          color="error" 
                          variant="outlined" 
                          block
                          @click="cancelPayment"
                          rounded="lg"
                          elevation="1"
                          height="48"
                        >
                          <v-icon start>mdi-close</v-icon>
                          Cancel
                        </v-btn>
                      </v-col>
                      
                      <v-col cols="6">
                        <v-btn 
                          color="primary" 
                          block
                          @click="currentStep = 'pin'"
                          :disabled="!isPaymentValid"
                          rounded="lg"
                          elevation="3"
                          height="48"
                          class="slide-button"
                        >
                          <v-icon start>mdi-lock</v-icon>
                          Continue
                        </v-btn>
                      </v-col>
                    </v-row>
                  </v-card-text>
                </v-card>
              </div>
            </div>
            
            <!-- PIN ENTRY STEP -->
            <div v-if="currentStep === 'pin'">
              <v-card class="card-container" elevation="0">
                <v-card-text class="text-center pa-6">
                  <h2 class="text-h6 font-weight-bold mb-2" :style="{color: theme.primary}">Enter PIN</h2>
                  <v-otp-input
                    v-model="pin"
                    length="4"
                    type="number"
                    :error="pinError"
                    variant="solo-filled"
                    color="primary"
                    class="mb-4"
                    @update:model-value="pinError = false"
                    @finish="validatePin"
                  ></v-otp-input>
                  <v-alert
                    v-if="pinError"
                    type="error"
                    text="Incorrect PIN. Please try again."
                    variant="tonal"
                    class="mb-4"
                    density="compact"
                    icon="mdi-alert-circle"
                    border="start"
                  ></v-alert>
                  <v-btn
                    color="error"
                    variant="text"
                    block
                    @click="cancelPayment"
                    class="mt-4"
                  >
                    <v-icon start>mdi-arrow-left</v-icon>
                    Back
                  </v-btn>
                </v-card-text>
              </v-card>
            </div>
            
            <!-- CONFIRMATION STEP -->
            <div v-else-if="currentStep === 'confirm'" key="confirm">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="pa-6">
                    <h2 class="text-h6 font-weight-bold mb-6 text-center" :style="{color: theme.primary}">Confirm Payment</h2>
                    
                    <v-list lines="two" class="bg-transparent pa-0 details-table">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-avatar color="primary" size="36" class="mr-3">
                            <v-icon icon="mdi-store" color="white" size="18"></v-icon>
                          </v-avatar>
                        </template>
                        <v-list-item-title class="font-weight-medium">Merchant</v-list-item-title>
                        <v-list-item-subtitle>{{ paymentData.merchantName }}</v-list-item-subtitle>
                      </v-list-item>
                      
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-avatar color="primary" size="36" class="mr-3">
                            <v-icon icon="mdi-tag" color="white" size="18"></v-icon>
                          </v-avatar>
                        </template>
                        <v-list-item-title class="font-weight-medium">Reference</v-list-item-title>
                        <v-list-item-subtitle>{{ paymentData.reference }}</v-list-item-subtitle>
                      </v-list-item>
                      
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-avatar color="primary" size="36" class="mr-3">
                            <v-icon icon="mdi-calendar-clock" color="white" size="18"></v-icon>
                          </v-avatar>
                        </template>
                        <v-list-item-title class="font-weight-medium">Date & Time</v-list-item-title>
                        <v-list-item-subtitle>{{ formattedTimestamp }}</v-list-item-subtitle>
                      </v-list-item>
                      
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-avatar color="primary" size="36" class="mr-3">
                            <v-icon icon="mdi-wallet" color="white" size="18"></v-icon>
                          </v-avatar>
                        </template>
                        <v-list-item-title class="font-weight-medium">Payment From</v-list-item-title>
                        <v-list-item-subtitle>My Credit Balance ({{ formattedBalance }})</v-list-item-subtitle>
                      </v-list-item>
                    </v-list>
                    
                    <v-card class="my-6 py-6" color="primary">
                      <div class="d-flex justify-space-between align-center px-4">
                        <span class="text-h6 font-weight-bold text-white">Total</span>
                        <span class="text-h4 font-weight-bold" :style="{color: theme.secondary}">{{ formattedAmount }}</span>
                      </div>
                    </v-card>
                    
                    <v-row>
                      <v-col cols="6">
                        <v-btn 
                          color="error" 
                          variant="outlined" 
                          block
                          @click="cancelPayment"
                          rounded="lg"
                          elevation="1"
                          height="48"
                        >
                          <v-icon start>mdi-close</v-icon>
                          Cancel
                        </v-btn>
                      </v-col>
                      
                      <v-col cols="6">
                        <v-btn 
                          color="success" 
                          block
                          @click="confirmPayment"
                          rounded="lg"
                          elevation="3"
                          height="48"
                          class="slide-button"
                        >
                          <v-icon start>mdi-check-circle</v-icon>
                          Pay Now
                        </v-btn>
                      </v-col>
                    </v-row>
                  </v-card-text>
                </v-card>
              </div>
            </div>
            
            <!-- PROCESSING STEP -->
            <div v-else-if="currentStep === 'processing'" key="processing">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="text-center pa-8">
                    <v-progress-circular
                      indeterminate
                      :color="theme.secondary"
                      size="80"
                      width="4"
                      class="mb-6 processing-icon"
                    ></v-progress-circular>
                    
                    <h2 class="text-h6 font-weight-bold mb-2" :style="{color: theme.primary}">Processing Payment</h2>
                    <p class="text-subtitle-2 text-medium-emphasis mb-4">Please wait while we process your transaction</p>
                    
                    <div class="d-flex align-center justify-center">
                      <v-icon color="warning" class="mr-2">mdi-clock-outline</v-icon>
                      <span class="text-body-2 font-weight-medium">Transaction will timeout in {{ timeLeft }}s</span>
                    </div>
                    
                    <v-chip class="mt-6 mx-auto" :color="theme.primary + '22'" size="small">
                      <span class="text-caption">Transaction ID: {{ receiptDetails.transactionId || 'Generating...' }}</span>
                    </v-chip>
                  </v-card-text>
                </v-card>
              </div>
            </div>
            
            <!-- SUCCESS STEP -->
            <div v-else-if="currentStep === 'success'" key="success">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="text-center pa-6">
                    <v-avatar
                      color="success"
                      size="80"
                      class="mb-4 success-icon"
                    >
                      <v-icon icon="mdi-check-bold" size="40" color="white"></v-icon>
                    </v-avatar>
                    
                    <h2 class="text-h5 font-weight-bold mb-2" style="color: var(--success);">Payment Successful!</h2>
                    <p class="text-subtitle-2 text-medium-emphasis mb-6">Your transaction has been completed successfully</p>
                    
                    <v-card class="receipt-card mb-6">
                      <div class="receipt-watermark">
                        <v-icon icon="mdi-check-circle" size="120"></v-icon>
                      </div>
                      
                      <div class="receipt-header">
                        <v-icon icon="mdi-receipt-text" color="success" size="24" class="mb-2"></v-icon>
                        <h3 class="text-h6 font-weight-bold mb-1" :style="{color: theme.primary}">Payment Receipt</h3>
                        <div class="success-badge">
                          <v-icon start size="small" color="success">mdi-check-circle</v-icon>
                          Completed
                        </div>
                      </div>
                      
                      <v-list class="bg-transparent pa-0" lines="two">
                        <v-list-item>
                          <v-list-item-title>Amount Paid</v-list-item-title>
                          <template v-slot:append>
                            <div class="text-h6 font-weight-bold" :style="{color: theme.primary}">{{ formattedAmount }}</div>
                          </template>
                        </v-list-item>
                        
                        <v-divider></v-divider>
                        
                        <v-list-item>
                          <v-list-item-title>Merchant</v-list-item-title>
                          <v-list-item-subtitle>{{ paymentData.merchantName }}</v-list-item-subtitle>
                        </v-list-item>
                        
                        <v-list-item>
                          <v-list-item-title>Transaction ID</v-list-item-title>
                          <v-list-item-subtitle class="transaction-id font-weight-medium">{{ receiptDetails.transactionId }}</v-list-item-subtitle>
                        </v-list-item>
                        
                        <v-list-item>
                          <v-list-item-title>Date & Time</v-list-item-title>
                          <v-list-item-subtitle>{{ formattedTransactionDate }}</v-list-item-subtitle>
                        </v-list-item>
                        
                        <v-list-item>
                          <v-list-item-title>Reference</v-list-item-title>
                          <v-list-item-subtitle>{{ paymentData.reference }}</v-list-item-subtitle>
                        </v-list-item>
                        
                        <v-list-item>
                          <v-list-item-title>New Balance</v-list-item-title>
                          <template v-slot:append>
                            <div class="font-weight-bold" :style="{color: theme.success}">{{ formattedBalance }}</div>
                          </template>
                        </v-list-item>
                      </v-list>
                    </v-card>
                    
                    <v-row>
                      <v-col cols="6">
                        <v-btn 
                          :color="theme.secondary" 
                          variant="outlined" 
                          block
                          @click="goToTransactions"
                          rounded="lg"
                          elevation="1"
                          height="48"
                        >
                          <v-icon start>mdi-history</v-icon>
                          View History
                        </v-btn>
                      </v-col>
                      
                      <v-col cols="6">
                        <v-btn 
                          color="primary" 
                          block
                          @click="resetPayment"
                          rounded="lg"
                          elevation="3"
                          height="48"
                          class="slide-button"
                        >
                          <v-icon start>mdi-qrcode-scan</v-icon>
                          New Payment
                        </v-btn>
                      </v-col>
                    </v-row>
                  </v-card-text>
                </v-card>
              </div>
            </div>
            
            <!-- ERROR STEP -->
            <div v-else-if="currentStep === 'error'" key="error">
              <div class="card-wrapper">
                <v-card class="card-container" elevation="0">
                  <div class="card-decoration top-right"></div>
                  <div class="card-decoration bottom-left"></div>
                  
                  <v-card-text class="text-center pa-6">
                    <v-avatar
                      color="error"
                      size="80"
                      class="mb-4 error-icon"
                    >
                      <v-icon icon="mdi-alert" size="40" color="white"></v-icon>
                    </v-avatar>
                    
                    <h2 class="text-h5 font-weight-bold mb-2" style="color: var(--error);">Payment Failed</h2>
                    <p class="text-subtitle-2 mb-6">{{ error || 'Something went wrong with your transaction.' }}</p>
                    
                    <v-alert
                      type="info"
                      variant="tonal"
                      icon="mdi-information"
                      class="mb-6 text-left"
                      border="start"
                    >
                      <div class="font-weight-bold">Your account has not been charged</div>
                      <div class="text-body-2">Your wallet balance remains at {{ formattedBalance }}</div>
                    </v-alert>
                    
                    <v-row>
                      <v-col cols="6">
                        <v-btn 
                          color="grey" 
                          variant="outlined" 
                          block
                          @click="goToHome"
                          rounded="lg"
                          elevation="1"
                          height="48"
                        >
                          <v-icon start>mdi-home</v-icon>
                          Home
                        </v-btn>
                      </v-col>
                      
                      <v-col cols="6">
                        <v-btn 
                          color="primary" 
                          block
                          @click="resetPayment"
                          rounded="lg"
                          elevation="3"
                          height="48"
                          class="slide-button"
                        >
                          <v-icon start>mdi-refresh</v-icon>
                          Try Again
                        </v-btn>
                      </v-col>
                    </v-row>
                  </v-card-text>
                </v-card>
              </div>
            </div>
          </transition-group>
        </v-container>
        
        <!-- Snackbar for notifications -->
        <v-snackbar
          v-model="showSnackbar"
          :color="snackbarColor"
          location="bottom"
          :timeout="3000"
          rounded="pill"
        >
          <div class="d-flex align-center">
            <v-icon 
              :icon="
                snackbarColor === 'success' ? 'mdi-check-circle' : 
                snackbarColor === 'error' ? 'mdi-alert-circle' : 
                'mdi-information'"
              class="mr-2" 
              size="small"
            ></v-icon>
            {{ snackbarText }}
          </div>
        </v-snackbar>
        
        <!-- Manual Entry Dialog -->
        <v-dialog v-model="manualEntryDialog" max-width="500" transition="dialog-bottom-transition">
          <v-card>
            <v-toolbar :color="theme.primary" dark>
              <v-toolbar-title>
                <v-icon start>mdi-keyboard</v-icon>
                Manual Payment Entry
              </v-toolbar-title>
              <v-spacer></v-spacer>
              <v-btn icon @click="manualEntryDialog = false">
                <v-icon>mdi-close</v-icon>
              </v-btn>
            </v-toolbar>
            
            <v-card-text class="pt-6">
              <v-form>
                <v-text-field
                  v-model="manualMerchantName"
                  label="Merchant Name"
                  variant="outlined"
                  density="comfortable"
                  prepend-inner-icon="mdi-store"
                  :rules="[v => !!v || 'Merchant name is required']"
                  required
                  class="mb-4"
                ></v-text-field>
                
                <v-text-field
                  v-model="manualAmount"
                  label="Amount (RM)"
                  variant="outlined"
                  density="comfortable"
                  prepend-inner-icon="mdi-cash"
                  type="number"
                  :rules="[
                    v => !!v || 'Amount is required',
                    v => v > 0 || 'Amount must be greater than 0'
                  ]"
                  required
                ></v-text-field>
                
                <div class="text-caption mt-4">
                  <v-icon start size="x-small" color="info">mdi-information</v-icon>
                  Enter the payment details manually when QR code scanning is not available.
                </div>
              </v-form>
            </v-card-text>
            
            <v-divider></v-divider>
            
            <v-card-actions class="pa-4">
              <v-spacer></v-spacer>
              
              <v-btn
                color="grey"
                variant="text"
                @click="manualEntryDialog = false"
              >
                Cancel
              </v-btn>
              
              <v-btn
                color="primary"
                @click="submitManualEntry"
                :disabled="!manualMerchantName || !manualAmount || parseFloat(manualAmount) <= 0"
                variant="elevated"
              >
                <v-icon start>mdi-check</v-icon>
                Submit
              </v-btn>
            </v-card-actions>
          </v-card>
        </v-dialog>
      </div>
    `
  }
  