export default {
  props: ["jsonData"],
  data() {
    return {
      apiData: null,

      moduleName: "Food & Beverage",
      moduleDescription: "Order your favorite meals and drinks",

      module: "food_order",
      method: "view",

      // Product categories
      categories: [],

      // Product list
      products: [],

      // Current selected category
      selectedCategory: null,

      // Shopping cart
      cart: [],

      // Order information
      orderInfo: {
        table_number: "",
        customer_name: "",
        phone: "",
        note: "",
        dining_type: "dine_in"
      },

      selectedPayment: 1,

      // Dialog controls
      showOrderDialog: false,
      showPaymentDialog: false,
      showCartSheet: false,


      showProductDetailSheet: false,
      productInDetailView: null,
      productOptionsData: [],
      selectedOptions: [],

      loading: false,

      // Mobile view state
      currentView: 'menu', // 'menu' or 'cart'


      orderSuccessDialog: false,
      orderSuccessDetails: {
        orderId: '',
        estimatedTime: '',
        newBalance: 0,
        diningType: ''
      },


      // scan
      scanInProgress: false,
      scanResultMessage: '',
      showScanResult: false,

      customerLookupLoading: false,

      printInProgress: false,
      printResultMessage: '',
      showPrintResult: false,
      printTimeout: null,

      // Member-specific data
      memberData: null,
      userLoggedIn: false,

    };
  },

  computed: {
    // Filter products
    filteredProducts() {
      if (!this.selectedCategory) return [];
      return this.products.filter(product => product.category_id === this.selectedCategory);
    },

    // Cart item count
    cartItemCount() {
      return this.cart.reduce((total, item) => total + item.quantity, 0);
    },

    // Cart total amount
    cartTotal() {
      return this.cart.reduce((total, item) => total + (item.price * item.quantity), 0);
    },

    // Calculate total price including selected options
    currentProductPriceWithOptions() {
      let basePrice = this.productInDetailView ? this.productInDetailView.price : 0;
      let optionsPrice = this.selectedOptions.reduce((sum, option) => sum + option.price, 0);
      return basePrice + optionsPrice;
    },

    // Formatted total amount
    formattedTotal() {
      return `RM ${this.cartTotal.toFixed(2)}`;
    }
  },


  watch: {

    'orderInfo.customer_name': {
      handler(newValue, oldValue) {

        if (newValue !== oldValue && newValue !== null && newValue.trim() !== '') {
          this.lookupCustomer(newValue);
        } else if (newValue === null || newValue.trim() === '') {

          this.customerDetails = { name: "", email: "", contact: "" };
          this.orderInfo.phone = "";
        }
      },

      immediate: false
    }
  },


  async mounted() {

    this.customerDetails = {
      name: "",
      email: "",
      contact: ""
    };

    // Load member data from local storage
    await this.loadMemberData();

    await this.loadData();

    if (this.categories.length > 0) {
      this.selectedCategory = this.categories[0].id;
    }


    // Set up the scan callback to handle results from Android
    this.setupScanCallback();
  },

  methods: {
    async loadMemberData() {
      // Load member data from localStorage
      const userId = localStorage.getItem('id');
      const fullname = localStorage.getItem('fullname');
      const memberNo = localStorage.getItem('member_no');
      const email = localStorage.getItem('email');
      const contact = localStorage.getItem('contact');
      const memberType = localStorage.getItem('member_type');

      if (userId && fullname) {
        this.memberData = {
          id: userId,
          name: fullname,
          member_code: memberNo || '',
          member_type: memberType || 'member',
          email: email || '',
          contact: contact || ''
        };

        this.userLoggedIn = true;
        this.orderInfo.customer_name = this.memberData.member_code;
        this.orderInfo.phone = this.memberData.contact;
        this.customerDetails = {
          name: this.memberData.name,
          email: this.memberData.email,
          contact: this.memberData.contact
        };

        console.log('Member data loaded from localStorage:', this.memberData);
      } else {
        console.log('No member data found in localStorage');
      }
    },

    async loadData() {
      this.loading = true;

      try {

        const queryData = {};

        const responseBody = await this.$root.callApi(this.module, this.method, queryData);

        if (responseBody.hasOwnProperty("data")) {
          this.apiData = responseBody.data;


          const actualData = this.apiData.data || this.apiData;

          console.log('API Data received:', actualData);


          if (actualData.categories) {
            this.categories = actualData.categories.map(category => ({
              id: parseInt(category.id),
              name: category.name,
              icon: category.icon ? this.getImageUrl(category.icon) : this.getCategoryIcon(category.name),
              iconType: category.icon ? 'image' : 'mdi',
              description: category.description
            }));

            console.log('Categories processed:', this.categories);
          }


          if (actualData.products) {
            this.products = actualData.products.map(product => ({
              id: parseInt(product.id),
              name: product.name,
              description: product.description,
              price: parseFloat(product.price),
              category_id: parseInt(product.category_id),
              image: this.getProductImage(product.image, product.name),
              status: product.status
            }));
          }

          this.loading = false;
        } else {
          alert("Failed to Load Data");
          this.loading = false;
        }
      } catch (error) {
        console.error('Error loading data:', error);
        alert("Error loading data: " + error.message);
        this.loading = false;
      }
    },


    getImageUrl(imagePath) {
      if (!imagePath) return '';

      // Check if imagePath is already a full URL
      if (imagePath.startsWith('http')) {
        return imagePath;
      }

      // Prepend base URL for relative paths - updated for member app
      const baseUrl = 'https://pcloudsb.tplinkdns.com/thetapangclub/pos/media';
      return `${baseUrl}/${imagePath}`;
    },


    getCategoryIcon(categoryName) {

      const iconMap = {
        'Hot Drinks': 'mdi-coffee',
        'Cold Drinks': 'mdi-cup',
        'Main Courses': 'mdi-food',
        'Desserts': 'mdi-cake'
      };

      return iconMap[categoryName] || 'mdi-food'; // default icon
    },


    getProductImage(apiImagePath, productName) {

      if (apiImagePath && apiImagePath.trim() !== '') {
        return this.getImageUrl(apiImagePath);
      }
      return ''; // If no valid image path, return empty string or a default placeholder
    },


    // Select category
    selectCategory(categoryId) {
      this.selectedCategory = categoryId;
    },


    async openProductDetailSheet(product) {
      this.showProductDetailSheet = true;

      // Prepare the data to send to the 'view_option' method
      const queryData = {
        product_id: product.id
      };

      try {
        const responseBody = await this.$root.callApi(this.module, "view_option", queryData);

        if (responseBody.status === "YES" && responseBody.data) {

          this.productInDetailView = {
            ...responseBody.data.product,
            image: this.getImageUrl(responseBody.data.product.image)
          };
          this.productOptionsData = responseBody.data.options || [];
          this.selectedOptions = [];
          console.log('API Data for product options received:', responseBody.data);
        } else {
          console.error("Failed to load product options data:", responseBody.message);
          alert("Failed to load product details: " + responseBody.message);
          this.showProductDetailSheet = false; // Close sheet on error
        }
      } catch (error) {
        console.error('Error calling view_option API:', error);
        alert("Error loading product details: " + error.message);
        this.showProductDetailSheet = false; // Close sheet on error
      }
    },


    addToCartFromDetailView() {

      const itemToAdd = {
        ...this.productInDetailView,
        quantity: 1,
        selectedOptions: JSON.parse(JSON.stringify(this.selectedOptions)),
        price: this.currentProductPriceWithOptions
      };

      this.addToCart(itemToAdd);

      this.showProductDetailSheet = false;
      this.selectedOptions = [];
      this.productOptionsData = [];
    },


    addToCart(itemToAdd) {

      const existingItemIndex = this.cart.findIndex(cartItem =>
        cartItem.id === itemToAdd.id &&
        JSON.stringify(cartItem.selectedOptions.map(o => o.id).sort()) === JSON.stringify(itemToAdd.selectedOptions.map(o => o.id).sort())
      );


      if (existingItemIndex !== -1) {
        this.cart[existingItemIndex].quantity += 1;
      } else {
        this.cart.push(itemToAdd);
      }


      this.$nextTick(() => {
        // toast notification here
      });
    },

    // Increase quantity
    increaseQuantity(item) {
      item.quantity += 1;
    },

    // Decrease quantity
    decreaseQuantity(item) {
      if (item.quantity > 1) {
        item.quantity -= 1;
      } else {
        this.removeFromCart(item);
      }
    },

    getFormattedPrice(item) {
      return `RM ${Number(item.price || 0).toFixed(2)}`;
    },
    getTotalPrice(item) {
      return `RM ${(Number(item.price || 0) * item.quantity).toFixed(2)}`;
    },

    // Remove from cart
    removeFromCart(item) {
      const index = this.cart.findIndex(cartItem => cartItem.id === item.id && JSON.stringify(cartItem.selectedOptions.map(o => o.id).sort()) === JSON.stringify(item.selectedOptions.map(o => o.id).sort()));
      if (index !== -1) {
        this.cart.splice(index, 1);
      }
    },

    formatMoney(amount) {
      return `RM ${Number(amount || 0).toFixed(2)}`;
    },

    // Clear cart
    clearCart() {
      this.cart = [];
    },

    // Toggle cart bottom sheet
    toggleCartSheet() {
      this.showCartSheet = !this.showCartSheet;
    },

    // Open order dialog
    openOrderDialog() {
      if (this.cart.length === 0) {
        alert("Cart is empty, please select items first");
        return;
      }
      this.showCartSheet = false;
      this.showOrderDialog = true;
    },

    // Close order dialog
    closeOrderDialog() {
      this.showOrderDialog = false;
    },

    // Submit order - updated for member experience
    submitOrder() {
      // Get session data from sessionStorage
      const sessionId = sessionStorage.getItem('id');
      const sessionSecret = sessionStorage.getItem('session_secret');

      // Check if user has valid session or member data
      if (!sessionId || !sessionSecret) {
        if (!this.memberData && !this.orderInfo.customer_name) {
          alert("Please log in to place an order");
          return;
        }
      }

      // Validate table number for dine-in orders
      if (this.orderInfo.dining_type === 'dine_in') {
        if (!this.orderInfo.table_number || this.orderInfo.table_number.trim() === '') {
          alert("Table number is required for dine-in orders. Please enter a table number.");
          return;
        }
        if (!Number.isInteger(Number(this.orderInfo.table_number)) || Number(this.orderInfo.table_number) <= 0) {
          alert("Please enter a valid table number (positive integer).");
          return;
        }
      }

      this.showOrderDialog = false;
      this.showPaymentDialog = true;
    },



    async processPayment() {
      // Data to send to the backend - updated for member app
      const orderData = {
        items: this.cart,
        total: this.cartTotal,
        orderInfo: {
          ...this.orderInfo,
          member_id: this.memberData ? this.memberData.id : null,
          is_member_order: true
        },
        paymentMethod: this.selectedPayment,
        timestamp: new Date().toISOString()
      };

      console.log("Sending member order data to backend:", orderData);

      try {
        const responseBody = await this.$root.callApi(this.module, "member_order", orderData);


        if (responseBody.data && responseBody.data.status === "YES") {

          const orderData = responseBody.data.data;
          const orderId = orderData.order_id;
          const newBalance = orderData.new_wallet_balance;
          const estimatedTime = orderData.estimated_time;

          console.log("Member order created successfully with ID:", orderId);
          console.log("New wallet balance:", newBalance);

          // Store order details for the success modal
          this.orderSuccessDetails = {
            orderId: orderId,
            estimatedTime: estimatedTime,
            newBalance: newBalance,
            diningType: orderData.dining_type
          };

          // Reset data after successful order
          this.cart = [];
          this.showPaymentDialog = false;
          this.orderSuccessDialog = true;

          // Update member balance if available
          if (this.memberData) {
            this.memberData.balance = newBalance;
          }

          // Reload data
          await this.loadData();

        } else if (responseBody.data && responseBody.data.status === "NO") {
          const errorMessage = responseBody.data.message || 'Unknown error occurred.';
          alert(`Order submission failed: ${errorMessage}`);
          console.error("Order API error:", errorMessage);
        } else {
          const errorMessage = responseBody.message || 'Unknown error occurred.';
          alert(`Order submission failed: ${errorMessage}`);
          console.error("Order API error:", errorMessage);
        }

      } catch (error) {

        alert(`An error occurred during order submission: ${error.message}`);
        console.error("Order API call failed:", error);
      }
    },

    // Cancel payment
    cancelPayment() {
      this.showPaymentDialog = false;
    },

    // Get category name
    getCategoryName(categoryId) {
      const category = this.categories.find(c => c.id === categoryId);
      return category ? category.name : "";
    },

    closeSuccessDialog() {
      this.orderSuccessDialog = false;
    },




    async lookupCustomer(customerCode) {
      if (!customerCode || customerCode.trim() === '') {
        // Clear customer details if no code
        this.customerDetails = { name: "", email: "", contact: "" };
        this.orderInfo.phone = "";
        return;
      }

      this.customerLookupLoading = true;


      const queryData = { customer_code: customerCode.trim() };

      try {
        const responseBody = await this.$root.callApi(this.module, "member_info", queryData);


        if (responseBody.data && responseBody.data.status === "YES" && responseBody.data.data) {
          const customer = responseBody.data.data;

          this.customerDetails = {
            name: customer.name || "",
            email: customer.email || "",
            contact: customer.contact || ""
          };
          this.orderInfo.phone = customer.contact || "";
          console.log('Customer lookup successful:', this.customerDetails, 'Phone:', this.orderInfo.phone);
        }
        else {
          console.log('Customer not found or API error:', responseBody.message);
          this.customerDetails = { name: "", email: "", contact: "" };
          this.orderInfo.phone = "";
        }
      } catch (error) {
        console.error('Error looking up customer:', error);
        this.customerDetails = { name: "", email: "", contact: "" };
        this.orderInfo.phone = "";
      } finally {
        this.customerLookupLoading = false;
      }
    },

    // Watch for changes in customer_name and trigger lookup
    async onCustomerCodeChange() {
      await this.lookupCustomer(this.orderInfo.customer_name);
    },

    // call android
    onScan() {
      console.log('onScan method called');
      this.scanInProgress = true;
      this.scanResultMessage = 'Starting QR code scanner...';
      this.showScanResult = true;

      if (typeof AndroidBridge !== 'undefined') {
        console.log('AndroidBridge is available, calling onScan()');

        try {
          AndroidBridge.onScan();
          console.log('AndroidBridge.onScan() called successfully');
        } catch (error) {
          console.error('Error calling AndroidBridge.onScan():', error);
          this.scanInProgress = false;
          this.scanResultMessage = 'Error calling scanner: ' + error.message;
        }

        // Add timeout to handle cases where callback never comes
        setTimeout(() => {
          if (this.scanInProgress) {
            console.log('Scan timeout reached');
            this.scanInProgress = false;
            this.scanResultMessage = 'Scan timeout - please try again';
            this.showScanResult = true;
            setTimeout(() => {
              this.showScanResult = false;
            }, 3000);
          }
        }, 30000); // 30 second timeout

      } else {
        console.log('AndroidBridge is not available');
        this.scanInProgress = false;
        this.scanResultMessage = 'Android bridge not available. Make sure you\'re running in the app\'s WebView.';
        this.showScanResult = true;
      }
    },

    cancelScan() {
      if (this.scanInProgress) {
        this.scanInProgress = false;
        this.scanResultMessage = 'Scan cancelled by user';
        this.showScanResult = true;
        setTimeout(() => {
          this.showScanResult = false;
        }, 2000);
      }
    },

    // New method to set up the scan callback
    setupScanCallback() {
      // Set up the global callback function that Android will call
      const self = this;

      window.onScanResult = function (result) {
        console.log('Scan result received:', result);
        self.handleScanResult(result);
      };

      // Also try alternative callback names in case Android uses different naming
      window.onScanCallback = function (result) {
        console.log('Scan callback received:', result);
        self.handleScanResult(result);
      };

      window.scanCallback = function (result) {
        console.log('scanCallback received:', result);
        self.handleScanResult(result);
      };

      // Debug: Log when callbacks are set up
      console.log('QR Scan callbacks set up successfully');
    },

    // Updated method to handle scan results based on YES/NO status
    handleScanResult(result) {
      console.log('Processing scan result:', result);
      this.scanInProgress = false;

      // Handle different result formats
      let status, message;

      if (typeof result === 'string') {
        // If result is just a string, treat it as successful scan
        status = 'YES';
        message = result;
      } else if (result && typeof result === 'object') {
        // If result is an object, extract status and message
        status = result.status || 'YES';
        message = result.message || result.data || result.result || result;
      } else {
        // Fallback for unexpected formats
        status = 'NO';
        message = 'Unknown result format';
      }

      if (status === 'YES') {
        // Success - we got a QR code result
        this.scanResultMessage = `Scanned successfully: ${message}`;
        this.showScanResult = true;

        // Process the scanned data
        this.processScanResult(message);

        // Hide the snackbar after 3 seconds
        setTimeout(() => {
          this.showScanResult = false;
        }, 3000);

      } else {
        // Error or cancelled - status is 'NO'
        this.scanResultMessage = `Scan failed: ${message}`;
        this.showScanResult = true;

        // Hide the error message after 5 seconds
        setTimeout(() => {
          this.showScanResult = false;
        }, 5000);
      }
    },

    // Updated method to process the actual scan result data
    async processScanResult(scannedData) {
      try {
        const qrData = JSON.parse(scannedData);
        if (qrData.customer_code) {
          this.orderInfo.customer_name = qrData.customer_code;
        } else if (qrData.member_id) {
          this.orderInfo.customer_name = qrData.member_id;
        }
      } catch (e) {
        // If not JSON, treat as plain text customer code
        this.orderInfo.customer_name = scannedData.trim();
      }

      // Trigger customer lookup
      await this.lookupCustomer(this.orderInfo.customer_name);
    },

    // New method to clear scan result
    clearScanResult() {
      this.showScanResult = false;
      this.scanResultMessage = '';
    },

    // Navigation methods for member app
    goBack() {
      this.$root.navigateTo('home');
    },

    viewOrderHistory() {
      this.$root.navigateTo('order_history');
    },

  },

  template: `
    <v-app>
      <v-main>
        <v-app-bar
          color="#2E3B2E"
          dark
          fixed
          app
          height="64"
          elevation="0"
        >
          <v-btn icon @click="goBack" class="mr-2">
            <v-icon>mdi-arrow-left</v-icon>
          </v-btn>
          
          <v-toolbar-title class="text-h6 font-weight-bold">
            {{ moduleName }}
          </v-toolbar-title>
          <v-spacer></v-spacer>

        <div
              v-if="cartItemCount > 0"
              class="d-flex align-center mr-2 cart-summary-header"
              @click="toggleCartSheet"
              style="cursor: pointer; padding: 6px 16px; border-radius: 24px; background: rgba(255,255,255,0.15);"
          >
          <v-badge
              :content="cartItemCount"
              color="error"
              text-color="white"
              location="top end"
              offset-x="4"
              offset-y="4"
              bordered
          >
              <v-icon color="white" size="24">mdi-cart</v-icon>
          </v-badge>

          <span class="text-body-1 font-weight-bold ml-3" style="letter-spacing: 0.5px;">
              {{ formattedTotal }}
          </span>
          </div>

        </v-app-bar>



        <v-container fluid class="pa-2" :style="cartItemCount > 0 ? { paddingTop: '130px' } : { paddingTop: '80px' }">
          
          <!-- Member Welcome Section -->
          <v-card v-if="memberData" class="mb-4" color="primary" dark>
            <v-card-text class="pa-4">
              <div class="d-flex align-center">
                <v-avatar size="48" class="mr-3">
                  <v-icon size="32">mdi-account-circle</v-icon>
                </v-avatar>
                <div>
                  <div class="text-h6 font-weight-bold">Welcome, {{ memberData.name }}</div>
                  <div class="text-body-2 opacity-80">Member ID: {{ memberData.member_code || memberData.id }}</div>
                </div>
                <v-spacer></v-spacer>
                <v-btn icon small @click="viewOrderHistory">
                  <v-icon>mdi-history</v-icon>
                </v-btn>
              </div>
            </v-card-text>
          </v-card>

          <div v-if="loading" class="text-center pa-8">
            <v-progress-circular indeterminate color="primary" size="64"></v-progress-circular>
            <p class="text-h6 mt-4">Loading menu...</p>
          </div>

          <div v-else>
            <v-chip-group
              v-model="selectedCategory"
              mandatory
              color="primary"
              class="mb-4"
              show-arrows
            >
              <v-chip
                v-for="category in categories"
                :key="category.id"
                :value="category.id"
                class="ma-1"
                small
              >

                <v-img
                  v-if="category.iconType === 'image'"
                  :src="category.icon"
                  width="16"
                  height="16"
                  class="mr-2"
                  :alt="category.name"
                  contain
                ></v-img>
                <v-icon v-else left small>{{ category.icon }}</v-icon>
                {{ category.name }}
              </v-chip>
            </v-chip-group>

           <v-row class="px-2">
            <v-col
                v-for="product in filteredProducts"
                :key="product.id"
                cols="6"
                sm="4"
                md="3"
                class="pa-2"
            >
                <v-card
                class="product-card rounded-xl"
                hover
                @click="openProductDetailSheet(product)" 
                height="180"
                elevation="3"
                >
                <v-card-text class="text-center pa-4">
                    <div
                    class="product-image mb-2 d-flex align-center justify-center"
                    style="border-radius: 16px; background-color: #f0f0f0; width: 56px; height: 56px; margin: 0 auto;"
                    >
                      <v-img
                        v-if="product.image && product.image.startsWith('http')"
                        :src="product.image"
                        width="48"
                        height="48"
                        contain
                        class="rounded-lg"
                      ></v-img>
                      <span v-else style="font-size: 36px;">{{ product.image }}</span>
                    </div>

                    <div
                    class="text-subtitle-2 font-weight-medium mb-2"
                    style="line-height: 1.3;"
                    >
                    {{ product.name }}
                    </div>

                    <div
                    class="text-caption text-grey mb-3"
                    style="font-size: 0.75rem; line-height: 1.2;"
                    >
                    {{ product.description }}
                    </div>

                    <v-chip color="success" x-small dense>
                    RM {{ Number(product.price || 0).toFixed(2) }}
                    </v-chip>
                </v-card-text>
                </v-card>
            </v-col>
            </v-row>
          </div>

        </v-container>

        <v-bottom-sheet v-model="showProductDetailSheet" persistent max-width="500">
            <v-card class="pb-3">
                <v-card-title class="pa-4 primary white--text d-flex align-center">
                    <h3 class="text-h6">{{ productInDetailView ? productInDetailView.name : 'Product Details' }}</h3>
                    <v-spacer></v-spacer>
                    <v-btn icon dark small @click="showProductDetailSheet = false">
                        <v-icon>mdi-close</v-icon>
                    </v-btn>
                </v-card-title>

                <v-card-text v-if="productInDetailView" class="pa-4 text-center">
                    <div
                        class="product-detail-image mb-4 d-flex align-center justify-center"
                        style="border-radius: 16px; background-color: #f0f0f0; width: 120px; height: 120px; margin: 0 auto;"
                    >
                        <v-img
                            v-if="productInDetailView.image && productInDetailView.image.startsWith('http')"
                            :src="productInDetailView.image"
                            width="100"
                            height="100"
                            contain
                            class="rounded-lg"
                        ></v-img>
                        <span v-else style="font-size: 64px;">{{ productInDetailView.image }}</span>
                    </div>

                    <h2 class="text-h5 font-weight-bold mb-2">{{ productInDetailView.name }}</h2>
                    <p class="text-body-1 text-grey-darken-1 mb-4">{{ productInDetailView.description }}</p>

                    <v-chip color="success" size="large" class="mb-4">
                        <span class="text-h6 font-weight-bold">RM {{ Number(productInDetailView.price || 0).toFixed(2) }}</span>
                    </v-chip>

                    <v-divider class="my-4"></v-divider>

                    <div v-if="productOptionsData && productOptionsData.length > 0">
                        <h3 class="text-h6 font-weight-medium text-left mb-3">Add-ons & Options:</h3>
                        <v-list dense class="pa-0">
                            <v-list-item v-for="option in productOptionsData" :key="option.id" class="px-0">
                                <v-checkbox
                                    v-model="selectedOptions"
                                    :label="option.name"
                                    :value="option"
                                    hide-details
                                    class="ma-0 pa-0 flex-grow-1"
                                >
                                    <template v-slot:append>
                                        <span class="font-weight-medium text-primary text-subtitle-1">
                                            + RM {{ Number(option.price || 0).toFixed(2) }}
                                        </span>
                                    </template>
                                </v-checkbox>
                            </v-list-item>
                        </v-list>
                        <v-divider class="my-4"></v-divider>
                    </div>
                    <div class="d-flex justify-space-between align-center mb-4">
                        <span class="text-h6 font-weight-bold">Total with options:</span>
                        <span class="text-h5 font-weight-bold primary--text">RM {{ currentProductPriceWithOptions.toFixed(2) }}</span>
                    </div>

                </v-card-text>

                  <v-card-actions class="pa-6 pt-1">
                    <v-btn
                      color="primary"
                      block 
                      large
                      height="64px"
                       variant="elevated"
                      rounded="lg"
                      elevation="2"
                      @click="addToCartFromDetailView()"
                    >
                      <v-icon left>mdi-cart-plus</v-icon>
                      Add to Cart (RM {{ currentProductPriceWithOptions.toFixed(2) }})
                    </v-btn>
                  </v-card-actions>


                
            </v-card>

        </v-bottom-sheet>


        <v-bottom-sheet v-model="showCartSheet" persistent max-width="500">
          <v-card class="pb-3">
            <v-card-title class="pa-4 primary white--text d-flex align-center">
              <v-icon left color="white">mdi-cart</v-icon>
              <h3 class="text-h6">Cart ({{ cartItemCount }} items)</h3>
              <v-spacer></v-spacer>
              <v-btn icon dark small @click="showCartSheet = false">
                <v-icon>mdi-close</v-icon>
              </v-btn>
            </v-card-title>

            <div v-if="cart.length === 0" class="text-center pa-8">
              <v-icon size="64" color="grey lighten-2">mdi-cart-outline</v-icon>
              <p class="text-h6 text-grey mt-4 mb-2">Your cart is empty</p>
              <p class="text-grey">Add some delicious items to get started!</p>
            </div>

            <div v-else>
             <v-card-text class="pa-0" style="max-height: 50vh; overflow-y: auto;">
              <v-list class="pa-0">
                 <v-list-item
                      v-for="(item, index) in cart"
                      :key="item.id + '-' + item.selectedOptions.map(o => o.id).sort().join(',')" class="px-4 py-3"
                      :class="{ 'bg-grey-lighten-5': index % 2 === 1 }"
                      style="border-bottom: 1px solid #eee;"
                  >
                      <div class="d-flex flex-row align-center" style="width: 100%;">
                          <div class="mr-3">
                              <v-avatar size="48" class="elevation-1">
                                  <div class="d-flex align-center justify-center" style="width: 48px; height: 48px; background: #f5f5f5; border-radius: 8px;">
                                      <v-img
                                        v-if="item.image && item.image.startsWith('http')"
                                        :src="item.image"
                                        width="40"
                                        height="40"
                                        contain
                                        class="rounded"
                                      ></v-img>
                                      <span v-else style="font-size: 22px;">{{ item.image }}</span>
                                  </div>
                              </v-avatar>
                          </div>

                          <div class="flex-grow-1">
                              <div class="d-flex justify-space-between align-center">
                                  <div class="text-subtitle-2 font-weight-medium">{{ item.name }}</div>
                                  <v-btn icon x-small color="grey" @click="removeFromCart(item)">
                                      <v-icon x-small>mdi-delete</v-icon>
                                  </v-btn>
                              </div>

                              <div class="mt-1 text-body-2 mb-2">
                                  <span class="text-success font-weight-medium">{{ getFormattedPrice(item) }}</span>
                                  <span class="mx-1">×</span>
                                  <span class="font-weight-medium">{{ item.quantity }}</span>
                                  <span class="mx-1">=</span>
                                  <span class="text-primary font-weight-bold">{{ getTotalPrice(item) }}</span>
                              </div>

                              <div v-if="item.selectedOptions && item.selectedOptions.length > 0" class="text-caption text-grey mt-1">
                                  Options: <span v-for="(option, optIndex) in item.selectedOptions" :key="option.id">{{ option.name }} (RM {{ Number(option.price).toFixed(2) }})<span v-if="optIndex < item.selectedOptions.length - 1">, </span></span>
                              </div>


                              <div class="d-flex justify-space-between align-center">
                                <!-- Quantity Controls -->
                                <div class="d-flex align-center bg-grey-lighten-4 rounded-pill pa-1">
                                  <v-btn
                                    icon
                                    size="small"
                                    color="error"
                                    variant="flat"
                                    @click="decreaseQuantity(item)"
                                    class="rounded-circle"
                                    style="width: 32px; height: 32px;"
                                  >
                                    <v-icon size="16">mdi-minus</v-icon>
                                  </v-btn>

                                  <div class="px-3 text-body-1 font-weight-bold" style="min-width: 40px; text-align: center;">
                                    {{ item.quantity }}
                                  </div>

                                  <v-btn
                                    icon
                                    size="small"
                                    color="success"
                                    variant="flat"
                                    @click="increaseQuantity(item)"
                                    class="rounded-circle"
                                    style="width: 32px; height: 32px;"
                                  >
                                    <v-icon size="16">mdi-plus</v-icon>
                                  </v-btn>
                                </div>
                              </div>


                          </div>
                      </div>
                  </v-list-item>
              </v-list>
              </v-card-text>


              <v-divider></v-divider>
              <v-card-actions class="pa-4">
                <div class="w-100">
                  <div class="d-flex justify-space-between align-center mb-3">
                    <span class="text-h6 font-weight-medium">Total Amount:</span>
                    <span class="text-h5 font-weight-bold text-success">{{ formattedTotal }}</span>
                  </div>

                 <div class="d-flex flex-row align-center mt-4">
                  <v-btn
                    color="error"
                    variant="elevated"
                    size="large"
                    rounded="lg"
                    class="flex-grow-1 mr-2"
                    @click="clearCart"
                    style="flex: 2;"
                    elevation="3"
                  >
                    <v-icon left small>mdi-delete</v-icon>
                    Clear All
                  </v-btn>

              

                  <v-btn
                    color="primary"
                    variant="elevated"
                    size="large"
                    rounded="lg"
                    @click="openOrderDialog"
                    style="flex: 2;"
                    elevation="3"
                  >
                    <v-icon left small>mdi-check-circle</v-icon>
                    Place Order
                  </v-btn>


                </div>
              </v-card-actions>
            </div>
          </v-card>
        </v-bottom-sheet>

        <v-dialog v-model="showOrderDialog" fullscreen hide-overlay transition="dialog-bottom-transition">
          <v-card>
            <v-toolbar color="primary" dark>
              <v-btn icon dark @click="closeOrderDialog">
                <v-icon>mdi-close</v-icon>
              </v-btn>
              <v-toolbar-title>Order Confirmation</v-toolbar-title>
            </v-toolbar>

            <v-card-text class="pa-4">
              <v-form>

                <!-- Member Info Display (read-only) -->
                <v-card v-if="memberData" class="mb-4" elevation="2" rounded="lg">
                  <v-card-text class="pa-4">
                    <div class="d-flex align-center mb-3">
                      <v-icon color="primary" class="mr-3">mdi-account-check</v-icon>
                      <span class="text-h6 font-weight-medium">Member Information</span>
                    </div>
                    <div class="pa-3 bg-success-lighten-5 rounded">
                      <div class="text-body-2">
                        <div><strong>Name:</strong> {{ memberData.name }}</div>
                        <div><strong>Member ID:</strong> {{ memberData.member_code || memberData.id }}</div>
                        <div v-if="memberData.email"><strong>Email:</strong> {{ memberData.email }}</div>
                        <div v-if="memberData.contact"><strong>Contact:</strong> {{ memberData.contact }}</div>
                      </div>
                    </div>
                  </v-card-text>
                </v-card>

                <!-- Table Selection for Members -->
                <v-card class="mb-4" elevation="2" rounded="lg">
                  <v-card-text class="pa-4">
                    <div class="d-flex align-center mb-3">
                      <v-icon color="primary" class="mr-3">mdi-table-furniture</v-icon>
                      <span class="text-h6 font-weight-medium">Dining Preference</span>
                    </div>
                    <v-radio-group v-model="orderInfo.dining_type" class="mt-2">
                      <v-radio label="Dine In - I'll select a table" value="dine_in"></v-radio>
                      <v-radio label="Takeaway - For pickup" value="takeaway"></v-radio>
                    </v-radio-group>
                    
                    <v-text-field
                      v-if="orderInfo.dining_type === 'dine_in'"
                      v-model="orderInfo.table_number"
                      label="Table Number *"
                      placeholder="Enter your table number"
                      variant="outlined"
                      type="number"
                      density="comfortable"
                      prepend-inner-icon="mdi-numeric"
                      class="mt-2"
                      required
                      :rules="[v => !!v || 'Table number is required for dine-in orders', v => (v && Number.isInteger(Number(v)) && Number(v) > 0) || 'Please enter a valid table number']"
                    ></v-text-field>
                  </v-card-text>
                </v-card>

                <!-- Order Notes -->
                <v-card class="mb-4" elevation="2" rounded="lg">
                  <v-card-text class="pa-4">
                    <div class="d-flex align-center mb-3">
                      <v-icon color="primary" class="mr-3">mdi-note-text</v-icon>
                      <span class="text-h6 font-weight-medium">Special Instructions</span>
                    </div>
                    <v-textarea
                      v-model="orderInfo.note"
                      label="Any special requests?"
                      placeholder="e.g., No spicy, extra sauce, etc."
                      variant="outlined"
                      rows="3"
                      density="comfortable"
                    ></v-textarea>
                  </v-card-text>
                </v-card>

              </v-form>

             <v-card outlined class="mb-3">
                <v-card-text class="pt-3 pb-2 px-4">
                  <div class="d-flex align-center mb-3">
                    <v-icon color="primary" class="mr-2" size="20">mdi-cart</v-icon>
                    <span class="text-h6 font-weight-bold">Order Summary</span>
                    <v-spacer></v-spacer>
                  </div>

                  <div
                    v-for="item in cart"
                    :key="item.id + '-' + item.selectedOptions.map(o => o.id).sort().join(',')"
                    class="d-flex justify-space-between mb-1"
                  >
                    <span class="font-weight-bold">
                      {{ item.name }} × {{ item.quantity }}
                      <span
                        v-if="item.selectedOptions && item.selectedOptions.length > 0"
                        class="text-caption text-grey ml-1 font-weight-regular"
                      >
                        (
                        <span
                          v-for="(option, optIndex) in item.selectedOptions"
                          :key="option.id"
                        >
                          {{ option.name }}<span v-if="optIndex < item.selectedOptions.length - 1">, </span>
                        </span>
                        )
                      </span>
                    </span>
                    <span class="font-weight-bold">{{ formatMoney(item.price * item.quantity) }}</span>
                  </div>

                  <v-divider class="my-3"></v-divider>

                  <div class="d-flex justify-space-between text-h6 font-weight-bold">
                    <span>Total:</span>
                    <span class="text-success">{{ formattedTotal }}</span>
                  </div>
                </v-card-text>
              </v-card>

            </v-card-text>

            <v-card-actions class="pa-4" style="border-top: 1px solid #eee;">
            <v-spacer></v-spacer>

            <v-btn
              variant="outlined"
              color="grey"
              class="mr-3"
              rounded="lg"
              size="large"
              @click="closeOrderDialog"
            >
              Cancel
            </v-btn>

            <v-btn
              color="primary"
              variant="elevated"
              rounded="lg"
              elevation="2"
              size="large"
              @click="submitOrder"
            >
              Confirm Order
            </v-btn>
          </v-card-actions>

          </v-card>
        </v-dialog>

        <v-dialog v-model="showPaymentDialog" max-width="500">
          <v-card>
            <v-toolbar color="primary" dark flat>
              <v-toolbar-title class="font-weight-bold">Order Confirmation</v-toolbar-title>
            </v-toolbar>

            <v-card-text class="pa-6 text-center">
              <div class="mb-6">
                <p class="text-h6 text-grey--text text--darken-1 mb-1">Total Amount:</p>
                <h1 class="text-h3 font-weight-bold primary--text mb-2">{{ formattedTotal }}</h1>
                <p class="text-subtitle-1 text-grey--text text--darken-1">
                  {{ orderInfo.dining_type === 'takeaway' ? 'Takeaway Order' : 'Dine In' }}
                  <span v-if="orderInfo.table_number && orderInfo.dining_type === 'dine_in'"> - Table: {{ orderInfo.table_number }}</span>
                </p>
              </div>

              <v-divider class="my-4"></v-divider>

              <p class="text-body-1 text-grey--text text--darken-1">
                Your order will be prepared once confirmed. You'll receive a notification when it's ready.
              </p>

             
            </v-card-text>

            <v-card-actions class="pa-6 mt-auto">
              <v-btn color="grey" text large @click="cancelPayment">
                Cancel
              </v-btn>
              <v-spacer></v-spacer>
             

              <v-btn
              color="primary"
              variant="elevated"
              rounded="lg"
              elevation="2"
              size="large"
              @click="processPayment"
            >


              
                <v-icon left>mdi-check-circle</v-icon>
                Place Order
              </v-btn>
            </v-card-actions>
          </v-card>
        </v-dialog>

    <v-dialog v-model="orderSuccessDialog" max-width="400px" persistent>
      <v-card class="text-center" style="border-radius: 16px;">
        <!-- Success Icon -->
        <div class="pt-6 pb-4">
          <v-icon color="success" size="64">mdi-check-circle-outline</v-icon>
        </div>

        <!-- Title -->
        <v-card-title class="px-4 pb-2 text-h6 font-weight-bold">
          Order Placed Successfully! 🎉
        </v-card-title>

        <!-- Order Details -->
        <v-card-text class="px-4 py-2">
          <!-- Order ID -->
          <div class="mb-3">
            <div class="text-caption text-grey-darken-1 mb-1">Order ID</div>
            <div class="text-h6 font-weight-bold text-primary">{{ orderSuccessDetails.orderId }}</div>
          </div>
          
          <!-- Ready Time -->
          <div class="mb-3">
            <div class="text-caption text-grey-darken-1 mb-1">Estimated Ready Time</div>
            <div class="text-subtitle-1 font-weight-bold">{{ orderSuccessDetails.estimatedTime }}</div>
          </div>
          
          <!-- New Balance -->
          <div class="mb-4">
            <div class="text-caption text-grey-darken-1 mb-1">New Wallet Balance</div>
            <div class="text-h6 font-weight-bold text-success">
              RM {{ Number(orderSuccessDetails.newBalance).toFixed(2) }}
            </div>
          </div>
          
          <!-- Notification Message -->
          <v-alert 
            color="info" 
            variant="tonal" 
            density="compact" 
            class="text-body-2 mb-0"
            style="border-radius: 12px;"
          >
            You will be notified when ready for 
            <strong>{{ orderSuccessDetails.diningType === 'takeaway' ? 'pickup' : 'serving' }}</strong>
          </v-alert>
        </v-card-text>

        <!-- Action Button -->
        <v-card-actions class="px-4 pb-6 pt-2">
          <v-btn
            color="primary"
            variant="elevated"
            block
            size="large"
            rounded="lg"
            @click="closeSuccessDialog"
          >
            Great!
          </v-btn>
        </v-card-actions>
      </v-card>
    </v-dialog>


        <v-snackbar
          v-model="showScanResult"
          :timeout="scanInProgress ? -1 : 5000"
          location="top"
          :color="scanResultMessage.includes('successfully') ? 'success' : 'primary'"
          elevation="6"
          multi-line
          max-width="90%"
        >
          <div class="d-flex align-center">
            <v-progress-circular 
              v-if="scanInProgress" 
              indeterminate 
              size="20" 
              class="mr-3"
            ></v-progress-circular>
            <v-icon v-else class="mr-3">
              {{ scanResultMessage.includes('successfully') ? 'mdi-check-circle' : 'mdi-information' }}
            </v-icon>
            <div class="flex-grow-1">
              <div style="word-wrap: break-word; white-space: pre-wrap;">{{ scanResultMessage }}</div>
            </div>
          </div>
          
          <template v-slot:actions>
            <v-btn
              color="white"
              variant="text"
              @click="clearScanResult"
            >
              Close
            </v-btn>
          </template>
        </v-snackbar>


      </v-main>
    </v-app>
  `
};