import getTropicalStyles from "../css/tropicalBookingStyles.js"; 


export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',
        secondary: '#DDA333',
        accent: '#A3BA95',
        lightBg: '#F8F5F0',
        creamBg: '#EADDCB',
        darkText: '#1E1E1E',
        lightText: '#FFFFFF',
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        muted: '#78909C',
        brown: '#6D4C41',
      },

      selectedDate: today,
      minDateForPicker: today,

      bookingTypes: [],
      selectedType: null,

      facilities: { // Will be populated from API
        sports: [],
        events: [],
        services: [],
        dining: []
      },

      selectedFacility: null,
      timeSlots: [],
      selectedTimeSlot: null,

      myBookings: [ // Data remains the same
        { id: 'bk1001', facilityId: 'tennis1', facilityName: 'Tennis Court 1', date: new Date(new Date().setDate(today.getDate() + 2)).toISOString().substring(0,10), timeSlot: '10:00 - 11:00', status: 'confirmed' },
        { id: 'bk1002', facilityId: 'restaurant', facilityName: 'The Canopy Restaurant', date: new Date(new Date().setDate(today.getDate() + 1)).toISOString().substring(0,10), timeSlot: '19:00 - 21:00', guests: 4, status: 'confirmed' },
        { id: 'bk1003', facilityId: 'spa', facilityName: 'Jungle Spa Retreat', date: new Date(new Date().setDate(today.getDate() + 5)).toISOString().substring(0,10), timeSlot: '14:00 - 15:00', status: 'pending' },
        { id: 'bk1004', facilityId: 'swim', facilityName: 'Swimming Pool', date: new Date(new Date().setDate(today.getDate() - 2)).toISOString().substring(0,10), timeSlot: '11:00 - 12:00', status: 'completed' },
      ],

      bookingForm: { guests: 1, notes: '', duration: 1 },
      loading: false,
      showBookingDialog: false,
      showCancelDialog: false,
      selectedBookingToCancel: null,
      activeTab: 'find',
      formValid: false,
    };
  },
  computed: { // All computed properties remain the same
    availableFacilities() {
      return this.facilities[this.selectedType] || [];
    },
    formattedSelectedDate() {
      if (!this.selectedDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return this.selectedDate.toLocaleDateString('en-US', options);
    },
    upcomingBookings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.myBookings.filter(booking => {
        const bookingDate = new Date(booking.date);
        bookingDate.setHours(0,0,0,0);
        return bookingDate >= todayNormalized && booking.status !== 'cancelled' && booking.status !== 'completed';
      }).sort((a, b) => new Date(a.date) - new Date(b.date) || a.timeSlot.localeCompare(b.timeSlot));
    },
    pastBookings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.myBookings.filter(booking => {
        const bookingDate = new Date(booking.date);
        bookingDate.setHours(0,0,0,0);
        return bookingDate < todayNormalized || booking.status === 'cancelled' || booking.status === 'completed';
      }).sort((a, b) => new Date(b.date) - new Date(a.date) || b.timeSlot.localeCompare(a.timeSlot));
    },
    isToday() {
      if (!this.selectedDate) return false;
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.selectedDate.getTime() === todayNormalized.getTime();
    },
    isPastDate() {
      if (!this.selectedDate) return false;
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.selectedDate < todayNormalized;
    },
    selectedTypeDetails() {
        return this.bookingTypes.find(bt => bt.id === this.selectedType);
    },
    selectedDateAsString() {
        if (!this.selectedDate) return null;
        // Use local timezone instead of UTC
        const year = this.selectedDate.getFullYear();
        const month = String(this.selectedDate.getMonth() + 1).padStart(2, '0');
        const day = String(this.selectedDate.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }
  },
  methods: { // All methods remain largely the same, just minor console.log fallbacks for snackbar
    // Remove the initializeBookingTypes method as it will be merged into fetchFacilities
    
    getFacilityTypeDetails(facilityId) {
        for (const typeKey in this.facilities) {
            const facility = this.facilities[typeKey].find(f => f.id === facilityId);
            if (facility) {
                return this.bookingTypes.find(bt => bt.id === facility.type) || this.bookingTypes[0];
            }
        }
        return this.bookingTypes[0];
    },
    
    // Remove updateMyBookingsColors as it will be merged into fetchFacilities
    
    updateDate(newDateVal) {
      if (this.selectedDate) {
        this.selectedDate.setHours(0,0,0,0);
      }
      if (this.selectedFacility) {
        this.generateTimeSlots();
      }
    },
    selectFacility(facility) {
      this.selectedFacility = facility;
      this.fetchFacilityTimeSlots(facility.id);
    },
    clearFacilitySelection() {
      this.selectedFacility = null;
      this.timeSlots = [];
      this.selectedTimeSlot = null;
    },
    // New method to fetch time slots for a specific facility
    async fetchFacilityTimeSlots(facilityId) {
      if (!this.selectedDate) {
        this.timeSlots = [];
        return;
      }
      
      this.loading = true;
      this.timeSlots = [];
      
      try {
        console.log("Fetching time slots for facility:", facilityId, "on date:", this.selectedDateAsString);
        
        const response = await this.$root.callApi('facilities', 'get_facilities', {
          date: this.selectedDateAsString,
          facility_id: facilityId
        });
        
        // Check if response has a data property that contains the actual response
        const actualResponse = response.hasOwnProperty('data') ? response.data : response;
        
        if (actualResponse && actualResponse.status === "YES") {
          // Find the facility in the response
          let facilityData = null;
          for (const typeKey in actualResponse.facilities) {
            const facilities = actualResponse.facilities[typeKey];
            const facility = facilities.find(f => f.id === facilityId);
            if (facility) {
              facilityData = facility;
              break;
            }
          }
          
          if (facilityData && facilityData.time_slots) {
            console.log("Received time slots:", facilityData.time_slots);
            this.timeSlots = facilityData.time_slots;
          } else {
            console.warn("No time slots found for facility", facilityId);
            this.timeSlots = [];
          }
        } else {
          console.error("Error fetching facility time slots:", 
            actualResponse?.message || response?.message || "Unknown error", 
            "Full response:", response);
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('Error loading time slots. Please try again.', 'error');
          }
        }
      } catch (error) {
        console.error("Error in fetchFacilityTimeSlots:", error);
        if (this.$root.showSnackbar) {
          this.$root.showSnackbar('Network error. Please check your connection.', 'error');
        }
      } finally {
        this.loading = false;
      }
    },
        // Add new method to fetch facilities from API
    async fetchFacilities() {
      this.loading = true;
      try {
        console.log("Calling API with:", {
          date: this.selectedDateAsString,
          category: this.selectedType
        });
        
        const response = await this.$root.callApi('facilities', 'get_facilities', {
          date: this.selectedDateAsString,
          category: this.selectedType
        });
        
        // Add more detailed logging to inspect the response
        console.log("API Response (full):", JSON.stringify(response));
        console.log("Response type:", typeof response);
        console.log("Response has status property:", response && response.hasOwnProperty('status'));
        console.log("Response has data property:", response && response.hasOwnProperty('data'));
        
        // Check if response has a data property that contains the actual response
        const actualResponse = response.hasOwnProperty('data') ? response.data : response;
        
        if (actualResponse && actualResponse.status === "YES") {
          // Update facilities with API data
          this.facilities = actualResponse.facilities;
          console.log("Updated facilities:", this.facilities);
          
          // Update booking types if categories are provided by the API
          if (actualResponse.categories && actualResponse.categories.length > 0) {
            console.log("Categories from API:", actualResponse.categories);
            this.bookingTypes = actualResponse.categories;
            console.log("Updated bookingTypes:", this.bookingTypes);
            
            // If no type is selected yet, select the first one
            if (!this.selectedType && this.bookingTypes.length > 0) {
              this.selectedType = this.bookingTypes[0].id;
              console.log("Selected first type:", this.selectedType);
            }
          } else {
            console.warn("No categories found in API response");
          }
          
          // Update my bookings colors (moved from updateMyBookingsColors)
          this.myBookings = this.myBookings.map(booking => {
            const typeDetails = this.getFacilityTypeDetails(booking.facilityId);
            return { ...booking, color: typeDetails.color, icon: typeDetails.icon };
          });
          
        } else {
          console.error("Error fetching facilities:", 
            actualResponse?.message || response?.message || "Unknown error", 
            "Full response:", response);
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('Error loading facilities. Please try again.', 'error');
          }
        }
      } catch (error) {
        console.error("Error in fetchFacilities:", error);
        if (this.$root.showSnackbar) {
          this.$root.showSnackbar('Network error. Please check your connection.', 'error');
        }
      } finally {
        this.loading = false;
      }
    },
    // Remove the old generateTimeSlots method as it's no longer needed
    // or keep it as a fallback if you prefer
    generateTimeSlots() {
      if (!this.selectedDate) {
          this.timeSlots = [];
          return;
      }
      this.loading = true;
      this.timeSlots = [];
      setTimeout(() => {
        const slots = [];
        const facilityTypeDetails = this.getFacilityTypeDetails(this.selectedFacility.id);
        const startHour = facilityTypeDetails.id === 'dining' ? 11 : 8;
        const endHour = facilityTypeDetails.id === 'dining' ? 22 : 20;
        const slotDuration = facilityTypeDetails.id === 'services' ? 60 : (facilityTypeDetails.id === 'dining' ? 120 : 60);

        const bookedSlots = new Set();
        const availability = this.selectedFacility.availability;
        const bookedCount = availability === 'low' ? Math.floor((endHour - startHour) * (60 / slotDuration) * 0.7) :
                           availability === 'medium' ? Math.floor((endHour - startHour) * (60 / slotDuration) * 0.4) :
                           Math.floor((endHour - startHour) * (60 / slotDuration) * 0.2);

        for (let i = 0; i < bookedCount; i++) {
          const randomHour = Math.floor(Math.random() * (endHour - startHour)) + startHour;
          const slotsInHour = 60 / slotDuration;
          const randomMinuteBlock = Math.floor(Math.random() * slotsInHour);
          const randomMinute = randomMinuteBlock * slotDuration;
          bookedSlots.add(`${String(randomHour).padStart(2, '0')}:${String(randomMinute).padStart(2, '0')}`);
        }

        for (let hour = startHour; hour < endHour; hour++) {
          for (let minute = 0; minute < 60; minute += slotDuration) {
            const startTime = `${String(hour).padStart(2, '0')}:${String(minute).padStart(2, '0')}`;
            let endTimeHour = hour;
            let endTimeMinute = minute + slotDuration;
            if (endTimeMinute >= 60) {
              endTimeHour += Math.floor(endTimeMinute / 60);
              endTimeMinute %= 60;
            }
            const endTime = `${String(endTimeHour).padStart(2, '0')}:${String(endTimeMinute).padStart(2, '0')}`;
            const timeSlot = `${startTime} - ${endTime}`;
            const isBooked = bookedSlots.has(startTime);

            if (this.isToday) {
              const now = new Date();
              const slotHour = parseInt(startTime.split(':')[0]);
              const slotMinute = parseInt(startTime.split(':')[1]);
              if (slotHour < now.getHours() || (slotHour === now.getHours() && slotMinute < now.getMinutes())) {
                continue;
              }
            }
            if (this.isPastDate) continue;

            slots.push({
              time: timeSlot,
              booked: isBooked,
              price: this.getSlotPrice(hour, facilityTypeDetails.id)
            });
          }
        }
        this.timeSlots = slots;
        this.loading = false;
      }, 800);
    },
    // Update getSlotPrice to use pricing from API if available
    getSlotPrice(hour, typeId) {
      // Check if we have pricing data from API
      if (this.selectedFacility && this.selectedFacility.pricing && this.selectedFacility.pricing[hour]) {
        const pricing = this.selectedFacility.pricing[hour];
        let price = pricing.base_price;
        
        // Apply peak hour surcharge if applicable (5pm-9pm)
        if (hour >= 17 && hour < 21) {
          price += pricing.peak_surcharge;
        }
        
        // Apply weekend surcharge if applicable
        const dayOfWeek = this.selectedDate.getDay();
        if (dayOfWeek === 0 || dayOfWeek === 6) { // 0 = Sunday, 6 = Saturday
          price += pricing.weekend_surcharge;
        }
        
        return price;
      }
      
      // Fallback to original pricing logic if API data not available
      let basePrice = 40;
      if (typeId === 'services') basePrice = 80;
      else if (typeId === 'events') basePrice = 200;
      else if (typeId === 'dining') basePrice = 50;
      return (hour >= 17 && hour < 21) ? basePrice * 1.5 : basePrice;
    },
    selectTimeSlot(slot) {
      if (!slot.booked && !this.isPastDate && this.selectedDate) {
        this.selectedTimeSlot = slot;
        this.bookingForm = { guests: (this.selectedType === 'dining' ? 2 : 1) , notes: '', duration: 1 };
        this.showBookingDialog = true;
      }
    },
    fetchMyBookings() {
      this.loading = true;
      
      // Prepare data for API call
      const requestData = {
        user_id: sessionStorage.getItem('id')
      };
      
      // Call the API
      this.$root.callApi('facilities', 'get_bookings', requestData)
        .then(response => {
          this.loading = false;
          
          if (response.data && response.data.status === "YES") {
            // Update the bookings array with the API response
            this.myBookings = response.data.data.bookings;
            console.log("Bookings loaded:", this.myBookings);
          } else {
            // Show error message
            console.error("Error fetching bookings:", 
              response.data?.message || "Unknown error", 
              "Full response:", response);
              
            if (this.$root.showSnackbar) {
              this.$root.showSnackbar(response.data?.message || 'Failed to fetch bookings', 'error');
            }
          }
        })
        .catch(error => {
          this.loading = false;
          console.error('Error:', error);
          
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('An unexpected error occurred while fetching bookings.', 'error');
          }
        });
    },
    submitBooking() {
      this.loading = true;
      
      // Prepare data for API call
      const requestData = {
        user_id: sessionStorage.getItem('id'),
        facility_id: this.selectedFacility.id,
        booking_date: this.selectedDateAsString,
        time_slot: this.selectedTimeSlot.time,
        guests: this.bookingForm.guests,
        notes: this.bookingForm.notes,
        duration: this.bookingForm.duration
      };
      
      // Call the API
      this.$root.callApi('facilities', 'submit_booking', requestData)
        .then(response => {
          this.loading = false;
          
          if (response.data && response.data.status === "YES") {
            const facilityTypeDetails = this.getFacilityTypeDetails(this.selectedFacility.id);
            const newBooking = {
              id: response.data.data.booking_id,
              facilityId: this.selectedFacility.id,
              facilityName: this.selectedFacility.name,
              date: this.selectedDateAsString,
              timeSlot: this.selectedTimeSlot.time,
              status: 'confirmed',
              icon: facilityTypeDetails.icon,
              color: facilityTypeDetails.color,
              guests: this.bookingForm.guests,
              notes: this.bookingForm.notes
            };
            
            this.myBookings.unshift(newBooking);
            this.bookingForm = { guests: 1, notes: '', duration: 1 };
            this.showBookingDialog = false;
            this.selectedTimeSlot = null;
            
            if (this.$root.showSnackbar) { 
              this.$root.showSnackbar('Booking confirmed in the rainforest!', 'success'); 
            } else { 
              console.info('Snackbar: Booking confirmed in the rainforest!'); 
            }
            
            this.activeTab = 'bookings';
          } else {
            // Show error message
            if (this.$root.showSnackbar) {
              this.$root.showSnackbar(response.data?.message || 'Failed to submit booking', 'error');
            } else {
              console.error('Error:', response.data?.message || 'Failed to submit booking');
            }
          }
        })
        .catch(error => {
          this.loading = false;
          console.error('Error:', error);
          
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('An unexpected error occurred. Please try again later.', 'error');
          } else {
            console.error('Snackbar: An unexpected error occurred. Please try again later.');
          }
        });
    },
    confirmCancelBooking() {
      this.loading = true;
      
      // Prepare data for API call
      const requestData = {
        user_id: sessionStorage.getItem('id'),
        booking_id: this.selectedBookingToCancel.id
      };
      
      // Call the API
      this.$root.callApi('facilities', 'cancel_booking', requestData)
        .then(response => {
          this.loading = false;
          
          if (response.data && response.data.status === "YES") {
            // Update the booking status in the UI
            const index = this.myBookings.findIndex(b => b.id === this.selectedBookingToCancel.id);
            if (index !== -1) {
              this.myBookings[index].status = 'cancelled';
            }
            
            this.showCancelDialog = false;
            this.selectedBookingToCancel = null;
            
            if (this.$root.showSnackbar) { 
              this.$root.showSnackbar('Booking successfully cancelled.', 'info'); 
            } else { 
              console.info('Snackbar: Booking successfully cancelled.'); 
            }
          } else {
            // Show error message
            if (this.$root.showSnackbar) {
              this.$root.showSnackbar(response.data?.message || 'Failed to cancel booking', 'error');
            } else {
              console.error('Error:', response.data?.message || 'Failed to cancel booking');
            }
          }
        })
        .catch(error => {
          this.loading = false;
          console.error('Error:', error);
          
          if (this.$root.showSnackbar) {
            this.$root.showSnackbar('An unexpected error occurred. Please try again later.', 'error');
          } else {
            console.error('Snackbar: An unexpected error occurred. Please try again later.');
          }
        });
    },
    formatDateForDisplay(dateStr_YYYY_MM_DD) {
      if (!dateStr_YYYY_MM_DD) return '';
      const parts = dateStr_YYYY_MM_DD.split('-');
      const dateObj = new Date(parts[0], parts[1] - 1, parts[2]);
      const options = { weekday: 'short', month: 'short', day: 'numeric' };
      return dateObj.toLocaleDateString('en-US', options);
    },
    getStatusDetails(status) {
      switch(status) {
        case 'confirmed': return { text: 'Confirmed', color: this.theme.success, icon: 'mdi-check-circle' };
        case 'pending': return { text: 'Pending', color: this.theme.warning, icon: 'mdi-clock-alert-outline' };
        case 'cancelled': return { text: 'Cancelled', color: this.theme.error, icon: 'mdi-cancel' };
        case 'completed': return { text: 'Completed', color: this.theme.muted, icon: 'mdi-history' };
        default: return { text: status, color: this.theme.muted, icon: 'mdi-information-outline' };
      }
    },
    getAvailabilityDetails(availability) {
        switch(availability) {
            case 'high': return { text: 'Many Slots', color: this.theme.success, icon: 'mdi-leaf-circle' };
            case 'medium': return { text: 'Some Slots', color: this.theme.warning, icon: 'mdi-leaf-circle-outline' };
            case 'low': return { text: 'Few Slots Left', color: this.theme.error, icon: 'mdi-alert-circle-outline' };
            default: return { text: 'Unknown', color: this.theme.muted, icon: 'mdi-help-circle-outline'};
        }
    }
  },
  async created() {
    // Fetch facilities first
    await this.fetchFacilities();
    
    // Then fetch user's bookings
    await this.fetchMyBookings();
  },
  watch: {
    selectedDate() {
      this.updateDate();
      if (this.selectedFacility) {
        this.fetchFacilityTimeSlots(this.selectedFacility.id);
      }
    },
    selectedType() {
      this.fetchFacilities();
      this.clearFacilitySelection();
    }
  },
  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent scrim="#1A3C28" :opacity="0.8">
        <div class="text-center">
            <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
            <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Loading rainforest data...</div>
        </div>
      </v-overlay>

      <v-tabs v-model="activeTab" :bg-color="theme.primary" grow slider-color="secondary" class="booking-tabs-nav" dark>
        <v-tab value="find" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'find'}">
          <v-icon start>mdi-compass-rose</v-icon> Explore & Book
        </v-tab>
        <v-tab value="bookings" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'bookings'}">
          <v-icon start>mdi-calendar-multiple-check</v-icon> My Reservations
        </v-tab>
      </v-tabs>

      <v-window v-model="activeTab" class="tropical-booking-main-content">
        <!-- FIND & BOOK TAB -->
        <v-window-item value="find" class="pa-sm-4 pa-2">
          <v-row>
            <!-- Left Column: Date & Type -->
            <v-col cols="12" md="4">
              
            
              <v-card class="content-card mb-4" elevation="6">
                <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
                  <v-icon start :color="theme.secondary">mdi-calendar-month</v-icon>
                  Choose Your Date
                </v-card-title>
                <div class="d-flex justify-center pt-5">
                    <v-date-picker
                        v-model="selectedDate"
                        :min="minDateForPicker"
                        :color="theme.primary"
                        :header-color="theme.primary"
                        event-color="secondary"
                        show-current="true"
                        elevation="0"
                        class="themed-date-picker"
                        @update:model-value="updateDate"
                        :hide-actions="$vuetify.display.xs"
                    ></v-date-picker>
                </div>
                <div class="selected-date-info pa-3 text-center" :style="{backgroundColor: theme.accent, color: theme.primary, borderTop: '1px solid ' + theme.primary}">
                    Selected: <strong>{{ formattedSelectedDate }}</strong>
                    <v-chip v-if="isToday" size="small" :color="theme.success" :text-color="theme.lightText" class="ml-2 elevation-2">TODAY</v-chip>
                    <v-chip v-else-if="isPastDate" size="small" :color="theme.error" :text-color="theme.lightText" class="ml-2 elevation-2">PAST</v-chip>
                </div>
              </v-card>

              <v-card class="content-card" elevation="6">
                 <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
                    <v-icon start :color="theme.secondary">mdi-shape-outline</v-icon>
                    Select Category
                </v-card-title>
                <v-list class="category-list pa-0" nav density="compact">
                    <v-list-item
                        v-for="type in bookingTypes"
                        :key="type.id"
                        :value="type.id"
                        @click="selectedType = type.id"
                        class="category-list-item"
                        :active="selectedType === type.id"
                        :active-color="type.color"
                        rounded="lg"
                    >
                        <template v-slot:prepend>
                            <v-avatar :color="selectedType === type.id ? type.color : theme.accent" :size="$vuetify.display.xs ? 32 : 36" class="mr-3 elevation-1">
                                <v-icon :color="selectedType === type.id ? type.textColor : theme.primary" :size="$vuetify.display.xs ? 'small' : 'default'">{{ type.icon }}</v-icon>
                            </v-avatar>
                        </template>
                        <v-list-item-title :style="{fontWeight: selectedType === type.id ? 'bold' : 'normal', color: selectedType === type.id ? type.color : theme.darkText, fontSize: $vuetify.display.xs ? '0.85rem' : '0.95rem' }">
                            {{ type.name }}
                        </v-list-item-title>
                         <template v-slot:append v-if="selectedType === type.id">
                            <v-icon :color="type.color">mdi-chevron-right</v-icon>
                        </template>
                    </v-list-item>
                </v-list>
              </v-card>
            </v-col>

            <!-- Right Column: Facilities & Time Slots -->
            <v-col cols="12" md="8">
              <v-card class="content-card" elevation="6" :min-height="selectedFacility ? 'auto' : ($vuetify.display.xs ? '400px' : '600px')">
                <v-card-title class="card-header d-flex align-center" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
                  <v-icon start :color="selectedTypeDetails ? selectedTypeDetails.color : theme.secondary">
                    {{ selectedTypeDetails ? selectedTypeDetails.icon : 'mdi-storefront-outline' }}
                  </v-icon>
                  <span class="text-truncate">{{ selectedFacility ? selectedFacility.name : (selectedTypeDetails ? selectedTypeDetails.name : 'Select Facility') }}</span>
                  <v-spacer></v-spacer>
                  <v-btn v-if="selectedFacility" icon="mdi-arrow-left-circle-outline" small @click="clearFacilitySelection" :color="theme.secondary" variant="text"></v-btn>
                </v-card-title>

                <v-card-text class="pa-sm-4 pa-2">
                  <v-scroll-y-transition mode="out-in">
                  <div v-if="!selectedFacility" key="facility-list">
                     <p v-if="isPastDate && selectedDate" class="text-center pa-4" :style="{color: theme.error}">
                        <v-icon large :color="theme.error" class="d-block mb-2">mdi-calendar-remove</v-icon>
                        Cannot book for past dates.
                     </p>
                     <p v-else-if="!selectedDate" class="text-center pa-4" :style="{color: theme.warning}">
                        <v-icon large :color="theme.warning" class="d-block mb-2">mdi-calendar-alert</v-icon>
                        Please select a date.
                     </p>
                     <v-list v-else class="facility-selection-list pa-0" :lines="$vuetify.display.xs ? 'two' : 'three'">
                        <v-list-item
                            v-for="facility in availableFacilities"
                            :key="facility.id"
                            @click="selectFacility(facility)"
                            class="facility-item elevation-1 mb-3"
                            rounded="lg"
                        >
                            <template v-slot:prepend>
                                <v-avatar :color="getFacilityTypeDetails(facility.id).color" :size="$vuetify.display.xs ? 40 : 50" class="mr-sm-4 mr-2 elevation-2">
                                    <v-icon color="white">{{ facility.icon }}</v-icon>
                                </v-avatar>
                            </template>
                            <v-list-item-title class="facility-name mb-1" :class="$vuetify.display.xs ? 'text-body-1' : 'text-h6'" :style="{color: theme.primary}">{{ facility.name }}</v-list-item-title>
                            <v-list-item-subtitle class="facility-availability d-flex align-center">
                                <v-icon small :color="getAvailabilityDetails(facility.availability).color" class="mr-1">{{ getAvailabilityDetails(facility.availability).icon }}</v-icon>
                                <span :style="{color: getAvailabilityDetails(facility.availability).color, fontSize: $vuetify.display.xs ? '0.75rem' : '0.875rem'}">{{ getAvailabilityDetails(facility.availability).text }}</span>
                            </v-list-item-subtitle>
                            <template v-slot:append>
                                <v-icon :color="theme.primary" :size="$vuetify.display.xs ? 24 : 28">mdi-chevron-right</v-icon>
                            </template>
                        </v-list-item>
                        <div v-if="!availableFacilities.length && selectedDate" class="text-center pa-5">
                            <v-icon size="60" :color="theme.muted" class="mb-2">mdi-magnify-remove-outline</v-icon>
                            <p :style="{color: theme.darkText}">No facilities found.</p>
                        </div>
                     </v-list>
                  </div>

                  <div v-else key="time-slots">
                    <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-1" :style="{color: theme.primary}">Available Time Slots</h3>
                    <p class="mb-4" :class="$vuetify.display.xs ? 'text-caption' : 'text-body-2'" :style="{color: theme.muted}">
                        For {{ formattedSelectedDate }}.
                    </p>
                    <div v-if="isPastDate && selectedDate" class="text-center pa-4" :style="{color: theme.error}">
                        <v-icon large :color="theme.error" class="d-block mb-2">mdi-calendar-remove</v-icon>
                        Cannot book for past dates.
                    </div>
                    <v-row v-else dense class="time-slot-grid">
                      <v-col
                        v-for="(slot, index) in timeSlots"
                        :key="index"
                        cols="6" sm="4" md="3"
                      >
                        <v-card
                          @click="selectTimeSlot(slot)"
                          :class="['time-slot-item', {'booked': slot.booked, 'available': !slot.booked}]"
                          :disabled="slot.booked"
                          elevation="2"
                          hover
                          rounded="lg"
                        >
                          <div class="pa-2 text-center">
                            <div class="font-weight-bold time-display" :style="{color: slot.booked ? theme.muted : theme.primary}">{{ slot.time }}</div>
                            <v-chip
                              size="x-small" 
                              :color="slot.booked ? theme.error : theme.success"
                              :text-color="theme.lightText"
                              class="mt-1 status-chip-small"
                              label
                            >
                              {{ slot.booked ? 'Booked' : 'Open' }}
                            </v-chip>
                            <div v-if="!slot.booked" class="price-display mt-1" :style="{color: theme.primary, fontWeight: '500'}">
                              RM {{ slot.price.toFixed(2) }}
                            </div>
                          </div>
                        </v-card>
                      </v-col>
                    </v-row>
                    <div v-if="!isPastDate && timeSlots.length === 0 && !loading && selectedDate" class="text-center pa-5">
                        <v-icon size="60" :color="theme.muted" class="mb-2">mdi-timer-sand-empty</v-icon>
                        <p :style="{color: theme.darkText}">No time slots available.</p>
                        <p :style="{color: theme.muted}" class="text-caption">Try another date/facility.</p>
                    </div>
                     <div v-if="loading && timeSlots.length === 0" class="text-center pa-5">
                         <v-progress-circular indeterminate :color="theme.secondary" size="48"></v-progress-circular>
                         <p class="mt-3" :style="{color: theme.muted}">Fetching slots...</p>
                     </div>
                  </div>
                  </v-scroll-y-transition>
                </v-card-text>
              </v-card>
            </v-col>
          </v-row>
        </v-window-item>

        <!-- MY BOOKINGS TAB -->
        <v-window-item value="bookings" class="pa-sm-4 pa-2">
            <v-card class="content-card" elevation="6">
                <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
                    <v-icon start :color="theme.secondary">mdi-view-list</v-icon> My Reservations
                </v-card-title>
                 <v-card-text class="pa-sm-4 pa-2">
                    <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3 d-flex align-center" :style="{color: theme.primary}">
                        <v-icon :color="theme.success" class="mr-2">mdi-calendar-arrow-right</v-icon>
                        Upcoming <v-chip class="ml-2" size="small" :color="theme.success" :text-color="theme.lightText" label>{{ upcomingBookings.length }}</v-chip>
                    </h3>
                    <div v-if="upcomingBookings.length === 0" class="empty-bookings-state text-center pa-5 mb-6">
                        <v-icon size="60" :color="theme.accent" class="mb-2">mdi-calendar-search</v-icon>
                        <p :class="$vuetify.display.xs ? 'text-body-1' : 'text-h6'" :style="{color: theme.darkText}">No upcoming adventures!</p>
                        <p :style="{color: theme.muted, fontSize: $vuetify.display.xs ? '0.8rem' : 'inherit'}">Time to explore.</p>
                        <v-btn :color="theme.primary" :text-color="theme.lightText" class="mt-4 elevation-2" @click="activeTab = 'find'" :size="$vuetify.display.xs ? 'small' : 'default'">
                            <v-icon start>mdi-plus-circle-outline</v-icon> Make a New Booking
                        </v-btn>
                    </div>
                    <v-expansion-panels v-else accordion class="mb-6 themed-expansion-panels" variant="accordion">
                      <v-expansion-panel
                        v-for="booking in upcomingBookings" :key="booking.id"
                        class="booking-panel-item elevation-2" :style="{'border-left-color': getStatusDetails(booking.status).color + ' !important'}"
                        rounded="lg"
                      >
                        <v-expansion-panel-title class="booking-panel-title" :class="{'booking-panel-title-mobile': $vuetify.display.xs}">
                            <v-avatar :color="booking.color" :size="$vuetify.display.xs ? 36 : 40" class="mr-sm-4 mr-2 elevation-1">
                                <v-icon :color="theme.lightText">{{ booking.icon }}</v-icon>
                            </v-avatar>
                            <div class="flex-grow-1">
                                <div class="font-weight-bold" :class="$vuetify.display.xs ? 'text-body-2' : 'text-subtitle-1'" :style="{color: theme.primary}">{{ booking.facilityName }}</div>
                                <div :class="$vuetify.display.xs ? 'text-caption' : 'text-body-2'" :style="{color: theme.muted}">{{ formatDateForDisplay(booking.date) }} • {{ booking.timeSlot }}</div>
                            </div>
                            <v-chip label :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="getStatusDetails(booking.status).color" :text-color="theme.lightText" class="status-chip-panel ml-1">
                                <v-icon start :size="$vuetify.display.xs ? 'x-small' : 'small'">{{ getStatusDetails(booking.status).icon }}</v-icon>
                                {{ getStatusDetails(booking.status).text }}
                            </v-chip>
                        </v-expansion-panel-title>
                        <v-expansion-panel-text class="booking-panel-content">
                            <v-list dense class="py-0 bg-transparent" :class="{'list-mobile-dense': $vuetify.display.xs}">
                                <v-list-item density="compact"><v-icon small start :color="theme.primary">mdi-calendar-blank</v-icon> <strong>Date:</strong> {{ formatDateForDisplay(booking.date) }}</v-list-item>
                                <v-list-item density="compact"><v-icon small start :color="theme.primary">mdi-clock-outline</v-icon> <strong>Time:</strong> {{ booking.timeSlot }}</v-list-item>
                                <v-list-item v-if="booking.guests && booking.guests > 1" density="compact"><v-icon small start :color="theme.primary">mdi-account-group</v-icon> <strong>Guests:</strong> {{ booking.guests }}</v-list-item>
                                <v-list-item v-if="booking.notes" density="compact"><v-icon small start :color="theme.primary">mdi-text-box-outline</v-icon> <strong>Notes:</strong> {{ booking.notes }}</v-list-item>
                            </v-list>
                            <v-card-actions v-if="booking.status === 'confirmed' || booking.status === 'pending'" class="pa-1 justify-end">
                                <v-btn :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="theme.error" variant="text" @click="selectedBookingToCancel = booking; showCancelDialog = true">
                                    <v-icon start>mdi-cancel</v-icon> Cancel
                                </v-btn>
                            </v-card-actions>
                        </v-expansion-panel-text>
                      </v-expansion-panel>
                    </v-expansion-panels>

                    <v-divider class="my-6"></v-divider>

                    <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3 d-flex align-center" :style="{color: theme.primary}">
                        <v-icon :color="theme.muted" class="mr-2">mdi-history</v-icon>
                        Past & Cancelled <v-chip class="ml-2" size="small" :color="theme.muted" :text-color="theme.lightText" label>{{ pastBookings.length }}</v-chip>
                    </h3>
                     <div v-if="pastBookings.length === 0" class="empty-bookings-state text-center pa-5">
                        <v-icon size="60" :color="theme.accent" class="mb-2">mdi-archive-outline</v-icon>
                        <p :style="{color: theme.darkText, fontSize: $vuetify.display.xs ? '0.9rem' : 'inherit'}">No past reservations.</p>
                    </div>
                    <v-expansion-panels v-else accordion class="themed-expansion-panels" variant="accordion">
                       <v-expansion-panel
                        v-for="booking in pastBookings" :key="booking.id"
                        class="booking-panel-item past-booking elevation-1" :style="{'border-left-color': getStatusDetails(booking.status).color + ' !important'}"
                        rounded="lg"
                      >
                        <v-expansion-panel-title class="booking-panel-title" :class="{'booking-panel-title-mobile': $vuetify.display.xs}">
                             <v-avatar :color="getStatusDetails(booking.status).color" :size="$vuetify.display.xs ? 36 : 40" class="mr-sm-4 mr-2 elevation-1" style="opacity: 0.7;">
                                <v-icon :color="theme.lightText">{{ booking.icon }}</v-icon>
                            </v-avatar>
                            <div class="flex-grow-1" style="opacity: 0.7;">
                                <div class="font-weight-medium" :class="$vuetify.display.xs ? 'text-body-2' : 'text-subtitle-1'" :style="{color: theme.darkText}">{{ booking.facilityName }}</div>
                                <div :class="$vuetify.display.xs ? 'text-caption' : 'text-body-2'" :style="{color: theme.muted}">{{ formatDateForDisplay(booking.date) }} • {{ booking.timeSlot }}</div>
                            </div>
                            <v-chip label :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="getStatusDetails(booking.status).color" :text-color="theme.lightText" class="status-chip-panel ml-1">
                                <v-icon start :size="$vuetify.display.xs ? 'x-small' : 'small'">{{ getStatusDetails(booking.status).icon }}</v-icon>
                                {{ getStatusDetails(booking.status).text }}
                            </v-chip>
                        </v-expansion-panel-title>
                         <v-expansion-panel-text class="booking-panel-content">
                            <v-list dense class="py-0 bg-transparent" :class="{'list-mobile-dense': $vuetify.display.xs}">
                                <v-list-item density="compact"><v-icon small start :color="theme.muted">mdi-calendar-blank</v-icon> <strong>Date:</strong> {{ formatDateForDisplay(booking.date) }}</v-list-item>
                                <v-list-item density="compact"><v-icon small start :color="theme.muted">mdi-clock-outline</v-icon> <strong>Time:</strong> {{ booking.timeSlot }}</v-list-item>
                                <v-list-item v-if="booking.guests && booking.guests > 1" density="compact"><v-icon small start :color="theme.muted">mdi-account-group</v-icon> <strong>Guests:</strong> {{ booking.guests }}</v-list-item>
                                <v-list-item v-if="booking.notes" density="compact"><v-icon small start :color="theme.muted">mdi-text-box-outline</v-icon> <strong>Notes:</strong> {{ booking.notes }}</v-list-item>
                            </v-list>
                          </v-expansion-panel-text>
                      </v-expansion-panel>
                    </v-expansion-panels>
                </v-card-text>
            </v-card>
        </v-window-item>
      </v-window>

      <!-- Booking Dialog -->
      <v-dialog
        v-model="showBookingDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '550px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        content-class="booking-dialog-card"
        persistent scrollable
        transition="dialog-bottom-transition"
      >
        <v-card v-if="selectedFacility && selectedTimeSlot">
          <v-toolbar :color="theme.primary" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2" :color="theme.secondary">mdi-calendar-plus</v-icon>
            <v-toolbar-title class="dialog-title">Confirm Adventure</v-toolbar-title>
            <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showBookingDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4">
            <v-alert border="start" :border-color="theme.accent" colored-border elevation="2" class="mb-4" :icon="getFacilityTypeDetails(selectedFacility.id).icon">
                Booking <strong>{{ selectedFacility.name }}</strong> for <strong>{{ selectedTimeSlot.time }}</strong> on <strong>{{ formattedSelectedDate }}</strong>.
            </v-alert>
            <div class="text-h5 font-weight-bold my-3 text-center" :style="{color: theme.primary}">
                Price: RM {{ selectedTimeSlot.price.toFixed(2) }}
            </div>
            <v-form v-model="formValid" class="mt-4">
              <v-text-field
                v-if="selectedType === 'dining' || selectedType === 'events'"
                v-model.number="bookingForm.guests"
                :rules="[v => !!v || 'Guests required', v => v > 0 || 'Min 1 guest']"
                label="Number of Guests" type="number" min="1"
                :color="theme.primary" variant="outlined" prepend-inner-icon="mdi-account-group-outline"
                density="comfortable"
              ></v-text-field>
              <v-select
                v-if="selectedType === 'services'"
                v-model.number="bookingForm.duration"
                :items="[{title: '1 Hour', value:1}, {title:'1.5 Hours', value:1.5}, {title:'2 Hours',value:2}]"
                label="Service Duration"
                :color="theme.primary" variant="outlined" prepend-inner-icon="mdi-clock-time-four-outline"
                density="comfortable"
              ></v-select>
              <v-textarea
                v-model="bookingForm.notes"
                label="Special Requests?"
                :color="theme.primary" variant="outlined" prepend-inner-icon="mdi-pencil-box-outline"
                rows="3" auto-grow
                density="comfortable"
              ></v-textarea>
            </v-form>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 dialog-actions-sticky">
            <v-btn :color="theme.muted" variant="text" @click="showBookingDialog = false" :size="$vuetify.display.xs ? 'small' : 'default'">Back</v-btn>
            <v-spacer></v-spacer>
            <v-btn :color="theme.primary" :text-color="theme.lightText" :size="$vuetify.display.xs ? 'small' : 'large'" elevation="2" @click="submitBooking" :loading="loading" :disabled="!formValid && (selectedType === 'dining' || selectedType === 'events')">
                <v-icon start>mdi-check-decagram</v-icon> Confirm
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>

      <!-- Cancel Booking Dialog -->
      <v-dialog
        v-model="showCancelDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '450px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        content-class="cancel-dialog-card"
        persistent
        transition="dialog-bottom-transition"
      >
        <v-card v-if="selectedBookingToCancel">
           <v-toolbar :color="theme.error" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2">mdi-alert-octagon-outline</v-icon>
            <v-toolbar-title class="dialog-title">Confirm Cancellation</v-toolbar-title>
             <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showCancelDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4 text-center">
            <p :class="$vuetify.display.xs ? 'text-body-2' : 'text-body-1'" class="mt-3">Cancel booking for:</p>
            <p :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="font-weight-medium my-2" :style="{color: theme.primary}">{{ selectedBookingToCancel.facilityName }}</p>
            <p class="text-body-2" :style="{color: theme.muted}">{{ formatDateForDisplay(selectedBookingToCancel.date) }} at {{ selectedBookingToCancel.timeSlot }}</p>
            <v-alert type="warning" density="compact" outlined class="mt-4 text-left text-caption">
                Cancellation policies may apply.
            </v-alert>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 dialog-actions-sticky">
            <v-btn :color="theme.muted" variant="text" @click="showCancelDialog = false" :size="$vuetify.display.xs ? 'small' : 'default'">Keep</v-btn>
            <v-spacer></v-spacer>
            <v-btn :color="theme.error" :text-color="theme.lightText" :size="$vuetify.display.xs ? 'small' : 'default'" elevation="2" @click="confirmCancelBooking()" :loading="loading">
                <v-icon start>mdi-trash-can-outline</v-icon> Yes, Cancel
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>

    </v-container>
  `,
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
    
    // Rest of your mounted code
    if (this.bookingTypes && this.bookingTypes.length > 0 && !this.selectedType) {
      this.selectedType = this.bookingTypes[0].id;
    }
  }
};