export default {
  props: ["jsonData"],
  data() {
    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C'
      },

      // UI states
      isLoading: false,
      isEditing: false,
      showNotification: false,
      notificationMessage: '',
      notificationType: 'success', // 'success', 'error', 'info'

      // Scroll and animation states
      scrollPosition: 0,
      animationReady: false,

      // Session state
      sessionValid: true,

      // Form validation
      valid: true,
      nameRules: [
        v => !!v || 'Name is required',
        v => (v && v.length >= 3) || 'Name must be at least 3 characters'
      ],
      emailRules: [
        v => !!v || 'Email is required',
        v => /.+@.+\..+/.test(v) || 'Email must be valid'
      ],
      phoneRules: [
        v => !!v || 'Phone number is required',
        v => /^\d{10,12}$/.test(v) || 'Phone number must be valid'
      ],

      // User data
      profileData: {
        name: '',
        email: '',
        phone: '',
        role: '',
        membershipId: '',
        address: '',
        dateJoined: '',
        avatar: null,
      },

      // Copy for editing
      editableProfile: {},

      // Photo upload
      photoFile: null,
      photoPreview: null,

      // Membership roles
      membershipRoles: [
        { value: 'platinum', text: 'Platinum Member', color: '#E5E4E2', icon: 'mdi-crown' },
        { value: 'gold', text: 'Gold Member', color: '#FFD700', icon: 'mdi-medal' },
        { value: 'silver', text: 'Silver Member', color: '#C0C0C0', icon: 'mdi-medal-outline' },
        { value: 'junior', text: 'Junior Member', color: '#A3BA95', icon: 'mdi-account-child' }
      ],

      // Animation control for floating leaves
      floatingLeaves: [
        { x: 20, y: 120, size: 60, rotation: 0, speed: 0.4 },
        { x: 80, y: 350, size: 40, rotation: 30, speed: 0.6 },
        { x: 70, y: 600, size: 50, rotation: 15, speed: 0.5 },
        { x: 30, y: 750, size: 45, rotation: -20, speed: 0.7 }
      ],

      // Account settings menu
      settingsMenu: [
        { title: 'Change Password', icon: 'mdi-lock-reset', action: 'changePassword' },
        { title: 'Change Pin', icon: 'mdi-credit-card-settings-outline', action: 'changePin' },

        // { title: 'Notification Preferences', icon: 'mdi-bell-outline', action: 'notificationPrefs' },
        // { title: 'Privacy Settings', icon: 'mdi-shield-account', action: 'privacySettings' },
        // { title: 'Linked Accounts', icon: 'mdi-link-variant', action: 'linkedAccounts' }
      ],

      // Change password dialog
      showPasswordDialog: false,
      passwordForm: {
        currentPassword: '',
        newPassword: '',
        confirmPassword: ''
      },
      passwordRules: {
        current: [v => !!v || 'Current password is required'],
        new: [
          v => !!v || 'New password is required',
          v => (v && v.length >= 6) || 'Password must be at least 6 characters'
        ],
        confirm: [
          v => !!v || 'Please confirm your password',
          v => v === this.passwordForm.newPassword || 'Passwords do not match'
        ]
      },

      // Change PIN dialog
      showPinDialog: false,
      pinForm: {
        currentPin: '',
        newPin: '',
        confirmPin: ''
      },
      pinRules: {
        current: [v => !!v || 'Current PIN is required'],
        new: [
          v => !!v || 'New PIN is required',
          v => /^\d{4,6}$/.test(v) || 'PIN must be 4-6 digits only'
        ],
        confirm: [
          v => !!v || 'Please confirm your PIN',
          v => v === this.pinForm.newPin || 'PINs do not match'
        ]
      }
    }
  },
  computed: {
    // Get user initials for avatar fallback
    userInitials() {
      return this.profileData.name.split(' ').map(name => name[0]).join('')
    },

    // Get membership badge details
    membershipBadge() {
      const role = this.membershipRoles.find(r => r.value === this.profileData.role) ||
        { color: '#A3BA95', text: 'Member', icon: 'mdi-account' };
      return {
        color: role.color,
        text: role.text,
        icon: role.icon,
        textColor: '#1A3C28'
      }
    },

    // Format the joined date
    formattedJoinDate() {
      const options = { year: 'numeric', month: 'long', day: 'numeric' };
      return new Date(this.profileData.dateJoined).toLocaleDateString('en-US', options);
    },

    // Parallax effect styles based on scroll position
    headerParallaxStyle() {
      return {
        transform: `translateY(${this.scrollPosition * 0.4}px)`
      }
    },

    bgLeaf1Style() {
      return {
        transform: `rotate(45deg) translate(${this.scrollPosition * 0.05}px, ${this.scrollPosition * -0.1}px)`
      }
    },

    bgLeaf2Style() {
      return {
        transform: `rotate(130deg) translate(${this.scrollPosition * -0.08}px, ${this.scrollPosition * 0.07}px)`
      }
    },

    silhouetteStyle() {
      return {
        transform: `translateY(${this.scrollPosition * -0.03}px)`
      }
    }
  },
  methods: {
    // Check if session is still valid
    checkSession() {
      const userId = sessionStorage.getItem('id');
      const secret = sessionStorage.getItem('session_secret');

      if (!userId || !secret) {
        // Session is invalid, redirect to login
        this.sessionValid = false;
        this.showSessionExpiredNotification();
        return false;
      }

      // In a real implementation, you might want to make a lightweight API call
      // to verify the session is still valid on the server
      return true;
    },


    //Call API (User Profile - START )-------------
    async fetchUserProfile() {
      this.isLoading = true;
      this.error = null;

      try {
        const response = await this.$root.callApi("settings", "profile");

        if (response && response.data && response.data.status == "YES") {
          console.log(response.data);

          // Update profileData with the response data
          if (response.data.profile) {
            this.profileData = {
              name: response.data.profile.fullname || '',
              email: response.data.profile.email || '',
              phone: response.data.profile.contact || '',
              role: response.data.profile.member_type || '',
              membershipId: response.data.profile.member_no || '',
              address: response.data.profile.adddress || '',
              dateJoined: response.data.profile.created_date || '',
              avatar: response.data.profile.cover_photo_path || null,
            };
          }

          // Also update session data if needed
          if (response.session && response.session.user_data) {
            // You might want to update session storage or other session-related data here
            console.log("Session data updated");
          }
        } else {
          this.error = "Failed to load data. Please try again";
          console.error("Invalid API response: ", response);
        }
      } catch (error) {
        this.error = "An error occurred while loading profile data";
        console.error("Error fetching profile data", error);
      } finally {
        this.isLoading = false;
      }

    },
    //Call API (User Profile - END )---------------

    // Show session expired notification and redirect
    showSessionExpiredNotification() {
      this.showNotification = true;
      this.notificationMessage = 'Session expired. Please log in again.';
      this.notificationType = 'error';

      setTimeout(() => {
        this.logout();
      }, 2000);
    },

    // Logout method
    logout() {
      // Add visual feedback before logout
      this.isLoading = true;

      setTimeout(() => {
        // Fade out the entire page
        document.querySelector('.tropical-profile').classList.add('page-exit');

        setTimeout(() => {
          // Call the root logout method
          this.$root.removeSession();
          this.$root.bottomNavDisplay = false;
          this.$root.loadComponent("modules/login.js");
        }, 400);
      }, 500);
    },

    // Handle settings menu actions
    openSetting(item) {
      // Check if session is valid before proceeding
      if (!this.checkSession()) return;

      // Handle special case for logout
      if (item.action === 'logout') {
        this.confirmLogout();
        return;
      }

      // Handle change password
      if (item.action === 'changePassword') {
        this.showChangePasswordDialog();
        return;
      }

      // Handle change PIN
      if (item.action === 'changePin') {
        this.showChangePinDialog();
        return;
      }

      // For other menu items
      this.showNotification = true;
      this.notificationMessage = `Opening ${item.title}...`;
      this.notificationType = 'info';

      setTimeout(() => {
        this.showNotification = false;
      }, 2000);
    },

    // Show logout confirmation
    confirmLogout() {
      if (confirm('Are you sure you want to log out?')) {
        this.logout();
      }
    },

    // Handle scroll events for parallax
    handleScroll() {
      this.scrollPosition = window.scrollY || window.pageYOffset;

      // Update floating leaves
      this.updateFloatingLeaves();
    },

    // Update floating leaves positions based on scroll
    updateFloatingLeaves() {
      const leaves = document.querySelectorAll('.floating-leaf');
      if (!leaves.length) return;

      this.floatingLeaves.forEach((leaf, index) => {
        if (index >= leaves.length) return;

        const yOffset = this.scrollPosition * leaf.speed;
        const rotationAdjust = Math.sin(this.scrollPosition / 500) * 5;

        leaves[index].style.transform = `
            translate(${leaf.x}px, ${leaf.y - yOffset}px) 
            rotate(${leaf.rotation + rotationAdjust}deg)
          `;
      });
    },

    // Start editing profile with animation
    editProfile() {
      // Check if session is valid before proceeding
      if (!this.checkSession()) return;

      this.editableProfile = JSON.parse(JSON.stringify(this.profileData));

      // Animate transition to edit mode
      const viewProfile = document.querySelector('.view-profile');
      if (viewProfile) {
        viewProfile.classList.add('fade-out');

        setTimeout(() => {
          this.isEditing = true;

          // Wait for edit form to be mounted
          this.$nextTick(() => {
            const editProfile = document.querySelector('.edit-profile');
            if (editProfile) {
              editProfile.classList.add('fade-in');
            }
          });
        }, 300);
      } else {
        this.isEditing = true;
      }
    },

    // Cancel editing and revert changes with animation
    cancelEdit() {
      const editProfile = document.querySelector('.edit-profile');
      if (editProfile) {
        editProfile.classList.add('fade-out');

        setTimeout(() => {
          this.isEditing = false;
          this.photoPreview = null;
          this.photoFile = null;

          // Wait for view profile to be mounted
          this.$nextTick(() => {
            const viewProfile = document.querySelector('.view-profile');
            if (viewProfile) {
              viewProfile.classList.add('fade-in');
            }
          });
        }, 300);
      } else {
        this.isEditing = false;
        this.photoPreview = null;
        this.photoFile = null;
      }
    },

    // Handle photo selection
    onPhotoSelected(e) {
      const file = e.target.files[0];
      if (!file) return;

      this.photoFile = file;

      // Create preview
      const reader = new FileReader();
      reader.onload = e => {
        this.photoPreview = e.target.result;

        // Add animation
        this.$nextTick(() => {
          const avatar = document.querySelector('.profile-avatar-editor .v-avatar');
          if (avatar) {
            avatar.classList.add('photo-updated');
            setTimeout(() => avatar.classList.remove('photo-updated'), 700);
          }
        });
      };
      reader.readAsDataURL(file);
    },

    // Remove selected photo
    removePhoto() {
      this.photoPreview = null;
      this.photoFile = null;
      this.editableProfile.avatar = null;

      // Add animation
      this.$nextTick(() => {
        const avatar = document.querySelector('.profile-avatar-editor .v-avatar');
        if (avatar) {
          avatar.classList.add('photo-removed');
          setTimeout(() => avatar.classList.remove('photo-removed'), 700);
        }
      });
    },

    // Save profile changes
    async saveProfile() {
      // Check if session is valid before proceeding
      if (!this.checkSession()) return;

      if (!this.$refs.form.validate()) return;

      this.isLoading = true;

      try {
        // Prepare data for API call
        const profileData = {
          fullname: this.editableProfile.name,
          email: this.editableProfile.email,
          contact: this.editableProfile.phone,
          address: this.editableProfile.address || '' // Make sure address is included
        };

        // Call the API to update profile
        const response = await this.$root.callApi("settings", "update_profile", profileData);

        // Check for success in the correct location of the response structure
        if (response && response.data && response.data.status === "YES") {
          // Update profile data on success
          this.profileData = { ...this.editableProfile };

          // If there's a new photo, in real implementation we'd upload it to server
          if (this.photoFile) {
            // In real implementation, this would be the URL returned from server
            this.profileData.avatar = this.photoPreview;
          }

          this.showNotification = true;
          this.notificationMessage = response.data.message || 'Profile updated successfully!';
          this.notificationType = 'success';
        } else {
          // Show error message from API
          this.showNotification = true;
          this.notificationMessage = (response.data && response.data.message) || 'Failed to update profile. Please try again.';
          this.notificationType = 'error';
        }

        setTimeout(() => {
          this.showNotification = false;
        }, 3000);

        // Animate transition back to view mode
        const editProfile = document.querySelector('.edit-profile');
        if (editProfile) {
          editProfile.classList.add('fade-out');

          setTimeout(() => {
            this.isEditing = false;

            // Wait for view profile to be mounted
            this.$nextTick(() => {
              const viewProfile = document.querySelector('.view-profile');
              if (viewProfile) {
                viewProfile.classList.add('fade-in');

                // Animate updated fields
                setTimeout(() => {
                  const detailValues = document.querySelectorAll('.detail-value');
                  detailValues.forEach((el, i) => {
                    setTimeout(() => {
                      el.classList.add('value-updated');
                      setTimeout(() => el.classList.remove('value-updated'), 1000);
                    }, i * 100);
                  });
                }, 300);
              }
            });
          }, 300);
        } else {
          this.isEditing = false;
        }
      } catch (error) {
        this.showNotification = true;
        this.notificationMessage = 'Failed to update profile. Please try again.';
        this.notificationType = 'error';

        setTimeout(() => {
          this.showNotification = false;
        }, 3000);
      } finally {
        this.isLoading = false;
      }
    },

    // Animate cards on scroll
    animateOnScroll() {
      const cards = document.querySelectorAll('.detail-card');

      cards.forEach(card => {
        const cardPosition = card.getBoundingClientRect().top;
        const screenPosition = window.innerHeight / 1.3;

        if (cardPosition < screenPosition) {
          card.classList.add('card-visible');
        }
      });
    },

    // API call to update profile
    async updateProfileAPI() {
      // In real implementation, this would call the API
      // const response = await this.$root.callApi('profile', 'update_profile', this.editableProfile);
      // return response;

      // For now, just simulate a successful response
      return { success: true };
    },

    // Navigation with animation
    navigateBack() {
      // Add fade-out animation to entire page
      document.querySelector('.tropical-profile').classList.add('page-exit');

      setTimeout(() => {
        this.$root.loadComponent('modules/dashboard.js');
      }, 400);
    },

    // Show change password dialog
    showChangePasswordDialog() {
      this.passwordForm = {
        currentPassword: '',
        newPassword: '',
        confirmPassword: ''
      };
      this.showPasswordDialog = true;
    },

    // Show change PIN dialog
    showChangePinDialog() {
      this.pinForm = {
        currentPin: '',
        newPin: '',
        confirmPin: ''
      };
      this.showPinDialog = true;
    },

    // Change password API call
    async changePassword() {
      if (!this.$refs.passwordForm.validate()) return;

      this.isLoading = true;

      try {
        const userId = sessionStorage.getItem('id');
        
        // Hash passwords with MD5 like in login
        const hashedCurrentPassword = this.$root.md5data(this.passwordForm.currentPassword).toString();
        const hashedNewPassword = this.$root.md5data(this.passwordForm.newPassword).toString();
        const hashedConfirmPassword = this.$root.md5data(this.passwordForm.confirmPassword).toString();
        
        const response = await this.$root.callApi("wallet", "change_password", {
          user_id: userId,
          current_password: hashedCurrentPassword,
          new_password: hashedNewPassword,
          confirm_password: hashedConfirmPassword
        });

        if (response && response.data && response.data.status === "YES") {
          this.showPasswordDialog = false;
          this.showNotification = true;
          this.notificationMessage = response.data.message;
          this.notificationType = 'success';
        } else {
          this.showNotification = true;
          this.notificationMessage = (response.data && response.data.message) || 'Failed to change password';
          this.notificationType = 'error';
        }
      } catch (error) {
        this.showNotification = true;
        this.notificationMessage = 'An error occurred while changing password';
        this.notificationType = 'error';
      } finally {
        this.isLoading = false;
        setTimeout(() => {
          this.showNotification = false;
        }, 3000);
      }
    },

    // Change PIN API call
    async changePin() {
      if (!this.$refs.pinForm.validate()) return;

      this.isLoading = true;

      try {
        const userId = sessionStorage.getItem('id');
        const requestData = {
          user_id: userId,
          new_pin: this.pinForm.newPin,
          confirm_pin: this.pinForm.confirmPin
        };

        // Only include current PIN if it's provided
        if (this.pinForm.currentPin) {
          requestData.current_pin = this.pinForm.currentPin;
        }

        const response = await this.$root.callApi("wallet", "change_pin", requestData);

        if (response && response.data && response.data.status === "YES") {
          this.showPinDialog = false;
          this.showNotification = true;
          this.notificationMessage = response.data.message;
          this.notificationType = 'success';
        } else {
          this.showNotification = true;
          this.notificationMessage = (response.data && response.data.message) || 'Failed to change PIN';
          this.notificationType = 'error';
        }
      } catch (error) {
        this.showNotification = true;
        this.notificationMessage = 'An error occurred while changing PIN';
        this.notificationType = 'error';
      } finally {
        this.isLoading = false;
        setTimeout(() => {
          this.showNotification = false;
        }, 3000);
      }
    }
  },
  mounted() {
    // Check session first
    if (!this.checkSession()) {
      return; // Stop loading if session is invalid
    }

    this.fetchUserProfile();

    // Load user data (in real implementation, this would come from API)
    this.isLoading = true;

    // Simulate API delay
    setTimeout(() => {
      // In real implementation, we'd call API to get the user profile data
      // const response = await this.$root.callApi('profile', 'get_profile', {});
      // this.profileData = response.data;

      this.isLoading = false;

      // Start animations after loading
      this.$nextTick(() => {
        this.animationReady = true;

        // Add entrance animation
        document.querySelector('.tropical-profile').classList.add('page-enter');

        // Animate initial cards
        setTimeout(() => {
          this.animateOnScroll();
        }, 300);
      });
    }, 800);

    // Setup scroll listener for parallax effects
    window.addEventListener('scroll', this.handleScroll);
    window.addEventListener('scroll', this.animateOnScroll);

    // Add the CSS styles for the tropical rainforest theme
    const styleElement = document.createElement('style');
    styleElement.textContent = `
        /* Base Styles */
        .tropical-profile {
          --primary: #1A3C28;
          --secondary: #DDA333;
          --accent: #A3BA95;
          --cream: #EADDCB;
          --muted: #889FAC;
          --text-dark: #1E1E1E;
          --text-light: #FFFFFF;
          
          font-family: 'Arial', sans-serif;
          background-color: var(--cream);
          min-height: 100vh;
          position: relative;
          overflow-x: hidden;
          opacity: 0;
          transition: opacity 0.6s ease-out;
        }
        
        .tropical-profile.page-enter {
          opacity: 1;
        }
        
        .tropical-profile.page-exit {
          opacity: 0;
          transform: translateY(20px);
          transition: opacity 0.4s ease-out, transform 0.4s ease-out;
        }
  
        /* Loading Overlay */
        .loading-overlay {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: rgba(26, 60, 40, 0.8);
          display: flex;
          justify-content: center;
          align-items: center;
          z-index: 1000;
        }
  
        .loading-spinner {
          display: flex;
          flex-direction: column;
          align-items: center;
          animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
          0% { transform: scale(1); }
          50% { transform: scale(1.05); }
          100% { transform: scale(1); }
        }
  
        .loading-text {
          margin-top: 16px;
          color: white;
          font-weight: 500;
        }
  
        /* Notification Toast */
        .notification-toast {
          position: fixed;
          bottom: 80px;
          left: 50%;
          transform: translateX(-50%);
          padding: 12px 24px;
          border-radius: 20px;
          font-weight: 600;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          z-index: 900;
          animation: toast-enter 0.5s ease-out forwards;
        }
        
        @keyframes toast-enter {
          from {
            opacity: 0;
            transform: translateX(-50%) translateY(30px);
          }
          to {
            opacity: 1;
            transform: translateX(-50%) translateY(0);
          }
        }
  
        .notification-toast.success {
          background-color: var(--accent);
          color: var(--primary);
        }
  
        .notification-toast.error {
          background-color: #B71C1C;
          color: white;
        }
  
        .notification-toast.info {
          background-color: var(--muted);
          color: white;
        }
  
        .fade-enter-active,
        .fade-leave-active {
          transition: opacity 0.3s, transform 0.3s;
        }
  
        .fade-enter-from,
        .fade-leave-to {
          opacity: 0;
          transform: translateX(-50%) translateY(20px);
        }
  
        /* Background Decorations */
        .background-decoration {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: -1;
          overflow: hidden;
        }
  
        .bg-leaf {
          position: absolute;
          border-radius: 50% 50% 0 50%;
          opacity: 0.07;
          background-color: var(--accent);
          transition: transform 0.5s ease-out;
        }
  
        .bg-leaf-1 {
          width: 400px;
          height: 400px;
          top: -200px;
          left: -100px;
        }
  
        .bg-leaf-2 {
          width: 300px;
          height: 300px;
          bottom: 10%;
          right: -150px;
        }
        
        /* Floating Leaves Animation */
        .floating-leaf {
          position: absolute;
          background-color: var(--accent);
          opacity: 0.05;
          border-radius: 50% 50% 0 50%;
          z-index: -1;
          transition: transform 0.5s ease-out;
        }
        
        @keyframes float {
          0% { transform: translateY(0) rotate(0deg); }
          50% { transform: translateY(-15px) rotate(5deg); }
          100% { transform: translateY(0) rotate(0deg); }
        }
        
        @keyframes sway {
          0% { transform: translateX(0) rotate(0deg); }
          50% { transform: translateX(10px) rotate(5deg); }
          100% { transform: translateX(0) rotate(0deg); }
        }
  
        .forest-silhouette {
          position: absolute;
          bottom: 0;
          left: 0;
          width: 100%;
          height: 100px;
          background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1440 320"><path fill="%231A3C28" fill-opacity="0.05" d="M0,224L48,213.3C96,203,192,181,288,181.3C384,181,480,203,576,197.3C672,192,768,160,864,165.3C960,171,1056,213,1152,208C1248,203,1344,149,1392,122.7L1440,96L1440,320L1392,320C1344,320,1248,320,1152,320C1056,320,960,320,864,320C768,320,672,320,576,320C480,320,384,320,288,320C192,320,96,320,48,320L0,320Z"></path></svg>');
          background-size: cover;
          opacity: 1;
          transition: transform 0.5s ease-out;
        }
  
        /* Profile Header */
        .profile-header {
          background-color: var(--primary);
          height: 180px;
          border-radius: 0 0 30px 30px;
          box-shadow: 0 8px 20px rgba(0,0,0,0.15);
          padding: 20px;
          position: relative;
          overflow: hidden;
          transition: transform 0.5s ease-out;
        }
  
        .leaf-decoration {
          position: absolute;
          width: 150px;
          height: 150px;
          background-color: rgba(255,255,255,0.05);
          border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
          z-index: 0;
          animation: pulse-slow 8s infinite alternate;
        }
        
        @keyframes pulse-slow {
          0% { transform: scale(1) rotate(35deg); }
          100% { transform: scale(1.1) rotate(40deg); }
        }
        
        @keyframes pulse-slow-reverse {
          0% { transform: scale(1) rotate(-15deg); }
          100% { transform: scale(1.1) rotate(-20deg); }
        }
  
        .leaf-decoration.left {
          top: -30px;
          left: -50px;
          animation: pulse-slow 8s infinite alternate;
        }
  
        .leaf-decoration.right {
          bottom: -60px;
          right: -30px;
          animation: pulse-slow-reverse 8s infinite alternate;
        }
  
        .profile-header-content {
          position: relative;
          z-index: 1;
          height: 100%;
          display: flex;
          flex-direction: column;
          justify-content: space-between;
        }
  
        .back-button {
          display: flex;
          align-items: center;
          color: white;
          opacity: 0.8;
          cursor: pointer;
          transition: all 0.3s;
          width: fit-content;
          overflow: hidden;
          position: relative;
        }
        
        .back-button::after {
          content: "";
          position: absolute;
          bottom: -2px;
          left: 0;
          width: 0;
          height: 1px;
          background-color: white;
          transition: width 0.3s ease;
        }
  
        .back-button:hover {
          opacity: 1;
          transform: translateX(-3px);
        }
        
        .back-button:hover::after {
          width: 100%;
        }
  
        .back-button span {
          margin-left: 8px;
        }
  
        .header-title {
          font-size: 32px;
          font-weight: 700;
          color: white;
          margin: 0;
          margin-top: 24px;
          margin-bottom: 16px;
          background: linear-gradient(90deg, #FFFFFF, #DDA333);
          -webkit-background-clip: text;
          background-clip: text;
          -webkit-text-fill-color: transparent;
          display: inline-block;
          animation: shimmer 4s infinite;
        }
        
        @keyframes shimmer {
          0% { background-position: -100% 0; }
          100% { background-position: 200% 0; }
        }
  
        /* Profile Content */
        .profile-content {
          padding: 0 16px 80px 16px;
          max-width: 800px;
          margin: 0 auto;
        }
  
        /* View Profile */
        .view-profile {
          margin-top: 20px;
          opacity: 1;
          transform: translateY(0);
          transition: opacity 0.3s, transform 0.3s;
        }
        
        .view-profile.fade-out {
          opacity: 0;
          transform: translateY(20px);
        }
        
        .view-profile.fade-in {
          animation: fade-in 0.5s forwards;
        }
        
        @keyframes fade-in {
          from {
            opacity: 0;
            transform: translateY(20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
  
        .profile-avatar-section {
          display: flex;
          align-items: center;
          margin-bottom: 32px;
        }
  
        .profile-avatar {
          margin-right: 24px;
          position: relative;
        }
        
        .profile-avatar .v-avatar {
          transition: transform 0.3s;
          box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .profile-avatar .v-avatar:hover {
          transform: scale(1.05);
        }
  
        .profile-name-section {
          display: flex;
          flex-direction: column;
        }
  
        .profile-name {
          font-size: 28px;
          font-weight: 700;
          color: var(--primary);
          margin: 0;
          margin-bottom: 8px;
          position: relative;
          display: inline-block;
        }
        
        .profile-name::after {
          content: "";
          position: absolute;
          bottom: 0;
          left: 0;
          width: 0;
          height: 2px;
          background-color: var(--secondary);
          transition: width 0.5s ease;
        }
        
        .profile-name:hover::after {
          width: 100%;
        }
  
        .membership-badge {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 4px 12px;
          border-radius: 20px;
          font-size: 14px;
          font-weight: 600;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
          width: fit-content;
          transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .membership-badge:hover {
          transform: translateY(-3px);
          box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
  
        .profile-details {
          display: flex;
          flex-direction: column;
          gap: 24px;
          margin-bottom: 32px;
        }
  
        .detail-card {
          background-color: white;
          border-radius: 16px;
          padding: 24px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          position: relative;
          overflow: hidden;
          transition: all 0.5s;
          opacity: 0;
          transform: translateY(30px);
        }
        
        .detail-card.card-visible {
          opacity: 1;
          transform: translateY(0);
        }
        
        .detail-card:hover {
          transform: translateY(-4px);
          box-shadow: 0 8px 15px rgba(0,0,0,0.1);
        }
  
        .detail-card::before {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          width: 5px;
          height: 100%;
          background-color: var(--accent);
          transition: height 0.5s;
        }
        
        .detail-card:hover::before {
          height: 100%;
          background-color: var(--secondary);
        }
  
        .detail-section-title {
          font-size: 18px;
          font-weight: 700;
          color: var(--primary);
          margin: 0;
          margin-bottom: 20px;
          position: relative;
          display: inline-block;
        }
        
        .detail-section-title::after {
          content: "";
          position: absolute;
          bottom: -5px;
          left: 0;
          width: 30px;
          height: 2px;
          background-color: var(--secondary);
          transition: width 0.3s ease;
        }
        
        .detail-card:hover .detail-section-title::after {
          width: 100%;
        }
  
        .detail-item {
          display: flex;
          flex-direction: column;
          margin-bottom: 16px;
          transition: transform 0.3s;
        }
        
        .detail-item:hover {
          transform: translateX(5px);
        }
  
        .detail-item:last-child {
          margin-bottom: 0;
        }
  
        .detail-label {
          display: flex;
          align-items: center;
          gap: 8px;
          font-size: 14px;
          color: var(--muted);
          margin-bottom: 4px;
        }
        
        .detail-label .v-icon {
          transition: transform 0.3s;
        }
        
        .detail-item:hover .detail-label .v-icon {
          transform: scale(1.2);
          color: var(--secondary) !important;
        }
  
        .detail-value {
          font-size: 16px;
          color: var(--text-dark);
          font-weight: 500;
          padding-left: 28px;
          transition: all 0.3s;
          position: relative;
        }
        
        .detail-value.value-updated {
          animation: highlight 1s ease;
        }
        
        @keyframes highlight {
          0% { background-color: transparent; }
          30% { background-color: rgba(221, 163, 51, 0.2); }
          100% { background-color: transparent; }
        }
  
        /* Account Settings Card */
        .settings-card {
          background-color: white;
          border-radius: 16px;
          padding: 24px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          position: relative;
          overflow: hidden;
          transition: all 0.5s;
          opacity: 0;
          transform: translateY(30px);
          margin-bottom: 32px;
        }
        
        .settings-card.card-visible {
          opacity: 1;
          transform: translateY(0);
        }
        
        .settings-card:hover {
          transform: translateY(-4px);
          box-shadow: 0 8px 15px rgba(0,0,0,0.1);
        }
        
        .settings-card::before {
          content: "";
          position: absolute;
          top: 0;
          left: 0;
          width: 5px;
          height: 100%;
          background-color: var(--secondary);
        }
        
        .settings-menu {
          list-style: none;
          padding: 0;
          margin: 0;
        }
        
        .settings-menu-item {
          padding: 12px 0;
          display: flex;
          align-items: center;
          border-bottom: 1px solid rgba(0,0,0,0.05);
          cursor: pointer;
          transition: all 0.3s;
        }
        
        .settings-menu-item:last-child {
          border-bottom: none;
        }
        
        .settings-menu-item:hover {
          transform: translateX(5px);
        }
        
        .settings-menu-item .item-icon {
          margin-right: 16px;
          display: flex;
          align-items: center;
          justify-content: center;
          width: 36px;
          height: 36px;
          border-radius: 50%;
          background-color: rgba(26, 60, 40, 0.05);
          transition: all 0.3s;
        }
        
        .settings-menu-item:hover .item-icon {
          background-color: var(--accent);
          transform: scale(1.1);
        }
        
        .settings-menu-item:hover .item-icon .v-icon {
          color: var(--primary) !important;
        }
        
        .settings-menu-item .item-title {
          flex: 1;
          font-size: 16px;
          font-weight: 500;
          color: var(--text-dark);
        }
        
        .logout-item {
          margin-top: 12px;
          padding-top: 12px;
          border-top: 1px dashed rgba(0,0,0,0.1);
        }
        
        .logout-item .item-icon {
          background-color: rgba(183, 28, 28, 0.05);
        }
        
        .logout-item:hover .item-icon {
          background-color: rgba(183, 28, 28, 0.2);
        }
        
        .logout-item .item-title {
          color: var(--error);
        }
  
        .profile-actions {
          display: flex;
          justify-content: center;
        }
  
        .edit-btn {
          min-width: 200px;
          box-shadow: 0 4px 10px rgba(26, 60, 40, 0.2) !important;
          font-weight: 600 !important;
          letter-spacing: 0.5px;
          text-transform: none !important;
          transition: all 0.3s !important;
          overflow: hidden;
          position: relative;
        }
        
        .edit-btn::after {
          content: "";
          position: absolute;
          top: 50%;
          left: 50%;
          width: 5px;
          height: 5px;
          background: rgba(255, 255, 255, 0.5);
          opacity: 0;
          border-radius: 100%;
          transform: scale(1, 1) translate(-50%);
          transform-origin: 50% 50%;
        }
  
        .edit-btn:hover {
          transform: translateY(-3px) scale(1.02);
          box-shadow: 0 6px 15px rgba(26, 60, 40, 0.3) !important;
        }
        
        .edit-btn:active::after {
          animation: ripple 0.6s ease-out;
        }
        
        @keyframes ripple {
          0% {
            transform: scale(0, 0);
            opacity: 0.5;
          }
          100% {
            transform: scale(20, 20);
            opacity: 0;
          }
        }
  
        /* Edit Profile */
        .edit-profile {
          background-color: white;
          border-radius: 16px;
          padding: 24px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          margin-top: -50px;
          position: relative;
          overflow: hidden;
          opacity: 0;
          transform: translateY(20px);
          transition: opacity 0.3s, transform 0.3s;
        }
        
        .edit-profile.fade-in {
          animation: fade-in 0.5s forwards;
        }
        
        .edit-profile.fade-out {
          opacity: 0;
          transform: translateY(20px);
        }
        
        .edit-profile::after {
          content: "";
          position: absolute;
          bottom: -80px;
          right: -80px;
          width: 200px;
          height: 200px;
          background-color: var(--accent);
          opacity: 0.05;
          border-radius: 50%;
          z-index: 0;
          animation: pulse 15s infinite alternate;
        }
  
        .photo-upload-section {
          display: flex;
          justify-content: center;
          margin-bottom: 32px;
        }
  
        .profile-avatar-editor {
          display: flex;
          flex-direction: column;
          align-items: center;
          position: relative;
        }
        
        .profile-avatar-editor .v-avatar {
          transition: all 0.3s;
        }
        
        .profile-avatar-editor .v-avatar.photo-updated {
          animation: photo-update 0.7s ease;
        }
        
        .profile-avatar-editor .v-avatar.photo-removed {
          animation: photo-remove 0.7s ease;
        }
        
        @keyframes photo-update {
          0% { transform: scale(1); box-shadow: 0 0 0 rgba(221, 163, 51, 0.5); }
          50% { transform: scale(1.1); box-shadow: 0 0 20px rgba(221, 163, 51, 0.5); }
          100% { transform: scale(1); box-shadow: 0 0 0 rgba(221, 163, 51, 0.5); }
        }
        
        @keyframes photo-remove {
          0% { transform: scale(1); }
          10% { transform: scale(0.9) rotate(-5deg); }
          20% { transform: scale(0.9) rotate(5deg); }
          30% { transform: scale(0.9) rotate(-5deg); }
          40% { transform: scale(0.9) rotate(5deg); }
          50% { transform: scale(1); }
          100% { transform: scale(1); }
        }
  
        .avatar-remove {
          position: absolute;
          top: 0;
          right: 0;
          background-color: var(--primary);
          color: white;
          width: 24px;
          height: 24px;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          cursor: pointer;
          transition: all 0.3s;
          z-index: 2;
        }
  
        .avatar-remove:hover {
          transform: scale(1.2);
          background-color: var(--error);
        }
  
        .upload-actions {
          margin-top: 16px;
        }
        
        .upload-actions .v-btn {
          transition: all 0.3s;
          overflow: hidden;
          position: relative;
        }
        
        .upload-actions .v-btn::after {
          content: "";
          position: absolute;
          top: 50%;
          left: 50%;
          width: 5px;
          height: 5px;
          background: rgba(255, 255, 255, 0.5);
          opacity: 0;
          border-radius: 100%;
          transform: scale(1, 1) translate(-50%);
          transform-origin: 50% 50%;
        }
        
        .upload-actions .v-btn:hover {
          transform: translateY(-3px);
          box-shadow: 0 4px 8px rgba(0,0,0,0.2) !important;
        }
        
        .upload-actions .v-btn:active::after {
          animation: ripple 0.6s ease-out;
        }
  
        .hidden-input {
          display: none;
        }
  
        .form-section {
          margin-bottom: 32px;
          position: relative;
          z-index: 1;
          animation: slideUp 0.5s ease-out;
          animation-fill-mode: both;
        }
        
        @keyframes slideUp {
          from {
            opacity: 0;
            transform: translateY(20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        .form-section:nth-child(2) {
          animation-delay: 0.1s;
        }
        
        .form-section:nth-child(3) {
          animation-delay: 0.2s;
        }
  
        .form-section-title {
          font-size: 18px;
          font-weight: 700;
          color: var(--primary);
          margin: 0;
          margin-bottom: 20px;
          position: relative;
          display: inline-block;
        }
        
        .form-section-title::after {
          content: "";
          position: absolute;
          bottom: -5px;
          left: 0;
          width: 30px;
          height: 2px;
          background-color: var(--secondary);
          transition: width 0.3s ease;
        }
        
        .form-section:hover .form-section-title::after {
          width: 100%;
        }
  
        .membership-info {
          background-color: var(--cream);
          padding: 20px;
          border-radius: 12px;
          margin-bottom: 32px;
          transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .membership-info:hover {
          transform: translateY(-5px);
          box-shadow: 0 8px 15px rgba(0,0,0,0.07);
        }
  
        .readonly-field {
          margin-bottom: 16px;
          display: flex;
          flex-direction: column;
          transition: transform 0.3s;
        }
        
        .readonly-field:hover {
          transform: translateX(5px);
        }
  
        .readonly-label {
          font-size: 14px;
          color: var(--muted);
          margin-bottom: 4px;
        }
  
        .readonly-value {
          font-size: 16px;
          color: var(--text-dark);
          font-weight: 500;
        }
  
        .membership-chip {
          width: fit-content;
          font-weight: 600;
          transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .membership-chip:hover {
          transform: translateY(-3px);
          box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
  
        .form-actions {
          display: flex;
          justify-content: flex-end;
          gap: 16px;
          position: relative;
          z-index: 1;
        }
  
        .cancel-btn, .save-btn {
          min-width: 120px;
          text-transform: none !important;
          font-weight: 600 !important;
          transition: all 0.3s !important;
          position: relative;
          overflow: hidden;
        }
        
        .cancel-btn::after, .save-btn::after {
          content: "";
          position: absolute;
          top: 50%;
          left: 50%;
          width: 5px;
          height: 5px;
          background: rgba(255, 255, 255, 0.5);
          opacity: 0;
          border-radius: 100%;
          transform: scale(1, 1) translate(-50%);
          transform-origin: 50% 50%;
        }
        
        .cancel-btn:hover, .save-btn:hover:not(:disabled) {
          transform: translateY(-3px);
        }
        
        .cancel-btn:active::after, .save-btn:active::after {
          animation: ripple 0.6s ease-out;
        }
        
        .save-btn {
          box-shadow: 0 4px 10px rgba(26, 60, 40, 0.2) !important;
        }
        
        .save-btn:hover:not(:disabled) {
          box-shadow: 0 6px 15px rgba(26, 60, 40, 0.3) !important;
        }
        
        /* Form field animations */
        .v-text-field {
          transition: transform 0.3s;
        }
        
        .v-text-field:focus-within {
          transform: translateY(-3px);
        }
        
        .v-text-field .v-field__outline {
          transition: all 0.3s !important;
        }
        
        .v-text-field:hover .v-field__outline {
          opacity: 0.8;
        }
  
        /* Responsive Adjustments */
        @media (max-width: 768px) {
          .profile-avatar-section {
            flex-direction: column;
            align-items: center;
            text-align: center;
          }
          
          .profile-avatar {
            margin-right: 0;
            margin-bottom: 16px;
          }
          
          .profile-name-section {
            align-items: center;
          }
          
          .detail-item {
            margin-bottom: 24px;
          }
          
          .form-actions {
            flex-direction: column-reverse;
            gap: 12px;
          }
          
          .cancel-btn, .save-btn {
            width: 100%;
          }
        }
      `;
    document.head.appendChild(styleElement);
  },
  beforeUnmount() {
    window.removeEventListener('scroll', this.handleScroll);
    window.removeEventListener('scroll', this.animateOnScroll);
  },
  template: `
    <div class="tropical-profile">
      <!-- Loading overlay -->
      <div class="loading-overlay" v-if="isLoading">
        <div class="loading-spinner">
          <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
          <div class="loading-text">Loading your profile...</div>
        </div>
      </div>
      
      <!-- Notification toast -->
      <transition name="fade">
        <div class="notification-toast" :class="notificationType" v-if="showNotification">
          {{ notificationMessage }}
        </div>
      </transition>
      
      <!-- Floating leaves decorations -->
      <div 
        v-for="(leaf, index) in floatingLeaves" 
        :key="'floating-' + index"
        class="floating-leaf"
        :style="{
          width: leaf.size + 'px',
          height: leaf.size + 'px',
          left: leaf.x + '%',
          top: leaf.y + 'px',
          transform: \`translate(\${leaf.x}px, \${leaf.y}px) rotate(\${leaf.rotation}deg)\`,
          animationName: index % 2 === 0 ? 'float' : 'sway',
          animationDuration: (3 + index) + 's',
          animationDelay: (index * 0.5) + 's',
          animationIterationCount: 'infinite',
          animationDirection: 'alternate'
        }"
      ></div>
      
      <!-- Background decorations with parallax -->
      <div class="background-decoration">
        <div class="bg-leaf bg-leaf-1" :style="bgLeaf1Style"></div>
        <div class="bg-leaf bg-leaf-2" :style="bgLeaf2Style"></div>
        <div class="forest-silhouette" :style="silhouetteStyle"></div>
      </div>
      
      <!-- Profile header with parallax -->
      <div class="profile-header" :style="headerParallaxStyle">
        <div class="leaf-decoration left"></div>
        <div class="leaf-decoration right"></div>
        
        <div class="profile-header-content">
          <!--<div class="back-button" @click="navigateBack">
            <v-icon color="white">mdi-arrow-left</v-icon>
            <span>Back</span>
          </div>-->
          
          <h1 class="header-title">My Profile</h1>
        </div>
      </div>
      
      <!-- Profile content -->
      <div class="profile-content">
        <!-- View mode -->
        <div v-if="!isEditing" class="view-profile">
          <!-- Profile avatar and membership -->
          <div class="profile-avatar-section">
            <div class="profile-avatar">
              <v-avatar size="100" v-if="profileData.avatar">
                <v-img :src="profileData.avatar" alt="Profile Picture"></v-img>
              </v-avatar>
              <v-avatar size="100" :color="theme.accent" v-else>
                <span class="text-h4" :style="{color: theme.primary}">{{ userInitials }}</span>
              </v-avatar>
            </div>
            
            <div class="profile-name-section">
              <h2 class="profile-name">{{ profileData.name }}</h2>
              
              <div class="membership-badge" :style="{backgroundColor: membershipBadge.color}">
                <v-icon size="18" :color="membershipBadge.textColor">{{ membershipBadge.icon }}</v-icon>
                <span :style="{color: membershipBadge.textColor}">{{ membershipBadge.text }}</span>
              </div>
            </div>
          </div>
          

          
          <!-- Profile details with scroll reveal -->
          <div class="profile-details">

            <!-- Edit button with ripple effect -->
            <div class="profile-actions">
                <v-btn 
                color="primary" 
                @click="editProfile"
                rounded="pill"
                size="large"
                class="edit-btn"
                >
                <v-icon start>mdi-pencil</v-icon>
                Edit Profile
                </v-btn>
            </div>

            <div class="detail-card card-visible">
              <h3 class="detail-section-title">Membership Details</h3>
              
              <div class="detail-item">
                <div class="detail-label">
                  <v-icon size="20" color="primary">mdi-card-account-details</v-icon>
                  <span>Membership ID</span>
                </div>
                <div class="detail-value">{{ profileData.membershipId }}</div>
              </div>
              
              <div class="detail-item">
                <div class="detail-label">
                  <v-icon size="20" color="primary">mdi-calendar</v-icon>
                  <span>Member Since</span>
                </div>
                <div class="detail-value">{{ formattedJoinDate }}</div>
              </div>
            </div>
            
            <div class="detail-card card-visible">
              <h3 class="detail-section-title">Contact Information</h3>
              
              <div class="detail-item">
                <div class="detail-label">
                  <v-icon size="20" color="primary">mdi-email</v-icon>
                  <span>Email</span>
                </div>
                <div class="detail-value">{{ profileData.email }}</div>
              </div>
              
              <div class="detail-item">
                <div class="detail-label">
                  <v-icon size="20" color="primary">mdi-phone</v-icon>
                  <span>Phone</span>
                </div>
                <div class="detail-value">{{ profileData.phone }}</div>
              </div>
              
              <div class="detail-item">
                <div class="detail-label">
                  <v-icon size="20" color="primary">mdi-map-marker</v-icon>
                  <span>Address</span>
                </div>
                <div class="detail-value">{{ profileData.address }}</div>
              </div>
            </div>
          </div>
          


          <!-- Account Settings -->
          <div class="detail-card card-visible">
            <h3 class="detail-section-title">Account Settings</h3>
            
            <ul class="settings-menu">
              <li 
                v-for="(item, i) in settingsMenu" 
                :key="i" 
                class="settings-menu-item"
                @click="openSetting(item)"
              >
                <div class="item-icon">
                  <v-icon :color="theme.primary">{{ item.icon }}</v-icon>
                </div>
                <div class="item-title">{{ item.title }}</div>
                <v-icon color="grey">mdi-chevron-right</v-icon>
              </li>
              
              <!-- Logout option -->
              <li class="settings-menu-item logout-item" @click="confirmLogout">
                <div class="item-icon">
                  <v-icon color="error">mdi-logout</v-icon>
                </div>
                <div class="item-title">Logout</div>
                <v-icon color="grey">mdi-chevron-right</v-icon>
              </li>
            </ul>
          </div>
        </div>


        <!-- Edit mode with animated transitions -->
        <div v-else class="edit-profile">
          <v-form ref="form" v-model="valid">
            <!-- Profile photo editor with animations -->
            <div class="photo-upload-section">
              <div class="profile-avatar-editor">
                <v-avatar size="100" v-if="photoPreview || editableProfile.avatar">
                  <v-img :src="photoPreview || editableProfile.avatar" alt="Profile Picture"></v-img>
                  <div class="avatar-remove" @click="removePhoto">
                    <v-icon size="small">mdi-close</v-icon>
                  </div>
                </v-avatar>
                <v-avatar size="100" :color="theme.accent" v-else>
                  <span class="text-h4" :style="{color: theme.primary}">{{ userInitials }}</span>
                </v-avatar>
                
                <div class="upload-actions">
                  <v-btn 
                    variant="outlined" 
                    color="secondary"
                    size="small"
                    prepend-icon="mdi-camera"
                    @click="$refs.photoInput.click()"
                  >
                    Change Photo
                  </v-btn>
                  <input 
                    ref="photoInput"
                    type="file"
                    accept="image/*"
                    class="hidden-input"
                    @change="onPhotoSelected"
                  />
                </div>
              </div>
            </div>
            
            <!-- Personal Information with staggered animations -->
            <div class="form-section">
              <h3 class="form-section-title">Personal Information</h3>
              
              <v-text-field
                v-model="editableProfile.name"
                label="Full Name"
                variant="outlined"
                :rules="nameRules"
                required
                color="primary"
                bg-color="white"
              ></v-text-field>
              
              <v-text-field
                v-model="editableProfile.email"
                label="Email Address"
                variant="outlined"
                :rules="emailRules"
                required
                color="primary"
                bg-color="white"
              ></v-text-field>
              
              <v-text-field
                v-model="editableProfile.phone"
                label="Phone Number"
                variant="outlined"
                :rules="phoneRules"
                required
                color="primary"
                bg-color="white"
              ></v-text-field>
              
              <v-textarea
                v-model="editableProfile.address"
                label="Address"
                variant="outlined"
                color="primary"
                bg-color="white"
                rows="3"
              ></v-textarea>
            </div>
            
            <!-- Membership Information (read-only) with hover effects -->
            <div class="form-section membership-info">
              <h3 class="form-section-title">Membership Information</h3>
              
              <div class="readonly-field">
                <div class="readonly-label">Membership ID</div>
                <div class="readonly-value">{{ editableProfile.membershipId }}</div>
              </div>
              
              <div class="readonly-field">
                <div class="readonly-label">Membership Type</div>
                <div class="readonly-value">
                  <v-chip 
                    color="primary" 
                    label
                    class="membership-chip"
                  >
                    <v-icon start size="small">{{ membershipBadge.icon }}</v-icon>
                    {{ membershipBadge.text }}
                  </v-chip>
                </div>
              </div>
            </div>
            
            <!-- Form Actions with ripple effects -->
            <div class="form-actions">
              <v-btn 
                variant="outlined" 
                color="primary"
                @click="cancelEdit"
                size="large"
                class="cancel-btn"
              >
                Cancel
              </v-btn>
              
              <v-btn 
                color="primary"
                @click="saveProfile"
                size="large"
                :loading="isLoading"
                :disabled="!valid"
                class="save-btn"
              >
                Save Changes
              </v-btn>
            </div>
          </v-form>
        </div>
      </div>
      
      <!-- Change Password Dialog -->
      <v-dialog v-model="showPasswordDialog" max-width="500px" persistent>
        <v-card>
          <v-card-title class="text-h5" :style="{color: theme.primary}">
            <v-icon :color="theme.primary" class="mr-2">mdi-lock-reset</v-icon>
            Change Password
          </v-card-title>
          
          <v-card-text>
            <v-form ref="passwordForm" v-model="valid">
              <v-text-field
                v-model="passwordForm.currentPassword"
                label="Current Password"
                type="password"
                variant="outlined"
                :rules="passwordRules.current"
                required
                color="primary"
                class="mb-3"
              ></v-text-field>
              
              <v-text-field
                v-model="passwordForm.newPassword"
                label="New Password"
                type="password"
                variant="outlined"
                :rules="passwordRules.new"
                required
                color="primary"
                class="mb-3"
              ></v-text-field>
              
              <v-text-field
                v-model="passwordForm.confirmPassword"
                label="Confirm New Password"
                type="password"
                variant="outlined"
                :rules="passwordRules.confirm"
                required
                color="primary"
              ></v-text-field>
            </v-form>
          </v-card-text>
          
          <v-card-actions>
            <v-spacer></v-spacer>
            <v-btn 
              variant="outlined" 
              @click="showPasswordDialog = false"
              :disabled="isLoading"
            >
              Cancel
            </v-btn>
            <v-btn 
              color="primary" 
              @click="changePassword"
              :loading="isLoading"
            >
              Change Password
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
      
      <!-- Change PIN Dialog -->
      <v-dialog v-model="showPinDialog" max-width="500px" persistent>
        <v-card>
          <v-card-title class="text-h5" :style="{color: theme.primary}">
            <v-icon :color="theme.primary" class="mr-2">mdi-credit-card-settings-outline</v-icon>
            Change PIN
          </v-card-title>
          
          <v-card-text>
            <v-form ref="pinForm" v-model="valid">
              <v-text-field
                v-model="pinForm.currentPin"
                label="Current PIN (if you have one)"
                type="password"
                variant="outlined"
                color="primary"
                class="mb-3"
                hint="Leave empty if this is your first time setting a PIN"
                persistent-hint
              ></v-text-field>
              
              <v-text-field
                v-model="pinForm.newPin"
                label="New PIN"
                type="password"
                variant="outlined"
                :rules="pinRules.new"
                required
                color="primary"
                class="mb-3"
                hint="4-6 digits only"
                persistent-hint
              ></v-text-field>
              
              <v-text-field
                v-model="pinForm.confirmPin"
                label="Confirm New PIN"
                type="password"
                variant="outlined"
                :rules="pinRules.confirm"
                required
                color="primary"
              ></v-text-field>
            </v-form>
          </v-card-text>
          
          <v-card-actions>
            <v-spacer></v-spacer>
            <v-btn 
              variant="outlined" 
              @click="showPinDialog = false"
              :disabled="isLoading"
            >
              Cancel
            </v-btn>
            <v-btn 
              color="primary" 
              @click="changePin"
              :loading="isLoading"
            >
              Change PIN
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
    </div>
    `
}
