export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C'
      },

      // View Details - States and Data
      isLoading: false,
      currentTab: 0,
      filterActive: false,
      dateRange: [],
      searchQuery: '',
      
      // User Data
      userData: {
        name: "Jason Tan",
        role: "platinum",
        membershipId: "TAPANG-PLAT-2023",
        balance: 0,
        rewards: 0,
        availableCredit: 0,
        pendingAmount: 0,
        loyaltyTier: "Platinum",
        loyaltyProgress: 85,
        nextTierPoints: 750
      },
      
      // Transaction History
      transactions: [
        {
          id: 'TRX-38291',
          title: "Tennis Court Booking",
          date: "June 3, 2023 • 10:30 AM",
          amount: -85.00,
          icon: "mdi-tennis",
          color: "light-green-darken-1",
          category: "Sports",
          status: "completed",
          details: "Court #3 - 2 hour session"
        },
        {
          id: 'TRX-38290',
          title: "Top Up - Credit Card",
          date: "June 2, 2023 • 6:15 PM",
          amount: 500.00,
          icon: "mdi-credit-card-plus",
          color: "blue-darken-1",
          category: "Top Up",
          status: "completed",
          details: "VISA ending 4832"
        },
        {
          id: 'TRX-38275',
          title: "Restaurant Purchase",
          date: "June 2, 2023 • 8:15 PM",
          amount: -176.40,
          icon: "mdi-silverware-fork-knife",
          color: "amber-darken-2",
          category: "Dining",
          status: "completed",
          details: "The Garden Restaurant - Table 15"
        },
        {
          id: 'TRX-38254',
          title: "Spa Reservation",
          date: "May 31, 2023 • 3:00 PM",
          amount: -320.00,
          icon: "mdi-spa",
          color: "teal-lighten-1",
          category: "Wellness",
          status: "completed",
          details: "Premium Massage Package - 90 min"
        },
        {
          id: 'TRX-38201',
          title: "Monthly Membership Fee",
          date: "May 30, 2023 • 12:00 AM",
          amount: -450.00,
          icon: "mdi-account-check",
          color: "purple-darken-1",
          category: "Membership",
          status: "completed",
          details: "Platinum Tier - June 2023"
        },
        {
          id: 'TRX-38190',
          title: "Golf Equipment Purchase",
          date: "May 29, 2023 • 11:20 AM",
          amount: -235.50,
          icon: "mdi-golf",
          color: "green-darken-2",
          category: "Sports",
          status: "completed",
          details: "Pro Shop - Premium Golf Balls"
        },
        {
          id: 'TRX-38156',
          title: "Top Up - Bank Transfer",
          date: "May 28, 2023 • 10:05 AM",
          amount: 1000.00,
          icon: "mdi-bank-transfer",
          color: "blue-darken-1",
          category: "Top Up",
          status: "completed",
          details: "Transfer from HSBC Savings"
        },
        {
          id: 'TRX-38111',
          title: "Event Ticket - Wine Tasting",
          date: "May 27, 2023 • 9:45 AM",
          amount: -120.00,
          icon: "mdi-glass-wine",
          color: "deep-purple-lighten-1",
          category: "Events",
          status: "completed",
          details: "2 tickets - Premium Wine Tasting"
        }
      ],
      
      // Financial Analytics
      financialData: {
        monthly: {
          totalSpent: 2850.00,
          totalSent: 1200.00,
          totalReceived: 1500.00,
          topCategory: "Dining",
          averageTransaction: 210.50,
          mostVisited: "The Garden Restaurant"
        },
        summary: [
          { month: "January", spent: 2450 },
          { month: "February", spent: 2100 },
          { month: "March", spent: 2300 },
          { month: "April", spent: 2650 },
          { month: "May", spent: 2850 },
          { month: "June", spent: 1250 } // Current month (partial)
        ]
      },
      
      // Filter options
      filterOptions: {
        categories: ["All", "Dining", "Sports", "Wellness", "Events", "Membership", "Top Up"],
        dateRanges: ["Last 7 days", "Last 30 days", "Last 3 months", "Custom range"],
        statuses: ["All", "Completed", "Pending", "Failed"]
      },
      
      // Selected filters
      selectedFilters: {
        category: "All",
        dateRange: "Last 30 days",
        status: "All"
      },
      
      // Notification system
      showNotification: false,
      notificationMessage: '',
      
      // Animation states
      hoveredTransaction: null
    }
  },
  computed: {
    // Format currency consistently
    formattedBalance() {
      return this.formatCredit(this.userData.balance)
    },
    
    // Format rewards with punctuation
    creditsSpent() {
      return this.formatCredit(this.userData.availableCredit)
    },

    creditsInTransit() {
      return this.formatCredit(this.userData.pendingAmount)
    },
    
    // Filter transactions based on selected filters and search query
    filteredTransactions() {
      let filtered = [...this.transactions]
      
      // Apply category filter
      if (this.selectedFilters.category !== "All") {
        filtered = filtered.filter(t => t.category === this.selectedFilters.category)
      }
      
      // Apply status filter
      if (this.selectedFilters.status !== "All") {
        filtered = filtered.filter(t => t.status === this.selectedFilters.status.toLowerCase())
      }
      
      // Apply search query
      if (this.searchQuery) {
        const query = this.searchQuery.toLowerCase()
        filtered = filtered.filter(t => 
          t.title.toLowerCase().includes(query) || 
          t.id.toLowerCase().includes(query) ||
          t.details.toLowerCase().includes(query)
        )
      }
      
      return filtered
    },
    
    // Calculate spending trends
    spendingTrend() {
      const data = this.financialData.summary
      const currentMonth = data[data.length - 1]
      const previousMonth = data[data.length - 2]
      
      // Calculate percentage change
      const percentageChange = ((currentMonth.spent - previousMonth.spent) / previousMonth.spent) * 100
      
      return {
        percentage: Math.abs(percentageChange).toFixed(1),
        direction: percentageChange >= 0 ? 'up' : 'down',
        icon: percentageChange >= 0 ? 'mdi-arrow-up' : 'mdi-arrow-down',
        color: percentageChange >= 0 ? 'error' : 'success' // up is red (spending more), down is green (saving)
      }
    }
  },
  methods: {
    formatCurrency(amount) {
      return "RM " + parseFloat(amount).toFixed(2)
    },

    formatCredit(amount) {
      return `<i class="mdi mdi-circle-multiple-outline"></i> ${parseFloat(amount).toFixed(2)}`
    },
    
    // Format amount with color based on positive/negative
    formatAmount(amount) {
      const isPositive = amount >= 0
      return {
        text: `${isPositive ? '+' : ''}${this.formatCredit(amount)}`,
        color: isPositive ? 'success' : 'darkText'
      }
    },
    
    // Handle downloading statement
    downloadStatement() {
      this.showNotification = true
      this.notificationMessage = 'Preparing your statement for download...'
      
      // Simulate download process
      setTimeout(() => {
        this.showNotification = true
        this.notificationMessage = 'Statement downloaded successfully!'
        setTimeout(() => {
          this.showNotification = false
        }, 3000)
      }, 2000)
    },
    
    // Apply selected filters
    applyFilters() {
      this.filterActive = false
      this.fetchTransactionHistory()
    },
    
    // Reset filters to defaults
    resetFilters() {
      this.selectedFilters = {
        category: "All",
        dateRange: "Last 30 days",
        status: "All"
      }
      this.searchQuery = ''
      this.fetchTransactionHistory()
    },
    
// Fetch wallet details (balance and transactions)
    async fetchWalletDetails() {
      this.isLoading = true;
      try {
        const params = {
          user_id: sessionStorage.getItem('id'),
          limit: 10,
          offset: 0
        };
        
        // Apply filters if active
        if (this.selectedFilters.category !== "All") {
          params.category = this.selectedFilters.category;
        }
        
        if (this.selectedFilters.status !== "All") {
          params.status = this.selectedFilters.status.toLowerCase();
        }
        
        if (this.searchQuery) {
          params.search = this.searchQuery;
        }
        
        // Apply date range if selected
        if (this.dateRange && this.dateRange.length === 2) {
          params.start_date = this.dateRange[0];
          params.end_date = this.dateRange[1];
        }
        
        const response = await this.$root.callApi('wallet', 'wallet_details', params);
        
        if (response.data && response.data.status === "YES") {
          // Update wallet balance
          this.userData.balance = parseFloat(response.data.balance);
          this.userData.availableCredit = parseFloat(response.data.available_credit);
          this.userData.pendingAmount = parseFloat(response.data.pending_amount);
          
          // Update transactions
          this.transactions = response.data.transactions || [];
          
          // Update financial data if available
          if (response.data.financial_data) {
            this.financialData = response.data.financial_data;
          }
        } else {
          const errorMessage = response.data ? response.data.message : response.message;
          console.log('API Error:', errorMessage);
          this.showNotification = true;
          this.notificationMessage = errorMessage || 'Failed to fetch wallet details';
          setTimeout(() => { this.showNotification = false }, 3000);
        }
      } catch (error) {
        console.error('Error fetching wallet details:', error);
        this.showNotification = true;
        this.notificationMessage = 'Error fetching wallet details';
        setTimeout(() => { this.showNotification = false }, 3000);
      } finally {
        this.isLoading = false;
      }
    },
    
    // View transaction details
    viewTransactionDetails(transaction) {
      this.showNotification = true
      this.notificationMessage = 'Viewing details for ' + transaction.id
      
      // Would normally navigate to transaction detail page or show modal
      setTimeout(() => {
        this.showNotification = false
      }, 3000)
    },
    
    // Load more transactions (pagination)
    loadMoreTransactions() {
      this.isLoading = true;
      
      const params = {
        user_id: sessionStorage.getItem('id'),
        limit: 10,
        offset: this.transactions.length
      };
      
      // Apply filters if active
      if (this.selectedFilters.category !== "All") {
        params.category = this.selectedFilters.category;
      }
      
      if (this.selectedFilters.status !== "All") {
        params.status = this.selectedFilters.status.toLowerCase();
      }
      
      if (this.searchQuery) {
        params.search = this.searchQuery;
      }
      
      // Apply date range if selected
      if (this.dateRange && this.dateRange.length === 2) {
        params.start_date = this.dateRange[0];
        params.end_date = this.dateRange[1];
      }
      
      this.$root.callApi('wallet', 'wallet_details', params)
        .then(response => {
          if (response.data.status === "YES" && response.data.transactions && response.data.transactions.length > 0) {
            this.transactions = [...this.transactions, ...response.data.transactions];
            this.showNotification = true;
            this.notificationMessage = 'Loaded additional transactions';
          } else {
            this.showNotification = true;
            this.notificationMessage = 'No more transactions to load';
          }
          setTimeout(() => { this.showNotification = false }, 3000);
        })
        .catch(error => {
          console.error('Error loading more transactions:', error);
          this.showNotification = true;
          this.notificationMessage = 'Error loading more transactions';
          setTimeout(() => { this.showNotification = false }, 3000);
        })
        .finally(() => {
          this.isLoading = false;
        });
    },
    
    // Navigation methods
    goBack() {
      //this.$root.loadComponent('modules/wallet/201-wallet.js')
      this.$root.loadComponent('modules/home.js');
    },
    
    navigateTo(screen) {
      this.showNotification = true
      this.notificationMessage = 'Loading...'
      setTimeout(() => {
        this.$root.loadComponent('modules/' + screen)
      }, 600)
    }
  },
  async mounted() {
    
    // IMPORTANT: Inject the CSS styles dynamically
    const styleElement = document.createElement('style')
    styleElement.textContent = `
      /* Base Styles - Mobile First Approach */
      .wallet-details {
        --primary: #1A3C28;
        --secondary: #DDA333;
        --accent: #A3BA95;
        --cream: #EADDCB;
        --muted: #889FAC;
        --text-dark: #1E1E1E;
        --text-light: #FFFFFF;
        
        font-family: 'Arial', sans-serif;
        background-color: var(--cream);
        min-height: 100vh;
        position: relative;
        overflow-x: hidden;
        -webkit-tap-highlight-color: transparent;
      }
      
      /* Loading Overlay */
      .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(26, 60, 40, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 1000;
      }
      
      .loading-spinner {
        display: flex;
        flex-direction: column;
        align-items: center;
      }
      
      .loading-text {
        margin-top: 16px;
        color: white;
        font-weight: 500;
      }
      
      /* Notification Toast - Mobile Optimized */
      .notification-toast {
        position: fixed;
        bottom: 24px;
        left: 50%;
        transform: translateX(-50%);
        background-color: var(--secondary);
        color: var(--primary);
        padding: 14px 20px;
        border-radius: 20px;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 900;
        min-width: 80%;
        max-width: 90%;
        text-align: center;
      }
      
      .fade-enter-active,
      .fade-leave-active {
        transition: opacity 0.3s, transform 0.3s;
      }
      
      .fade-enter-from,
      .fade-leave-to {
        opacity: 0;
        transform: translateX(-50%) translateY(20px);
      }
      
      /* Background Decorations */
      .background-decoration {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: -1;
        overflow: hidden;
      }
      
      .bg-leaf {
        position: absolute;
        border-radius: 50% 50% 0 50%;
        opacity: 0.07;
        background-color: var(--accent);
        transform: rotate(45deg);
      }
      
      .bg-leaf-1 {
        width: 300px;
        height: 300px;
        top: -150px;
        left: -150px;
      }
      
      .bg-leaf-2 {
        width: 250px;
        height: 250px;
        bottom: 10%;
        right: -100px;
        transform: rotate(130deg);
      }
      
      .bg-leaf-3 {
        width: 180px;
        height: 180px;
        top: 30%;
        left: -90px;
        transform: rotate(80deg);
      }
      
      /* Header Section - Mobile Optimized */
      .wallet-header {
        background-color: var(--primary);
        padding: 16px;
        border-radius: 0 0 24px 24px;
        box-shadow: 0 6px 16px rgba(0,0,0,0.15);
        color: var(--text-light);
        position: relative;
        overflow: hidden;
      }
      
      .leaf-decoration {
        position: absolute;
        width: 120px;
        height: 120px;
        background-color: rgba(255,255,255,0.05);
        border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
        z-index: 0;
      }
      
      .leaf-decoration.left {
        top: -30px;
        left: -50px;
        transform: rotate(35deg);
      }
      
      .leaf-decoration.right {
        bottom: -60px;
        right: -30px;
        transform: rotate(-15deg);
      }
      
      .header-content {
        position: relative;
        z-index: 1;
      }
      
      .back-button {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        cursor: pointer;
        min-height: 44px; /* Mobile touch target */
      }
      
      .back-button-text {
        margin-left: 8px;
        font-weight: 500;
        font-size: 16px;
      }
      
      .page-title {
        font-size: 22px;
        font-weight: 700;
        margin-bottom: 12px;
      }
      
      /* Balance Card - Mobile Optimized */
      .balance-card {
        background-color: white;
        border-radius: 16px;
        margin: -15px 16px 20px 16px;
        padding: 16px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        position: relative;
        overflow: hidden;
      }
      
      .card-decoration {
        position: absolute;
        top: -20px;
        right: -20px;
        width: 120px;
        height: 120px;
        background-color: var(--secondary);
        opacity: 0.1;
        border-radius: 50%;
      }
      
      .balance-title {
        color: var(--muted);
        font-size: 14px;
        margin-bottom: 6px;
      }
      
      .balance-amount {
        font-size: 26px;
        font-weight: 700;
        color: var(--primary);
        margin-bottom: 14px;
      }
      
      .balance-details {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 10px;
      }
      
      .balance-detail-item {
        border-radius: 10px;
        padding: 10px;
        background-color: rgba(211, 211, 211, 0.1);
        border: 1px solid rgba(0,0,0,0.05);
      }
      
      .detail-label {
        font-size: 12px;
        color: var(--muted);
        margin-bottom: 3px;
      }
      
      .detail-value {
        font-size: 15px;
        font-weight: 600;
        color: var(--text-dark);
      }
      
      /* Tabs and Content - Mobile Optimized */
      .content-area {
        padding: 0 16px 80px 16px;
        margin-top: 20px; /* Add this line to create space below the balance-card */
      }
      
      .custom-tabs {
        background-color: transparent;
        margin-bottom: 16px;
      }
      
      .tab-item {
        font-weight: 600;
        color: var(--text-dark);
        opacity: 0.6;
      }
      
      .tab-item.active {
        color: var(--primary);
        opacity: 1;
      }
      
      /* Transaction List - Mobile Optimized */
      .transaction-list {
        margin-bottom: 24px;
      }
      
      .list-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 12px;
      }
      
      .header-title {
        font-size: 18px;
        font-weight: 700;
        color: var(--text-dark);
      }
      
      .search-and-filter {
        display: flex;
        gap: 10px;
        margin-bottom: 14px;
      }
      
      .search-input {
        flex: 1;
      }
      
      .filter-panel {
        position: relative;
        z-index: 100;
      }
      
      .filter-chip {
        cursor: pointer;
        margin-right: 6px;
        margin-bottom: 6px;
        height: 32px;
      }
      
      /* Transaction Card - Mobile Optimized */
      .transaction-card {
        background-color: white;
        border-radius: 12px;
        padding: 14px;
        margin-bottom: 10px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        cursor: pointer;
        transition: all 0.3s;
        min-height: 80px; /* Ensure good tap target */
      }
      
      .transaction-card:active {
        transform: scale(0.98);
        background-color: rgba(0,0,0,0.02);
      }
      
      .transaction-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 10px;
      }
      
      .transaction-icon-wrapper {
        width: 50px;
        height: 50px;
        border-radius: 50%; /* Changed from 10px to 50% for perfect circle */
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        background-color: #FF5252; /* Default background color */
        color: white;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
      }
      
      /* Color-specific styling for icon backgrounds */
      .light-green-darken-1 {
        background-color: #7CB342;
      }
      
      .blue-darken-1 {
        background-color: #1976D2;
      }
      
      .amber-darken-2 {
        background-color: #FF8F00;
      }
      
      .teal-lighten-1 {
        background-color: #26A69A;
      }
      
      .purple-darken-1 {
        background-color: #8E24AA;
      }
      
      .green-darken-2 {
        background-color: #388E3C;
      }
      
      .deep-purple-lighten-1 {
        background-color: #7E57C2;
      }
      
      .primary {
        background-color: #1A3C28; /* Your primary color */
      }
      
      .transaction-content {
        flex: 1;
        margin: 0 10px;
      }
      
      .transaction-title {
        font-weight: 600;
        font-size: 15px;
        color: var(--text-dark);
        margin-bottom: 2px;
      }
      
      .transaction-date {
        font-size: 12px;
        color: var(--muted);
      }
      
      .transaction-id {
        font-size: 12px;
        color: #666;
        margin-bottom: 2px;
      }

      .transaction-category {
        font-weight: 500;
        margin-bottom: 4px;
        color: var(--v-primary-base);
      }
      
      .transaction-amount {
        font-weight: 700;
        font-size: 15px;
      }
      
      .amount-positive {
        color: #4CAF50;
      }
      
      .amount-negative {
        color: var(--text-dark);
      }
      
      .transaction-right {
        display: flex;
        flex-direction: column;
        align-items: flex-end;
        gap: 6px;
      }
      
      .transaction-category-pill {
        display: inline-block;
        padding: 3px 8px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 600;
        background-color: #E3F2FD;
        color: #1565C0;
      }
      
      /* Category-specific colors */
      .category-dining {
        background-color: #FFF3E0;
        color: #E65100;
      }
      
      .category-sports {
        background-color: #E8F5E9;
        color: #2E7D32;
      }
      
      .category-wellness {
        background-color: #E0F7FA;
        color: #00838F;
      }
      
      .category-events {
        background-color: #F3E5F5;
        color: #6A1B9A;
      }
      
      .category-membership {
        background-color: #EDE7F6;
        color: #4527A0;
      }
      
      .category-top-up {
        background-color: #E8EAF6;
        color: #283593;
      }
      
      .transaction-details {
        font-size: 12px;
        color: var(--muted);
        margin-top: 8px;
        padding-top: 8px;
        border-top: 1px dashed rgba(0,0,0,0.05);
      }
      
      .transaction-status {
        display: inline-block;
        padding: 3px 8px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 600;
      }
      
      .status-completed {
        background-color: #E8F5E9;
        color: #2E7D32;
      }
      
      .status-pending {
        background-color: #FFF8E1;
        color: #F57F17;
      }
      
      .status-failed {
        background-color: #FFEBEE;
        color: #C62828;
      }
      
      .load-more-btn {
        display: flex;
        justify-content: center;
        width: 100%;
        margin-top: 8px;
        margin-bottom: 24px;
      }
      
      /* Financial Summary - Mobile Optimized */
      .financial-summary {
        background-color: white;
        border-radius: 16px;
        padding: 16px;
        margin-bottom: 24px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
      }
      
      .summary-header {
        margin-bottom: 14px;
        display: flex;
        flex-direction: column;
        gap: 10px;
      }
      
      .summary-title {
        font-size: 18px;
        font-weight: 600;
        color: var(--text-dark);
      }
      
      .month-selector {
        display: flex;
        align-items: center;
        gap: 8px;
      }
      
      .month-chip {
        font-size: 13px;
        font-weight: 500;
      }
      
      .summary-stats {
        display: grid;
        grid-template-columns: repeat(2, 1fr); /* Change from 1fr to 2 columns */
        gap: 12px;
        margin-bottom: 20px;
      }
      
      .stat-card {
        background-color: rgba(211, 211, 211, 0.1);
        border-radius: 10px;
        padding: 14px;
        transition: transform 0.2s ease;
      }
      
      .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.05);
      }
      
      /* Make the Total Spent card span full width */
      .stat-card.full-width {
        grid-column: 1 / -1;
      }
      
      .stat-label {
        font-size: 12px;
        color: var(--muted);
        margin-bottom: 6px;
      }
      
      .stat-value {
        font-size: 16px;
        font-weight: 600;
        color: var(--text-dark);
        display: flex;
        align-items: center;
        gap: 8px;
      }
      
      .trend-indicator {
        display: flex;
        align-items: center;
        font-size: 12px;
        font-weight: 500;
      }
      
      .spending-chart {
        height: 180px;
        margin-top: 16px;
      }
      
      /* Action Buttons - Mobile Optimized */
      .action-buttons {
        display: flex;
        flex-direction: column;
        gap: 12px;
        margin-top: 24px;
      }
      
      .action-btn {
        height: 48px; /* Increased touch target */
        font-size: 15px !important;
        font-weight: 600 !important;
        text-transform: none !important;
      }
      
      /* Media Queries for larger screens */
      @media (min-width: 600px) {
        .summary-header {
          flex-direction: row;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 16px;
        }
        
        .summary-stats {
          gap: 16px;
        }
        
        .action-buttons {
          flex-direction: row;
          justify-content: center;
        }
        
        .action-btn {
          min-width: 180px;
        }
        
        .spending-chart {
          height: 200px;
        }
      }
      
      @media (min-width: 960px) {
        .wallet-header {
          padding: 20px;
        }
        
        .page-title {
          font-size: 24px;
        }
        
        .balance-card {
          padding: 20px;
          margin: -20px 20px 24px 20px;
        }
        
        .balance-amount {
          font-size: 28px;
        }
        
        .content-area {
          padding: 0 20px 80px 20px;
          margin-top: 30px; /* Slightly larger margin for desktop */
        }
      }
    `
    document.head.appendChild(styleElement)

    // Fetch wallet data
    await this.fetchWalletDetails();
  },
  template: `
  <div class="wallet-details">
    <!-- Loading overlay -->
    <div class="loading-overlay" v-if="isLoading">
      <div class="loading-spinner">
        <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
        <div class="loading-text">Loading wallet details...</div>
      </div>
    </div>
    
    <!-- Notification toast -->
    <transition name="fade">
      <div class="notification-toast" v-if="showNotification">
        {{ notificationMessage }}
      </div>
    </transition>
    
    <!-- Animated Background Decoration -->
    <div class="background-decoration">
      <div class="bg-leaf bg-leaf-1"></div>
      <div class="bg-leaf bg-leaf-2"></div>
      <div class="bg-leaf bg-leaf-3"></div>
    </div>

    <!-- Header Section -->
    <div class="wallet-header">
      <div class="leaf-decoration left"></div>
      <div class="leaf-decoration right"></div>
      
      <div class="header-content">
        <div class="back-button" @click="goBack">
          <v-icon color="white" size="24">mdi-arrow-left</v-icon>
          <span class="back-button-text">Back</span>
        </div>
        
        <h1 class="page-title">Wallet Details</h1>
      </div>
    </div>
    
    <!-- Balance Card -->
    <div class="balance-card">
      <div class="card-decoration"></div>
      
      <div class="balance-title">Available Credit Balance</div>
      <div class="balance-amount" v-html="formattedBalance"></div>
      
      <div class="balance-details">
        <div class="balance-detail-item">
          <div class="detail-label">Credits Purchased</div>
          <div class="detail-value" v-html="creditsSpent"></div>
        </div>
        
        <div class="balance-detail-item">
          <div class="detail-label">Credits In Transit</div>
          <div class="detail-value" v-html="creditsInTransit"></div>
        </div>
      </div>
      
      <!--<v-divider class="my-3"></v-divider>
      
      <div class="balance-detail-item">
        <div class="detail-label">Daily/Monthly Spending</div>
        <div>
          <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
            <span class="detail-value">{{ userData.loyaltyProgress }}%</span>
            <span class="detail-label">{{ userData.nextTierPoints }} points to next tier</span>
          </div>
          <v-progress-linear
            v-model="userData.loyaltyProgress"
            color="secondary"
            height="8"
            rounded
          ></v-progress-linear>
        </div>
      </div>-->
    </div>
    
    <!-- Content Area -->
    <div class="content-area">
      <!-- Tabs -->
      <v-tabs
        v-model="currentTab"
        color="primary"
        centered
        class="custom-tabs"
        grow
      >
        <v-tab value="0">Transactions</v-tab>
        <v-tab value="1">Summary</v-tab>
      </v-tabs>
      
      <v-window v-model="currentTab">
        <!-- Transactions Tab -->
        <v-window-item value="0">
          <!-- Transaction List -->
          <div class="transaction-list">
            
            <!-- Search and Filter -->
            <!--<div class="search-and-filter">
              <v-text-field
                v-model="searchQuery"
                density="comfortable"
                variant="outlined"
                label="Search"
                prepend-inner-icon="mdi-magnify"
                class="search-input"
                hide-details
                clearable
              ></v-text-field>
              
              <v-menu v-model="filterActive" :close-on-content-click="false">
                <template v-slot:activator="{ props }">
                  <v-btn
                    icon
                    color="primary"
                    v-bind="props"
                    size="large"
                  >
                    <v-icon>mdi-filter-variant</v-icon>
                  </v-btn>
                </template>
                
                <div class="filter-panel">
                  <v-card min-width="300" class="pa-4">
                    <div class="text-subtitle-1 font-weight-bold mb-2">Filter Transactions</div>
                    
                    <v-select
                      v-model="selectedFilters.category"
                      :items="filterOptions.categories"
                      label="Category"
                      variant="outlined"
                      density="comfortable"
                      class="mb-3"
                      hide-details
                    ></v-select>
                    
                    <v-select
                      v-model="selectedFilters.dateRange"
                      :items="filterOptions.dateRanges"
                      label="Date Range"
                      variant="outlined"
                      density="comfortable"
                      class="mb-3"
                      hide-details
                    ></v-select>
                    
                    <v-select
                      v-model="selectedFilters.status"
                      :items="filterOptions.statuses"
                      label="Status"
                      variant="outlined"
                      density="comfortable"
                      class="mb-4"
                      hide-details
                    ></v-select>
                    
                    <div class="d-flex gap-2">
                      <v-btn
                        block
                        color="primary"
                        variant="elevated"
                        @click="applyFilters"
                      >
                        Apply
                      </v-btn>
                      
                      <v-btn
                        block
                        color="primary"
                        variant="outlined"
                        @click="resetFilters"
                      >
                        Reset
                      </v-btn>
                    </div>
                  </v-card>
                </div>
              </v-menu>
            </div>-->
            
            <!-- Filter chips -->
           <!-- <div v-if="selectedFilters.category !== 'All' || selectedFilters.status !== 'All'" class="mb-3">
              <v-chip
                v-if="selectedFilters.category !== 'All'"
                class="filter-chip"
                closable
                size="small"
                @click:close="selectedFilters.category = 'All'"
              >
                Category: {{ selectedFilters.category }}
              </v-chip>
              
              <v-chip
                v-if="selectedFilters.status !== 'All'"
                class="filter-chip"
                closable
                size="small"
                @click:close="selectedFilters.status = 'All'"
              >
                Status: {{ selectedFilters.status }}
              </v-chip>
            </div>-->
            
            <!-- Transactions -->
            <div v-if="filteredTransactions.length > 0">
              <div
                v-for="(transaction, i) in filteredTransactions"
                :key="i"
                class="transaction-card"
                :class="{ 'card-hovered': hoveredTransaction === i }"
                @mouseenter="hoveredTransaction = i"
                @mouseleave="hoveredTransaction = null"
                @click="viewTransactionDetails(transaction)"
              >
                <div class="transaction-header">
                  <div style="display: flex; align-items: flex-start;">
                    <div :class="['transaction-icon-wrapper', transaction.color || 'primary']">
                      <v-icon color="white" size="24">{{ transaction.icon || 'mdi-cash' }}</v-icon>
                    </div>
                    
                    <div class="transaction-content">
                      <div class="transaction-title">{{ transaction.title || transaction.category }}</div>
                      <div class="transaction-id">{{ transaction.id }}</div>
                      <div class="transaction-date">{{ transaction.date }}</div>
                    </div>
                  </div>
                  
                  <div class="transaction-right">
                    <span 
                      :class="['transaction-category-pill', 'category-' + transaction.category.toLowerCase().replace(/\s+/g, '-')]">
                      {{ transaction.category }}
                    </span>
                    <div :class="['transaction-amount', transaction.amount >= 0 ? 'amount-positive' : 'amount-negative']">
                      <span v-html="formatCredit(transaction.amount)"></span>
                    </div>
                  </div>
                </div>
                
                <div class="transaction-details">
                  <div>{{ transaction.details }}</div>
                  <div class="mt-2">
                    <span 
                      :class="['transaction-status', 'status-' + transaction.status.toLowerCase()]">
                      {{ transaction.status.charAt(0).toUpperCase() + transaction.status.slice(1).toLowerCase() }}
                    </span>
                  </div>
                </div>
              </div>
              
              <!-- Load more button -->
              <div class="load-more-btn">
                <v-btn
                  variant="outlined"
                  color="primary"
                  @click="loadMoreTransactions"
                  :loading="isLoading"
                  block
                >
                  Load More
                </v-btn>
              </div>
            </div>
            
            <div v-else class="d-flex flex-column align-center py-8">
              <v-icon color="muted" size="64">mdi-receipt-text-outline</v-icon>
              <div class="text-subtitle-1 font-weight-bold mt-4 text-center">No transactions found</div>
              <div class="text-caption text-center">Please make a transaction</div>
              <!--<v-btn 
                variant="text" 
                color="primary" 
                class="mt-4" 
                @click="resetFilters"
              >
                Reset Filters
              </v-btn>-->
            </div>
          </div>
        </v-window-item>
        
        <!-- Financial Summary Tab -->
        <v-window-item value="1">
          <div class="financial-summary">
            <div class="summary-header">
              <div class="summary-title">{{ new Date().toLocaleString('default', { month: 'long' }) }} Summary</div>
            </div>
            
            <div class="summary-stats">
              <div class="stat-card full-width">
                <div class="stat-label">Total Spent</div>
                <div class="stat-value">
                  {{ formatCurrency(financialData.monthly.totalSpent) }}
                  <div class="trend-indicator" :style="{color: spendingTrend.color === 'error' ? '#B71C1C' : '#2E7D32'}">
                    <v-icon size="16" :color="spendingTrend.color">{{ spendingTrend.icon }}</v-icon>
                    {{ spendingTrend.percentage }}%
                  </div>
                </div>
              </div>
              
              <div class="stat-card">
                <div class="stat-label">Total Sent</div>
                <div class="stat-value">{{ formatCurrency(financialData.monthly.totalSent) }}</div>
              </div>
              
              <div class="stat-card">
                <div class="stat-label">Total Received</div>
                <div class="stat-value">{{ formatCurrency(financialData.monthly.totalReceived) }}</div>
              </div>
              
              <div class="stat-card">
                <div class="stat-label">Top Category</div>
                <div class="stat-value">{{ financialData.monthly.topCategory }}</div>
              </div>
              
              <div class="stat-card">
                <div class="stat-label">Average Transaction</div>
                <div class="stat-value">{{ formatCurrency(financialData.monthly.averageTransaction) }}</div>
              </div>
              
            </div>
            
          </div>
          
        </v-window-item>
      </v-window>
    </div>
  </div>
  `
}
