export default {
    props: ["jsonData"],
    data() {
      return {
        // Tropical Rainforest Premium Color Scheme
        theme: {
          primary: '#1A3C28',       // Rich rainforest green
          secondary: '#DDA333',     // Golden amber
          accent: '#A3BA95',        // Sage green
          lightBg: '#F8F5F0',       // Luxury paper
          darkText: '#1E1E1E',
          cream: '#EADDCB',         // Cream
          error: '#B71C1C'
        },
        
        // Top Up - States and Data
        isLoading: false,
        topUpAmount: 100,
        selectedMethod: null,
        paymentStep: 1,
        bankTransferDetails: false,
        showCardForm: false,
        isProcessing: false,
        showSPayAnimation: false,
        
        // New properties for receipt upload
        receiptFile: null,
        receiptPreview: null,
        receiptUploaded: false,
        
        // Predefined amounts
        predefinedAmounts: [50, 100, 200, 500, 1000],
        
        // Response data
        response: {
          data: {
            transaction_id: ''
          }
        },
        
        // User Data
      userData: {
        name: "",
        role: "",
        membershipId: "",
        balance: 0,
        availableCredit: 0
      },
        
        // Payment Methods
        paymentMethods: [
          /*{
            id: 'credit_card',
            title: 'Credit/Debit Card',
            icon: 'mdi-credit-card',
            description: 'Top up using your saved or new card',
            color: 'blue-darken-1'
          },*/
          {
            id: 's_pay',
            title: 'SarawakPay',
            icon: 'mdi-wallet',
            description: 'Top-Up from SarawakPay',
            color: 'green-darken-1'
          },
          {
            id: 'bank_transfer',
            title: 'Bank Transfer',
            icon: 'mdi-bank',
            description: 'Transfer funds from your bank account',
            color: 'green-darken-1'
          },
          /*{
            id: 'e_wallet',
            title: 'E-Wallet',
            icon: 'mdi-wallet',
            description: 'Top up using your e-wallet',
            color: 'purple-darken-1'
          },
          {
            id: 'points_conversion',
            title: 'Convert Points',
            icon: 'mdi-swap-horizontal',
            description: 'Convert reward points to wallet balance',
            color: 'deep-orange-darken-1'
          }*/
        ],
        
        // Saved Cards
        savedCards: [
          {
            id: 'card_1',
            type: 'visa',
            last4: '4832',
            expMonth: 10,
            expYear: 25,
            default: true
          },
          {
            id: 'card_2',
            type: 'mastercard',
            last4: '7631',
            expMonth: 8,
            expYear: 26,
            default: false
          }
        ],
        
        // Bank Transfer Details
        bankDetails: {
          bankName: 'Maybank',
          accountNumber: '514231876321',
          accountName: 'Tropical Rainforest Club',
          referenceID: 'TRFOP-8721'
        },
        
        // User input for new card
        newCard: {
          number: '',
          name: '',
          expiry: '',
          cvc: ''
        },
        
        // Form validation rules
        rules: {
          required: value => !!value || 'Required.',
          cardNumber: value => /^[0-9]{16}$/.test(value) || 'Card number must be 16 digits',
          cardName: value => /^[A-Za-z ]+$/.test(value) || 'Name should contain only letters',
          expiry: value => /^(0[1-9]|1[0-2])\/[0-9]{2}$/.test(value) || 'Format must be MM/YY',
          cvc: value => /^[0-9]{3,4}$/.test(value) || 'CVC must be 3-4 digits'
        },
        
        // Transaction history (recent top ups)
        recentTopUps: [
          {
            date: 'June 2, 2023',
            amount: 500.00,
            method: 'Credit Card •••• 4832',
            status: 'completed'
          },
          {
            date: 'May 15, 2023',
            amount: 1000.00,
            method: 'Bank Transfer',
            status: 'completed'
          },
          {
            date: 'April 28, 2023',
            amount: 300.00,
            method: 'E-Wallet (Touch n Go)',
            status: 'completed'
          }
        ],
        
        // Notification system
        showNotification: false,
        notificationMessage: '',
        
        // Animation states
        hoveredMethod: null
      }
    },
    computed: {
      // Format current balance
      formattedBalance() {
        return this.formatCredit(this.userData.balance)
      },
      
      // Format top up amount
      formattedTopUpAmount() {
        return this.formatCurrency(this.topUpAmount)
      },
      
      // Get current selected payment method
      currentMethod() {
        return this.paymentMethods.find(m => m.id === this.selectedMethod) || {}
      },
      
      // Check if amount is valid
      isAmountValid() {
        return Number.isInteger(this.topUpAmount) && this.topUpAmount >= 10 && this.topUpAmount <= 10000
      },
      
      // Card icon based on card type
      cardIcon() {
        return card => {
          return card.type === 'visa' ? 'mdi-credit-card' : 
                 card.type === 'mastercard' ? 'mdi-credit-card' : 'mdi-credit-card'
        }
      },
      
      // Selected payment method title
      paymentTitle() {
        return this.currentMethod.title || 'Select a payment method'
      },
      
      // Calculate new balance after top up
      newBalance() {
        return this.userData.balance + this.topUpAmount
      },
      
      // Filter payment methods based on amount
      filteredPaymentMethods() {
        if (this.topUpAmount >= 500) {
          // For amounts >= 500, show only Bank Transfer
          return this.paymentMethods.filter(method => method.id === 'bank_transfer');
        } else {
          // For amounts < 500, show only SarawakPay
          return this.paymentMethods.filter(method => method.id === 's_pay');
        }
      }
    },
    methods: {
    formatCurrency(amount) {
      return "RM " + parseFloat(amount).toFixed(0)
    },

    formatCredit(amount) {
      return `<i class="mdi mdi-circle-multiple-outline"></i> ${parseFloat(amount).toFixed(0)}`
    },
      
      // Add these new methods for payment method display
      getMethodClass(method) {
        if (method === 's_pay') return 'method-sarawakpay';
        if (method === 'bank_transfer') return 'method-bank';
        return 'method-default';
      },
      
      getMethodDisplay(method) {
        if (method === 's_pay') return 'SarawakPay';
        if (method === 'bank_transfer') return 'Bank Transfer';
        return method; // Return original for other methods
      },
    
    // Replace the existing fetchWalletBalance method with this:
    async fetchWalletBalance() {
      this.isLoading = true;
      try {
        const response = await this.$root.callApi('wallet', 'recent_topup', {
          user_id: sessionStorage.getItem('id')
        });
        
        // Check if data exists and has status
        if (response.data && response.data.status == "YES") {
          this.userData.balance = parseFloat(response.data.balance);
          console.log('Wallet Balance:', this.userData.balance);
          this.userData.availableCredit = parseFloat(response.data.available_credit);
          
          // Update recent topups from API response
          if (response.data.recent_topups && response.data.recent_topups.length > 0) {
            this.recentTopUps = response.data.recent_topups.map(topup => ({
              date: topup.date,
              amount: parseFloat(topup.amount),
              method: topup.payment_method,
              status: topup.status.toLowerCase()
            }));
          }
        } else {
          const errorMessage = response.data ? response.data.message : response.message;
          console.log('API Error:', errorMessage);
          this.showNotification = true;
          this.notificationMessage = errorMessage || 'Failed to fetch wallet balance';
          setTimeout(() => { this.showNotification = false }, 3000);
        }
      } catch (error) {
        console.error('Error fetching wallet balance:', error);
        this.showNotification = true;
        this.notificationMessage = 'Error fetching wallet balance';
        setTimeout(() => { this.showNotification = false }, 3000);
      } finally {
        this.isLoading = false;
      }
    },
      
      // Set predefined amount
      setAmount(amount) {
        this.topUpAmount = amount
        // Reset selected method when amount changes
        if (this.paymentStep === 2) {
          this.selectedMethod = null
          this.bankTransferDetails = false
          this.showSPayAnimation = false
        }
      },
      
      // Select payment method
      selectMethod(methodId) {
        this.selectedMethod = methodId
        this.paymentStep = 2
        
        // If bank transfer selected, show details
        if (methodId === 'bank_transfer') {
          this.bankTransferDetails = true
        } else {
          this.bankTransferDetails = false
        }
        
        // If credit card selected, prepare card options
        if (methodId === 'credit_card') {
          this.showCardForm = this.savedCards.length === 0
        }
        
        // If SarawakPay selected, show animation and redirect
        if (methodId === 's_pay') {
          this.showSPayAnimation = true
          setTimeout(() => {
            this.processPayment()
          }, 3000) // Redirect after 3 seconds
        }
      },
      
      // Go back to previous step
      goBack() {
        if (this.paymentStep > 1) {
          this.paymentStep--
          this.bankTransferDetails = false
          this.showCardForm = false
          this.selectedMethod = null
        } else {
          //this.$root.loadComponent('modules/wallet/201-wallet.js')
          this.$root.loadComponent('modules/home.js');
        }
      },
      
      // Show new card form
      showNewCardForm() {
        this.showCardForm = true
      },
      
      // Select existing card
      selectCard(cardId) {
        this.showCardForm = false
        this.processPayment()
      },
      
      // Cancel SarawakPay redirect
      cancelSPayRedirect() {
        this.showSPayAnimation = false
        this.selectedMethod = null
        this.paymentStep = 2
      },
      
      // Handle receipt file selection
    handleReceiptUpload(event) {
      const file = event.target.files[0];
      if (!file) return;
      
      // Check file type
      if (!file.type.startsWith("image/") && file.type !== 'application/pdf') {
        this.showNotification = true;
        this.notificationMessage = 'Please upload an image or PDF file';
        setTimeout(() => { this.showNotification = false }, 3000);
        return;
      }
      
      // Check file size (max 5MB)
      if (file.size > 5 * 1024 * 1024) {
        this.showNotification = true;
        this.notificationMessage = 'File size should not exceed 5MB';
        setTimeout(() => { this.showNotification = false }, 3000);
        return;
      }
      
      this.receiptFile = file;
      
      // Create preview for image files
      if (file.type.startsWith("image/")) {
        const reader = new FileReader();
        reader.onload = (e) => {
          this.receiptPreview = e.target.result;
          this.receiptUploaded = true;
        };
        reader.readAsDataURL(file);
      } else {
        // For PDF, just show the filename
        this.receiptPreview = null;
        this.receiptUploaded = true;
      }
    },
    
    // Remove uploaded receipt
    removeReceipt() {
      this.receiptFile = null;
      this.receiptPreview = null;
      this.receiptUploaded = false;
    },
    
    // Process the payment
    async processPayment() {
      console.log('processPayment called');
      console.log('User ID:', sessionStorage.getItem('id'));
      if (!this.isAmountValid) {
        this.showNotification = true;
        this.notificationMessage = 'Please enter a valid whole number amount between RM 10 and RM 10,000';
        setTimeout(() => {
          this.showNotification = false;
        }, 3000);
        return;
      }
      
      // Check if receipt is uploaded for bank transfer
      if (this.selectedMethod === 'bank_transfer' && !this.receiptUploaded) {
        this.showNotification = true;
        this.notificationMessage = 'Please upload your bank transfer receipt';
        setTimeout(() => {
          this.showNotification = false;
        }, 3000);
        return;
      }
      
      this.isProcessing = true;
      this.showNotification = true;
      this.notificationMessage = 'Processing your top up...';
      
      try {
        // Prepare reference based on payment method
        let reference = '';
        if (this.selectedMethod === 'credit_card') {
          const selectedCard = this.savedCards.find(card => card.default);
          reference = selectedCard ? `${selectedCard.type.toUpperCase()} ending ${selectedCard.last4}` : 'New card';
        } else if (this.selectedMethod === 'bank_transfer') {
          reference = this.bankDetails.referenceID;
        }
        
        // Prepare API request parameters
        const requestParams = {
          user_id: sessionStorage.getItem('id'),
          amount: this.topUpAmount,
          payment_method: this.selectedMethod,
          reference: reference
        };
        
        // For bank transfer with receipt, include the file content directly
        if (this.selectedMethod === 'bank_transfer' && this.receiptFile) {
          // Read the file content
          if (typeof this.receiptFile !== 'string') {
            const reader = new FileReader();
            const filePromise = new Promise((resolve, reject) => {
              reader.onload = (e) => {
                let fileContent = e.target.result;
                resolve(fileContent);
              };
              reader.onerror = () => reject(new Error('Failed to read file'));
              reader.readAsDataURL(this.receiptFile);
            });
            
            const fileContent = await filePromise;
            requestParams.receipt_file_content = fileContent;
            requestParams.receipt_file_name = this.receiptFile.name;
          } else {
            // Already a data URL
            requestParams.receipt_file_content = this.receiptFile;
            requestParams.receipt_file_name = 'receipt_' + Date.now() + '.jpg';
          }
        }
        
        const apiResponse = await this.$root.callApi('wallet', 'topup', requestParams);
        
        // Store the entire response in the component's data
        this.response = apiResponse;
        
        if (apiResponse.data && apiResponse.data.status === "YES") {
          // Update balance
          this.userData.balance = parseFloat(apiResponse.data.new_balance);
          
          // Show success screen
          this.paymentStep = 3;
          
          this.showNotification = true;
          this.notificationMessage = 'Top up successful!';
          setTimeout(() => { this.showNotification = false }, 3000);
        } else {
          this.showNotification = true;
          this.notificationMessage = apiResponse.data?.message || 'Top up failed';
          setTimeout(() => { this.showNotification = false }, 3000);
        }
      } catch (error) {
        console.error('Error processing top up:', error);
        this.showNotification = true;
        this.notificationMessage = 'Error processing top up';
        setTimeout(() => { this.showNotification = false }, 3000);
      } finally {
        this.isProcessing = false;
      }
    },
      
      // Complete the top up process
      completeTopUp() {
        this.$root.loadComponent('modules/home.js')
      },
      
      // Add new card
      addNewCard() {
        this.isProcessing = true
        this.showNotification = true
        this.notificationMessage = 'Adding your card...'
        
        // Simulate card processing
        setTimeout(() => {
          // Add new card to saved cards
          this.savedCards.push({
            id: 'card_new',
            type: 'visa',
            last4: this.newCard.number.slice(-4),
            expMonth: parseInt(this.newCard.expiry.split('/')[0]),
            expYear: parseInt(this.newCard.expiry.split('/')[1]),
            default: this.savedCards.length === 0
          })
          
          this.showCardForm = false
          this.isProcessing = false
          this.showNotification = true
          this.notificationMessage = 'Card added successfully'
          
          // Process payment with new card
          setTimeout(() => {
            this.processPayment()
          }, 1000)
        }, 2000)
      },
      
      // Copy to clipboard
      copyToClipboard(text) {
        navigator.clipboard.writeText(text).then(() => {
          this.showNotification = true
          this.notificationMessage = 'Copied to clipboard'
          setTimeout(() => {
            this.showNotification = false
          }, 3000)
        })
      },
      
      // Navigation methods
      navigateTo(screen) {
        this.showNotification = true
        this.notificationMessage = 'Loading...'
        setTimeout(() => {
          this.$root.loadComponent('modules/' + screen)
        }, 600)
      }
    },
    mounted() {
    // Fetch wallet data
    this.fetchWalletBalance();
    
    // Simulate initial data loading
    this.isLoading = true
    setTimeout(() => { this.isLoading = false }, 800)
      
      // IMPORTANT: Inject the CSS styles dynamically
      const styleElement = document.createElement('style')
      styleElement.textContent = `
        /* Base Styles - Mobile First Approach */
        .top-up-container {
          --primary: #1A3C28;
          --secondary: #DDA333;
          --accent: #A3BA95;
          --cream: #EADDCB;
          --muted: #889FAC;
          --text-dark: #1E1E1E;
          --text-light: #FFFFFF;
          
          font-family: 'Arial', sans-serif;
          background-color: var(--cream);
          min-height: 100vh;
          position: relative;
          overflow-x: hidden;
          -webkit-tap-highlight-color: transparent;
        }
        
        /* Loading Overlay */
        .loading-overlay {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background-color: rgba(26, 60, 40, 0.8);
          display: flex;
          justify-content: center;
          align-items: center;
          z-index: 1000;
        }
        
        .loading-spinner {
          display: flex;
          flex-direction: column;
          align-items: center;
        }
        
        .loading-text {
          margin-top: 16px;
          color: white;
          font-weight: 500;
        }
        
        /* Notification Toast - Mobile Optimized */
        .notification-toast {
          position: fixed;
          bottom: 24px;
          left: 50%;
          transform: translateX(-50%);
          background-color: var(--secondary);
          color: var(--primary);
          padding: 14px 20px;
          border-radius: 20px;
          font-weight: 600;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          z-index: 900;
          min-width: 80%;
          max-width: 90%;
          text-align: center;
        }
        
        .fade-enter-active,
        .fade-leave-active {
          transition: opacity 0.3s, transform 0.3s;
        }
        
        .fade-enter-from,
        .fade-leave-to {
          opacity: 0;
          transform: translateX(-50%) translateY(20px);
        }
        
        /* Background Decorations */
        .background-decoration {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: -1;
          overflow: hidden;
        }
        
        .bg-leaf {
          position: absolute;
          border-radius: 50% 50% 0 50%;
          opacity: 0.07;
          background-color: var(--accent);
          transform: rotate(45deg);
        }
        
        .bg-leaf-1 {
          width: 300px;
          height: 300px;
          top: -150px;
          left: -150px;
        }
        
        .bg-leaf-2 {
          width: 250px;
          height: 250px;
          bottom: 10%;
          right: -100px;
          transform: rotate(130deg);
        }
        
        .bg-leaf-3 {
          width: 180px;
          height: 180px;
          top: 30%;
          left: -90px;
          transform: rotate(80deg);
        }
        
        /* Header Section - Mobile Optimized */
        .topup-header {
          background-color: var(--primary);
          padding: 16px;
          border-radius: 0 0 24px 24px;
          box-shadow: 0 6px 16px rgba(0,0,0,0.15);
          color: var(--text-light);
          position: relative;
          overflow: hidden;
        }
        
        .leaf-decoration {
          position: absolute;
          width: 120px;
          height: 120px;
          background-color: rgba(255,255,255,0.05);
          border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
          z-index: 0;
        }
        
        .leaf-decoration.left {
          top: -30px;
          left: -50px;
          transform: rotate(35deg);
        }
        
        .leaf-decoration.right {
          bottom: -60px;
          right: -30px;
          transform: rotate(-15deg);
        }
        
        .header-content {
          position: relative;
          z-index: 1;
        }
        
        .back-button {
          display: flex;
          align-items: center;
          margin-bottom: 16px;
          cursor: pointer;
          min-height: 44px; /* Mobile touch target */
        }
        
        .back-button-text {
          margin-left: 8px;
          font-weight: 500;
          font-size: 16px;
        }
        
        .page-title {
          font-size: 22px;
          font-weight: 700;
          margin-bottom: 12px;
        }
        
        /* Balance Summary - Mobile Optimized */
        .balance-summary {
          background-color: white;
          border-radius: 16px;
          margin: -15px 16px 20px 16px;
          padding: 16px;
          box-shadow: 0 4px 12px rgba(0,0,0,0.1);
          position: relative;
          overflow: hidden;
        }
        
        .balance-title {
          color: var(--muted);
          font-size: 14px;
          margin-bottom: 6px;
        }
        
        .balance-amount {
          font-size: 26px;
          font-weight: 700;
          color: var(--primary);
        }
        
        /* Content Area - Mobile Optimized */
        .content-area {
          padding: 0 16px 80px 16px;
          margin-top: 20px;
        }
        
        /* Step indicator - Mobile Optimized */
        .step-indicator {
          display: flex;
          justify-content: center;
          margin: 0 0 20px 0;
        }
        
        .step {
          display: flex;
          flex-direction: column;
          align-items: center;
          position: relative;
          flex: 1;
        }
        
        .step:not(:last-child)::after {
          content: '';
          position: absolute;
          top: 16px;
          left: calc(50% + 16px);
          width: calc(100% - 32px);
          height: 2px;
          background-color: #D8D8D8;
          z-index: 0;
        }
        
        .step.active:not(:last-child)::after {
          background-color: var(--secondary);
        }
        
        .step-circle {
          width: 32px;
          height: 32px;
          border-radius: 50%;
          background-color: #D8D8D8;
          display: flex;
          align-items: center;
          justify-content: center;
          font-weight: 600;
          font-size: 14px;
          color: white;
          margin-bottom: 8px;
          position: relative;
          z-index: 1;
        }
        
        .step.active .step-circle {
          background-color: var(--secondary);
        }
        
        .step.completed .step-circle {
          background-color: var(--primary);
        }
        
        .step-label {
          font-size: 12px;
          color: var(--muted);
          text-align: center;
        }
        
        .step.active .step-label,
        .step.completed .step-label {
          color: var(--text-dark);
          font-weight: 500;
        }
        
        /* Amount Selection - Mobile Optimized */
        .amount-selection {
          background-color: white;
          border-radius: 16px;
          padding: 16px;
          box-shadow: 0 4px 12px rgba(0,0,0,0.1);
          margin-bottom: 24px;
        }
        
        .section-title {
          font-size: 18px;
          font-weight: 600;
          color: var(--text-dark);
          margin-bottom: 16px;
        }
        
        .amount-chips {
          display: grid;
          grid-template-columns: repeat(3, 1fr);
          gap: 10px;
          margin-bottom: 20px;
        }
        
        .amount-chip {
          background-color: rgba(211, 211, 211, 0.2);
          color: var(--text-dark);
          padding: 12px 8px;
          border-radius: 10px;
          cursor: pointer;
          transition: all 0.2s;
          border: 1px solid transparent;
          font-weight: 500;
          text-align: center;
          min-height: 60px; /* Increased height to accommodate two lines */
          display: flex;
          flex-direction: column; /* Changed to column for vertical layout */
          align-items: center;
          justify-content: center;
        }
        
        .amount-chip:active {
          transform: scale(0.96);
        }
        
        .amount-chip.selected {
          background-color: rgba(26, 60, 40, 0.05);
          border-color: var(--primary);
          color: var(--primary);
          font-weight: 600;
        }
        
        /* New styles for credit display in amount chips */
        .amount-chip-credit {
          color: #2E7D32; /* Green color */
          font-weight: 600;
          font-size: 14px;
          margin-bottom: 2px;
        }
        
        .amount-chip-currency {
          font-size: 13px;
        }
        
        .amount-chip.selected .amount-chip-credit,
        .amount-chip.selected .amount-chip-currency {
          color: var(--primary);
        }
        
        .custom-amount {
          margin-top: 16px;
        }
        
        /* Payment Method Selection - Mobile Optimized */
        .payment-methods {
          margin-top: 20px;
        }
        
        .payment-method-card {
          background-color: white;
          border-radius: 16px;
          margin-bottom: 12px;
          padding: 16px;
          display: flex;
          align-items: center;
          gap: 16px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          cursor: pointer;
          transition: all 0.3s;
          position: relative;
          overflow: hidden;
          min-height: 80px; /* Good mobile touch target */
        }
        
        .payment-method-card:active {
          transform: scale(0.98);
          background-color: rgba(0,0,0,0.02);
        }
        
        .payment-icon-wrapper {
          width: 44px;
          height: 44px;
          border-radius: 10px;
          display: flex;
          align-items: center;
          justify-content: center;
          flex-shrink: 0;
        }
        
        .payment-content {
          flex: 1;
        }
        
        .payment-title {
          font-weight: 600;
          font-size: 15px;
          color: var(--text-dark);
          margin-bottom: 4px;
        }
        
        .payment-description {
          font-size: 12px;
          color: var(--muted);
        }
        
        /* Card Selection - Mobile Optimized */
        .saved-cards {
          margin-bottom: 20px;
        }
        
        .card-option {
          background-color: rgba(211, 211, 211, 0.1);
          border-radius: 12px;
          padding: 14px;
          margin-bottom: 10px;
          cursor: pointer;
          display: flex;
          align-items: center;
          gap: 12px;
          transition: all 0.2s;
          border: 1px solid rgba(0,0,0,0.05);
          min-height: 72px; /* Good mobile touch target */
        }
        
        .card-option:active {
          transform: scale(0.98);
          background-color: rgba(211, 211, 211, 0.2);
        }
        
        .card-option.selected {
          background-color: rgba(26, 60, 40, 0.05);
          border-color: var(--primary);
        }
        
        .card-icon {
          width: 40px;
          height: 40px;
          border-radius: 8px;
          background-color: var(--primary);
          display: flex;
          align-items: center;
          justify-content: center;
        }
        
        .card-details {
          flex: 1;
        }
        
        .card-number {
          font-weight: 600;
          font-size: 14px;
          color: var(--text-dark);
        }
        
        .card-expiry {
          font-size: 12px;
          color: var(--muted);
        }
        
        .card-default-badge {
          background-color: var(--secondary);
          color: var(--primary);
          padding: 2px 6px;
          border-radius: 4px;
          font-size: 11px;
          font-weight: 600;
        }
        
        .add-new-card {
          display: flex;
          align-items: center;
          gap: 10px;
          cursor: pointer;
          padding: 12px 0;
          color: var(--primary);
          font-weight: 500;
          min-height: 44px; /* Good mobile touch target */
        }
        
        /* Card Form - Mobile Optimized */
        .card-form {
          margin-top: 24px;
        }
        
        .form-row {
          display: flex;
          flex-direction: column;
          gap: 12px;
          margin-bottom: 16px;
        }
        
        .form-col {
          flex: 1;
        }
        
        /* Bank Transfer Details - Mobile Optimized */
        .bank-details {
          background-color: white;
          border-radius: 16px;
          padding: 16px;
          margin-bottom: 24px;
        }
        
        .bank-detail-row {
          display: flex;
          flex-direction: column;
          padding: 12px 0;
          border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .bank-detail-row:last-child {
          border-bottom: none;
        }
        
        .detail-key {
          font-size: 13px;
          color: var(--muted);
          margin-bottom: 4px;
        }
        
        .detail-value {
          font-size: 15px;
          font-weight: 600;
          color: var(--text-dark);
          display: flex;
          align-items: center;
          justify-content: space-between;
        }
        
        .copy-button {
          cursor: pointer;
          min-width: 44px; /* Good mobile touch target */
          min-height: 44px;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        
        .bank-instructions {
          background-color: rgba(221, 163, 51, 0.1);
          border-left: 4px solid var(--secondary);
          padding: 12px;
          margin-top: 16px;
          border-radius: 4px;
        }
        
        .instruction-title {
          font-weight: 600;
          font-size: 14px;
          color: var(--text-dark);
          margin-bottom: 8px;
        }
        
        .instruction-list {
          list-style-position: outside;
          padding-left: 16px;
          font-size: 13px;
          color: var(--text-dark);
        }
        
        .instruction-list li {
          margin-bottom: 8px;
        }
        
        /* Top Up Success - Mobile Optimized */
        .topup-success {
          display: flex;
          flex-direction: column;
          align-items: center;
          text-align: center;
          padding: 20px 0;
        }
        
        .success-icon {
          width: 72px;
          height: 72px;
          border-radius: 50%;
          background-color: var(--primary);
          display: flex;
          align-items: center;
          justify-content: center;
          margin-bottom: 20px;
        }
        
        .success-title {
          font-size: 22px;
          font-weight: 700;
          color: var(--text-dark);
          margin-bottom: 8px;
        }
        
        .success-message {
          font-size: 15px;
          color: var(--muted);
          margin-bottom: 16px;
        }
        
        .transaction-details {
          width: 100%;
          background-color: rgba(211, 211, 211, 0.1);
          border-radius: 16px;
          padding: 16px;
          margin-top: 24px;
          margin-bottom: 24px;
        }
        
        .transaction-row {
          display: flex;
          justify-content: space-between;
          margin-bottom: 12px;
        }
        
        .transaction-label {
          font-size: 13px;
          color: var(--muted);
        }
        
        .transaction-value {
          font-size: 14px;
          font-weight: 600;
          color: var(--text-dark);
        }
        
        .green-highlight {
          color: #2E7D32;
        }
        
        /* Recent Top Ups - Mobile Optimized */
        .recent-topups {
          margin-top: 32px;
        }
        
        .recent-topup-item {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 14px 0;
          border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .topup-info {
          display: flex;
          flex-direction: column;
          align-items: flex-end;
        }
        
        .topup-date {
          font-size: 12px;
          color: var(--muted);
          margin-top: 4px;
        }
        
        .topup-method {
          font-size: 0.85rem;
          color: rgba(0, 0, 0, 0.6);
          margin-top: 2px;
        }
        
        .topup-status {
          font-size: 12px;
          font-weight: 600;
          padding: 2px 8px;
          border-radius: 12px;
          text-transform: uppercase;
        }
        
        .topup-status.completed {
          background-color: rgba(46, 125, 50, 0.1);
          color: #2E7D32;
        }
        
        .topup-status.pending {
          background-color: rgba(221, 163, 51, 0.1);
          color: #DDA333;
        }
        
        .topup-amount {
          font-size: 16px;
          font-weight: 700;
          color: var(--primary);
        }
        
        /* Action Buttons - Mobile Optimized */
        .action-buttons {
          display: flex;
          flex-direction: column;
          gap: 12px;
          margin-top: 24px;
        }
        
        .action-btn {
          height: 48px; /* Increased touch target */
          font-size: 15px !important;
          font-weight: 600 !important;
          text-transform: none !important;
        }
        
        /* SarawakPay Animation */
        .spay-animation {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          margin: 20px 0;
        }
        
        .spay-logo {
          width: 120px;
          height: 120px;
          background-color: #DDA333; /* SarawakPay green */
          border-radius: 60px;
          display: flex;
          align-items: center;
          justify-content: center;
          animation: pulse 1.5s infinite;
        }
        
        .spay-logo-inner {
          width: 80px;
          height: 80px;
          background-color: white;
          border-radius: 40px;
          position: relative;
        }
        
        .spay-logo-inner:before {
          content: "S";
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          font-size: 48px;
          font-weight: bold;
          color: #DDA333;
        }
        
        .spay-loading {
          display: flex;
          gap: 8px;
          margin-top: 20px;
        }
        
        .spay-loading-dot {
          width: 12px;
          height: 12px;
          background-color: #DDA333;
          border-radius: 50%;
          animation: loading 1.5s infinite;
        }
        
        .spay-loading-dot:nth-child(2) {
          animation-delay: 0.5s;
        }
        
        .spay-loading-dot:nth-child(3) {
          animation-delay: 1s;
        }
        
        @keyframes pulse {
          0% {
            transform: scale(1);
            box-shadow: 0 0 0 0 #DDA333;
          }
          
          70% {
            transform: scale(1.05);
            box-shadow: 0 0 0 10px #DDA333;
          }
          
          100% {
            transform: scale(1);
            box-shadow: 0 0 0 0 #DDA333;
          }
        }
        
        @keyframes loading {
          0%, 100% {
            transform: scale(0.6);
            opacity: 0.6;
          }
          
          50% {
            transform: scale(1);
            opacity: 1;
          }
        }
        
        /* Media Queries for larger screens */
        @media (min-width: 600px) {
          .amount-chips {
            display: flex;
            flex-wrap: wrap;
            justify-content: center;
            gap: 12px;
          }
          
          .amount-chip {
            flex: 0 0 auto;
            min-width: 120px;
            padding: 12px 20px;
          }
          
          .bank-detail-row {
            flex-direction: row;
            justify-content: space-between;
            align-items: center;
          }
          
          .detail-key {
            margin-bottom: 0;
          }
          
          .form-row {
            flex-direction: row;
          }
          
          .action-buttons {
            flex-direction: row;
            justify-content: center;
          }
          
          .action-btn {
            min-width: 180px;
          }
        }
        
        @media (min-width: 960px) {
          .topup-header {
            padding: 20px;
          }
          
          .page-title {
            font-size: 24px;
          }
          
          .balance-summary {
            padding: 20px;
            margin: -20px 20px 24px 20px;
          }
          
          .balance-amount {
            font-size: 28px;
          }
          
          .content-area {
            padding: 0 20px 80px 20px;
            margin-top: 30px;
          }
        }




        /* Receipt Upload Styles */
        .receipt-upload-section {
          margin-top: 24px;
          background-color: rgba(255,255,255,0.7);
          border-radius: 12px;
          padding: 16px;
          border: 1px dashed var(--primary);
        }
        
        .upload-title {
          font-size: 16px;
          font-weight: 600;
          color: var(--primary);
          margin-bottom: 8px;
        }
        
        .upload-description {
          font-size: 14px;
          color: var(--muted);
          margin-bottom: 16px;
        }
        
        .upload-area {
          display: flex;
          justify-content: center;
          align-items: center;
          padding: 24px;
          background-color: rgba(26, 60, 40, 0.05);
          border-radius: 8px;
          cursor: pointer;
        }
        
        .upload-button {
          display: flex;
          flex-direction: column;
          align-items: center;
          cursor: pointer;
        }
        
        .upload-button span {
          margin-top: 8px;
          color: var(--primary);
          font-weight: 500;
        }
        
        .receipt-preview {
          border-radius: 8px;
          overflow: hidden;
          background-color: white;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .preview-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 8px 12px;
          background-color: var(--primary);
          color: white;
        }
        
        .file-name {
          font-size: 14px;
          white-space: nowrap;
          overflow: hidden;
          text-overflow: ellipsis;
          max-width: 80%;
        }
        
        .remove-button {
          cursor: pointer;
        }
        
        .image-preview {
          max-height: 200px;
          overflow: hidden;
        }
        
        .image-preview img {
          width: 100%;
          object-fit: contain;
        }
        
        .pdf-preview {
          height: 120px;
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          background-color: #f5f5f5;
          color: var(--primary);
        }

        /* Payment Method Pills */
        .topup-method-pill {
          font-size: 11px;
          font-weight: 600;
          padding: 3px 8px;
          border-radius: 4px;
          text-transform: uppercase;
          margin-bottom: 4px;
          display: inline-block;
        }
        
        .method-sarawakpay {
          background-color: #FF7A00;
          color: #ffffff;
        }
        
        .method-bank {
          background-color: #000000;
          color: #FFFFFF;
        }
        
        .method-default {
          background-color: #E0E0E0;
          color: #424242;
        }

        /* Topup Amounts - Vertical Layout */
        .topup-amounts {
          display: flex;
          flex-direction: column;
          align-items: flex-start;
        }
        
        .topup-credit {
          font-size: 16px;
          font-weight: 700;
          color: #2E7D32; /* Green color */
          margin-bottom: 2px;
        }
        
        .topup-currency {
          font-size: 14px;
          color: var(--text-dark);
        }
      `
      document.head.appendChild(styleElement)
    },
    template: `
    <div class="top-up-container">
      <!-- Loading overlay -->
      <div class="loading-overlay" v-if="isLoading || isProcessing">
        <div class="loading-spinner">
          <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
          <div class="loading-text">{{ isProcessing ? 'Processing your top up...' : 'Loading...' }}</div>
        </div>
      </div>
      
      <!-- Notification toast -->
      <transition name="fade">
        <div class="notification-toast" v-if="showNotification">
          {{ notificationMessage }}
        </div>
      </transition>
      
      <!-- Animated Background Decoration -->
      <div class="background-decoration">
        <div class="bg-leaf bg-leaf-1"></div>
        <div class="bg-leaf bg-leaf-2"></div>
        <div class="bg-leaf bg-leaf-3"></div>
      </div>
  
      <!-- Header Section -->
      <div class="topup-header">
        <div class="leaf-decoration left"></div>
        <div class="leaf-decoration right"></div>
        
        <div class="header-content">
          <div class="back-button" @click="goBack">
            <v-icon color="white" size="24">mdi-arrow-left</v-icon>
            <span class="back-button-text">Back</span>
          </div>
          
          <h1 class="page-title">Top Up Wallet</h1>
        </div>
      </div>
      
      <!-- Balance Summary -->
      <div class="balance-summary">
        <div class="balance-title">Current Credit Balance</div>
        <div class="balance-amount" v-html="formattedBalance"></div>
      </div>
      
      <!-- Content Area -->
      <div class="content-area">
        <!-- Step indicator -->
        <div class="step-indicator">
          <div :class="['step', paymentStep >= 1 ? 'active' : '', paymentStep > 1 ? 'completed' : '']">
            <div class="step-circle">
              <span v-if="paymentStep <= 1">1</span>
              <v-icon v-else size="16">mdi-check</v-icon>
            </div>
            <div class="step-label">Amount</div>
          </div>
          
          <div :class="['step', paymentStep >= 2 ? 'active' : '', paymentStep > 2 ? 'completed' : '']">
            <div class="step-circle">
              <span v-if="paymentStep <= 2">2</span>
              <v-icon v-else size="16">mdi-check</v-icon>
            </div>
            <div class="step-label">Payment</div>
          </div>
          
          <div :class="['step', paymentStep >= 3 ? 'active' : '']">
            <div class="step-circle">
              <span v-if="paymentStep < 3">3</span>
              <v-icon v-else size="16">mdi-check</v-icon>
            </div>
            <div class="step-label">Complete</div>
          </div>
        </div>
        
        <!-- Step 1: Amount Selection -->
        <div v-if="paymentStep === 1" class="amount-selection">
          <div class="section-title">Select amount to top up</div>
          
          <div class="amount-chips">
            <div 
              v-for="amount in predefinedAmounts" 
              :key="amount"
              :class="['amount-chip', topUpAmount === amount ? 'selected' : '']"
              @click="setAmount(amount)"
            >
              <div class="amount-chip-credit" v-html="formatCredit(amount)"></div>
              <div class="amount-chip-currency">{{ formatCurrency(amount) }}</div>
            </div>
          </div>
          
          <div class="custom-amount">
            <v-text-field
              v-model.number="topUpAmount"
              label="Custom Amount"
              type="number"
              min="10"
              max="10000"
              step="1"
              :rules="[rules.required, value => Number.isInteger(value) || 'Please enter whole numbers only', value => value >= 10 || 'Minimum amount is RM 10', value => value <= 10000 || 'Maximum amount is RM 10,000']"
              hint="Enter whole number amount between RM 10 - RM 10,000"
              variant="outlined"
              prefix="RM"
              density="comfortable"
            ></v-text-field>
          </div>
          
          <div class="action-buttons">
            <v-btn 
              color="primary" 
              class="action-btn"
              size="large"
              :disabled="!isAmountValid"
              @click="paymentStep = 2"
              block
            >
              Continue
            </v-btn>
          </div>
        </div>
        
        <!-- Step 2: Payment Method -->
        <div v-if="paymentStep === 2">
          <div v-if="!selectedMethod" class="payment-methods">
            <div class="section-title">Select payment method</div>
            
            <div class="payment-method-info mb-4">
              <v-alert
                density="comfortable"
                type="info"
                variant="tonal"
                icon="mdi-information"
              >
                {{ topUpAmount >= 500 ? 
                  'For amounts RM 500 and above, only Bank Transfer is available.' : 
                  'For amounts below RM 500, please use SarawakPay.' }}
              </v-alert>
            </div>
            
            <div 
              v-for="(method, i) in filteredPaymentMethods" 
              :key="i"
              class="payment-method-card"
              @click="selectMethod(method.id)"
              :class="{ 'card-hovered': hoveredMethod === i }"
              @mouseenter="hoveredMethod = i"
              @mouseleave="hoveredMethod = null"
            >
              <div :class="['payment-icon-wrapper', method.color]">
                <v-icon color="black" size="24">{{ method.icon }}</v-icon>
              </div>
              
              <div class="payment-content">
                <div class="payment-title">{{ method.title }}</div>
                <div class="payment-description">{{ method.description }}</div>
              </div>
              
              <v-icon color="muted">mdi-chevron-right</v-icon>
            </div>
          </div>
          
          <!-- Credit Card Payment -->
          <div v-if="selectedMethod === 'credit_card'" class="payment-methods">
            <div class="section-title">Top up with {{ paymentTitle }}</div>
            
            <!-- Saved Cards -->
            <div v-if="savedCards.length > 0 && !showCardForm" class="saved-cards">
              <div class="mb-3">Select a saved card:</div>
              
              <div 
                v-for="card in savedCards" 
                :key="card.id"
                class="card-option"
                @click="selectCard(card.id)"
              >
                <div class="card-icon">
                  <v-icon color="white" size="20">{{ cardIcon(card) }}</v-icon>
                </div>
                
                <div class="card-details">
                  <div class="card-number">{{ card.type.toUpperCase() }} •••• {{ card.last4 }}</div>
                  <div class="card-expiry">Expires {{ card.expMonth }}/{{ card.expYear }}</div>
                </div>
                
                <div v-if="card.default" class="card-default-badge">Default</div>
              </div>
              
              <div class="add-new-card mt-4" @click="showNewCardForm">
                <v-icon>mdi-plus-circle</v-icon>
                Add new card
              </div>
            </div>
            
            <!-- New Card Form -->
            <div v-if="showCardForm" class="card-form">
              <v-form @submit.prevent="addNewCard">
                <v-text-field
                  v-model="newCard.number"
                  label="Card Number"
                  :rules="[rules.required, rules.cardNumber]"
                  placeholder="1234 5678 9012 3456"
                  maxlength="16"
                  variant="outlined"
                  density="comfortable"
                  type="tel"
                  inputmode="numeric"
                ></v-text-field>
                
                <v-text-field
                  v-model="newCard.name"
                  label="Name on Card"
                  :rules="[rules.required, rules.cardName]"
                  placeholder="John Doe"
                  variant="outlined"
                  density="comfortable"
                ></v-text-field>
                
                <div class="form-row">
                  <div class="form-col">
                    <v-text-field
                      v-model="newCard.expiry"
                      label="Expiry Date"
                      :rules="[rules.required, rules.expiry]"
                      placeholder="MM/YY"
                      maxlength="5"
                      variant="outlined"
                      density="comfortable"
                      type="tel"
                      inputmode="numeric"
                    ></v-text-field>
                  </div>
                  
                  <div class="form-col">
                    <v-text-field
                      v-model="newCard.cvc"
                      label="CVC"
                      :rules="[rules.required, rules.cvc]"
                      placeholder="123"
                      maxlength="4"
                      type="password"
                      variant="outlined"
                      density="comfortable"
                      inputmode="numeric"
                    ></v-text-field>
                  </div>
                </div>
                
                <div class="action-buttons">
                  <v-btn
                    color="primary"
                    variant="outlined"
                    class="action-btn"
                    @click="showCardForm = false"
                    block
                  >
                    Cancel
                  </v-btn>
                  
                  <v-btn
                    color="primary"
                    class="action-btn"
                    type="submit"
                    block
                  >
                    Add Card
                  </v-btn>
                </div>
              </v-form>
            </div>
            
            <!-- Card Top Up Summary -->
            <div v-if="!showCardForm && savedCards.length > 0" class="mt-8">
              <div class="transaction-details">
                <div class="transaction-row">
                  <div class="transaction-label">Amount to Top Up</div>
                  <div class="transaction-value">{{ formattedTopUpAmount }}</div>
                </div>
                
                <div class="transaction-row">
                  <div class="transaction-label">Processing Fee</div>
                  <div class="transaction-value">RM 0.00</div>
                </div>
                
                <v-divider class="my-3"></v-divider>
                
                <div class="transaction-row">
                  <div class="transaction-label">Total</div>
                  <div class="transaction-value">{{ formattedTopUpAmount }}</div>
                </div>
              </div>
              
              <div class="action-buttons">
                <v-btn
                  color="primary"
                  class="action-btn"
                  @click="processPayment"
                  block
                >
                  Confirm Payment
                </v-btn>
              </div>
            </div>
          </div>
          
          <!-- Bank Transfer Payment -->
          <div v-if="bankTransferDetails" class="bank-details">
            <div class="section-title">Bank Transfer Details</div>
            
            <div class="bank-detail-row">
              <div class="detail-key">Bank Name</div>
              <div class="detail-value">
                {{ bankDetails.bankName }}
                <v-icon class="copy-button" size="16" @click="copyToClipboard(bankDetails.bankName)">mdi-content-copy</v-icon>
              </div>
            </div>
            
            <div class="bank-detail-row">
              <div class="detail-key">Account Number</div>
              <div class="detail-value">
                {{ bankDetails.accountNumber }}
                <v-icon class="copy-button" size="16" @click="copyToClipboard(bankDetails.accountNumber)">mdi-content-copy</v-icon>
              </div>
            </div>
            
            <div class="bank-detail-row">
              <div class="detail-key">Account Name</div>
              <div class="detail-value">
                {{ bankDetails.accountName }}
                <v-icon class="copy-button" size="16" @click="copyToClipboard(bankDetails.accountName)">mdi-content-copy</v-icon>
              </div>
            </div>
            
            <div class="bank-detail-row">
              <div class="detail-key">Amount</div>
              <div class="detail-value">
                {{ formattedTopUpAmount }}
                <v-icon class="copy-button" size="16" @click="copyToClipboard(topUpAmount.toString())">mdi-content-copy</v-icon>
              </div>
            </div>
            
            <div class="bank-detail-row">
              <div class="detail-key">Reference ID</div>
              <div class="detail-value">
                <span class="green-highlight">{{ bankDetails.referenceID }}</span>
                <v-icon class="copy-button" size="16" @click="copyToClipboard(bankDetails.referenceID)">mdi-content-copy</v-icon>
              </div>
            </div>
            
            <div class="bank-instructions">
              <div class="instruction-title">Bank Transfer Instructions:</div>
              <ol class="instruction-list">
                <li>Transfer the exact amount shown above.</li>
                <li>Include the Reference ID in your transfer remarks.</li>
                <li>Your wallet will be credited once we verify your transfer.</li>
                <li>Processing time: 1-2 business days.</li>
              </ol>
            </div>
            
            <!-- Receipt Upload Section -->
            <div class="receipt-upload-section">
              <div class="upload-title">Upload Transfer Receipt - PDF</div>
              <div class="upload-description">Please upload a screenshot or photo of your bank transfer receipt</div>
              
              <div v-if="!receiptUploaded" class="upload-area">
                <input 
                  type="file" 
                  id="receipt-upload" 
                  accept="image/*,application/pdf" 
                  @change="handleReceiptUpload" 
                  style="display: none"
                >
                <label for="receipt-upload" class="upload-button">
                  <v-icon size="24">mdi-upload</v-icon>
                  <span>Select Receipt File</span>
                </label>
              </div>
              
              <div v-else class="receipt-preview">
                <div class="preview-header">
                  <div class="file-name">{{ receiptFile.name }}</div>
                  <v-icon @click="removeReceipt" class="remove-button">mdi-close</v-icon>
                </div>
                
                <div v-if="receiptPreview" class="image-preview">
                  <img :src="receiptPreview" alt="Receipt Preview">
                </div>
                <div v-else class="pdf-preview">
                  <v-icon size="48">mdi-file-pdf-box</v-icon>
                  <div>PDF Document</div>
                </div>
              </div>
            </div>
            
            <div class="action-buttons">
              <v-btn
                color="primary"
                class="action-btn"
                @click="processPayment"
                :disabled="!receiptUploaded"
                block
              >
                I've Made The Transfer
              </v-btn>
            </div>
          </div>
          
          <!-- SarawakPay Payment -->
          <div v-if="selectedMethod === 's_pay'" class="payment-methods">
            <div class="section-title">Redirecting to SarawakPay</div>
            
            <div class="d-flex flex-column align-center py-6">
              <div class="spay-animation">
                <div class="spay-logo">
                  <div class="spay-logo-inner"></div>
                </div>
                <div class="spay-loading">
                  <div class="spay-loading-dot"></div>
                  <div class="spay-loading-dot"></div>
                  <div class="spay-loading-dot"></div>
                </div>
              </div>
              
              <div class="text-subtitle-1 font-weight-bold mt-6">Connecting to SarawakPay</div>
              <div class="text-caption text-center mt-2 mb-6 px-4">
                Please wait while we redirect you to SarawakPay to complete your payment of {{ formattedTopUpAmount }}
              </div>
              
              <div class="transaction-details mt-4">
                <div class="transaction-row">
                  <div class="transaction-label">Amount to Top Up</div>
                  <div class="transaction-value">{{ formattedTopUpAmount }}</div>
                </div>
                
                <div class="transaction-row">
                  <div class="transaction-label">Processing Fee</div>
                  <div class="transaction-value">RM 0.00</div>
                </div>
                
                <v-divider class="my-3"></v-divider>
                
                <div class="transaction-row">
                  <div class="transaction-label">Total</div>
                  <div class="transaction-value">{{ formattedTopUpAmount }}</div>
                </div>
              </div>
              
              <div class="action-buttons">
                <v-btn
                  color="error"
                  variant="outlined"
                  class="action-btn"
                  @click="cancelSPayRedirect"
                  block
                >
                  Cancel
                </v-btn>
              </div>
            </div>
          </div>
          
          <!-- Other Payment Methods -->
          <div v-if="selectedMethod === 'e_wallet'" class="payment-methods">
            <div class="section-title">Top up with E-Wallet</div>
            
            <div class="d-flex flex-column align-center py-6">
              <v-img
                src="https://via.placeholder.com/300x200?text=QR+Code"
                width="220"
                height="220"
                class="mb-4"
              ></v-img>
              <div class="text-subtitle-1 font-weight-bold">Scan with your e-wallet app</div>
              <div class="text-caption text-center mt-2 mb-6 px-4">Open your e-wallet app and scan this QR code to top up {{ formattedTopUpAmount }}</div>
              
              <div class="action-buttons">
                <v-btn
                  color="primary"
                  class="action-btn"
                  @click="processPayment"
                  block
                >
                  I've Completed Payment
                </v-btn>
              </div>
            </div>
          </div>
          
          <div v-if="selectedMethod === 'points_conversion'" class="payment-methods">
            <div class="section-title">Convert Reward Points</div>
            
            <div class="transaction-details">
              <div class="transaction-row">
                <div class="transaction-label">Conversion Rate</div>
                <div class="transaction-value">10 points = RM 1.00</div>
              </div>
              
              <div class="transaction-row">
                <div class="transaction-label">Available Points</div>
                <div class="transaction-value">3,250 points</div>
              </div>
              
              <div class="transaction-row">
                <div class="transaction-label">Points Required</div>
                <div class="transaction-value">{{ topUpAmount * 10 }} points</div>
              </div>
              
              <v-divider class="my-3"></v-divider>
              
              <div class="transaction-row">
                <div class="transaction-label">Amount to Add</div>
                <div class="transaction-value">{{ formattedTopUpAmount }}</div>
              </div>
            </div>
            
            <div class="action-buttons">
              <v-btn
                color="primary"
                class="action-btn"
                :disabled="topUpAmount * 10 > 3250"
                @click="processPayment"
                block
              >
                Convert Points
              </v-btn>
            </div>
            
            <div v-if="topUpAmount * 10 > 3250" class="text-caption text-center mt-4 text-error">
              You don't have enough points for this conversion. Please reduce the amount or choose another payment method.
            </div>
          </div>
        </div>
        
        <!-- Step 3: Success -->
        <div v-if="paymentStep === 3" class="topup-success">
          <div class="success-icon">
            <v-icon color="white" size="36">mdi-check-circle</v-icon>
          </div>
          
          <!-- Different titles based on payment method -->
          <div class="success-title" v-if="selectedMethod === 's_pay'">Top Up Successful!</div>
          <div class="success-title" v-else>Top Up Submitted!</div>
          
          <!-- Different descriptions based on payment method -->
          <div class="success-message" v-if="selectedMethod === 's_pay'">Your wallet has been topped up successfully via SarawakPay.</div>
          <div class="success-message" v-else-if="selectedMethod === 'bank_transfer'">Please allow 1-2 business days for your bank transfer to be processed.</div>
          <div class="success-message" v-else>Please allow 1-2 business days for your top up to be processed.</div>
          
          <div class="transaction-details">
            <!-- Hide Amount Added for bank transfer -->
            <div class="transaction-row" v-if="selectedMethod !== 'bank_transfer'">
              <div class="transaction-label">Amount Added</div>
              <div class="transaction-value">{{ formattedTopUpAmount }}</div>
            </div>
            
            <div class="transaction-row">
              <div class="transaction-label">Method</div>
              <div class="transaction-value">{{ paymentTitle }}</div>
            </div>
            
            <div class="transaction-row">
              <div class="transaction-label">Date & Time</div>
              <div class="transaction-value">{{ new Date().toLocaleString() }}</div>
            </div>
            
            <div class="transaction-row">
              <div class="transaction-label">Transaction ID</div>
              <div class="transaction-value">{{ response.data.transaction_id }}</div>
            </div>
            
            <v-divider class="my-3"></v-divider>
            
            <!-- Hide Expected New Balance for SarawakPay -->
            <div class="transaction-row" v-if="selectedMethod !== 's_pay'">
              <div class="transaction-label">Expected New Balance</div>
              <div class="transaction-value green-highlight">{{ formatCurrency(newBalance) }}</div>
            </div>
          </div>
          
          <div class="action-buttons">
            <v-btn
              color="primary"
              variant="outlined"
              class="action-btn"
              @click="navigateTo('screen_008_wallet_detail.js')"
              block
            >
              View Transactions
            </v-btn>
            
            <v-btn
              color="primary"
              class="action-btn"
              @click="completeTopUp"
              block
            >
              Back to Wallet
            </v-btn>
          </div>
        </div>
        
        <!-- Recent Top Ups -->
        <div v-if="paymentStep === 1" class="recent-topups">
          <div class="section-title">Recent Top Ups</div>
          
          <div v-for="(topup, i) in recentTopUps" :key="i" class="recent-topup-item">
            <div class="topup-amounts">
              <div class="topup-credit" v-html="formatCredit(topup.amount)"></div>
              <div class="topup-currency">{{ formatCurrency(topup.amount) }}</div>
            </div>
            
            <div class="topup-info">
              <div class="topup-method-pill" :class="getMethodClass(topup.method)">
                {{ getMethodDisplay(topup.method) }}
              </div>
              <div class="topup-status" :class="topup.status.toLowerCase()">
                {{ topup.status }}
              </div>
              <div class="topup-date">{{ topup.date }}</div>
            </div>
          </div>
        </div>
      </div>
    </div>
    `
  }
  