import getTropicalStyles from "../css/tropicalBookingStyles.js"; 

export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',      // Dark green
        secondary: '#DDA333',    // Golden
        accent: '#A3BA95',       // Light sage green
        lightBg: '#F8F5F0',      // Off-white
        creamBg: '#EADDCB',      // Cream
        darkText: '#1E1E1E',     // Almost black
        lightText: '#FFFFFF',    // White
        error: '#B71C1C',        // Red
        success: '#2E7D32',      // Green
        warning: '#FF8F00',      // Orange
        muted: '#78909C',        // Grayish blue
        brown: '#6D4C41',        // Brown
      },

      // Check-in process control
      currentStep: 1,
      totalSteps: 5,
      
      // Rest of your data properties remain the same
      lookupMethod: 'reservation',
      lookupForm: {
        reservationNumber: 'RES123456', // Pre-filled for demo
        lastName: 'Smith', // Pre-filled for demo
        email: '',
        phone: '',
      },
      lookupFormValid: true, // Pre-set for demo
      reservationFound: false,
      lookupError: null,
      
      // Reservation data
      reservation: {
        id: 'RES123456',
        status: 'confirmed',
        checkInDate: '2025-06-10',
        checkOutDate: '2025-06-15',
        nights: 5,
        roomType: 'Jungle View Suite',
        roomRate: 250.00,
        adults: 2,
        children: 1,
        guestDetails: {
          firstName: 'John',
          lastName: 'Smith',
          email: 'jsmith@example.com',
          phone: '+1234567890',
        },
        amount: 1250.00,
        amountPaid: 1250.00,
        paymentStatus: 'Paid',
        specialRequests: 'High floor preferred',
        bookedDate: '2025-01-15',
      },
      
      // Guest information form
      guestForm: {
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        address: '',
        city: '',
        country: '',
        idType: '',
        idNumber: '',
      },
      guestFormValid: false,
      
      // Room preferences
      roomPreferences: {
        floorLevel: 'any',      // 'low', 'middle', 'high', 'any'
        bedType: 'unchanged',   // 'king', 'queen', 'twin', 'unchanged'
        pillowType: 'standard', // 'standard', 'firm', 'soft', 'hypoallergenic'
        view: 'forest',         // 'garden', 'pool', 'forest', 'unchanged'
        smoking: false,
        accessibility: false,
        quietZone: true,
        turndown: true,
      },
      specialRequests: '',
      
      // Add-on services
      additionalServices: [
        { id: 'breakfast', name: 'Rainforest Breakfast Buffet', price: 25, selected: false, icon: 'mdi-food-variant' },
        { id: 'parking', name: 'Valet Parking (Per Day)', price: 15, selected: false, icon: 'mdi-car' },
        { id: 'wifi', name: 'Premium High-Speed WiFi', price: 10, selected: false, icon: 'mdi-wifi' },
        { id: 'spa', name: 'Welcome Spa Treatment', price: 85, selected: false, icon: 'mdi-spa' },
        { id: 'airport', name: 'Airport Transfer', price: 60, selected: false, icon: 'mdi-airplane' },
        { id: 'earlyCheckin', name: 'Early Check-in (Before 2PM)', price: 40, selected: false, icon: 'mdi-clock-fast' },
        { id: 'lateCheckout', name: 'Late Check-out (Until 4PM)', price: 40, selected: false, icon: 'mdi-clock-time-four' },
        { id: 'minibar', name: 'Premium Minibar Package', price: 30, selected: false, icon: 'mdi-bottle-wine' },
      ],
      
      // Terms acceptance
      termsAccepted: false,
      dataConfirmed: false,
      
      // Check-in completion data
      checkInComplete: false,
      roomDetails: null,
      digitalKeyCode: null,
      
      // UI control
      loading: false,
      showSnackbar: false,
      snackbar: {
        text: '',
        color: 'success',
        timeout: 3000
      },
    };
  },
  
  computed: {
    // All computed properties remain the same
    formattedCheckInDate() {
      if (!this.reservation || !this.reservation.checkInDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return new Date(this.reservation.checkInDate).toLocaleDateString('en-US', options);
    },
    
    formattedCheckOutDate() {
      if (!this.reservation || !this.reservation.checkOutDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return new Date(this.reservation.checkOutDate).toLocaleDateString('en-US', options);
    },
    
    dateRange() {
      if (!this.reservation) return '';
      const checkIn = new Date(this.reservation.checkInDate);
      const checkOut = new Date(this.reservation.checkOutDate);
      const options = { month: 'short', day: 'numeric' };
      return `${checkIn.toLocaleDateString('en-US', options)} - ${checkOut.toLocaleDateString('en-US', options)} (${this.reservation.nights} nights)`;
    },
    
    isToday() {
      if (!this.reservation || !this.reservation.checkInDate) return false;
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const checkInDate = new Date(this.reservation.checkInDate);
      checkInDate.setHours(0, 0, 0, 0);
      return checkInDate.getTime() === today.getTime();
    },
    
    isFutureDate() {
      if (!this.reservation || !this.reservation.checkInDate) return false;
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const checkInDate = new Date(this.reservation.checkInDate);
      checkInDate.setHours(0, 0, 0, 0);
      return checkInDate > today;
    },
    
    isPastDate() {
      if (!this.reservation || !this.reservation.checkInDate) return false;
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const checkInDate = new Date(this.reservation.checkInDate);
      checkInDate.setHours(0, 0, 0, 0);
      return checkInDate < today;
    },
    
    isEarlyCheckIn() {
      if (!this.isToday) return false;
      const now = new Date();
      return now.getHours() < 14; // Standard check-in time is 2 PM (14:00)
    },
    
    checkInReadinessMessage() {
      if (this.isFutureDate) {
        return "Your check-in date is in the future. Standard check-in time is 2:00 PM on your arrival date.";
      } else if (this.isPastDate) {
        return "Your reservation was for a past date. Please contact the front desk for assistance.";
      } else if (this.isEarlyCheckIn) {
        return "You are checking in early. An early check-in fee may apply if your room is ready.";
      } else {
        return "You can check in now. Your room will be ready shortly.";
      }
    },
    
    selectedServicesTotal() {
      return this.additionalServices
        .filter(service => service.selected)
        .reduce((total, service) => total + service.price, 0);
    },
    
    totalAmount() {
      return this.selectedServicesTotal + (this.reservation ? this.reservation.amount : 0);
    }
  },
  
  methods: {
    // Navigation methods - Fixed to ensure they properly update the currentStep
    nextStep() {
      if (this.currentStep < this.totalSteps) {
        this.currentStep++;
      }
      // Scroll to top for better UX when navigating
      window.scrollTo(0, 0);
    },
    
    prevStep() {
      if (this.currentStep > 1) {
        this.currentStep--;
      }
      // Scroll to top for better UX when navigating
      window.scrollTo(0, 0);
    },
    
    // Other methods remain the same
    lookupReservation() {
      this.loading = true;
      this.lookupError = null;
      
      setTimeout(() => {
        // Simulate API call
        if (this.lookupMethod === 'reservation' && 
            this.lookupForm.reservationNumber === this.reservation.id && 
            this.lookupForm.lastName.toLowerCase() === 'smith') {
          this.reservationFound = true;
          this.guestForm = { ...this.reservation.guestDetails };
          this.nextStep();
          this.showSnackbarMessage('Reservation found!', 'success');
        } else if (this.lookupMethod === 'email' && 
                  this.lookupForm.email === this.reservation.guestDetails.email) {
          this.reservationFound = true;
          this.guestForm = { ...this.reservation.guestDetails };
          this.nextStep();
          this.showSnackbarMessage('Reservation found!', 'success');
        } else if (this.lookupMethod === 'phone' && 
                  this.lookupForm.phone === this.reservation.guestDetails.phone) {
          this.reservationFound = true;
          this.guestForm = { ...this.reservation.guestDetails };
          this.nextStep();
          this.showSnackbarMessage('Reservation found!', 'success');
        } else {
          this.reservationFound = false;
          this.lookupError = 'No matching reservation found. Please check your information or contact the front desk.';
        }
        this.loading = false;
      }, 1200);
    },
    
    completeCheckIn() {
      this.loading = true;
      
      setTimeout(() => {
        // Simulate API check-in process
        this.checkInComplete = true;
        this.roomDetails = {
          roomNumber: '314',
          floor: '3rd Floor',
          wing: 'Orchid Wing',
          accessInstructions: 'Use the elevators near the main lobby. Your room is to the right when exiting the elevator.',
        };
        this.digitalKeyCode = 'DK-' + Math.floor(10000 + Math.random() * 90000);
        this.loading = false;
        
        this.showSnackbarMessage('Check-in completed successfully!', 'success');
      }, 2000);
    },
    
    showSnackbarMessage(text, color = 'success', timeout = 3000) {
      this.snackbar.text = text;
      this.snackbar.color = color;
      this.snackbar.timeout = timeout;
      this.showSnackbar = true;
    },
    
    formatDate(dateString) {
      if (!dateString) return '';
      const options = { weekday: 'short', year: 'numeric', month: 'short', day: 'numeric' };
      return new Date(dateString).toLocaleDateString('en-US', options);
    },
    
    getStatusColor(status) {
      switch(status.toLowerCase()) {
        case 'confirmed': return this.theme.success;
        case 'pending': return this.theme.warning;
        case 'cancelled': return this.theme.error;
        case 'completed': return this.theme.muted;
        default: return this.theme.muted;
      }
    }
  },
  
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
    
    // Initialize the guest form from the reservation to avoid it being empty
    if (this.reservation && this.reservation.guestDetails) {
      this.guestForm = { ...this.reservation.guestDetails };
      this.guestFormValid = true; // Set to true for demo purposes
    }
  },
  
  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent :scrim="theme.primary" :opacity="0.8">
        <div class="text-center">
          <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
          <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Processing your check-in...</div>
        </div>
      </v-overlay>

      <!-- Hotel Logo & Header -->
      <v-row>
        <v-col cols="12" class="text-center pt-6 pb-2">
          <v-img src="https://via.placeholder.com/180x120?text=Rainforest+Resort" height="120" contain class="mx-auto mb-3"></v-img>
          <h1 :style="{color: theme.primary, fontFamily: 'Playfair Display, serif', fontSize: '2.5rem', fontWeight: 700, marginBottom: 0, letterSpacing: '-0.5px'}">Tapang Club</h1>
          <h2 :style="{color: theme.secondary, fontFamily: 'Poppins, sans-serif', fontSize: '1.5rem', fontWeight: 300, marginTop: 0, letterSpacing: '1px'}">Express Check-In</h2>
        </v-col>
      </v-row>

      <!-- Main Content -->
      <v-row>
        <v-col cols="12" sm="10" md="8" lg="6" class="mx-auto px-sm-4 px-2">
          <!-- Steps Indicator -->
          <v-card class="content-card mb-4" flat>
            <v-card-text class="pa-2">
              <div class="d-flex justify-space-between align-center">
                <div v-for="step in totalSteps" :key="step" class="text-center" style="flex: 1;">
                  <v-avatar
                    :color="step === currentStep ? theme.primary : (step < currentStep ? theme.accent : theme.creamBg)"
                    :size="step === currentStep ? 40 : 32"
                    class="mb-1"
                    :class="{'elevation-2': step === currentStep}"
                  >
                    <v-icon :color="step <= currentStep ? theme.lightText : theme.primary">
                      {{ step === 1 ? 'mdi-magnify' : 
                         step === 2 ? 'mdi-account-check' : 
                         step === 3 ? 'mdi-bed' : 
                         step === 4 ? 'mdi-plus-circle' : 
                         'mdi-check-circle' }}
                    </v-icon>
                  </v-avatar>
                  <div class="text-caption d-none d-sm-block" :style="{color: step === currentStep ? theme.primary : theme.muted, fontWeight: step === currentStep ? 'bold' : 'normal'}">
                    {{ step === 1 ? 'Find' : 
                       step === 2 ? 'Verify' : 
                       step === 3 ? 'Room' : 
                       step === 4 ? 'Add-ons' : 
                       'Done' }}
                  </div>
                </div>
              </div>
              <v-progress-linear
                :model-value="((currentStep - 1) / (totalSteps - 1)) * 100"
                :color="theme.primary"
                height="6"
                rounded
                class="mt-2"
              ></v-progress-linear>
            </v-card-text>
          </v-card>

          <!-- Content Card with Steps -->
          <v-card class="content-card" elevation="6">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">
                {{ currentStep === 1 ? 'mdi-magnify' : 
                   currentStep === 2 ? 'mdi-account-check' : 
                   currentStep === 3 ? 'mdi-bed' : 
                   currentStep === 4 ? 'mdi-plus-circle' : 
                   'mdi-check-circle' }}
              </v-icon>
              {{ currentStep === 1 ? 'Find Your Reservation' : 
                 currentStep === 2 ? 'Verify Information' : 
                 currentStep === 3 ? 'Room Preferences' : 
                 currentStep === 4 ? 'Additional Services' : 
                 'Complete Check-in' }}
            </v-card-title>

            <!-- Step 1: Find Reservation -->
            <v-card-text v-if="currentStep === 1" class="pa-sm-4 pa-2">
              <v-form v-model="lookupFormValid" @submit.prevent="lookupReservation">
                <v-tabs
                  v-model="lookupMethod"
                  :color="theme.primary"
                  align-tabs="center"
                  grow
                  slider-color="secondary"
                  class="mb-4"
                >
                  <v-tab value="reservation" class="booking-tab-item">Confirmation #</v-tab>
                  <v-tab value="email" class="booking-tab-item">Email</v-tab>
                  <v-tab value="phone" class="booking-tab-item">Phone</v-tab>
                </v-tabs>
                
                <v-window v-model="lookupMethod">
                  <!-- Reservation Number Search -->
                  <v-window-item value="reservation">
                    <br>
                    <v-text-field
                      v-model="lookupForm.reservationNumber"
                      label="Confirmation Number"
                      placeholder="e.g., RES123456"
                      :rules="[v => !!v || 'Confirmation number is required']"
                      variant="outlined"
                      :color="theme.primary"
                      prepend-inner-icon="mdi-ticket-confirmation"
                      density="comfortable"
                    ></v-text-field>
                    
                    <v-text-field
                      v-model="lookupForm.lastName"
                      label="Last Name"
                      placeholder="e.g., Smith"
                      :rules="[v => !!v || 'Last name is required']"
                      variant="outlined"
                      :color="theme.primary"
                      prepend-inner-icon="mdi-account"
                      density="comfortable"
                    ></v-text-field>
                  </v-window-item>
                  
                  <!-- Email Search -->
                  <v-window-item value="email">
                    <br>
                    <v-text-field
                      v-model="lookupForm.email"
                      label="Email Address"
                      placeholder="e.g., you@example.com"
                      :rules="[
                        v => !!v || 'Email is required',
                        v => /.+@.+\..+/.test(v) || 'Email must be valid'
                      ]"
                      variant="outlined"
                      :color="theme.primary"
                      prepend-inner-icon="mdi-email"
                      density="comfortable"
                    ></v-text-field>
                  </v-window-item>
                  
                  <!-- Phone Search -->
                  <v-window-item value="phone">
                    <br>
                    <v-text-field
                      v-model="lookupForm.phone"
                      label="Phone Number"
                      placeholder="e.g., +1234567890"
                      :rules="[v => !!v || 'Phone number is required']"
                      variant="outlined"
                      :color="theme.primary"
                      prepend-inner-icon="mdi-phone"
                      density="comfortable"
                    ></v-text-field>
                  </v-window-item>
                </v-window>
                
                <v-alert
                  v-if="lookupError"
                  type="error"
                  variant="tonal"
                  icon="mdi-alert-circle"
                  class="mt-4 mb-4"
                  closable
                  title="Not Found"
                >
                  {{ lookupError }}
                </v-alert>
                
                <div class="d-flex mt-6">
                  <v-spacer></v-spacer>
                  <v-btn
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    type="submit"
                    :loading="loading"
                    :disabled="!lookupFormValid"
                    size="large"
                    elevation="2"
                  >
                    <v-icon start>mdi-magnify</v-icon>
                    Find My Reservation
                  </v-btn>
                </div>
              </v-form>
              
              <v-divider class="my-4"></v-divider>
              
              <div class="text-center pb-3">
                <p class="text-body-2" :style="{color: theme.primary}">Don't have a reservation?</p>
                <v-btn
                  :color="theme.secondary"
                  :text-color="theme.primary"
                  variant="outlined"
                  class="mt-2"
                >
                  <v-icon start>mdi-book-plus</v-icon>
                  Book Now
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 2: Verify Information -->
            <v-card-text v-if="currentStep === 2" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Please review your details before proceeding</p>
              
              <v-alert
                :color="theme.accent"
                icon="mdi-information"
                variant="tonal"
                class="mb-4"
                density="comfortable"
              >
                Your reservation is confirmed for <strong>{{ formattedCheckInDate }}</strong>
              </v-alert>
              
              <v-card class="mb-4 reservation-summary" flat variant="outlined">
                <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">Reservation Details</v-card-title>
                <v-card-text class="py-2">
                  <v-row dense>
                    <v-col cols="12" sm="6">
                      <v-list density="compact" class="bg-transparent">
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                          </template>
                          <v-list-item-title class="text-body-2">{{ dateRange }}</v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-bed</v-icon>
                          </template>
                          <v-list-item-title class="text-body-2">{{ reservation.roomType }}</v-list-item-title>
                        </v-list-item>
                      </v-list>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-list density="compact" class="bg-transparent">
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-account-group</v-icon>
                          </template>
                          <v-list-item-title class="text-body-2">
                            {{ reservation.adults }} Adult{{ reservation.adults > 1 ? 's' : '' }}
                            <span v-if="reservation.children > 0">, {{ reservation.children }} Child{{ reservation.children > 1 ? 'ren' : '' }}</span>
                          </v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-cash</v-icon>
                          </template>
                          <v-list-item-title class="text-body-2">
                            Total: RM {{ reservation.amount.toFixed(2) }} ({{ reservation.paymentStatus }})
                          </v-list-item-title>
                        </v-list-item>
                      </v-list>
                    </v-col>
                  </v-row>
                </v-card-text>
              </v-card>
              
              <v-form v-model="guestFormValid">
                <h4 class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Guest Information</h4>
                
                <v-row>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.firstName"
                      label="First Name"
                      :rules="[v => !!v || 'First name is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-account"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.lastName"
                      label="Last Name"
                      :rules="[v => !!v || 'Last name is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.email"
                      label="Email"
                      :rules="[
                        v => !!v || 'Email is required',
                        v => /.+@.+\..+/.test(v) || 'Email must be valid'
                      ]"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-email"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.phone"
                      label="Phone"
                      :rules="[v => !!v || 'Phone is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-phone"
                    ></v-text-field>
                  </v-col>
                  
                  <v-col cols="12">
                    <v-divider class="my-2"></v-divider>
                    <h4 class="text-subtitle-2 mb-2 mt-2" :style="{color: theme.primary}">ID Verification</h4>
                  </v-col>
                  
                  <v-col cols="12" sm="6">
                    <v-select
                      v-model="guestForm.idType"
                      :items="['Passport', 'ID Card', 'Driver\\'s License']"
                      label="ID Type"
                      :rules="[v => !!v || 'ID Type is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-card-account-details"
                    ></v-select>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.idNumber"
                      label="ID Number"
                      :rules="[v => !!v || 'ID Number is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                    ></v-text-field>
                  </v-col>
                </v-row>
              </v-form>
                
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  :disabled="!guestFormValid"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 3: Room Preferences -->
            <v-card-text v-if="currentStep === 3" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Customize your rainforest experience</p>
              
              <v-alert
                :color="theme.accent"
                icon="mdi-leaf"
                variant="tonal"
                class="mb-4"
                density="comfortable"
              >
                We'll do our best to accommodate your preferences based on availability.
              </v-alert>
              
              <v-row>
                <v-col cols="12" sm="6">
                  <v-select
                    v-model="roomPreferences.floorLevel"
                    :items="[
                      { title: 'No Preference', value: 'any' },
                      { title: 'Lower Floor (1-3)', value: 'low' },
                      { title: 'Middle Floor (4-7)', value: 'middle' },
                      { title: 'Higher Floor (8+)', value: 'high' }
                    ]"
                    label="Floor Preference"
                    variant="outlined"
                    :color="theme.primary"
                    density="comfortable"
                    prepend-inner-icon="mdi-elevator"
                  ></v-select>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-select
                    v-model="roomPreferences.bedType"
                    :items="[
                      { title: 'No Change', value: 'unchanged' },
                      { title: 'King Bed', value: 'king' },
                      { title: 'Queen Bed', value: 'queen' },
                      { title: 'Twin Beds', value: 'twin' }
                    ]"
                    label="Bed Type"
                    variant="outlined"
                    :color="theme.primary"
                    density="comfortable"
                    prepend-inner-icon="mdi-bed"
                  ></v-select>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-select
                    v-model="roomPreferences.pillowType"
                    :items="[
                      { title: 'Standard Pillows', value: 'standard' },
                      { title: 'Firm Pillows', value: 'firm' },
                      { title: 'Soft Pillows', value: 'soft' },
                      { title: 'Hypoallergenic Pillows', value: 'hypoallergenic' }
                    ]"
                    label="Pillow Type"
                    variant="outlined"
                    :color="theme.primary"
                    density="comfortable"
                    prepend-inner-icon="mdi-pillow"
                  ></v-select>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-select
                    v-model="roomPreferences.view"
                    :items="[
                      { title: 'No Preference', value: 'unchanged' },
                      { title: 'Garden View', value: 'garden' },
                      { title: 'Pool View', value: 'pool' },
                      { title: 'Rainforest View', value: 'forest' }
                    ]"
                    label="Room View"
                    variant="outlined"
                    :color="theme.primary"
                    density="comfortable"
                    prepend-inner-icon="mdi-image-filter-hdr"
                  ></v-select>
                </v-col>
              </v-row>
              
              <v-divider class="my-4"></v-divider>
              
              <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Special Requirements</h4>
              
              <v-row>
                <v-col cols="12" sm="6">
                  <v-switch
                    v-model="roomPreferences.quietZone"
                    :color="theme.primary"
                    label="Quiet Zone Room"
                    hint="Room in a quieter area of the resort"
                    persistent-hint
                  ></v-switch>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-switch
                    v-model="roomPreferences.accessibility"
                    :color="theme.primary"
                    label="Accessibility Features"
                    hint="Room with accessibility accommodations"
                    persistent-hint
                  ></v-switch>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-switch
                    v-model="roomPreferences.smoking"
                    :color="theme.primary"
                    label="Smoking Room"
                    hint="If available (most rooms are non-smoking)"
                    persistent-hint
                  ></v-switch>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-switch
                    v-model="roomPreferences.turndown"
                    :color="theme.primary"
                    label="Turndown Service"
                    hint="Evening room refresh with jungle treats"
                    persistent-hint
                  ></v-switch>
                </v-col>
              </v-row>
              
              <v-textarea
                v-model="specialRequests"
                label="Special Requests"
                hint="Please let us know if you have any other special requests"
                variant="outlined"
                :color="theme.primary"
                density="comfortable"
                prepend-inner-icon="mdi-text-box"
                class="mt-4"
                rows="3"
              ></v-textarea>
              
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 4: Additional Services -->
            <v-card-text v-if="currentStep === 4" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Enhance your rainforest experience</p>
              
              <v-row>
                <v-col cols="12">
                  <v-alert
                    :color="theme.accent"
                    icon="mdi-leaf-circle"
                    variant="tonal"
                    class="mb-4"
                    density="comfortable"
                  >
                    Select any extra services to enhance your stay at Rainforest Resort
                  </v-alert>
                  
                  <div class="services-grid">
                    <v-card
                      v-for="service in additionalServices"
                      :key="service.id"
                      :class="{ 'selected-service': service.selected }"
                      class="service-card mb-3"
                      @click="service.selected = !service.selected"
                      elevation="1"
                      :color="service.selected ? theme.accent + '15' : ''"
                      variant="outlined"
                    >
                      <div class="d-flex align-center pa-3">
                        <v-avatar :color="service.selected ? theme.primary : theme.accent" size="42" class="mr-3">
                          <v-icon :color="service.selected ? theme.lightText : theme.primary">{{ service.icon }}</v-icon>
                        </v-avatar>
                        
                        <div class="flex-grow-1">
                          <div class="font-weight-medium" :style="{color: theme.primary}">{{ service.name }}</div>
                          <div class="text-caption" :style="{color: theme.muted}">RM {{ service.price.toFixed(2) }}</div>
                        </div>
                        
                        <v-checkbox
                          v-model="service.selected"
                          :color="theme.primary"
                          hide-details
                          density="compact"
                        ></v-checkbox>
                      </div>
                    </v-card>
                  </div>
                  
                  <v-divider class="my-4"></v-divider>
                  
                  <v-card class="summary-card pa-4" variant="outlined">
                    <div class="d-flex justify-space-between mb-2">
                      <span class="text-body-1">Room Total:</span>
                      <span class="text-body-1">RM {{ reservation.amount.toFixed(2) }}</span>
                    </div>
                    <div class="d-flex justify-space-between mb-2">
                      <span class="text-body-1">Additional Services:</span>
                      <span class="text-body-1">RM {{ selectedServicesTotal.toFixed(2) }}</span>
                    </div>
                    <v-divider class="my-2"></v-divider>
                    <div class="d-flex justify-space-between font-weight-bold" :style="{color: theme.primary}">
                      <span class="text-h6">Total Amount:</span>
                      <span class="text-h6">RM {{ totalAmount.toFixed(2) }}</span>
                    </div>
                  </v-card>
                  
                  <v-alert
                    type="info"
                    variant="tonal"
                    density="comfortable"
                    class="mt-4"
                    icon="mdi-information"
                  >
                    Additional charges will be applied to your room bill. You can modify these selections during your stay.
                  </v-alert>
                </v-col>
              </v-row>
              
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 5: Complete Check-in -->
            <v-card-text v-if="currentStep === 5" class="pa-sm-4 pa-2">
              <div v-if="!checkInComplete">
                <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Finalize your jungle adventure</p>
                
                <v-alert
                  :color="theme.success"
                  icon="mdi-check-circle"
                  variant="tonal"
                  class="mb-4"
                >
                  You're almost done! Review your details and complete your check-in.
                </v-alert>
                
                <v-card class="mb-4" variant="outlined">
                  <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">Check-in Summary</v-card-title>
                  <v-card-text class="py-2">
                    <v-list density="compact" bg-color="transparent">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-account</v-icon>
                        </template>
                        <v-list-item-title>{{ guestForm.firstName }} {{ guestForm.lastName }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-bed</v-icon>
                        </template>
                        <v-list-item-title>{{ reservation.roomType }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                        </template>
                        <v-list-item-title>{{ dateRange }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item v-if="selectedServicesTotal > 0">
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-plus-circle</v-icon>
                        </template>
                        <v-list-item-title>
                          Additional Services: RM {{ selectedServicesTotal.toFixed(2) }}
                        </v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-card-text>
                </v-card>
                
                <v-alert
                  :color="theme.warning"
                  :icon="isEarlyCheckIn ? 'mdi-clock-alert' : 'mdi-information'"
                  variant="tonal"
                  class="mb-4"
                >
                  {{ checkInReadinessMessage }}
                </v-alert>
                
                <v-divider class="my-4"></v-divider>
                
                <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Terms & Conditions</h4>
                
                <v-checkbox
                  v-model="termsAccepted"
                  :color="theme.primary"
                  label="I accept the hotel's terms and policies"
                  :rules="[v => !!v || 'You must accept the terms to continue']"
                ></v-checkbox>
                
                <v-checkbox
                  v-model="dataConfirmed"
                  :color="theme.primary"
                  label="I confirm that all the information provided is correct"
                  :rules="[v => !!v || 'You must confirm your information']"
                ></v-checkbox>
                
                <div class="d-flex mt-6">
                  <v-btn 
                    :color="theme.muted" 
                    variant="text"
                    @click="prevStep"
                  >
                    <v-icon start>mdi-arrow-left</v-icon>
                    Back
                  </v-btn>
                  <v-spacer></v-spacer>
                  <v-btn
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    @click="completeCheckIn"
                    size="large"
                    elevation="2"
                    :loading="loading"
                    :disabled="!termsAccepted || !dataConfirmed"
                  >
                    <v-icon start>mdi-check-circle</v-icon>
                    Complete Check-in
                  </v-btn>
                </div>
              </div>
              
              <!-- Check-in Success Card -->
              <div v-else>
                <div class="text-center py-4" :style="{ backgroundColor: theme.success, color: theme.lightText, borderRadius: '8px 8px 0 0' }">
                  <v-icon size="64" class="mb-2">mdi-check-circle</v-icon>
                  <h2 class="text-h5 font-weight-bold">Check-in Successful!</h2>
                </div>
                
                <div class="pa-4">
                  <div class="text-center mb-4">
                    <h3 class="text-h6" :style="{ color: theme.primary }">Welcome to Tapang Resort</h3>
                    <p class="text-body-1">Your room is ready, enjoy your stay!</p>
                  </div>
                  
                  <v-row>
                    <v-col cols="12" sm="6">
                      <v-card class="facility-item mb-4 mb-sm-0" elevation="1">
                        <v-card-text>
                          <div class="d-flex align-center mb-3">
                            <v-icon :color="theme.primary" size="large" class="mr-2">mdi-door</v-icon>
                            <span class="text-h5 font-weight-bold" :style="{ color: theme.primary }">Room {{ roomDetails.roomNumber }}</span>
                          </div>
                          <v-list density="compact" bg-color="transparent">
                            <v-list-item>
                              <template v-slot:prepend>
                                <v-icon small :color="theme.primary">mdi-home-floor-1</v-icon>
                              </template>
                              <v-list-item-title>{{ roomDetails.floor }}, {{ roomDetails.wing }}</v-list-item-title>
                            </v-list-item>
                            <v-list-item>
                              <template v-slot:prepend>
                                <v-icon small :color="theme.primary">mdi-directions</v-icon>
                              </template>
                              <v-list-item-title class="text-wrap">{{ roomDetails.accessInstructions }}</v-list-item-title>
                            </v-list-item>
                          </v-list>
                        </v-card-text>
                      </v-card>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-card class="facility-item" elevation="1">
                        <v-card-text class="text-center">
                          <div class="d-flex align-center justify-center mb-3">
                            <v-icon :color="theme.secondary" size="large" class="mr-2">mdi-key-wireless</v-icon>
                            <span class="text-h6 font-weight-bold" :style="{ color: theme.secondary }">Digital Room Key</span>
                          </div>
                          <v-divider class="mb-3"></v-divider>
                          <p class="text-body-2 mb-2">Your digital key code:</p>
                          <div class="digital-key-code pa-2 mb-3" :style="{ backgroundColor: theme.creamBg }">
                            <span class="text-h6 font-weight-bold">{{ digitalKeyCode }}</span>
                          </div>
                          <p class="text-caption">Use this code at the elevator and your room door</p>
                          
                          <v-btn
                            block
                            :color="theme.secondary"
                            :text-color="theme.primary"
                            class="mt-3"
                            prepend-icon="mdi-cellphone-nfc"
                          >
                            Send to Phone
                          </v-btn>
                        </v-card-text>
                      </v-card>
                    </v-col>
                  </v-row>
                  
                  <v-divider class="my-4"></v-divider>
                  
                  <div class="important-info">
                    <h4 class="text-subtitle-1 mb-2" :style="{ color: theme.primary }">Important Information:</h4>
                    <v-list density="compact" bg-color="transparent">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-clock</v-icon>
                        </template>
                        <v-list-item-title>Check-out time: 12:00 PM</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-silverware</v-icon>
                        </template>
                        <v-list-item-title>Breakfast: 6:30 AM - 10:30 AM at Canopy Restaurant</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-wifi</v-icon>
                        </template>
                        <v-list-item-title>Wi-Fi Network: RainforestGuest / Password: jungle2025</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-phone</v-icon>
                        </template>
                        <v-list-item-title>For assistance: Dial 0 from your room phone</v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </div>
                
                  <div class="d-flex mt-6 justify-center">
                    <v-btn
                      :color="theme.secondary"
                      :text-color="theme.primary"
                      elevation="2"
                      prepend-icon="mdi-file-document"
                    >
                      Receipt
                    </v-btn>
                    <v-btn
                      :color="theme.primary"
                      :text-color="theme.lightText"
                      elevation="2"
                      class="ml-4"
                      prepend-icon="mdi-map"
                    >
                      Resort Map
                    </v-btn>
                  </div>
                </div>
              </div>
            </v-card-text>
          </v-card>
        </v-col>
      </v-row>
      
      <v-snackbar
        v-model="showSnackbar"
        :color="snackbar.color"
        :timeout="snackbar.timeout"
        location="top"
      >
        {{ snackbar.text }}
        <template v-slot:actions>
          <v-btn
            variant="text"
            @click="showSnackbar = false"
          >
            Close
          </v-btn>
        </template>
      </v-snackbar>
    </v-container>
  `
};
