export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C'
      },

      // Transfer - States and Data
      isLoading: false,
      transferAmount: 100,
      transferStep: 1,
      transferPin: '',
      isProcessing: false,
      showReceipt: false,
      isErrorNotification: false,

      // Predefined amounts
      predefinedAmounts: [50, 100, 200, 500, 1000],

      // User Data
      userData: {
        name: "Jason Tan",
        role: "platinum",
        membershipId: "TAPANG-PLAT-2023",
        balance: 350.00,
        availableCredit: 0
      },

      // Recipient Data
      recipient: null,
      recentRecipients: [
        {
          id: 'MEM-1234',
          name: 'Sarah Wong',
          membershipId: 'TAPANG-GOLD-2023',
          avatar: null,
          lastTransfer: 'May 15, 2023'
        },
        {
          id: 'MEM-2345',
          name: 'Michael Lee',
          membershipId: 'TAPANG-PLAT-2023',
          avatar: null,
          lastTransfer: 'April 28, 2023'
        },
        {
          id: 'MEM-3456',
          name: 'Jennifer Tan',
          membershipId: 'TAPANG-SILV-2023',
          avatar: null,
          lastTransfer: 'March 10, 2023'
        }
      ],

      // Search functionality
      searchQuery: '',
      searchResults: [],
      isSearching: false,

      // Transfer details
      transferNote: '',
      transferReference: '',

      // Transaction receipt
      receipt: {
        transactionId: '',
        date: '',
        amount: 0,
        recipientName: '',
        recipientId: '',
        status: 'completed',
        note: ''
      },

      // Notification system
      showNotification: false,
      notificationMessage: '',

      // Animation states
      hoveredRecipient: null
    }
  },
  computed: {
    // Format current balance
    formattedBalance() {
      return this.formatCurrency(this.userData.balance)
    },

    // Format transfer amount
    formattedTransferAmount() {
      return this.formatCurrency(this.transferAmount)
    },

    // Check if amount is valid
    isAmountValid() {
      return this.transferAmount >= 10 &&
        this.transferAmount <= 5000 &&
        this.transferAmount <= this.userData.balance
    },

    // Calculate new balance after transfer
    newBalance() {
      return this.userData.balance - this.transferAmount
    },

    // Format new balance
    formattedNewBalance() {
      return this.formatCurrency(this.newBalance)
    },

    // Filter recent recipients based on search query
    filteredRecipients() {
      if (!this.searchQuery) return this.recentRecipients

      const query = this.searchQuery.toLowerCase()
      return this.recentRecipients.filter(r =>
        r.name.toLowerCase().includes(query) ||
        r.membershipId.toLowerCase().includes(query)
      )
    },

    // Check if transfer is ready to proceed
    canProceed() {
      return this.isAmountValid && this.recipient !== null
    },

    // Check if PIN is valid (must be 4-6 digits to match the change PIN validation)
    isPinValid() {
      return this.transferPin.length === 6 && this.transferPin.length <= 6 && /^\d+$/.test(this.transferPin)
    }
  },
  methods: {
    formatCurrency(amount) {
      return "RM " + parseFloat(amount).toFixed(2)
    },

    // Set predefined amount
    setAmount(amount) {
      this.transferAmount = amount
    },

    // Search for recipients
    async searchRecipients() {
      if (!this.searchQuery) return

      this.isSearching = true

      try {
        // Call the search users API
        const response = await this.$root.callApi('wallet', 'search_users', {
          search_query: this.searchQuery,
          user_id: sessionStorage.getItem('id'),
          limit: 10
        })

        if (response.data && response.data.status === "YES") {
          this.searchResults = response.data.users
        } else {
          // this.showErrorNotification(response.data.message || 'Error searching recipients')
        }
      } catch (error) {
        console.error('Error searching recipients:', error)
        this.showErrorNotification('Error searching recipients')
      } finally {
        this.isSearching = false
      }
    },

    // Show error notification (centered modal)
    showErrorNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = true

      // Auto-hide after 4 seconds for errors (longer than regular notifications)
      setTimeout(() => {
        this.showNotification = false
        this.isErrorNotification = false
      }, 4000)
    },

    // Method to manually close error notification
    closeErrorNotification() {
      this.showNotification = false
      this.isErrorNotification = false
    },

    // Show regular notification
    showRegularNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = false
      setTimeout(() => {
        this.showNotification = false
      }, 3000)
    },

    // Select a recipient
    selectRecipient(recipient) {
      this.recipient = recipient
      this.searchQuery = ''
      this.searchResults = []
    },

    // Clear selected recipient
    clearRecipient() {
      this.recipient = null
    },

    // Proceed to next step
    nextStep() {
      if (this.transferStep === 1 && this.canProceed) {
        this.transferStep = 2
      } else if (this.transferStep === 2) {
        this.transferStep = 3
      }
    },

    // Go back to previous step
    goBack() {
      if (this.transferStep > 1) {
        this.transferStep--
      } else {
        this.$root.loadComponent('modules/home.js')
      }
    },

    // Process the transfer
    async processTransfer() {
      if (!this.isPinValid) {
        this.showErrorNotification('Please enter a valid 6-digit PIN')
        return
      }

      this.isProcessing = true
      this.showRegularNotification('Processing your transfer...')

      try {
        const response = await this.$root.callApi('wallet', 'transfer', {
          user_id: sessionStorage.getItem('id'),
          recipient_id: this.recipient.id, // We still need to use ID for the API call
          amount: this.transferAmount,
          note: this.transferNote,
          pin: this.transferPin
        })

        if (response.data && response.data.status === "YES") {
          // Update balance
          this.userData.balance = parseFloat(response.data.new_balance)

          // Generate transaction receipt
          this.receipt = {
            transactionId: response.data.transaction_id,
            date: response.data.transaction_date,
            amount: this.transferAmount,
            recipientName: response.data.recipient_name,
            recipientId: response.data.recipient_member_no || response.data.recipient_id, // Use member_no instead of id
            status: 'completed',
            note: this.transferNote || 'No note provided'
          }

          // Show success screen
          this.transferStep = 4
          this.showReceipt = true

          this.showRegularNotification('Transfer successful!')
        } else {
          this.showErrorNotification(response.data.message || 'Transfer failed')
        }
      } catch (error) {
        console.error('Error processing transfer:', error)
        this.showErrorNotification('Error processing transfer')
      } finally {
        this.isProcessing = false
      }
    },

    // Complete the transfer process
    completeTransfer() {
      this.$root.loadComponent('modules/home.js')
    },

    // Download receipt as PDF
    downloadReceiptPDF() {
      this.showNotification = true
      this.notificationMessage = 'Generating PDF...'

      // Create a clone of the receipt element to modify before capturing
      const originalReceipt = document.querySelector('.receipt-screen')

      if (!originalReceipt) {
        this.showErrorNotification('Could not generate PDF receipt')
        return
      }

      // Clone the receipt element
      const receiptClone = originalReceipt.cloneNode(true)

      // Remove the action buttons from the clone
      const actionButtons = receiptClone.querySelector('.action-buttons')
      if (actionButtons) {
        actionButtons.remove()
      }

      // Add the clone to the document temporarily (hidden)
      receiptClone.style.position = 'absolute'
      receiptClone.style.left = '-9999px'
      receiptClone.style.top = '-9999px'
      document.body.appendChild(receiptClone)

      // Use html2canvas to capture the modified receipt clone
      html2canvas(receiptClone, {
        scale: 2, // Higher scale for better quality
        useCORS: true,
        backgroundColor: '#ffffff'
      }).then(canvas => {
        // Remove the clone from the document
        document.body.removeChild(receiptClone)

        // Create a new jsPDF instance
        const { jsPDF } = window.jspdf
        const pdf = new jsPDF('p', 'mm', 'a4')

        // Calculate dimensions to fit the receipt on the page
        const imgWidth = 210 - 40 // A4 width (210mm) minus margins
        const pageHeight = 297 // A4 height
        const imgHeight = (canvas.height * imgWidth) / canvas.width

        // Add the Tapang Club header
        pdf.setFontSize(22)
        pdf.setTextColor(26, 60, 40) // var(--primary) color
        pdf.text('Tapang Club', 105, 20, { align: 'center' })

        pdf.setFontSize(16)
        pdf.setTextColor(0, 0, 0)
        pdf.text('Transfer Receipt', 105, 30, { align: 'center' })

        // Add the canvas image to the PDF
        const imgData = canvas.toDataURL('image/png')
        pdf.addImage(imgData, 'PNG', 20, 40, imgWidth, imgHeight)

        // Add footer with transaction ID and date
        pdf.setFontSize(10)
        pdf.setTextColor(100, 100, 100)
        pdf.text(`Transaction ID: ${this.receipt.transactionId}`, 20, pageHeight - 20)
        pdf.text(`Date: ${this.receipt.date}`, 20, pageHeight - 15)
        pdf.text('Tapang Club © 2025', 105, pageHeight - 10, { align: 'center' })

        // Generate the PDF as a blob with proper MIME type
        const pdfBlob = pdf.output('blob', {
          filename: `Tapang_Transfer_Receipt_${this.receipt.transactionId}.pdf`
        })

        // Create a proper PDF file with correct MIME type
        const pdfFile = new File(
          [pdfBlob],
          `Tapang_Transfer_Receipt_${this.receipt.transactionId}.pdf`,
          { type: 'application/pdf' }
        )

        // Check if the browser supports the modern File System Access API
        if (navigator.canShare && navigator.canShare({ files: [pdfFile] })) {
          // Use Web Share API for mobile devices
          navigator.share({
            title: 'Tapang Club Transfer Receipt',
            files: [pdfFile]
          }).then(() => {
            this.showNotification = true
            this.notificationMessage = 'PDF receipt shared successfully!'
            setTimeout(() => {
              this.showNotification = false
            }, 3000)
          }).catch(error => {
            // If sharing fails, fall back to download
            console.log('Sharing failed, falling back to download', error)
            this.downloadFile(pdfBlob)
          })
        } else {
          // Fall back to direct download for browsers that don't support sharing
          this.downloadFile(pdfBlob)
        }
      }).catch(error => {
        console.error('Error generating PDF:', error)
        this.showErrorNotification('Error generating PDF receipt')
      })
    },

    // Helper method to download the PDF file
    downloadFile(blob) {
      // Create a URL for the blob
      const url = URL.createObjectURL(blob)

      // Create a link element
      const link = document.createElement('a')
      link.href = url
      link.download = `Tapang_Transfer_Receipt_${this.receipt.transactionId}.pdf`

      // Append to the document, click it, and remove it
      document.body.appendChild(link)
      link.click()
      document.body.removeChild(link)

      // Clean up the URL object
      setTimeout(() => URL.revokeObjectURL(url), 100)

      // Show success notification
      this.showNotification = true
      this.notificationMessage = 'PDF receipt downloaded successfully!'
      setTimeout(() => {
        this.showNotification = false
      }, 3000)
    },

    // Navigation methods
    navigateTo(screen) {
      this.showNotification = true
      this.notificationMessage = 'Loading...'
      setTimeout(() => {
        this.$root.loadComponent('modules/' + screen)
      }, 600)
    },

    // Fetch wallet balance
    async fetchWalletBalance() {
      this.isLoading = true
      try {
        const response = await this.$root.callApi('wallet', 'get_balance', {
          user_id: sessionStorage.getItem('id')
        })

        if (response.data && response.data.status === "YES") {
          this.userData.balance = parseFloat(response.data.balance)
          this.userData.availableCredit = parseFloat(response.data.available_credit)
        } else {
          this.showErrorNotification(response.data.message || 'Failed to fetch wallet balance')
        }
      } catch (error) {
        console.error('Error fetching wallet balance:', error)
        this.showErrorNotification('Error fetching wallet balance')
      } finally {
        this.isLoading = false
      }
    }
  },
  mounted() {
    // Fetch wallet data
    this.fetchWalletBalance()

    // IMPORTANT: Inject the CSS styles dynamically
    const styleElement = document.createElement('style')
    styleElement.textContent = `
      /* Base Styles - Mobile First Approach */
      .transfer-container {
        --primary: #1A3C28;
        --secondary: #DDA333;
        --accent: #A3BA95;
        --cream: #EADDCB;
        --muted: #889FAC;
        --text-dark: #1E1E1E;
        --text-light: #FFFFFF;
        --error: #B71C1C;
        --light-green: #F0F7F2;
        --success: #2E7D32;
        
        font-family: 'Arial', sans-serif;
        background-color: var(--cream);
        min-height: 100vh;
        position: relative;
        overflow-x: hidden;
        -webkit-tap-highlight-color: transparent;
      }
      
      /* Loading Overlay */
      .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(26, 60, 40, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 1000;
      }
      
      .loading-spinner {
        display: flex;
        flex-direction: column;
        align-items: center;
      }
      
      .loading-text {
        margin-top: 16px;
        color: white;
        font-weight: 500;
      }
      
      /* Notification Toast - Mobile Optimized */
      .notification-toast {
        position: fixed;
        bottom: 24px;
        left: 50%;
        transform: translateX(-50%);
        background-color: var(--secondary);
        color: var(--primary);
        padding: 14px 20px;
        border-radius: 20px;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 900;
        min-width: 80%;
        max-width: 90%;
        text-align: center;
        transition: all 0.3s ease;
      }
      
      /* Error notification - Centered modal overlay */
      .notification-toast.error-notification {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background-color: var(--error);
        color: white;
        padding: 20px 24px;
        border-radius: 16px;
        box-shadow: 0 8px 24px rgba(183, 28, 28, 0.3);
        z-index: 1001;
        min-width: 85%;
        max-width: 95%;
        font-size: 16px;
        line-height: 1.4;
        position: relative;
      }

      /* Error notification backdrop */
      .error-notification-backdrop {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        backdrop-filter: blur(2px);
      }

      /* Close button for error notifications */
      .error-close-button {
        position: absolute;
        top: 8px;
        right: 12px;
        background: none;
        border: none;
        color: white;
        font-size: 20px;
        cursor: pointer;
        padding: 4px;
        line-height: 1;
        opacity: 0.8;
      }

      .error-close-button:hover {
        opacity: 1;
      }
      
      .fade-enter-active,
      .fade-leave-active {
        transition: opacity 0.3s, transform 0.3s;
      }
      
      .fade-enter-from,
      .fade-leave-to {
        opacity: 0;
        transform: translateX(-50%) translateY(20px);
      }
      
      /* Background Decorations */
      .background-decoration {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        z-index: -1;
        overflow: hidden;
      }
      
      .bg-leaf {
        position: absolute;
        border-radius: 50% 50% 0 50%;
        opacity: 0.07;
        background-color: var(--accent);
        transform: rotate(45deg);
      }
      
      .bg-leaf-1 {
        width: 300px;
        height: 300px;
        top: -150px;
        left: -150px;
      }
      
      .bg-leaf-2 {
        width: 250px;
        height: 250px;
        bottom: 10%;
        right: -100px;
        transform: rotate(130deg);
      }
      
      .bg-leaf-3 {
        width: 180px;
        height: 180px;
        top: 30%;
        left: -90px;
        transform: rotate(80deg);
      }
      
      /* Header Section - Mobile Optimized */
      .transfer-header {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        padding: 16px;
        border-radius: 0 0 24px 24px;
        box-shadow: 0 6px 16px rgba(0,0,0,0.15);
        color: var(--text-light);
        position: relative;
        overflow: hidden;
      }
      
      .leaf-decoration {
        position: absolute;
        width: 120px;
        height: 120px;
        background-color: rgba(255,255,255,0.05);
        border-radius: 70% 30% 50% 50% / 50% 50% 70% 30%;
        z-index: 0;
      }
      
      .leaf-decoration.left {
        top: -30px;
        left: -50px;
        transform: rotate(35deg);
      }
      
      .leaf-decoration.right {
        bottom: -60px;
        right: -30px;
        transform: rotate(-15deg);
      }
      
      .header-content {
        position: relative;
        z-index: 1;
      }
      
      .back-button {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        cursor: pointer;
        min-height: 44px; /* Mobile touch target */
        transition: opacity 0.2s ease;
      }
      
      .back-button:hover {
        opacity: 0.8;
      }
      
      .back-button-text {
        margin-left: 8px;
        font-weight: 500;
        font-size: 16px;
      }
      
      .page-title {
        font-size: 22px;
        font-weight: 700;
        margin-bottom: 12px;
      }
      
      /* Balance Summary - Enhanced Design */
      .balance-summary {
        background: linear-gradient(135deg, white 0%, #f8f9fa 100%);
        border-radius: 20px;
        margin: -15px 16px 20px 16px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.12);
        position: relative;
        overflow: hidden;
        border: 1px solid rgba(26, 60, 40, 0.08);
      }
      
      .balance-summary::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 3px;
        background: linear-gradient(90deg, var(--primary), var(--secondary), var(--accent));
      }
      
      .balance-title {
        color: var(--muted);
        font-size: 14px;
        margin-bottom: 8px;
        font-weight: 500;
      }
      
      .balance-amount {
        font-size: 28px;
        font-weight: 700;
        color: var(--primary);
        text-shadow: 0 1px 2px rgba(0,0,0,0.1);
      }
      
      /* Content Area - Mobile Optimized */
      .content-area {
        padding: 0 16px 80px 16px;
        margin-top: 20px;
      }
      
      /* Step indicator - Enhanced Design */
      .step-indicator {
        display: flex;
        justify-content: center;
        margin: 0 0 24px 0;
        padding: 16px;
        background: white;
        border-radius: 16px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.08);
      }
      
      .step {
        display: flex;
        flex-direction: column;
        align-items: center;
        position: relative;
        flex: 1;
      }
      
      .step:not(:last-child)::after {
        content: '';
        position: absolute;
        top: 18px;
        left: calc(50% + 20px);
        width: calc(100% - 40px);
        height: 3px;
        background: linear-gradient(90deg, #E0E0E0, #F0F0F0);
        border-radius: 2px;
        z-index: 0;
      }
      
      .step.active:not(:last-child)::after,
      .step.completed:not(:last-child)::after {
        background: linear-gradient(90deg, var(--secondary), var(--accent));
      }
      
      .step-circle {
        width: 36px;
        height: 36px;
        border-radius: 50%;
        background: linear-gradient(135deg, #E0E0E0 0%, #D0D0D0 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        font-size: 14px;
        color: white;
        margin-bottom: 8px;
        position: relative;
        z-index: 1;
        transition: all 0.3s ease;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      }
      
      .step.active .step-circle {
        background: linear-gradient(135deg, var(--secondary) 0%, #E6B800 100%);
        transform: scale(1.1);
        box-shadow: 0 4px 12px rgba(221, 163, 51, 0.3);
      }
      
      .step.completed .step-circle {
        background: linear-gradient(135deg, var(--primary) 0%, var(--success) 100%);
        box-shadow: 0 4px 12px rgba(26, 60, 40, 0.3);
      }
      
      .step-label {
        font-size: 12px;
        color: var(--muted);
        text-align: center;
        font-weight: 500;
        transition: all 0.2s ease;
      }
      
      .step.active .step-label,
      .step.completed .step-label {
        color: var(--text-dark);
        font-weight: 600;
      }
      
      /* Recipient Selection - Enhanced Design */
      .recipient-selection {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .section-title {
        font-size: 18px;
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 16px;
      }
      
      /* Search Input - FIXED DESIGN */
      .search-container {
        margin-bottom: 20px;
      }
      
      .search-input-wrapper {
        position: relative;
        border-radius: 16px;
        overflow: hidden;
        box-shadow: 0 4px 16px rgba(26, 60, 40, 0.08);
        transition: all 0.3s ease;
        background: white;
        border: 2px solid transparent;
      }
      
      .search-input-wrapper:focus-within {
        box-shadow: 0 6px 24px rgba(26, 60, 40, 0.15);
        border-color: var(--accent);
        transform: translateY(-1px);
      }
      
      .search-input-wrapper .search-input {
        width: 100%;
        border: none;
        padding: 16px 60px 16px 20px;
        font-size: 16px;
        background: transparent;
        color: var(--text-dark);
        outline: none;
      }
      
      .search-input-wrapper .search-input::placeholder {
        color: var(--muted);
        opacity: 0.7;
      }
      
      .search-button {
        position: absolute;
        right: 8px;
        top: 50%;
        transform: translateY(-50%);
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        color: white;
        border: none;
        border-radius: 12px;
        width: 40px;
        height: 40px;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: all 0.2s ease;
        box-shadow: 0 2px 8px rgba(26, 60, 40, 0.2);
      }
      
      .search-button:hover:not(:disabled) {
        background: linear-gradient(135deg, var(--secondary) 0%, #E6B800 100%);
        color: var(--primary);
        transform: translateY(-50%) scale(1.05);
      }
      
      .search-button:active {
        transform: translateY(-50%) scale(0.95);
      }
      
      .search-button:disabled {
        background: #ccc;
        color: #666;
        cursor: not-allowed;
        transform: translateY(-50%);
        box-shadow: none;
      }
      
      .recipient-list {
        margin-top: 16px;
      }
      
      .recipient-card {
        display: flex;
        align-items: center;
        padding: 16px;
        border-radius: 16px;
        margin-bottom: 12px;
        cursor: pointer;
        transition: all 0.3s ease;
        background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        border: 2px solid transparent;
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
      }
      
      .recipient-card:hover {
        background: linear-gradient(135deg, var(--light-green) 0%, #ffffff 100%);
        border-color: var(--accent);
        transform: translateY(-2px);
        box-shadow: 0 4px 16px rgba(26, 60, 40, 0.1);
      }
      
      .recipient-card:active {
        transform: translateY(0);
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
      }
      
      .recipient-card.selected {
        background: linear-gradient(135deg, rgba(26, 60, 40, 0.05) 0%, rgba(163, 186, 149, 0.1) 100%);
        border-color: var(--primary);
        box-shadow: 0 4px 16px rgba(26, 60, 40, 0.15);
      }
      
      .recipient-avatar {
        width: 48px;
        height: 48px;
        border-radius: 50%;
        background: linear-gradient(135deg, var(--accent) 0%, #8FA582 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 16px;
        font-weight: 700;
        color: white;
        font-size: 18px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      }
      
      .recipient-info {
        flex: 1;
      }
      
      .recipient-name {
        font-weight: 600;
        font-size: 16px;
        color: var(--text-dark);
        margin-bottom: 4px;
      }
      
      .recipient-id {
        font-size: 13px;
        color: var(--muted);
        font-weight: 500;
      }
      
      .recipient-last-transfer {
        font-size: 12px;
        color: var(--muted);
        margin-top: 2px;
      }
      
      /* Amount Selection - Enhanced Design */
      .amount-selection {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .amount-chips {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 12px;
        margin-bottom: 24px;
      }
      
      .amount-chip {
        background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        color: var(--text-dark);
        padding: 16px 8px;
        border-radius: 16px;
        cursor: pointer;
        transition: all 0.3s ease;
        border: 2px solid transparent;
        font-weight: 600;
        text-align: center;
        min-height: 56px; /* Good mobile touch target */
        display: flex;
        align-items: center;
        justify-content: center;
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
      }
      
      .amount-chip:hover {
        background: linear-gradient(135deg, var(--light-green) 0%, #ffffff 100%);
        border-color: var(--accent);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(26, 60, 40, 0.1);
      }
      
      .amount-chip:active {
        transform: translateY(0);
      }
      
      .amount-chip.selected {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        border-color: var(--primary);
        color: white;
        font-weight: 700;
        box-shadow: 0 4px 16px rgba(26, 60, 40, 0.2);
      }
      
      .custom-amount {
        margin-top: 16px;
      }
      
      /* Transfer Note - Enhanced Design */
      .transfer-note {
        margin-top: 20px;
      }
      
      /* Enhanced Form Controls */
      .form-control {
        border: 2px solid rgba(26, 60, 40, 0.1);
        border-radius: 12px;
        padding: 14px 16px;
        font-size: 16px;
        transition: all 0.2s ease;
        background: white;
      }
      
      .form-control:focus {
        border-color: var(--accent);
        box-shadow: 0 0 0 3px rgba(163, 186, 149, 0.1);
        outline: none;
      }
      
      .form-label {
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 8px;
      }
      
      /* Confirmation Screen - Enhanced Design */
      .confirmation-screen {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .confirmation-item {
        display: flex;
        justify-content: space-between;
        padding: 16px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
      }
      
      .confirmation-item:last-child {
        border-bottom: none;
      }
      
      .confirmation-label {
        color: var(--muted);
        font-size: 14px;
        font-weight: 500;
      }
      
      .confirmation-value {
        font-weight: 600;
        font-size: 14px;
        color: var(--text-dark);
        text-align: right;
      }
      
      /* PIN Entry - Enhanced Design */
      .pin-entry {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        text-align: center;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .pin-description {
        color: var(--muted);
        font-size: 14px;
        margin-bottom: 16px;
        font-weight: 500;
      }
      
      .pin-input {
        margin: 0 auto;
        max-width: 240px;
      }
      
      /* Receipt Screen - Enhanced Design */
      .receipt-screen {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        position: relative;
        overflow: hidden;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .receipt-header {
        text-align: center;
        margin-bottom: 24px;
      }
      
      .receipt-status {
        display: inline-block;
        padding: 8px 16px;
        border-radius: 20px;
        font-size: 14px;
        font-weight: 600;
        margin-bottom: 12px;
      }
      
      .status-completed {
        background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%);
        color: #2E7D32;
        border: 1px solid #4CAF50;
      }
      
      .receipt-title {
        font-size: 22px;
        font-weight: 700;
        color: var(--text-dark);
        margin-bottom: 4px;
      }
      
      .receipt-subtitle {
        color: var(--muted);
        font-size: 14px;
        font-weight: 500;
      }
      
      .receipt-amount {
        font-size: 36px;
        font-weight: 700;
        color: var(--primary);
        margin: 24px 0;
        text-align: center;
        text-shadow: 0 2px 4px rgba(0,0,0,0.1);
      }
      
      .receipt-details {
        margin-top: 24px;
      }
      
      .receipt-item {
        display: flex;
        justify-content: space-between;
        padding: 12px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
      }
      
      .receipt-item:last-child {
        border-bottom: none;
      }
      
      .receipt-label {
        color: var(--muted);
        font-size: 14px;
        font-weight: 500;
      }
      
      .receipt-value {
        font-weight: 600;
        font-size: 14px;
        color: var(--text-dark);
        text-align: right;
      }
      
      /* Action Buttons - Enhanced Design */
      .action-buttons {
        display: flex;
        gap: 16px;
        margin-top: 24px;
      }
      
      .action-button {
        flex: 1;
        text-align: center;
      }
      
      /* FIXED Continue/Action Button Styles */
      .btn-primary {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        border: none;
        border-radius: 16px;
        padding: 18px 28px;
        font-weight: 700;
        font-size: 16px;
        text-transform: capitalize;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
        box-shadow: 0 6px 20px rgba(26, 60, 40, 0.2);
        position: relative;
        overflow: hidden;
        min-height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        text-decoration: none;
      }
      
      .btn-primary::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
        transition: left 0.5s ease;
      }
      
      .btn-primary:hover:not(:disabled) {
        transform: translateY(-3px);
        box-shadow: 0 8px 28px rgba(26, 60, 40, 0.3);
        background: linear-gradient(135deg, #2d5a3d 0%, var(--secondary) 100%);
        color: white;
      }
      
      .btn-primary:hover:not(:disabled)::before {
        left: 100%;
      }
      
      .btn-primary:active:not(:disabled) {
        transform: translateY(-1px);
        box-shadow: 0 4px 16px rgba(26, 60, 40, 0.2);
      }
      
      .btn-primary:disabled {
        background: linear-gradient(135deg, #e0e0e0 0%, #bdbdbd 100%);
        color: #9e9e9e;
        cursor: not-allowed;
        transform: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        opacity: 0.6;
      }
      
      .btn-primary:disabled::before {
        display: none;
      }
      
      /* Secondary Button Enhancement */
      .btn-outline-secondary {
        border: 2px solid var(--accent);
        color: var(--primary);
        background: linear-gradient(135deg, transparent 0%, rgba(163, 186, 149, 0.05) 100%);
        border-radius: 16px;
        padding: 16px 24px;
        font-weight: 600;
        transition: all 0.3s ease;
        min-height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        text-decoration: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.05);
      }
      
      .btn-outline-secondary:hover {
        background: linear-gradient(135deg, var(--accent) 0%, #8FA582 100%);
        color: white;
        border-color: var(--accent);
        transform: translateY(-2px);
        box-shadow: 0 4px 16px rgba(163, 186, 149, 0.2);
      }
      
      .btn-outline-secondary:active {
        transform: translateY(0);
      }
      
      /* Grid button alignment improvements */
      .d-grid .btn {
        display: flex;
        align-items: center;
        justify-content: center;
        text-align: center;
      }
      
      /* Bottom Navigation - Enhanced Design */
      .bottom-navigation {
        position: fixed;
        bottom: 0;
        left: 0;
        width: 100%;
        background: linear-gradient(135deg, white 0%, #f8f9fa 100%);
        box-shadow: 0 -6px 24px rgba(0,0,0,0.1);
        padding: 20px 16px;
        z-index: 100;
        border-top: 1px solid rgba(26, 60, 40, 0.1);
      }
      
      /* Enhanced Input Group for Search */
      .input-group {
        position: relative;
        display: flex;
        flex-wrap: wrap;
        align-items: stretch;
        width: 100%;
      }
      
      .input-group-text {
        display: flex;
        align-items: center;
        padding: 0.375rem 0.75rem;
        font-size: 1rem;
        font-weight: 400;
        line-height: 1.5;
        color: var(--muted);
        text-align: center;
        white-space: nowrap;
        background-color: #f8f9fa;
        border: 2px solid rgba(26, 60, 40, 0.1);
        border-radius: 12px 0 0 12px;
      }
      
      .input-group .form-control {
        position: relative;
        flex: 1 1 auto;
        width: 1%;
        min-width: 0;
        border-radius: 0 12px 12px 0;
        border-left: none;
      }
      
      .input-group .form-control:focus {
        z-index: 3;
        border-color: var(--accent);
        box-shadow: 0 0 0 3px rgba(163, 186, 149, 0.1);
      }
      
      /* Loading states */
      .spinner-border-sm {
        width: 1rem;
        height: 1rem;
        border-width: 0.2em;
      }
      
      /* Text utilities */
      .text-muted {
        color: var(--muted) !important;
      }
      
      .text-center {
        text-align: center !important;
      }
      
      .text-danger {
        color: var(--error) !important;
      }
      
      /* Spacing utilities */
      .mb-3 {
        margin-bottom: 1rem !important;
      }
      
      .mt-2 {
        margin-top: 0.5rem !important;
      }
      
      .ms-2 {
        margin-left: 0.5rem !important;
      }
      
      .me-1 {
        margin-right: 0.25rem !important;
      }
      
      .me-2 {
        margin-right: 0.5rem !important;
      }
      
      .py-3 {
        padding-top: 1rem !important;
        padding-bottom: 1rem !important;
      }
      
      /* Display utilities */
      .d-flex {
        display: flex !important;
      }
      
      .d-grid {
        display: grid !important;
      }
      
      .align-items-center {
        align-items: center !important;
      }
      
      .justify-content-center {
        justify-content: center !important;
      }
      
      .gap-2 {
        gap: 0.5rem !important;
      }
      
      /* Width utilities */
      .w-100 {
        width: 100% !important;
      }
      
      /* Font size utilities */
      .fs-6 {
        font-size: 1rem !important;
      }
      
      /* Responsive enhancements */
      @media (max-width: 576px) {
        .step-indicator {
          padding: 12px;
          margin-bottom: 20px;
        }
        
        .step-circle {
          width: 32px;
          height: 32px;
          font-size: 13px;
        }
        
        .step-label {
          font-size: 11px;
        }
        
        .amount-chips {
          grid-template-columns: repeat(2, 1fr);
        }
        
        .receipt-amount {
          font-size: 32px;
        }
        
        .action-buttons {
          flex-direction: column;
          gap: 12px;
        }
      }
      
      /* Accessibility improvements */
      @media (prefers-reduced-motion: reduce) {
        * {
          transition: none !important;
          animation: none !important;
        }
      }
      
      /* Focus visible for better accessibility */
      .btn:focus-visible,
      .form-control:focus-visible,
      .recipient-card:focus-visible,
      .amount-chip:focus-visible {
        outline: 2px solid var(--secondary);
        outline-offset: 2px;
      }
    `
    document.head.appendChild(styleElement)
  },
  template: `
    <div class="transfer-container">
      <!-- Background Decorations -->
      <div class="background-decoration">
        <div class="bg-leaf bg-leaf-1"></div>
        <div class="bg-leaf bg-leaf-2"></div>
        <div class="bg-leaf bg-leaf-3"></div>
      </div>
      
      <!-- Loading Overlay -->
      <div class="loading-overlay" v-if="isLoading || isProcessing">
        <div class="loading-spinner">
          <div class="spinner-border text-light" role="status">
            <span class="visually-hidden">Loading...</span>
          </div>
          <div class="loading-text">{{ isProcessing ? 'Processing...' : 'Loading...' }}</div>
        </div>
      </div>
      
      <!-- Error Notification Backdrop (only for errors) -->
      <div class="error-notification-backdrop" 
           v-if="showNotification && isErrorNotification" 
           @click="closeErrorNotification">
      </div>
      
      <!-- Header -->
      <div class="transfer-header">
        <div class="leaf-decoration left"></div>
        <div class="leaf-decoration right"></div>
        
        <div class="header-content">
          <div class="back-button" @click="goBack">
            <i class="mdi mdi-arrow-left"></i>
            <span class="back-button-text">Back</span>
          </div>
          
          <h1 class="page-title">Transfer Funds</h1>
        </div>
      </div>
      
      <!-- Balance Summary -->
      <div class="balance-summary">
        <div class="balance-title">Available Balance</div>
        <div class="balance-amount">{{ formattedBalance }}</div>
      </div>
      
      <!-- Content Area -->
      <div class="content-area">
        <!-- Step Indicator -->
        <div class="step-indicator">
          <div class="step" :class="{ active: transferStep >= 1, completed: transferStep > 1 }">
            <div class="step-circle">1</div>
            <div class="step-label">Recipient</div>
          </div>
          
          <div class="step" :class="{ active: transferStep >= 2, completed: transferStep > 2 }">
            <div class="step-circle">2</div>
            <div class="step-label">Amount</div>
          </div>
          
          <div class="step" :class="{ active: transferStep >= 3, completed: transferStep > 3 }">
            <div class="step-circle">3</div>
            <div class="step-label">Confirm</div>
          </div>
          
          <div class="step" :class="{ active: transferStep >= 4 }">
            <div class="step-circle">4</div>
            <div class="step-label">Receipt</div>
          </div>
        </div>
        
        <!-- Step 1: Recipient Selection -->
        <div v-if="transferStep === 1">
          <div class="recipient-selection">
            <h2 class="section-title">Select Recipient</h2>
            
            <!-- FIXED Search Input -->
            <div class="search-container">
              <div class="search-input-wrapper">
                <input type="text" 
                       class="search-input" 
                       placeholder="Search by name or member ID" 
                       v-model="searchQuery" 
                       @input="searchRecipients"
                       @keyup.enter="searchRecipients">
                <button class="search-button" 
                        type="button" 
                        @click="searchRecipients"
                        :disabled="!searchQuery.trim()">
                  <i class="mdi mdi-magnify"></i>
                </button>
              </div>
            </div>
            
            <!-- Selected Recipient -->
            <div v-if="recipient" class="selected-recipient mb-3">
              <div class="d-flex align-items-center">
                <div class="recipient-card selected">
                  <div class="recipient-avatar">
                    {{ recipient.name.charAt(0) }}
                  </div>
                  <div class="recipient-info">
                    <div class="recipient-name">{{ recipient.name }}</div>
                    <div class="recipient-id">{{ recipient.membershipId }}</div>
                  </div>
                </div>
                <button class="btn btn-sm btn-outline-danger ms-2" @click="clearRecipient">
                  <i class="mdi mdi-close"></i>
                </button>
              </div>
            </div>
            
            <!-- Recent Recipients -->
            <div v-if="!recipient" class="recipient-list">
              <h3 class="mb-3 fs-6">{{ searchQuery ? 'Search Results' : 'Recent Recipients' }}</h3>
              
              <div v-if="isSearching" class="text-center py-3">

                <div class="mt-2 text-muted">Searching...</div>
              </div>
              
              <div v-else-if="searchQuery && searchResults.length === 0" class="text-center py-3">
                <div class="text-muted">No recipients found</div>
              </div>
              
              <div v-else-if="!searchQuery && filteredRecipients.length === 0" class="text-center py-3">
                <div class="text-muted">No recent recipients</div>
              </div>
              
              <div v-else>
                <div v-for="(rec, index) in searchQuery ? searchResults : filteredRecipients" :key="rec.id"
                     class="recipient-card" @click="selectRecipient(rec)"
                     :class="{ 'hovered': hoveredRecipient === index }"
                     @mouseenter="hoveredRecipient = index"
                     @mouseleave="hoveredRecipient = null">
                  <div class="recipient-avatar">
                    {{ rec.name.charAt(0) }}
                  </div>
                  <div class="recipient-info">
                    <div class="recipient-name">{{ rec.name }}</div>
                    <div class="recipient-id">{{ rec.membershipId }}</div>
                    <div class="recipient-last-transfer" v-if="rec.lastTransfer">Last transfer: {{ rec.lastTransfer }}</div>
                    <div class="recipient-role" v-if="rec.role">{{ rec.role }}</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          
          <!-- FIXED Continue Button -->
          <div class="d-grid gap-2">
            <button class="btn btn-primary" @click="nextStep" :disabled="!recipient">
              <i class="mdi mdi-arrow-right me-2"></i>
              Continue to Amount
            </button>
          </div>
        </div>
        
        <!-- Step 2: Amount Selection -->
        <div v-if="transferStep === 2">
          <div class="amount-selection">
            <h2 class="section-title">Transfer Amount</h2>
            
            <!-- Predefined Amounts -->
            <div class="amount-chips">
              <div v-for="amount in predefinedAmounts" :key="amount"
                   class="amount-chip" :class="{ 'selected': transferAmount === amount }"
                   @click="setAmount(amount)">
                {{ formatCurrency(amount) }}
              </div>
            </div>
            
            <!-- Custom Amount -->
            <div class="custom-amount">
              <label for="customAmount" class="form-label">Custom Amount</label>
              <div class="input-group mb-3">
                <span class="input-group-text">RM</span>
                <input type="number" class="form-control" id="customAmount"
                       v-model="transferAmount" min="10" max="5000">
              </div>
              
              <div v-if="!isAmountValid" class="text-danger mt-2">
                <small v-if="transferAmount > userData.balance">
                  Amount exceeds your available balance
                </small>
                <small v-else-if="transferAmount < 10">
                  Minimum transfer amount is RM 10
                </small>
                <small v-else-if="transferAmount > 5000">
                  Maximum transfer amount is RM 5,000
                </small>
              </div>
            </div>
            
            <!-- Transfer Note -->
            <!--<div class="transfer-note">
              <label for="transferNote" class="form-label">Note (Optional)</label>
              <textarea class="form-control" id="transferNote" rows="2"
                        v-model="transferNote" placeholder="Add a note for this transfer"></textarea>
            </div>-->
          </div>
          
          <!-- Next Button -->
          <div class="d-grid gap-2">
            <button class="btn btn-primary" @click="nextStep" :disabled="!isAmountValid">
              <i class="mdi mdi-check-circle me-2"></i>
              Continue to Confirm
            </button>
          </div>
        </div>
        
        <!-- Step 3: Confirmation -->
        <div v-if="transferStep === 3">
          <div class="confirmation-screen">
            <h2 class="section-title">Confirm Transfer</h2>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Recipient</div>
              <div class="confirmation-value">{{ recipient.name }}</div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Membership ID</div>
              <div class="confirmation-value">{{ recipient.membershipId }}</div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Amount</div>
              <div class="confirmation-value">{{ formattedTransferAmount }}</div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">New Balance</div>
              <div class="confirmation-value">{{ formattedNewBalance }}</div>
            </div>
            
            <div class="confirmation-item" v-if="transferNote">
              <div class="confirmation-label">Note</div>
              <div class="confirmation-value">{{ transferNote }}</div>
            </div>
          </div>
          
          <!-- PIN Entry -->
          <div class="pin-entry">
            <h2 class="section-title">Enter PIN</h2>
            <p class="pin-description">Please enter your 6-digit PIN to authorize this transfer</p>
            
            <div class="pin-input">
              <input type="password" class="form-control" placeholder="6-digit PIN"
                     v-model="transferPin" maxlength="6" inputmode="numeric" pattern="[0-9]*">
            </div>
          </div>
          
          <!-- Process Button -->
          <div class="d-grid gap-2">
            <button class="btn btn-primary" @click="processTransfer" :disabled="!isPinValid">
              <i class="mdi mdi-send me-2"></i>
              Transfer {{ formattedTransferAmount }}
            </button>
          </div>
        </div>
        
        <!-- Step 4: Receipt -->
        <div v-if="transferStep === 4 && showReceipt">
          <div class="receipt-screen">
            <div class="receipt-header">
              <div class="receipt-status status-completed">
                <i class="mdi mdi-check-circle me-1"></i>
                {{ receipt.status }}
              </div>
              <h2 class="receipt-title">Transfer Successful</h2>
              <p class="receipt-subtitle">{{ receipt.date }}</p>
            </div>
            
            <div class="receipt-amount">
              {{ formatCurrency(receipt.amount) }}
            </div>
            
            <div class="receipt-details">
              <div class="receipt-item">
                <div class="receipt-label">Transaction ID</div>
                <div class="receipt-value">{{ receipt.transactionId }}</div>
              </div>
              
              <div class="receipt-item">
                <div class="receipt-label">Recipient</div>
                <div class="receipt-value">{{ receipt.recipientName }}</div>
              </div>
              
              <div class="receipt-item">
                <div class="receipt-label">Membership ID</div>
                <div class="receipt-value">{{ receipt.recipientId }}</div>
              </div>
              
              <div class="receipt-item">
                <div class="receipt-label">Note</div>
                <div class="receipt-value">{{ receipt.note }}</div>
              </div>
              
              <div class="receipt-item">
                <div class="receipt-label">New Balance</div>
                <div class="receipt-value">{{ formattedBalance }}</div>
              </div>
            </div>
            
            <div class="action-buttons">
              <div class="action-button">
                <button class="btn btn-outline-secondary w-100" @click="downloadReceiptPDF">
                  <i class="mdi mdi-file-pdf-box me-1"></i>
                  Download PDF
                </button>
              </div>
              
              <div class="action-button">
                <button class="btn btn-primary w-100" @click="completeTransfer">
                  Done
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Notification Toast -->
      <transition name="fade">
        <div class="notification-toast" 
             :class="{ 'error-notification': isErrorNotification }" 
             v-if="showNotification">
          
          <!-- Close button for error notifications -->
          <button v-if="isErrorNotification" 
                  class="error-close-button" 
                  @click="closeErrorNotification">
            <i class="mdi mdi-close"></i>
          </button>
          
          {{ notificationMessage }}
        </div>
      </transition>
    </div>
  `
}