// screen_011_gym.js
import getTropicalStyles from "../css/tropicalBookingStyles.js"; 

export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',       // Dark green
        secondary: '#DDA333',     // Golden
        accent: '#A3BA95',        // Light sage green
        lightBg: '#F8F5F0',       // Off-white
        creamBg: '#EADDCB',       // Cream
        darkText: '#1E1E1E',      // Almost black
        lightText: '#FFFFFF',     // White
        error: '#B71C1C',         // Red
        success: '#2E7D32',       // Green
        warning: '#FF8F00',       // Orange
        muted: '#78909C',         // Grayish blue
        brown: '#6D4C41',         // Brown
      },
      
      // Navigation state
      currentView: 'home',
      previousView: null,
      
      // User data
      user: {
        id: "M1001",
        name: "Alex Johnson",
        memberLevel: "Premium",
        memberSince: "2023-06-15",
        profileImage: "https://randomuser.me/api/portraits/men/32.jpg",
        membershipValid: "2025-06-15",
        daysRemaining: 369,
        stats: {
          totalVisits: 87,
          thisMonth: 12,
          streakDays: 4,
          minutesThisMonth: 840,
          recordedWorkouts: 42
        }
      },
      
      // Check-in state
      isCheckedIn: false,
      checkInTime: null,
      scanResult: null,
      qrValue: "GYM-ENTRY-AB12345XYZ",
      showQrScanner: false,
      
      // Workout tracking
      workoutActive: false,
      workoutStartTime: null,
      workoutDuration: 0,
      workoutTimer: null,
      selectedWorkout: null,
      showWorkoutTimer: false,
      
      // Class booking
      selectedDate: today,
      minDateForPicker: today,
      selectedClass: null,
      bookedClasses: [
        {
          id: "c1001",
          name: "HIIT Extreme",
          trainer: "Mike Chen",
          time: "07:00 - 08:00",
          date: new Date(today.getTime() + 86400000).toISOString().split('T')[0], // Tomorrow
          room: "Studio 2",
          spots: "5 spots left",
          intensityLevel: 5,
          color: "#F44336",
          icon: "mdi-fire"
        },
        {
          id: "c1002",
          name: "Yoga Flow",
          trainer: "Sarah Williams",
          time: "18:30 - 19:30",
          date: new Date(today.getTime() + 86400000 * 2).toISOString().split('T')[0], // Day after tomorrow
          room: "Studio 1",
          spots: "8 spots left",
          intensityLevel: 2,
          color: "#4CAF50",
          icon: "mdi-yoga"
        }
      ],
      
      // Classes schedule
      classSchedule: {
        monday: [
          { id: "m1", name: "Morning HIIT", time: "06:30 - 07:30", trainer: "Mike Chen", color: this.theme?.error || "#B71C1C", icon: "mdi-dumbbell" },
          { id: "m2", name: "Power Yoga", time: "12:00 - 13:00", trainer: "Sarah Williams", color: this.theme?.success || "#2E7D32", icon: "mdi-yoga" },
          { id: "m3", name: "Boxing", time: "18:00 - 19:00", trainer: "James Wilson", color: this.theme?.warning || "#FF8F00", icon: "mdi-boxing-glove" }
        ],
        tuesday: [
          { id: "t1", name: "Spin Class", time: "07:00 - 08:00", trainer: "Ryan Smith", color: this.theme?.warning || "#FF8F00", icon: "mdi-bike" },
          { id: "t2", name: "Pilates", time: "12:30 - 13:30", trainer: "Emma Davis", color: this.theme?.accent || "#A3BA95", icon: "mdi-yoga" },
          { id: "t3", name: "Strength Circuit", time: "18:30 - 19:30", trainer: "Mike Chen", color: this.theme?.primary || "#1A3C28", icon: "mdi-weight-lifter" }
        ],
        wednesday: [
          { id: "w1", name: "Morning Yoga", time: "06:30 - 07:30", trainer: "Sarah Williams", color: this.theme?.success || "#2E7D32", icon: "mdi-yoga" },
          { id: "w2", name: "HIIT Express", time: "12:15 - 13:00", trainer: "Mike Chen", color: this.theme?.error || "#B71C1C", icon: "mdi-dumbbell" },
          { id: "w3", name: "Zumba", time: "18:00 - 19:00", trainer: "Sofia Rodriguez", color: this.theme?.secondary || "#DDA333", icon: "mdi-dance-ballroom" }
        ],
        thursday: [
          { id: "th1", name: "Spin Class", time: "07:00 - 08:00", trainer: "Ryan Smith", color: this.theme?.warning || "#FF8F00", icon: "mdi-bike" },
          { id: "th2", name: "Core Focus", time: "12:30 - 13:15", trainer: "Lisa Chen", color: this.theme?.accent || "#A3BA95", icon: "mdi-human" },
          { id: "th3", name: "Boxing", time: "18:30 - 19:30", trainer: "James Wilson", color: this.theme?.warning || "#FF8F00", icon: "mdi-boxing-glove" }
        ],
        friday: [
          { id: "f1", name: "Morning HIIT", time: "06:30 - 07:30", trainer: "Mike Chen", color: this.theme?.error || "#B71C1C", icon: "mdi-dumbbell" },
          { id: "f2", name: "Pilates", time: "12:00 - 13:00", trainer: "Emma Davis", color: this.theme?.accent || "#A3BA95", icon: "mdi-yoga" },
          { id: "f3", name: "Happy Hour Yoga", time: "17:30 - 18:30", trainer: "Sarah Williams", color: this.theme?.success || "#2E7D32", icon: "mdi-yoga" }
        ],
        saturday: [
          { id: "s1", name: "Weekend Warrior", time: "09:00 - 10:00", trainer: "Mike Chen", color: this.theme?.error || "#B71C1C", icon: "mdi-dumbbell" },
          { id: "s2", name: "Yoga Flow", time: "11:00 - 12:00", trainer: "Sarah Williams", color: this.theme?.success || "#2E7D32", icon: "mdi-yoga" },
          { id: "s3", name: "Strength & Sculpt", time: "14:00 - 15:00", trainer: "James Wilson", color: this.theme?.primary || "#1A3C28", icon: "mdi-weight-lifter" }
        ],
        sunday: [
          { id: "su1", name: "Gentle Yoga", time: "09:30 - 10:30", trainer: "Sarah Williams", color: this.theme?.success || "#2E7D32", icon: "mdi-yoga" },
          { id: "su2", name: "Spin Express", time: "11:00 - 11:45", trainer: "Ryan Smith", color: this.theme?.warning || "#FF8F00", icon: "mdi-bike" },
          { id: "su3", name: "Pilates", time: "14:00 - 15:00", trainer: "Emma Davis", color: this.theme?.accent || "#A3BA95", icon: "mdi-yoga" }
        ]
      },
      
      // Personal trainers
      trainers: [
        {
          id: "t1",
          name: "Mike Chen",
          specialty: "HIIT, Strength Training",
          image: "https://randomuser.me/api/portraits/men/22.jpg",
          rating: 4.9,
          experience: "8 years",
          bio: "Olympic weightlifting champion with expertise in functional training and athletic performance.",
          availability: "Mon, Wed, Fri"
        },
        {
          id: "t2",
          name: "Sarah Williams",
          specialty: "Yoga, Pilates",
          image: "https://randomuser.me/api/portraits/women/65.jpg",
          rating: 4.8,
          experience: "10 years",
          bio: "Certified yoga instructor specialized in mind-body connection and stress reduction techniques.",
          availability: "Tue, Thu, Sat"
        },
        {
          id: "t3",
          name: "James Wilson",
          specialty: "Boxing, Functional Training",
          image: "https://randomuser.me/api/portraits/men/32.jpg",
          rating: 4.7,
          experience: "7 years",
          bio: "Former professional boxer with a passion for functional training and athletic conditioning.",
          availability: "Mon, Thu, Sat"
        },
        {
          id: "t4",
          name: "Emma Davis",
          specialty: "Pilates, Rehabilitation",
          image: "https://randomuser.me/api/portraits/women/44.jpg",
          rating: 4.9,
          experience: "12 years",
          bio: "Physical therapist and Pilates expert specializing in rehabilitation and posture correction.",
          availability: "Tue, Fri, Sun"
        }
      ],
      
      // Recent workouts
      recentWorkouts: [
        {
          date: "2025-06-09",
          duration: "45 min",
          type: "Strength",
          details: "Upper Body Focus",
          calories: 320,
          color: this.theme?.primary || "#1A3C28"
        },
        {
          date: "2025-06-07",
          duration: "30 min",
          type: "Cardio",
          details: "Treadmill Intervals",
          calories: 280,
          color: this.theme?.error || "#B71C1C"
        },
        {
          date: "2025-06-05",
          duration: "60 min",
          type: "Yoga",
          details: "Vinyasa Flow",
          calories: 220,
          color: this.theme?.success || "#2E7D32"
        }
      ],
      
      // Workout templates
      workoutTemplates: [
        {
          id: "w1",
          name: "Full Body Blast",
          duration: "45-60 min",
          difficulty: "Intermediate",
          category: "Strength",
          color: this.theme?.primary || "#1A3C28",
          exercises: [
            { name: "Squats", sets: 3, reps: "12-15", rest: "60 sec" },
            { name: "Push-ups", sets: 3, reps: "10-12", rest: "60 sec" },
            { name: "Bent-over Rows", sets: 3, reps: "12", rest: "60 sec" },
            { name: "Shoulder Press", sets: 3, reps: "10", rest: "60 sec" },
            { name: "Lunges", sets: 3, reps: "12 each", rest: "60 sec" },
            { name: "Plank", sets: 3, reps: "30-45 sec", rest: "30 sec" }
          ]
        },
        {
          id: "w2",
          name: "HIIT Cardio",
          duration: "25-30 min",
          difficulty: "Advanced",
          category: "Cardio",
          color: this.theme?.error || "#B71C1C",
          exercises: [
            { name: "High Knees", sets: 4, reps: "30 sec", rest: "10 sec" },
            { name: "Mountain Climbers", sets: 4, reps: "30 sec", rest: "10 sec" },
            { name: "Jumping Jacks", sets: 4, reps: "30 sec", rest: "10 sec" },
            { name: "Burpees", sets: 4, reps: "30 sec", rest: "10 sec" },
            { name: "Jump Squats", sets: 4, reps: "30 sec", rest: "10 sec" },
            { name: "Rest", sets: 1, reps: "60 sec", rest: "0 sec" },
            { name: "Repeat circuit", sets: "2-3", reps: "", rest: "60 sec between circuits" }
          ]
        },
        {
          id: "w3",
          name: "Core Crusher",
          duration: "20 min",
          difficulty: "Beginner-Intermediate",
          category: "Core",
          color: this.theme?.accent || "#A3BA95",
          exercises: [
            { name: "Crunches", sets: 3, reps: "15-20", rest: "30 sec" },
            { name: "Russian Twists", sets: 3, reps: "20 total", rest: "30 sec" },
            { name: "Plank", sets: 3, reps: "30-45 sec", rest: "30 sec" },
            { name: "Bicycle Crunches", sets: 3, reps: "20 total", rest: "30 sec" },
            { name: "Leg Raises", sets: 3, reps: "12-15", rest: "30 sec" },
            { name: "Side Planks", sets: 3, reps: "20-30 sec each", rest: "30 sec" }
          ]
        }
      ],
      
      // Membership options
      membershipOptions: [
        {
          id: "m1",
          name: "Basic",
          price: 49.99,
          period: "monthly",
          features: [
            "Access to gym equipment",
            "Locker room access",
            "2 guest passes per month",
            "Basic fitness assessment"
          ],
          color: this.theme?.muted || "#78909C",
          recommended: false
        },
        {
          id: "m2",
          name: "Premium",
          price: 79.99,
          period: "monthly",
          features: [
            "All Basic features",
            "Unlimited group classes",
            "1 session with personal trainer monthly",
            "Access to pool and sauna",
            "Fitness app premium features"
          ],
          color: this.theme?.secondary || "#DDA333",
          recommended: true
        },
        {
          id: "m3",
          name: "Elite",
          price: 129.99,
          period: "monthly",
          features: [
            "All Premium features",
            "4 sessions with personal trainer monthly",
            "Nutrition consultation",
            "Priority class booking",
            "Exclusive members-only events",
            "Discounts on merchandise"
          ],
          color: this.theme?.primary || "#1A3C28",
          recommended: false
        }
      ],
      
      // UI state
      loading: false,
      showSnackbar: false,
      snackbarText: '',
      snackbarColor: '',
    };
  },
  
  computed: {
    formattedMembershipDate() {
      if (!this.user.membershipValid) return "";
      
      const date = new Date(this.user.membershipValid);
      return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
      });
    },
    
    formattedSelectedDate() {
      if (!this.selectedDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return this.selectedDate.toLocaleDateString('en-US', options);
    },
    
    formattedTime() {
      if (!this.checkInTime) return null;
      
      return this.checkInTime.toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
      });
    },
    
    formattedWorkoutDuration() {
      const minutes = Math.floor(this.workoutDuration / 60);
      const seconds = this.workoutDuration % 60;
      return `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    },
    
    membershipStatus() {
      if (!this.user.membershipValid) return "Unknown";
      
      const validUntil = new Date(this.user.membershipValid);
      const today = new Date();
      
      if (validUntil < today) {
        return "Expired";
      } else {
        // Display days remaining
        const diffTime = validUntil - today;
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        
        if (diffDays <= 7) {
          return "Expiring Soon";
        } else {
          return "Active";
        }
      }
    },
    
    membershipStatusColor() {
      switch (this.membershipStatus) {
        case "Active": return this.theme.success;
        case "Expiring Soon": return this.theme.warning;
        case "Expired": return this.theme.error;
        default: return this.theme.muted;
      }
    },
    
    dayOfWeek() {
      const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
      return days[this.selectedDate.getDay()];
    },
    
    todayClasses() {
      return this.classSchedule[this.dayOfWeek] || [];
    },
    
    canCheckIn() {
      return !this.isCheckedIn;
    },
    
    canCheckOut() {
      return this.isCheckedIn;
    },
    
    isToday() {
      if (!this.selectedDate) return false;
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.selectedDate.getTime() === todayNormalized.getTime();
    }
  },
  
  methods: {
    // Navigation
    navigateTo(view) {
      this.previousView = this.currentView;
      this.currentView = view;
    },
    
    goBack() {
      if (this.previousView) {
        this.currentView = this.previousView;
        this.previousView = 'home';
      } else {
        this.currentView = 'home';
      }
    },
    
    // Check-in/out functions
    checkIn() {
      this.loading = true;
      
      setTimeout(() => {
        this.isCheckedIn = true;
        this.checkInTime = new Date();
        this.showSnackbarMessage("Successfully checked in!", "success");
        this.loading = false;
      }, 1500);
    },
    
    checkOut() {
      this.loading = true;
      
      setTimeout(() => {
        this.isCheckedIn = false;
        const checkInTime = this.checkInTime;
        const checkOutTime = new Date();
        const durationMs = checkOutTime - checkInTime;
        const durationMinutes = Math.round(durationMs / 60000);
        
        this.showSnackbarMessage(`Checked out after ${durationMinutes} minutes`, "info");
        this.checkInTime = null;
        this.loading = false;
      }, 1500);
    },
    
    toggleQrScanner() {
      this.showQrScanner = !this.showQrScanner;
    },
    
    onScanSuccess(decodedText) {
      this.scanResult = decodedText;
      this.showQrScanner = false;
      // In reality, this would validate against a backend service
      if (decodedText.includes("GYM-ENTRY")) {
        this.checkIn();
      } else {
        this.showSnackbarMessage("Invalid QR code", "error");
      }
    },
    
    // Simulated QR scan (for demo)
    simulateScan() {
      this.loading = true;
      setTimeout(() => {
        this.onScanSuccess(this.qrValue);
        this.loading = false;
      }, 1500);
    },
    
    // Workout functions
    startWorkout(workout) {
      this.selectedWorkout = workout;
      this.workoutActive = true;
      this.workoutStartTime = new Date();
      this.workoutDuration = 0;
      this.workoutTimer = setInterval(() => {
        this.workoutDuration++;
      }, 1000);
      this.showWorkoutTimer = true;
      this.showSnackbarMessage(`${workout.name} workout started!`, "success");
    },
    
    completeWorkout() {
      clearInterval(this.workoutTimer);
      const duration = this.formattedWorkoutDuration;
      
      // Add to recent workouts
      const today = new Date().toISOString().split('T')[0];
      this.recentWorkouts.unshift({
        date: today,
        duration: `${Math.floor(this.workoutDuration / 60)} min`,
        type: this.selectedWorkout.category,
        details: this.selectedWorkout.name,
        calories: Math.floor(this.workoutDuration / 60 * 7.5),
        color: this.selectedWorkout.color
      });
      
      this.workoutActive = false;
      this.showWorkoutTimer = false;
      this.selectedWorkout = null;
      this.showSnackbarMessage(`Workout completed! Duration: ${duration}`, "success");
    },
    
    cancelWorkout() {
      clearInterval(this.workoutTimer);
      this.workoutActive = false;
      this.showWorkoutTimer = false;
      this.selectedWorkout = null;
      this.showSnackbarMessage("Workout cancelled", "info");
    },
    
    // Class booking
    bookClass(classItem) {
      this.selectedClass = classItem;
      this.loading = true;
      
      setTimeout(() => {
        // Add to booked classes
        const bookingDate = this.selectedDate.toISOString().split('T')[0];
        this.bookedClasses.push({
          ...classItem,
          date: bookingDate,
          id: "c" + Math.floor(1000 + Math.random() * 9000),
          room: classItem.room || "Main Studio"
        });
        
        this.showSnackbarMessage(`Successfully booked ${classItem.name}!`, "success");
        this.selectedClass = null;
        this.loading = false;
      }, 1500);
    },
    
    cancelClassBooking(classId) {
      this.loading = true;
      
      setTimeout(() => {
        const index = this.bookedClasses.findIndex(c => c.id === classId);
        if (index !== -1) {
          const className = this.bookedClasses[index].name;
          this.bookedClasses.splice(index, 1);
          this.showSnackbarMessage(`${className} booking cancelled`, "success");
        }
        this.loading = false;
      }, 1000);
    },
    
    updateDate(newDateVal) {
      if (this.selectedDate) {
        this.selectedDate.setHours(0,0,0,0);
      }
    },
    
    // PT booking
    bookTrainer(trainer) {
      this.loading = true;
      
      setTimeout(() => {
        this.showSnackbarMessage(`Consultation with ${trainer.name} requested`, "success");
        this.loading = false;
      }, 1000);
    },
    
    // Membership functions
    selectMembership(membership) {
      this.loading = true;
      
      setTimeout(() => {
        this.showSnackbarMessage(`${membership.name} plan selected for upgrade`, "success");
        this.loading = false;
      }, 1000);
    },
    
    renewMembership() {
      this.loading = true;
      
      setTimeout(() => {
        // Extend membership by 1 year
        const currentExpiry = new Date(this.user.membershipValid);
        const newExpiry = new Date(currentExpiry.setFullYear(currentExpiry.getFullYear() + 1));
        this.user.membershipValid = newExpiry.toISOString().split('T')[0];
        this.user.daysRemaining = 365 + this.user.daysRemaining;
        
        this.showSnackbarMessage("Membership renewed successfully!", "success");
        this.loading = false;
      }, 2000);
    },
    
    // Utility functions
    showSnackbarMessage(text, color) {
      this.snackbarText = text;
      this.snackbarColor = color === 'info' ? this.theme.secondary : this.theme[color];
      this.showSnackbar = true;
    },
    
    formatDateShort(dateString) {
      const date = new Date(dateString);
      return date.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric'
      });
    },
    
    getDayName(dateString) {
      const date = new Date(dateString);
      return date.toLocaleDateString('en-US', { weekday: 'short' });
    },
    
    getRelativeDay(dateString) {
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      
      const date = new Date(dateString);
      date.setHours(0, 0, 0, 0);
      
      const diffTime = date - today;
      const diffDays = Math.round(diffTime / (1000 * 60 * 60 * 24));
      
      if (diffDays === 0) return "Today";
      if (diffDays === 1) return "Tomorrow";
      if (diffDays > 1 && diffDays < 7) return date.toLocaleDateString('en-US', { weekday: 'long' });
      return this.formatDateShort(dateString);
    },
    
    getRatingStars(rating) {
      const fullStars = Math.floor(rating);
      const halfStar = rating % 1 >= 0.5;
      const emptyStars = 5 - fullStars - (halfStar ? 1 : 0);
      
      return {
        full: fullStars,
        half: halfStar ? 1 : 0,
        empty: emptyStars
      };
    },
    
    // For exercise breakdown
    formatExercise(exercise) {
      return `${exercise.sets} × ${exercise.reps}`;
    }
  },
  
  mounted() {
    const styleElement = document.createElement('style');
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
  },
  
  beforeUnmount() {
    if (this.workoutTimer) {
      clearInterval(this.workoutTimer);
    }
  },
  
  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent scrim="#1A3C28" :opacity="0.8">
        <div class="text-center">
          <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
          <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Loading...</div>
        </div>
      </v-overlay>
      
      <!-- Back button if not on home screen -->
      <div v-if="currentView !== 'home'" class="back-navigation pa-4">
        <v-btn 
          icon="mdi-arrow-left" 
          @click="goBack" 
          :color="theme.primary" 
          variant="text" 
          class="back-btn elevation-0" 
          size="large"
        ></v-btn>
      </div>

      <!-- HOME VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'home'">
          <!-- Welcome Header -->
          <div class="text-center pt-6 pb-3 px-4">
            <h1 :style="{color: theme.primary, fontFamily: 'Playfair Display, serif', fontSize: '2.5rem', fontWeight: 700, marginBottom: 0, letterSpacing: '-0.5px'}">Tapang Club Gym</h1>
            <p class="text-subtitle-1" :style="{color: theme.secondary}">Welcome back, {{user.name}}</p>
          </div>
          
          <!-- User & Check-in Status Card -->
          <v-card class="mx-4 mb-5 rounded-lg overflow-hidden elevation-3">
            <div class="position-relative">
              <div class="user-card-header pa-4" :style="{ backgroundColor: theme.primary, color: theme.lightText }">
                <div class="d-flex align-center">
                  <v-avatar size="64" class="mr-4 elevation-2">
                    <v-img :src="user.profileImage" alt="User Profile"></v-img>
                  </v-avatar>
                  <div>
                    <div class="text-subtitle-1 font-weight-medium">
                      <v-chip 
                        size="small" 
                        class="mr-2"
                        :color="theme.secondary"
                        label
                      >{{user.memberLevel}}</v-chip>
                      <span class="text-caption">Member since {{new Date(user.memberSince).getFullYear()}}</span>
                    </div>
                    <div class="text-body-2 mt-1 d-flex align-center">
                      <v-icon size="small" :color="membershipStatusColor" class="mr-1">mdi-circle</v-icon>
                      <span>{{membershipStatus}}, valid until {{formattedMembershipDate}}</span>
                    </div>
                  </div>
                </div>
              </div>
              
              <div 
                class="check-in-status pa-3" 
                :style="{ 
                  backgroundColor: isCheckedIn ? theme.success : theme.accent, 
                  color: isCheckedIn ? theme.lightText : theme.primary 
                }"
              >
                <div class="d-flex align-center justify-space-between">
                  <div class="d-flex align-center">
                    <v-icon :color="isCheckedIn ? theme.lightText : theme.primary" class="mr-2">
                      {{ isCheckedIn ? 'mdi-check-circle' : 'mdi-login-variant' }}
                    </v-icon>
                    <span class="font-weight-medium">{{ isCheckedIn ? 'Currently Checked In' : 'Not Checked In' }}</span>
                  </div>
                  <v-btn 
                    v-if="isCheckedIn" 
                    size="small" 
                    variant="tonal" 
                    :color="theme.error" 
                    @click="checkOut"
                  >Check Out</v-btn>
                  <v-btn 
                    v-else 
                    size="small" 
                    variant="tonal" 
                    :color="theme.primary" 
                    @click="navigateTo('checkin')"
                  >Check In</v-btn>
                </div>
                <div v-if="isCheckedIn" class="text-caption mt-1">
                  Since {{formattedTime}}
                </div>
              </div>
            </div>
          </v-card>
          
          <!-- Quick Actions -->
          <div class="px-4 mb-5">
            <h2 class="text-h6 mb-3" :style="{color: theme.primary}">Quick Actions</h2>
            <v-row>
              <v-col cols="4" @click="navigateTo('checkin')">
                <v-card class="action-card text-center pa-3 rounded-lg" flat>
                  <v-avatar :color="theme.primary" size="52" class="mb-2">
                    <v-icon color="white" size="24">mdi-qrcode-scan</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">Check In</div>
                </v-card>
              </v-col>
              
              <v-col cols="4" @click="navigateTo('classes')">
                <v-card class="action-card text-center pa-3 rounded-lg" flat>
                  <v-avatar :color="theme.secondary" size="52" class="mb-2">
                    <v-icon color="white" size="24">mdi-calendar</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">Classes</div>
                </v-card>
              </v-col>
              
              <v-col cols="4" @click="navigateTo('workouts')">
                <v-card class="action-card text-center pa-3 rounded-lg" flat>
                  <v-avatar :color="theme.accent" size="52" class="mb-2">
                    <v-icon :color="theme.primary" size="24">mdi-dumbbell</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">Workouts</div>
                </v-card>
              </v-col>
            </v-row>
            
            <v-row class="mt-3">
              <v-col cols="4" @click="navigateTo('trainers')">
                <v-card class="action-card text-center pa-3 rounded-lg" flat>
                  <v-avatar :color="theme.brown" size="52" class="mb-2">
                    <v-icon color="white" size="24">mdi-account-tie</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">Trainers</div>
                </v-card>
              </v-col>
              
              <v-col cols="4" @click="workoutActive ? navigateTo('activeWorkout') : navigateTo('workouts')">
                <v-card 
                  class="action-card text-center pa-3 rounded-lg" 
                  flat
                  :style="workoutActive ? {backgroundColor: theme.error + '22'} : {}"
                >
                  <v-avatar :color="workoutActive ? theme.error : theme.muted" size="52" class="mb-2">
                    <v-icon color="white" size="24">{{workoutActive ? 'mdi-timer-outline' : 'mdi-history'}}</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">{{workoutActive ? 'Active Workout' : 'History'}}</div>
                </v-card>
              </v-col>
              
              <v-col cols="4" @click="navigateTo('membership')">
                <v-card class="action-card text-center pa-3 rounded-lg" flat>
                  <v-avatar :color="theme.secondary" size="52" class="mb-2">
                    <v-icon color="white" size="24">mdi-card-account-details</v-icon>
                  </v-avatar>
                  <div class="text-caption font-weight-medium">Membership</div>
                </v-card>
              </v-col>
            </v-row>
          </div>
          
          <!-- Today's Classes -->
          <div class="px-4 mb-5">
            <div class="d-flex justify-space-between align-center mb-3">
              <h2 class="text-h6" :style="{color: theme.primary}">Today's Classes</h2>
              <v-btn 
                variant="text" 
                :color="theme.secondary" 
                density="comfortable"
                @click="navigateTo('classes')"
              >
                View All
              </v-btn>
            </div>
            
            <div v-if="todayClasses.length === 0" class="text-center pa-4 rounded-lg" :style="{backgroundColor: theme.lightBg}">
              <v-icon size="40" :color="theme.muted" class="mb-2">mdi-calendar-blank</v-icon>
              <div class="text-subtitle-2">No classes today</div>
              <div class="text-caption" :style="{color: theme.muted}">Check back tomorrow!</div>
            </div>
            
            <v-card 
              v-else
              v-for="(classItem, index) in todayClasses.slice(0, 2)"
              :key="index"
              class="mb-3 elevation-1 rounded-lg"
              :style="{borderLeft: '4px solid ' + classItem.color}"
            >
              <v-card-text class="pa-3">
                <div class="d-flex justify-space-between">
                  <div>
                    <div class="text-subtitle-1 font-weight-medium mb-1">{{ classItem.name }}</div>
                    <div class="text-caption d-flex align-center flex-wrap">
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-clock-outline</v-icon>
                        {{ classItem.time }}
                      </span>
                      <span class="d-flex align-center">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-account</v-icon>
                        {{ classItem.trainer }}
                      </span>
                    </div>
                  </div>
                  <v-btn 
                    :color="theme.secondary" 
                    size="small"
                    class="align-self-center"
                    @click="bookClass(classItem)"
                  >Book</v-btn>
                </div>
              </v-card-text>
            </v-card>
            
            <div v-if="todayClasses.length > 2" class="text-center mt-3">
              <v-btn 
                variant="text" 
                :color="theme.primary" 
                size="small"
                @click="navigateTo('classes')"
              >
                View {{todayClasses.length - 2}} more classes
              </v-btn>
            </div>
          </div>
          
          <!-- Stats Summary -->
          <v-card class="mx-4 mb-5 content-card rounded-lg" elevation="1">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-chart-box</v-icon>
              Your Activity
            </v-card-title>
            <v-card-text class="pa-0">
              <v-row no-gutters>
                <v-col cols="6" class="pa-4 text-center" :style="{borderRight: '1px solid ' + theme.lightBg, borderBottom: '1px solid ' + theme.lightBg}">
                  <div class="text-h5 font-weight-bold" :style="{color: theme.secondary}">{{user.stats.thisMonth}}</div>
                  <div class="text-caption" :style="{color: theme.muted}">Visits this month</div>
                </v-col>
                <v-col cols="6" class="pa-4 text-center" :style="{borderBottom: '1px solid ' + theme.lightBg}">
                  <div class="text-h5 font-weight-bold" :style="{color: theme.secondary}">{{user.stats.streakDays}}</div>
                  <div class="text-caption" :style="{color: theme.muted}">Day streak</div>
                </v-col>
                <v-col cols="6" class="pa-4 text-center" :style="{borderRight: '1px solid ' + theme.lightBg}">
                  <div class="text-h5 font-weight-bold" :style="{color: theme.secondary}">{{Math.round(user.stats.minutesThisMonth/60)}}</div>
                  <div class="text-caption" :style="{color: theme.muted}">Hours this month</div>
                </v-col>
                <v-col cols="6" class="pa-4 text-center">
                  <div class="text-h5 font-weight-bold" :style="{color: theme.secondary}">{{user.stats.recordedWorkouts}}</div>
                  <div class="text-caption" :style="{color: theme.muted}">Workouts logged</div>
                </v-col>
              </v-row>
            </v-card-text>
          </v-card>
          
          <!-- My Bookings -->
          <div class="px-4 mb-5">
            <div class="d-flex justify-space-between align-center mb-3">
              <h2 class="text-h6" :style="{color: theme.primary}">My Bookings</h2>
            </div>
            
            <div v-if="bookedClasses.length === 0" class="text-center pa-4 rounded-lg" :style="{backgroundColor: theme.lightBg}">
              <v-icon size="40" :color="theme.muted" class="mb-2">mdi-calendar-blank</v-icon>
              <div class="text-subtitle-2">No upcoming bookings</div>
              <div class="text-caption" :style="{color: theme.muted}">Book a class to get started</div>
            </div>
            
            <v-card 
              v-else
              v-for="(booking, index) in bookedClasses.slice(0, 3)"
              :key="booking.id"
              class="mb-3 elevation-1 rounded-lg"
              variant="outlined"
            >
              <v-card-text class="pa-3">
                <div class="d-flex align-center">
                  <v-avatar :color="booking.color || theme.primary" class="mr-3">
                    <v-icon color="white">{{ booking.icon || 'mdi-calendar' }}</v-icon>
                  </v-avatar>
                  <div>
                    <div class="text-subtitle-1 font-weight-medium mb-1">{{ booking.name }}</div>
                    <div class="text-caption d-flex flex-wrap">
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-calendar</v-icon>
                        {{ getRelativeDay(booking.date) }}
                      </span>
                      <span class="d-flex align-center">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-clock-outline</v-icon>
                        {{ booking.time }}
                      </span>
                    </div>
                  </div>
                  <v-spacer></v-spacer>
                  <v-btn
                    icon="mdi-close"
                    size="x-small"
                    variant="text"
                    :color="theme.error"
                    @click="cancelClassBooking(booking.id)"
                  ></v-btn>
                </div>
              </v-card-text>
            </v-card>
            
            <div v-if="bookedClasses.length > 3" class="text-center mt-3">
              <v-btn 
                variant="text" 
                :color="theme.primary" 
                size="small"
                @click="navigateTo('classes')"
              >
                View {{bookedClasses.length - 3}} more bookings
              </v-btn>
            </div>
          </div>
        </div>
      </v-fade-transition>
      
      <!-- CHECK-IN VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'checkin'" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Check In/Out</h1>
          
          <v-card class="rounded-lg mb-5 overflow-hidden content-card" elevation="3">
            <v-card-text class="text-center pa-5">
              <div v-if="isCheckedIn" class="checked-in-status">
                <v-avatar :color="theme.success" class="mb-3" size="72">
                  <v-icon size="40" color="white">mdi-check</v-icon>
                </v-avatar>
                <h2 class="text-h6 mb-1">Currently Checked In</h2>
                <p class="text-body-2 mb-3">Since {{ formattedTime }}</p>
                
                <v-btn
                  :color="theme.primary"
                  variant="flat"
                  size="large"
                  block
                  @click="checkOut"
                  class="mb-2"
                >
                  <v-icon start>mdi-exit-run</v-icon>
                  Check Out Now
                </v-btn>
              </div>
              
              <div v-else>
                <div v-if="showQrScanner" class="qr-scanner-container mb-4">
                  <div class="qr-scanner-placeholder d-flex flex-column align-center justify-center py-5">
                    <v-progress-circular
                      indeterminate
                      :color="theme.secondary"
                      size="56"
                      class="mb-3"
                    ></v-progress-circular>
                    <div class="text-subtitle-1 mb-1">Scanning for QR code...</div>
                    <div class="text-caption" :style="{color: theme.muted}">Point your camera at the gym entry QR code</div>
                    
                    <!-- For demo, we'll add a simulate button -->
                    <v-btn
                      :color="theme.secondary"
                      variant="outlined"
                      class="mt-4"
                      @click="simulateScan"
                    >
                      Simulate Scan
                    </v-btn>
                    
                    <v-btn
                      :color="theme.muted"
                      variant="text"
                      class="mt-2"
                      @click="toggleQrScanner"
                    >
                      Cancel
                    </v-btn>
                  </div>
                </div>
                
                <div v-else>
                  <v-avatar :color="theme.primary" class="mb-3" size="72">
                    <v-icon size="40" color="white">mdi-qrcode-scan</v-icon>
                  </v-avatar>
                  <h2 class="text-h6 mb-3">Ready to check in?</h2>
                  <p class="text-body-2 mb-5">Scan the QR code at the gym entrance or use quick check-in</p>
                  
                  <v-btn
                    :color="theme.primary"
                    variant="flat"
                    size="large"
                    block
                    @click="toggleQrScanner"
                    class="mb-3"
                  >
                    <v-icon start>mdi-qrcode-scan</v-icon>
                    Scan QR Code
                  </v-btn>
                  
                  <v-btn
                    :color="theme.secondary"
                    variant="outlined"
                    block
                    @click="checkIn"
                  >
                    <v-icon start>mdi-login</v-icon>
                    Quick Check In
                  </v-btn>
                </div>
              </div>
            </v-card-text>
          </v-card>
          
          <!-- Recent Check-ins -->
          <h2 class="text-subtitle-1 font-weight-bold mb-3" :style="{color: theme.primary}">Recent Activity</h2>
          
          <v-timeline class="mb-0 checkin-timeline" density="compact" align="start">
            <v-timeline-item
              v-for="i in 5"
              :key="i"
              :dot-color="i === 1 && isCheckedIn ? theme.success : theme.secondary"
              :icon="i === 1 && isCheckedIn ? 'mdi-login' : 'mdi-clock-time-four-outline'"
              :size="i === 1 && isCheckedIn ? 'small' : 'x-small'"
              :line-color="theme.accent + '40'"
            >
              <div class="d-flex justify-space-between">
                <div>
                  <div class="text-subtitle-2 mb-1">
                    {{ i === 1 && isCheckedIn ? 'Currently Checked In' : 'Check-in / Check-out' }}
                  </div>
                  <div class="text-caption" :style="{color: theme.muted}">
                    {{ new Date(new Date().setDate(new Date().getDate() - i + 1)).toLocaleDateString() }}
                  </div>
                </div>
                <div class="text-caption" :style="{color: theme.muted}">
                  {{ i === 1 && isCheckedIn 
                    ? 'Since ' + new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})
                    : new Date(new Date().setHours(9, 0, 0)).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'}) + 
                      ' - ' + 
                      new Date(new Date().setHours(10, 30, 0)).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})
                  }}
                </div>
              </div>
            </v-timeline-item>
          </v-timeline>
        </div>
      </v-fade-transition>
      
      <!-- CLASSES VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'classes'" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Class Schedule</h1>
          
          <!-- Date picker for classes -->
          <v-card class="content-card mb-4" elevation="2">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-calendar</v-icon>
              Select Date
            </v-card-title>
            <v-card-text class="pa-0">
              <!-- Date selector -->
              <div class="week-day-selector d-flex pa-4 overflow-x-auto">
                <div 
                  v-for="i in 7" 
                  :key="i"
                  class="day-item"
                  :class="{ 'active': selectedDate.getDate() === new Date(new Date().setDate(new Date().getDate() + i - 1)).getDate() }"
                  @click="selectedDate = new Date(new Date().setDate(new Date().getDate() + i - 1)); updateDate()"
                >
                  <div class="day-name">{{ new Date(new Date().setDate(new Date().getDate() + i - 1)).toLocaleDateString('en-US', { weekday: 'short' }) }}</div>
                  <div class="day-number">{{ new Date(new Date().setDate(new Date().getDate() + i - 1)).getDate() }}</div>
                </div>
              </div>
              
              <div class="selected-date-info pa-3 text-center" :style="{backgroundColor: theme.accent + '40', color: theme.primary}">
                Selected: <strong>{{ formattedSelectedDate }}</strong>
                <v-chip v-if="isToday" size="x-small" :color="theme.success" :text-color="theme.lightText" class="ml-2">TODAY</v-chip>
              </div>
            </v-card-text>
          </v-card>
          
          <!-- Classes for selected day -->
          <div class="classes-for-date mb-5">
            <h2 class="text-subtitle-1 font-weight-bold mb-3" :style="{color: theme.primary}">Available Classes</h2>
            
            <div v-if="todayClasses.length === 0" class="text-center pa-4 rounded-lg" :style="{backgroundColor: theme.lightBg}">
              <v-icon size="40" :color="theme.muted" class="mb-2">mdi-calendar-blank</v-icon>
              <div class="text-subtitle-2">No classes scheduled</div>
              <div class="text-caption" :style="{color: theme.muted}">Try another day!</div>
            </div>
            
            <v-card 
              v-for="(classItem, index) in todayClasses"
              :key="index"
              class="mb-3 elevation-2 rounded-lg"
              :style="{borderLeft: '4px solid ' + classItem.color}"
            >
              <v-card-text class="pa-4">
                <div class="d-flex justify-space-between mb-2">
                  <div>
                    <div class="text-subtitle-1 font-weight-bold">{{ classItem.name }}</div>
                    <div class="text-caption d-flex align-center flex-wrap">
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-clock-outline</v-icon>
                        {{ classItem.time }}
                      </span>
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-account</v-icon>
                        {{ classItem.trainer }}
                      </span>
                    </div>
                  </div>
                  <v-btn 
                    :color="theme.secondary" 
                    size="small"
                    class="align-self-start"
                    @click="bookClass(classItem)"
                  >Book</v-btn>
                </div>
                
                <!-- Additional class details -->
                <v-chip 
                  size="x-small" 
                  :color="classItem.color + '20'" 
                  :text-color="classItem.color"
                  class="mt-2"
                >
                  <v-icon start size="x-small">{{ classItem.icon }}</v-icon>
                  {{ classItem.name.split(' ')[0] }}
                </v-chip>
              </v-card-text>
            </v-card>
          </div>
          
          <!-- My Bookings -->
          <div class="my-bookings">
            <h2 class="text-subtitle-1 font-weight-bold mb-3" :style="{color: theme.primary}">My Bookings</h2>
            
            <div v-if="bookedClasses.length === 0" class="text-center pa-4 rounded-lg" :style="{backgroundColor: theme.lightBg}">
              <v-icon size="40" :color="theme.muted" class="mb-2">mdi-calendar-blank</v-icon>
              <div class="text-subtitle-2">No bookings yet</div>
              <div class="text-caption" :style="{color: theme.muted}">Book a class to get started</div>
            </div>
            
            <v-card 
              v-else
              v-for="(booking, index) in bookedClasses"
              :key="booking.id"
              class="mb-3 elevation-1 rounded-lg"
              variant="outlined"
            >
              <v-card-text class="pa-3">
                <div class="d-flex align-center">
                  <v-avatar :color="booking.color || theme.primary" class="mr-3">
                    <v-icon color="white">{{ booking.icon || 'mdi-calendar' }}</v-icon>
                  </v-avatar>
                  <div class="flex-grow-1">
                    <div class="text-subtitle-1 font-weight-medium mb-1">{{ booking.name }}</div>
                    <div class="text-caption d-flex flex-wrap">
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-calendar</v-icon>
                        {{ getRelativeDay(booking.date) }}
                      </span>
                      <span class="d-flex align-center mr-3">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-clock-outline</v-icon>
                        {{ booking.time }}
                      </span>
                      <span v-if="booking.room" class="d-flex align-center">
                        <v-icon size="14" :color="theme.muted" class="mr-1">mdi-map-marker</v-icon>
                        {{ booking.room }}
                      </span>
                    </div>
                  </div>
                  <v-btn
                    icon="mdi-close"
                    size="x-small"
                    variant="text"
                    :color="theme.error"
                    @click="cancelClassBooking(booking.id)"
                  ></v-btn>
                </div>
              </v-card-text>
            </v-card>
          </div>
        </div>
      </v-fade-transition>
      
      <!-- WORKOUTS VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'workouts'" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Workout Library</h1>
          
          <!-- Active Workout Notice -->
          <v-card v-if="workoutActive" class="mb-4 content-card" :color="theme.error" dark>
            <v-card-text class="pa-3">
              <div class="d-flex align-center">
                <v-avatar color="white" class="mr-3">
                  <v-icon :color="theme.error">mdi-timer</v-icon>
                </v-avatar>
                <div class="flex-grow-1">
                  <div class="text-subtitle-1 font-weight-medium">Active Workout: {{ selectedWorkout?.name }}</div>
                  <div class="text-caption">Time: {{ formattedWorkoutDuration }}</div>
                </div>
                <v-btn
                  variant="tonal"
                  :color="theme.lightText"
                  :text-color="theme.error"
                  @click="navigateTo('activeWorkout')"
                >
                  View
                </v-btn>
              </div>
            </v-card-text>
          </v-card>
          
          <!-- Workout Templates -->
          <v-card
            v-for="(workout, index) in workoutTemplates"
            :key="workout.id"
            class="mb-4 content-card rounded-lg overflow-hidden"
            elevation="2"
          >
            <v-card-title :style="{backgroundColor: workout.color, color: theme.lightText}">
              <div class="d-flex align-center justify-space-between">
                <div>{{ workout.name }}</div>
                <v-btn 
                  variant="tonal" 
                  :disabled="workoutActive"
                  @click="workoutActive ? null : startWorkout(workout)"
                >
                  Start
                </v-btn>
              </div>
            </v-card-title>
            <v-card-text class="pt-3">
              <div class="d-flex mb-3 flex-wrap">
                <v-chip class="mr-2 mb-2" size="small" :color="workout.color + '20'" :text-color="workout.color">
                  <v-icon start size="small">mdi-clock-outline</v-icon>
                  {{ workout.duration }}
                </v-chip>
                <v-chip class="mr-2 mb-2" size="small" :color="workout.color + '20'" :text-color="workout.color">
                  <v-icon start size="small">mdi-weight</v-icon>
                  {{ workout.category }}
                </v-chip>
                <v-chip class="mb-2" size="small" :color="workout.color + '20'" :text-color="workout.color">
                  <v-icon start size="small">mdi-signal</v-icon>
                  {{ workout.difficulty }}
                </v-chip>
              </div>
              
              <v-expansion-panels variant="accordion">
                <v-expansion-panel title="View Exercises">
                  <template v-slot:text>
                    <v-list density="compact" lines="one">
                      <v-list-item v-for="(exercise, i) in workout.exercises" :key="i">
                        <template v-slot:prepend>
                          <v-avatar :color="workout.color + '33'" size="32">{{ i + 1 }}</v-avatar>
                        </template>
                        <v-list-item-title>{{ exercise.name }}</v-list-item-title>
                        <v-list-item-subtitle>{{ formatExercise(exercise) }}</v-list-item-subtitle>
                      </v-list-item>
                    </v-list>
                  </template>
                </v-expansion-panel>
              </v-expansion-panels>
            </v-card-text>
          </v-card>
          
          <!-- Recent Workouts -->
          <h2 class="text-subtitle-1 font-weight-bold my-3" :style="{color: theme.primary}">Recent Workouts</h2>
          
          <div v-if="recentWorkouts.length === 0" class="text-center pa-4 rounded-lg" :style="{backgroundColor: theme.lightBg}">
            <v-icon size="40" :color="theme.muted" class="mb-2">mdi-dumbbell</v-icon>
            <div class="text-subtitle-2">No workout history yet</div>
            <div class="text-caption" :style="{color: theme.muted}">Start a workout to track your progress</div>
          </div>
          
          <v-list class="pa-0" v-else>
            <v-list-item
              v-for="(workout, index) in recentWorkouts"
              :key="index"
              class="mb-2 rounded-lg"
              :ripple="false"
              density="compact"
              :style="{backgroundColor: theme.lightBg}"
            >
              <template v-slot:prepend>
                <v-avatar :color="workout.color" size="36" class="mr-2">
                  <v-icon size="16" color="white">{{ 
                    workout.type === 'Strength' ? 'mdi-dumbbell' : 
                    workout.type === 'Yoga' ? 'mdi-yoga' : 'mdi-run'
                  }}</v-icon>
                </v-avatar>
              </template>
              <v-list-item-title class="font-weight-medium">{{ workout.type }}: {{ workout.details }}</v-list-item-title>
              <v-list-item-subtitle class="d-flex flex-wrap">
                <span class="mr-3">{{ formatDateShort(workout.date) }}</span>
                <span class="mr-3">{{ workout.duration }}</span>
                <span>{{ workout.calories }} cal</span>
              </v-list-item-subtitle>
            </v-list-item>
          </v-list>
        </div>
      </v-fade-transition>
      
      <!-- ACTIVE WORKOUT VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'activeWorkout' && workoutActive" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Active Workout</h1>
          
          <v-card class="mb-4 content-card timer-card rounded-lg overflow-hidden" elevation="3">
            <v-card-title :style="{backgroundColor: selectedWorkout.color, color: theme.lightText}">
              {{ selectedWorkout.name }}
            </v-card-title>
            <v-card-text class="pa-0">
              <div class="text-center py-6">
                <div class="text-h2 font-weight-bold mb-1">{{ formattedWorkoutDuration }}</div>
                <div class="text-caption" :style="{color: theme.muted}">
                  Started at {{ workoutStartTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }) }}
                </div>
              </div>
              
              <div class="d-flex pa-4 pt-0 justify-space-between">
                <v-btn 
                  :color="theme.error"
                  variant="outlined"
                  @click="cancelWorkout"
                  width="48%"
                >
                  Cancel
                </v-btn>
                <v-btn 
                  :color="theme.success"
                  width="48%"
                  @click="completeWorkout"
                >
                  Complete
                </v-btn>
              </div>
            </v-card-text>
          </v-card>
          
          <!-- Exercise list -->
          <v-card class="content-card rounded-lg" elevation="1">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-view-list</v-icon>
              Exercise List
            </v-card-title>
            <v-list lines="one">
              <v-list-item v-for="(exercise, i) in selectedWorkout.exercises" :key="i">
                <template v-slot:prepend>
                  <v-avatar :color="selectedWorkout.color + '33'" size="32">{{ i + 1 }}</v-avatar>
                </template>
                <v-list-item-title>{{ exercise.name }}</v-list-item-title>
                <v-list-item-subtitle>
                  {{ exercise.sets }} sets × {{ exercise.reps }}
                </v-list-item-subtitle>
                <v-list-item-subtitle class="text-caption" :style="{color: theme.muted}">
                  Rest: {{ exercise.rest }}
                </v-list-item-subtitle>
              </v-list-item>
            </v-list>
          </v-card>
        </div>
      </v-fade-transition>
      
      <!-- TRAINERS VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'trainers'" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Personal Trainers</h1>
          
          <v-card
            v-for="(trainer, index) in trainers"
            :key="trainer.id"
            class="mb-4 content-card rounded-lg overflow-hidden"
            elevation="2"
          >
            <div class="trainer-header d-flex pa-4" :style="{backgroundColor: theme.lightBg}">
              <v-avatar size="80" class="mr-4">
                <v-img :src="trainer.image" alt="Trainer"></v-img>
              </v-avatar>
              <div>
                <div class="text-h6 font-weight-medium">{{ trainer.name }}</div>
                <div class="text-caption mb-1">{{ trainer.specialty }}</div>
                <div class="d-flex align-center">
                  <v-rating
                    :model-value="trainer.rating"
                    color="amber"
                    density="compact"
                    half-increments
                    readonly
                    size="x-small"
                  ></v-rating>
                  <span class="text-caption ml-1">{{ trainer.rating }}/5</span>
                </div>
              </div>
            </div>
            
            <v-card-text class="pa-4">
              <div class="text-body-2 mb-3">{{ trainer.bio }}</div>
              <div class="d-flex align-center mb-3 flex-wrap">
                <v-chip class="mr-2 mb-2" size="small" :color="theme.primary + '20'" :text-color="theme.primary">
                  <v-icon start size="small">mdi-clock-outline</v-icon>
                  {{ trainer.experience }}
                </v-chip>
                <v-chip class="mb-2" size="small" :color="theme.primary + '20'" :text-color="theme.primary">
                  <v-icon start size="small">mdi-calendar-check</v-icon>
                  Available: {{ trainer.availability }}
                </v-chip>
              </div>
              
              <v-btn
                block
                :color="theme.secondary"
                @click="bookTrainer(trainer)"
              >
                <v-icon start>mdi-calendar-plus</v-icon>
                Book Consultation
              </v-btn>
            </v-card-text>
          </v-card>
        </div>
      </v-fade-transition>
      
      <!-- MEMBERSHIP VIEW -->
      <v-fade-transition>
        <div v-if="currentView === 'membership'" class="pa-4 pt-12">
          <h1 class="text-h5 font-weight-bold text-center mb-4" :style="{color: theme.primary}">Membership</h1>
          
          <!-- Current membership -->
          <v-card class="content-card rounded-lg mb-4" elevation="3">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-card-account-details</v-icon>
              Your Membership
            </v-card-title>
            <v-card-text class="pa-4">
              <div class="d-flex align-items-center mb-3">
                <v-avatar :color="theme.secondary" class="mr-3">
                  <v-icon color="white">mdi-crown</v-icon>
                </v-avatar>
                <div>
                  <div class="text-h6 font-weight-bold">{{ user.memberLevel }}</div>
                  <div class="text-caption">
                    <v-chip
                      size="x-small"
                      :color="membershipStatusColor"
                      text-color="white"
                    >{{ membershipStatus }}</v-chip>
                  </div>
                </div>
              </div>
              
              <div class="mb-3">
                <div class="text-body-2 d-flex justify-space-between">
                  <span>Valid until:</span>
                  <span class="font-weight-medium">{{ formattedMembershipDate }}</span>
                </div>
                <div class="text-body-2 d-flex justify-space-between">
                  <span>Days remaining:</span>
                  <span class="font-weight-medium">{{ user.daysRemaining }} days</span>
                </div>
              </div>
              
              <v-btn
                block
                :color="theme.secondary"
                @click="renewMembership"
              >
                <v-icon start>mdi-refresh</v-icon>
                Renew Membership
              </v-btn>
            </v-card-text>
          </v-card>
          
          <!-- Membership options -->
          <h2 class="text-subtitle-1 font-weight-bold mb-3" :style="{color: theme.primary}">Membership Plans</h2>
          
          <v-card
            v-for="(plan, index) in membershipOptions"
            :key="plan.id"
            class="mb-4 content-card rounded-lg"
            elevation="2"
            :style="plan.recommended ? {border: '2px solid ' + theme.secondary} : {}"
          >
            <div v-if="plan.recommended" 
              class="recommended-banner text-center py-1" 
              :style="{backgroundColor: theme.secondary, color: theme.primary}"
            >
              <span class="text-caption font-weight-bold">RECOMMENDED</span>
            </div>
            
            <v-card-text class="pa-4">
              <div class="d-flex align-items-center mb-3">
                <div :style="{
                    width: '48px', 
                    height: '48px', 
                    borderRadius: '50%', 
                    backgroundColor: plan.color,
                    color: theme.lightText,
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontWeight: 'bold',
                    fontSize: '20px',
                    marginRight: '16px'
                  }"
                >
                  {{ plan.name.charAt(0) }}
                </div>
                <div>
                  <div class="text-h6 font-weight-bold">{{ plan.name }}</div>
                  <div class="text-subtitle-1">
                    <span class="font-weight-medium">RM {{ plan.price.toFixed(2) }}</span>
                    <span class="text-caption">/{{ plan.period }}</span>
                  </div>
                </div>
              </div>
              
              <v-list density="compact" class="pa-0 bg-transparent mb-3">
                <v-list-item v-for="(feature, i) in plan.features" :key="i" density="compact" class="px-0 py-1">
                  <template v-slot:prepend>
                    <v-icon :color="theme.success" size="small">mdi-check-circle</v-icon>
                  </template>
                  <v-list-item-title class="text-body-2">{{ feature }}</v-list-item-title>
                </v-list-item>
              </v-list>
              
              <v-btn
                block
                :color="plan.recommended ? theme.secondary : plan.color"
                :variant="plan.recommended ? 'flat' : 'outlined'"
                @click="selectMembership(plan)"
                :disabled="user.memberLevel === plan.name"
              >
                {{ user.memberLevel === plan.name ? 'Current Plan' : 'Select Plan' }}
              </v-btn>
            </v-card-text>
          </v-card>
        </div>
      </v-fade-transition>
      
      <!-- Snackbar Notifications -->
      <v-snackbar
        v-model="showSnackbar"
        :color="snackbarColor"
        :timeout="3000"
        location="top"
      >
        {{ snackbarText }}
        <template v-slot:actions>
          <v-btn variant="text" @click="showSnackbar = false">Close</v-btn>
        </template>
      </v-snackbar>
    </v-container>



    <style>
.back-navigation {
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  z-index: 5;
}

.back-btn {
  border-radius: 50%;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.action-card {
  transition: all 0.2s ease;
}

.action-card:active {
  transform: scale(0.95);
}

.day-item {
  display: flex;
  flex-direction: column;
  align-items: center;
  min-width: 48px;
  cursor: pointer;
  padding: 8px 4px;
  margin-right: 8px;
  border-radius: 8px;
  position: relative;
}

.day-item.active {
  background-color: rgba(0, 0, 0, 0.05);
  font-weight: bold;
}

.day-item.active:after {
  content: '';
  position: absolute;
  bottom: -2px;
  left: 50%;
  transform: translateX(-50%);
  width: 8px;
  height: 8px;
  border-radius: 50%;
  background-color: currentColor;
}

.day-name {
  font-size: 0.7rem;
  text-transform: uppercase;
}

.day-number {
  font-size: 1.1rem;
}

.week-day-selector {
  -ms-overflow-style: none;  /* IE and Edge */
  scrollbar-width: none;  /* Firefox */
}

.week-day-selector::-webkit-scrollbar {
  display: none;
}

.timer-card {
  background: linear-gradient(135deg, rgba(255,255,255,0.7) 0%, rgba(243,243,243,0.7) 100%);
}

.checkin-timeline .v-timeline-divider__dot {
  background-color: white !important;
}

.qr-scanner-container {
  width: 100%;
  height: 300px;
  border-radius: 12px;
  overflow: hidden;
  position: relative;
}

.qr-scanner-placeholder {
  width: 100%;
  height: 100%;
  background-color: rgba(0,0,0,0.03);
  border: 1px dashed rgba(0,0,0,0.2);
  border-radius: 12px;
}

@media (max-width: 600px) {
  .day-item {
    min-width: 40px;
  }
}
</style>

  `
};
