export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C'
      },

      // QR Scanner States
      isScanning: false,
      scannerStream: null,
      scannerVideo: null,
      isInitializing: false,
      scanError: null,
      lastScanTime: 0,
      scanCooldown: 2000, // 2 second cooldown between scans

      // Transfer States and Data
      isLoading: false,
      transferAmount: 100,
      transferStep: 1, // 1: QR Scan, 2: Amount, 3: Confirm, 4: Receipt
      transferPin: '',
      isProcessing: false,
      showReceipt: false,
      isErrorNotification: false,

      // Predefined amounts (matching wallet topup)
      predefinedAmounts: [50, 100, 200, 500, 1000],

      // User Data
      userData: {
        name: "Jason Tan",
        role: "platinum", 
        membershipId: "TAPANG-PLAT-2023",
        balance: 350.00,
        availableCredit: 0
      },

      // Recipient Data (from QR scan)
      recipient: null,
      scannedMemberNo: null,

      // Transfer details
      transferNote: '',
      transferReference: '',

      // Transaction receipt
      receipt: {
        transactionId: '',
        date: '',
        amount: 0,
        recipientName: '',
        recipientId: '',
        status: 'completed',
        note: ''
      },

      // Notification system
      showNotification: false,
      notificationMessage: '',

      // Animation states
      scannerActive: false
    }
  },
  computed: {
    // Format current balance
    formattedBalance() {
      return this.userData.balance.toFixed(2)
    },

    // Format transfer amount
    formattedTransferAmount() {
      return this.transferAmount.toFixed(0)
    },

    // Check if amount is valid
    isAmountValid() {
      return Number.isInteger(this.transferAmount) &&
        this.transferAmount >= 10 &&
        this.transferAmount <= 5000 &&
        this.transferAmount <= this.userData.balance
    },

    // Calculate new balance after transfer
    newBalance() {
      return this.userData.balance - this.transferAmount
    },

    // Format new balance
    formattedNewBalance() {
      return this.newBalance.toFixed(2)
    },

    // Check if transfer is ready to proceed
    canProceed() {
      return this.isAmountValid && this.recipient !== null
    },

    // Check if PIN is valid
    isPinValid() {
      return this.transferPin.length === 6 && /^\d+$/.test(this.transferPin)
    }
  },
  methods: {
    formatCurrency(amount) {
      return parseFloat(amount).toFixed(2)
    },

    // Set predefined amount
    setAmount(amount) {
      this.transferAmount = amount
    },

    // Get amount validation error message
    getAmountErrorMessage() {
      if (!Number.isInteger(this.transferAmount)) return 'Please enter whole numbers only'
      if (this.transferAmount < 10) return 'Minimum amount is 10 credits'
      if (this.transferAmount > 5000) return 'Maximum amount is 5,000 credits'
      if (this.transferAmount > this.userData.balance) return 'Amount exceeds your balance'
      return ''
    },

    // Initialize QR Scanner
    async initQRScanner() {
      if (this.isInitializing) return
      
      this.isInitializing = true
      this.scanError = null

      try {
        // Check if getUserMedia is supported
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
          throw new Error('Camera access not supported')
        }

        // Request camera access
        const stream = await navigator.mediaDevices.getUserMedia({
          video: {
            facingMode: 'environment', // Use back camera
            width: { ideal: 1280 },
            height: { ideal: 720 }
          }
        })

        this.scannerStream = stream
        this.scannerVideo = this.$refs.scannerVideo
        this.scannerVideo.srcObject = stream
        
        await this.scannerVideo.play()
        this.isScanning = true
        this.scannerActive = true
        
        // Start scanning for QR codes
        this.startQRDetection()

      } catch (error) {
        console.error('QR Scanner initialization failed:', error)
        this.scanError = error.message.includes('Permission denied') 
          ? 'Camera permission denied. Please allow camera access and try again.'
          : 'Failed to access camera. Please check your device settings.'
        this.showErrorNotification(this.scanError)
      } finally {
        this.isInitializing = false
      }
    },

    // Start QR code detection
    startQRDetection() {
      if (!this.scannerVideo || !this.isScanning) return

      const canvas = document.createElement('canvas')
      const context = canvas.getContext('2d')
      
      const detectQR = () => {
        if (!this.isScanning || !this.scannerVideo) return

        const video = this.scannerVideo
        if (video.readyState === video.HAVE_ENOUGH_DATA) {
          canvas.width = video.videoWidth
          canvas.height = video.videoHeight
          context.drawImage(video, 0, 0, canvas.width, canvas.height)
          
          const imageData = context.getImageData(0, 0, canvas.width, canvas.height)
          
          // Use jsQR library for QR detection (we need to add this)
          if (window.jsQR) {
            const code = window.jsQR(imageData.data, imageData.width, imageData.height)
            if (code && code.data) {
              this.handleQRCodeDetected(code.data)
              return
            }
          }
        }
        
        // Continue scanning
        if (this.isScanning) {
          requestAnimationFrame(detectQR)
        }
      }
      
      detectQR()
    },

    // Handle QR code detection
    async handleQRCodeDetected(qrData) {
      const now = Date.now()
      if (now - this.lastScanTime < this.scanCooldown) return
      
      this.lastScanTime = now
      
      try {
        // The QR code should contain the member_no as plain text
        const memberNo = qrData.trim()
        
        if (!memberNo) {
          this.showErrorNotification('Invalid QR code format')
          return
        }

        this.scannedMemberNo = memberNo
        this.showRegularNotification(`Scanned member: ${memberNo}`)
        
        // Look up member details
        await this.lookupMemberDetails(memberNo)
        
      } catch (error) {
        console.error('QR processing error:', error)
        this.showErrorNotification('Failed to process QR code')
      }
    },

    // Look up member details from scanned QR
    async lookupMemberDetails(memberNo) {
      try {
        const response = await this.$root.callApi('wallet', 'search_users', {
          search_query: memberNo,
          user_id: sessionStorage.getItem('id'),
          limit: 1
        })

        if (response.data && response.data.status === "YES" && response.data.users.length > 0) {
          const member = response.data.users[0]
          this.recipient = {
            id: member.id,
            name: member.name || member.fullname,
            membershipId: member.member_no || memberNo,
            role: member.member_type || 'member'
          }
          
          // Stop scanning and proceed to amount step
          this.stopQRScanner()
          this.transferStep = 2
          this.showRegularNotification(`Found: ${this.recipient.name}`)
        } else {
          this.showErrorNotification('Member not found. Please try again.')
        }
      } catch (error) {
        console.error('Error looking up member:', error)
        this.showErrorNotification('Error looking up member details')
      }
    },

    // Stop QR Scanner
    stopQRScanner() {
      this.isScanning = false
      this.scannerActive = false
      
      if (this.scannerStream) {
        this.scannerStream.getTracks().forEach(track => track.stop())
        this.scannerStream = null
      }
      
      if (this.scannerVideo) {
        this.scannerVideo.srcObject = null
      }
    },

    // Restart QR scanning (for rescan)
    restartQRScanning() {
      this.recipient = null
      this.scannedMemberNo = null
      this.transferStep = 1
      this.initQRScanner()
    },

    // Show error notification (centered modal)
    showErrorNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = true

      setTimeout(() => {
        this.showNotification = false
        this.isErrorNotification = false
      }, 4000)
    },

    // Method to manually close error notification
    closeErrorNotification() {
      this.showNotification = false
      this.isErrorNotification = false
    },

    // Show regular notification
    showRegularNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = false
      setTimeout(() => {
        this.showNotification = false
      }, 3000)
    },

    // Proceed to next step
    nextStep() {
      if (this.transferStep === 2 && this.isAmountValid) {
        this.transferStep = 3
      } else if (this.transferStep === 3) {
        this.transferStep = 4
      }
    },

    // Go back to previous step
    goBack() {
      if (this.transferStep > 1) {
        this.transferStep--
        if (this.transferStep === 1) {
          this.restartQRScanning()
        }
      } else {
        this.stopQRScanner()
        this.$root.loadComponent('modules/home.js')
      }
    },

    // Process the transfer (same as original)
    async processTransfer() {
      if (!this.isPinValid) {
        this.showErrorNotification('Please enter a valid 6-digit PIN')
        return
      }

      this.isProcessing = true
      this.showRegularNotification('Processing your transfer...')

      try {
        const response = await this.$root.callApi('wallet', 'transfer', {
          user_id: sessionStorage.getItem('id'),
          recipient_id: this.recipient.id,
          amount: this.transferAmount,
          note: this.transferNote,
          pin: this.transferPin
        })

        if (response.data && response.data.status === "YES") {
          // Update balance
          this.userData.balance = parseFloat(response.data.new_balance)

          // Generate transaction receipt
          this.receipt = {
            transactionId: response.data.transaction_id,
            date: response.data.transaction_date,
            amount: this.transferAmount,
            recipientName: response.data.recipient_name,
            recipientId: response.data.recipient_member_no || response.data.recipient_id,
            status: 'completed',
            note: this.transferNote || 'QR Transfer'
          }

          // Show success screen
          this.transferStep = 4
          this.showReceipt = true
          this.showRegularNotification('Transfer successful!')
        } else {
          this.showErrorNotification(response.data.message || 'Transfer failed')
        }
      } catch (error) {
        console.error('Error processing transfer:', error)
        this.showErrorNotification('Error processing transfer')
      } finally {
        this.isProcessing = false
      }
    },

    // Complete the transfer process
    completeTransfer() {
      this.$root.loadComponent('modules/home.js')
    },

    // Navigation methods
    navigateTo(screen) {
      this.stopQRScanner()
      this.showNotification = true
      this.notificationMessage = 'Loading...'
      setTimeout(() => {
        this.$root.loadComponent('modules/' + screen)
      }, 600)
    },

    // Fetch wallet balance
    async fetchWalletBalance() {
      this.isLoading = true
      try {
        const response = await this.$root.callApi('wallet', 'get_balance', {
          user_id: sessionStorage.getItem('id')
        })

        if (response.data && response.data.status === "YES") {
          this.userData.balance = parseFloat(response.data.balance)
          this.userData.availableCredit = parseFloat(response.data.available_credit)
        } else {
          this.showErrorNotification(response.data.message || 'Failed to fetch wallet balance')
        }
      } catch (error) {
        console.error('Error fetching wallet balance:', error)
        this.showErrorNotification('Error fetching wallet balance')
      } finally {
        this.isLoading = false
      }
    }
  },
  async mounted() {
    // Fetch wallet data
    await this.fetchWalletBalance()
    
    // Start QR scanning immediately
    this.initQRScanner()

    // IMPORTANT: Inject the CSS styles dynamically
    const styleElement = document.createElement('style')
    styleElement.textContent = `
      /* Base Styles - QR Transfer */
      .qr-transfer-container {
        --primary: #1A3C28;
        --secondary: #DDA333;
        --accent: #A3BA95;
        --cream: #EADDCB;
        --muted: #889FAC;
        --text-dark: #1E1E1E;
        --text-light: #FFFFFF;
        --error: #B71C1C;
        --success: #2E7D32;
        
        font-family: 'Arial', sans-serif;
        background-color: var(--cream);
        min-height: 100vh;
        position: relative;
        overflow-x: hidden;
      }
      
      /* QR Scanner Styles */
      .qr-scanner-container {
        position: relative;
        background: black;
        border-radius: 20px;
        overflow: hidden;
        margin-bottom: 24px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.3);
      }
      
      .scanner-video {
        width: 100%;
        height: 300px;
        object-fit: cover;
        display: block;
      }
      
      .scanner-overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(0,0,0,0.3);
      }
      
      .scanner-frame {
        width: 200px;
        height: 200px;
        border: 3px solid var(--secondary);
        border-radius: 16px;
        position: relative;
        background: transparent;
      }
      
      .scanner-frame::before,
      .scanner-frame::after {
        content: '';
        position: absolute;
        width: 30px;
        height: 30px;
      }
      
      .scanner-frame::before {
        top: -3px;
        left: -3px;
        border-top: 3px solid var(--secondary);
        border-left: 3px solid var(--secondary);
        border-radius: 16px 0 0 0;
      }
      
      .scanner-frame::after {
        bottom: -3px;
        right: -3px;
        border-bottom: 3px solid var(--secondary);
        border-right: 3px solid var(--secondary);
        border-radius: 0 0 16px 0;
      }
      
      .scanner-instructions {
        position: absolute;
        bottom: 20px;
        left: 20px;
        right: 20px;
        text-align: center;
        color: white;
        background: rgba(0,0,0,0.7);
        padding: 12px 16px;
        border-radius: 12px;
        font-size: 14px;
      }
      
      .scanner-error {
        background: white;
        border-radius: 20px;
        padding: 24px;
        text-align: center;
        margin-bottom: 24px;
        border: 2px solid var(--error);
      }
      
      .scanner-error-icon {
        font-size: 48px;
        color: var(--error);
        margin-bottom: 16px;
      }
      
      .scanner-error-message {
        color: var(--error);
        font-weight: 600;
        margin-bottom: 16px;
      }
      
      /* Scanning Animation */
      .scanning-line {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 2px;
        background: var(--secondary);
        animation: scanningAnimation 2s infinite;
      }
      
      @keyframes scanningAnimation {
        0% { transform: translateY(0); opacity: 1; }
        50% { transform: translateY(198px); opacity: 0.7; }
        100% { transform: translateY(0); opacity: 1; }
      }
      
      /* Recipient Display */
      .scanned-recipient {
        background: white;
        border-radius: 20px;
        padding: 20px;
        margin-bottom: 24px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        border: 2px solid var(--accent);
      }
      
      .recipient-header {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
      }
      
      .recipient-avatar {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background: linear-gradient(135deg, var(--accent) 0%, #8FA582 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 16px;
        font-weight: 700;
        color: white;
        font-size: 24px;
      }
      
      .recipient-info h3 {
        margin: 0 0 4px 0;
        color: var(--text-dark);
        font-weight: 700;
      }
      
      .recipient-info p {
        margin: 0;
        color: var(--muted);
        font-size: 14px;
      }
      
      .rescan-button {
        background: var(--accent);
        color: white;
        border: none;
        border-radius: 12px;
        padding: 8px 16px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
      }
      
      .rescan-button:hover {
        background: #8FA582;
        transform: translateY(-2px);
      }
      
      /* Loading States */
      .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(26, 60, 40, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 1000;
      }
      
      .loading-spinner {
        display: flex;
        flex-direction: column;
        align-items: center;
      }
      
      .loading-text {
        margin-top: 16px;
        color: white;
        font-weight: 500;
      }
      
      /* Notification Styles */
      .notification-toast {
        position: fixed;
        bottom: 24px;
        left: 50%;
        transform: translateX(-50%);
        background-color: var(--secondary);
        color: var(--primary);
        padding: 14px 20px;
        border-radius: 20px;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 900;
        min-width: 80%;
        max-width: 90%;
        text-align: center;
      }
      
      .notification-toast.error-notification {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background-color: var(--error);
        color: white;
        padding: 20px 24px;
        border-radius: 16px;
        box-shadow: 0 8px 24px rgba(183, 28, 28, 0.3);
        z-index: 1001;
        min-width: 85%;
        max-width: 95%;
      }
      
      .error-notification-backdrop {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        backdrop-filter: blur(2px);
      }
      
      .error-close-button {
        position: absolute;
        top: 8px;
        right: 12px;
        background: none;
        border: none;
        color: white;
        font-size: 20px;
        cursor: pointer;
        padding: 4px;
        opacity: 0.8;
      }
      
      /* Header Section */
      .qr-transfer-header {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        padding: 16px;
        border-radius: 0 0 24px 24px;
        box-shadow: 0 6px 16px rgba(0,0,0,0.15);
        color: var(--text-light);
        position: relative;
        overflow: hidden;
      }
      
      .header-content {
        position: relative;
        z-index: 1;
      }
      
      .back-button {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        cursor: pointer;
        min-height: 44px;
        transition: opacity 0.2s ease;
      }
      
      .back-button:hover {
        opacity: 0.8;
      }
      
      .back-button-text {
        margin-left: 8px;
        font-weight: 500;
        font-size: 16px;
      }
      
      .page-title {
        font-size: 22px;
        font-weight: 700;
        margin-bottom: 12px;
      }
      
      /* Balance Summary */
      .balance-summary {
        background: linear-gradient(135deg, white 0%, #f8f9fa 100%);
        border-radius: 20px;
        margin: -15px 16px 20px 16px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.12);
        position: relative;
        overflow: hidden;
        border: 1px solid rgba(26, 60, 40, 0.08);
      }
      
      .balance-summary::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 3px;
        background: linear-gradient(90deg, var(--primary), var(--secondary), var(--accent));
      }
      
      .balance-title {
        color: var(--muted);
        font-size: 14px;
        margin-bottom: 8px;
        font-weight: 500;
      }
      
      .balance-amount {
        font-size: 28px;
        font-weight: 700;
        color: var(--primary);
        display: flex;
        align-items: center;
        gap: 8px;
      }
      
      .balance-icon {
        margin-right: 4px;
      }
      
      /* Content Area */
      .content-area {
        padding: 0 16px 80px 16px;
        margin-top: 20px;
      }
      
      /* Reuse existing form styles from transfer page */
      .amount-selection {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .section-title {
        font-size: 18px;
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 16px;
      }
      
      .amount-chips {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 10px;
        margin-bottom: 20px;
      }
      
      .amount-chip {
        background: rgba(255, 255, 255, 0.9);
        border: 2px solid rgba(26, 60, 40, 0.08);
        border-radius: 16px;
        padding: 16px 12px;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
        text-align: center;
        min-height: 80px;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        backdrop-filter: blur(10px);
        position: relative;
        overflow: hidden;
      }
      
      .amount-chip::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(135deg, rgba(26, 60, 40, 0.02) 0%, rgba(163, 186, 149, 0.03) 100%);
        opacity: 0;
        transition: opacity 0.3s ease;
      }
      
      .amount-chip:hover::before {
        opacity: 1;
      }
      
      .amount-chip:active {
        transform: scale(0.96);
      }
      
      .amount-chip.selected {
        background: linear-gradient(135deg, rgba(26, 60, 40, 0.08) 0%, rgba(163, 186, 149, 0.05) 100%);
        border-color: var(--primary);
        box-shadow: 0 6px 20px rgba(26, 60, 40, 0.15);
        transform: translateY(-2px);
      }
      
      .amount-chip-credit {
        color: #2E7D32;
        font-weight: 700;
        font-size: 16px;
        position: relative;
        z-index: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 6px;
      }
      
      .credit-icon {
        color: #2E7D32 !important;
      }
      
      .amount-chip.selected .amount-chip-credit {
        color: var(--primary);
      }
      
      .amount-chip.selected .credit-icon {
        color: var(--primary) !important;
      }
      
      .custom-amount-section {
        margin-top: 24px;
      }
      
      .custom-amount-input {
        margin-top: 8px;
      }
      
      .custom-amount-input .v-field__prepend-inner {
        margin-right: 8px !important;
      }
      
      .custom-amount-input .v-field__input {
        font-weight: 600 !important;
        font-size: 16px !important;
      }
      
      .form-control {
        border: 2px solid rgba(26, 60, 40, 0.1);
        border-radius: 12px;
        padding: 14px 16px;
        font-size: 16px;
        transition: all 0.2s ease;
        background: white;
      }
      
      .form-control:focus {
        border-color: var(--accent);
        box-shadow: 0 0 0 3px rgba(163, 186, 149, 0.1);
        outline: none;
      }
      
      .form-label {
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 8px;
      }
      
      .input-group-text {
        display: flex;
        align-items: center;
        padding: 0.375rem 0.75rem;
        font-size: 1rem;
        font-weight: 400;
        line-height: 1.5;
        color: var(--muted);
        background-color: #f8f9fa;
        border: 2px solid rgba(26, 60, 40, 0.1);
        border-radius: 12px 0 0 12px;
      }
      
      .input-group .form-control {
        border-radius: 0 12px 12px 0;
        border-left: none;
      }
      
      /* Button Styles */
      .btn-primary {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        border: none;
        border-radius: 16px;
        padding: 18px 28px;
        font-weight: 700;
        font-size: 16px;
        text-transform: capitalize;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
        box-shadow: 0 6px 20px rgba(26, 60, 40, 0.2);
        min-height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        text-decoration: none;
      }
      
      .btn-primary:hover:not(:disabled) {
        transform: translateY(-3px);
        box-shadow: 0 8px 28px rgba(26, 60, 40, 0.3);
        background: linear-gradient(135deg, #2d5a3d 0%, var(--secondary) 100%);
        color: white;
      }
      
      .btn-primary:disabled {
        background: linear-gradient(135deg, #e0e0e0 0%, #bdbdbd 100%);
        color: #9e9e9e;
        cursor: not-allowed;
        transform: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        opacity: 0.6;
      }
      
      .d-grid {
        display: grid !important;
      }
      
      .text-danger {
        color: var(--error) !important;
      }
      
      .mb-3 {
        margin-bottom: 1rem !important;
      }
      
      .mt-2 {
        margin-top: 0.5rem !important;
      }
      
      /* Confirmation Screen */
      .confirmation-screen {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .confirmation-item {
        display: flex;
        justify-content: space-between;
        padding: 16px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
      }
      
      .confirmation-item:last-child {
        border-bottom: none;
      }
      
      .confirmation-label {
        color: var(--muted);
        font-size: 14px;
        font-weight: 500;
      }
      
      .confirmation-value {
        font-weight: 600;
        font-size: 14px;
        color: var(--text-dark);
        text-align: right;
        display: flex;
        align-items: center;
        justify-content: flex-end;
        gap: 4px;
      }
      
      /* PIN Entry */
      .pin-entry {
        background: white;
        border-radius: 20px;
        padding: 20px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        margin-bottom: 24px;
        text-align: center;
        border: 1px solid rgba(26, 60, 40, 0.05);
      }
      
      .pin-description {
        color: var(--muted);
        font-size: 14px;
        margin-bottom: 16px;
        font-weight: 500;
      }
      
      .pin-input {
        margin: 0 auto;
        max-width: 240px;
      }
    `
    document.head.appendChild(styleElement)
  },
  beforeUnmount() {
    // Clean up scanner when component is destroyed
    this.stopQRScanner()
  },
  template: `
    <div class="qr-transfer-container">
      <!-- Loading Overlay -->
      <div class="loading-overlay" v-if="isLoading || isProcessing">
        <div class="loading-spinner">
          <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
          <div class="loading-text">{{ isProcessing ? 'Processing...' : 'Loading...' }}</div>
        </div>
      </div>
      
      <!-- Error Notification Backdrop -->
      <div class="error-notification-backdrop" 
           v-if="showNotification && isErrorNotification" 
           @click="closeErrorNotification">
      </div>
      
      <!-- Header -->
      <div class="qr-transfer-header">
        <div class="header-content">
          <div class="back-button" @click="goBack">
            <v-icon color="white">mdi-arrow-left</v-icon>
            <span class="back-button-text">Back</span>
          </div>
          
          <h1 class="page-title">QR Scan Transfer</h1>
        </div>
      </div>
      
      <!-- Balance Summary -->
      <div class="balance-summary">
        <div class="balance-title">Available Balance</div>
        <div class="balance-amount">
          <v-icon size="24" color="#1A3C28" class="balance-icon">mdi-circle-multiple-outline</v-icon>
          {{ formattedBalance }}
        </div>
      </div>
      
      <!-- Content Area -->
      <div class="content-area">
        <!-- Step 1: QR Scanner -->
        <div v-if="transferStep === 1">
          <div v-if="scanError" class="scanner-error">
            <div class="scanner-error-icon">
              <v-icon size="48" color="error">mdi-camera-off</v-icon>
            </div>
            <div class="scanner-error-message">{{ scanError }}</div>
            <button class="btn-primary" @click="initQRScanner" :disabled="isInitializing">
              <v-icon start>mdi-camera</v-icon>
              {{ isInitializing ? 'Initializing...' : 'Try Again' }}
            </button>
          </div>
          
          <div v-else class="qr-scanner-container">
            <video ref="scannerVideo" class="scanner-video" playsinline></video>
            
            <div class="scanner-overlay" v-if="scannerActive">
              <div class="scanner-frame">
                <div class="scanning-line" v-if="isScanning"></div>
              </div>
            </div>
            
            <div class="scanner-instructions" v-if="isScanning">
              Point your camera at a member's QR code to scan
            </div>
            
            <div class="scanner-instructions" v-else-if="isInitializing">
              Initializing camera...
            </div>
          </div>
        </div>
        
        <!-- Step 2: Amount Selection -->
        <div v-if="transferStep === 2">
          <!-- Show scanned recipient -->
          <div class="scanned-recipient" v-if="recipient">
            <div class="recipient-header">
              <div class="recipient-avatar">
                {{ recipient.name.charAt(0) }}
              </div>
              <div class="recipient-info">
                <h3>{{ recipient.name }}</h3>
                <p>{{ recipient.membershipId }}</p>
                <p><small>{{ recipient.role }} member</small></p>
              </div>
              <button class="rescan-button" @click="restartQRScanning">
                <v-icon size="16">mdi-qrcode-scan</v-icon>
                Rescan
              </button>
            </div>
          </div>
          
          <div class="amount-selection">
            <h2 class="section-title">Transfer Amount</h2>
            
            <!-- Predefined Amounts -->
            <div class="amount-chips">
              <div v-for="amount in predefinedAmounts" :key="amount"
                   class="amount-chip" :class="{ 'selected': transferAmount === amount }"
                   @click="setAmount(amount)">
                <div class="amount-chip-credit">
                  <v-icon size="16" class="credit-icon">mdi-circle-multiple-outline</v-icon>
                  {{ amount.toFixed(0) }}
                </div>
              </div>
            </div>
            
            <!-- Custom Amount -->
            <div class="custom-amount-section">
              <h4 class="mb-2">Custom Amount</h4>
              <v-text-field
                v-model.number="transferAmount"
                type="number"
                min="10"
                max="5000"
                step="1"
                placeholder="Enter credits (whole numbers only)"
                variant="outlined"
                density="comfortable"
                color="primary"
                class="custom-amount-input"
                :error="!isAmountValid && transferAmount > 0"
                :error-messages="!isAmountValid && transferAmount > 0 ? getAmountErrorMessage() : []"
              >
                <template v-slot:prepend-inner>
                  <v-icon size="18" color="#2E7D32">mdi-circle-multiple-outline</v-icon>
                </template>
              </v-text-field>
            </div>
          </div>
          
          <!-- Next Button -->
          <div class="d-grid gap-2">
            <button class="btn btn-primary" @click="nextStep" :disabled="!isAmountValid">
              <v-icon start>mdi-check-circle</v-icon>
              Continue to Confirm
            </button>
          </div>
        </div>
        
        <!-- Step 3: Confirmation -->
        <div v-if="transferStep === 3">
          <div class="confirmation-screen">
            <h2 class="section-title">Confirm Transfer</h2>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Recipient</div>
              <div class="confirmation-value">{{ recipient.name }}</div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Membership ID</div>
              <div class="confirmation-value">{{ recipient.membershipId }}</div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">Amount</div>
              <div class="confirmation-value">
                <v-icon size="16" color="#2E7D32">mdi-circle-multiple-outline</v-icon>
                {{ transferAmount.toFixed(0) }}
              </div>
            </div>
            
            <div class="confirmation-item">
              <div class="confirmation-label">New Balance</div>
              <div class="confirmation-value">
                <v-icon size="16" color="#2E7D32">mdi-circle-multiple-outline</v-icon>
                {{ newBalance.toFixed(2) }}
              </div>
            </div>
          </div>
          
          <!-- PIN Entry -->
          <div class="pin-entry">
            <h2 class="section-title">Enter PIN</h2>
            <p class="pin-description">Please enter your 6-digit PIN to authorize this transfer</p>
            
            <div class="pin-input">
              <input type="password" class="form-control" placeholder="6-digit PIN"
                     v-model="transferPin" maxlength="6" inputmode="numeric" pattern="[0-9]*">
            </div>
          </div>
          
          <!-- Process Button -->
          <div class="d-grid gap-2">
            <button class="btn btn-primary" @click="processTransfer" :disabled="!isPinValid">
              <v-icon start>mdi-send</v-icon>
              Transfer
              <v-icon size="16" class="ml-1">mdi-circle-multiple-outline</v-icon>
              {{ transferAmount.toFixed(0) }}
            </button>
          </div>
        </div>
        
        <!-- Step 4: Receipt (reuse from original transfer) -->
        <div v-if="transferStep === 4 && showReceipt">
          <div class="confirmation-screen">
            <div style="text-align: center; margin-bottom: 24px;">
              <div style="display: inline-block; padding: 8px 16px; border-radius: 20px; font-size: 14px; font-weight: 600; background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%); color: #2E7D32; border: 1px solid #4CAF50;">
                <v-icon size="16" color="success">mdi-check-circle</v-icon>
                Transfer Completed
              </div>
              <h2 style="font-size: 22px; font-weight: 700; color: var(--text-dark); margin: 16px 0 4px 0;">Transfer Successful</h2>
              <p style="color: var(--muted); font-size: 14px; font-weight: 500;">{{ receipt.date }}</p>
            </div>
            
            <div style="font-size: 36px; font-weight: 700; color: var(--primary); margin: 24px 0; text-align: center; display: flex; align-items: center; justify-content: center; gap: 8px;">
              <v-icon size="32" color="#1A3C28">mdi-circle-multiple-outline</v-icon>
              {{ receipt.amount.toFixed(0) }}
            </div>
            
            <div>
              <div class="confirmation-item">
                <div class="confirmation-label">Transaction ID</div>
                <div class="confirmation-value">{{ receipt.transactionId }}</div>
              </div>
              
              <div class="confirmation-item">
                <div class="confirmation-label">Recipient</div>
                <div class="confirmation-value">{{ receipt.recipientName }}</div>
              </div>
              
              <div class="confirmation-item">
                <div class="confirmation-label">Membership ID</div>
                <div class="confirmation-value">{{ receipt.recipientId }}</div>
              </div>
              
              <div class="confirmation-item">
                <div class="confirmation-label">Transfer Method</div>
                <div class="confirmation-value">QR Scan</div>
              </div>
              
              <div class="confirmation-item">
                <div class="confirmation-label">New Balance</div>
                <div class="confirmation-value">
                  <v-icon size="16" color="#2E7D32">mdi-circle-multiple-outline</v-icon>
                  {{ formattedBalance }}
                </div>
              </div>
            </div>
            
            <div style="display: flex; gap: 16px; margin-top: 24px;">
              <button class="btn btn-primary" @click="completeTransfer" style="flex: 1;">
                Done
              </button>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Notification Toast -->
      <transition name="fade">
        <div class="notification-toast" 
             :class="{ 'error-notification': isErrorNotification }" 
             v-if="showNotification">
          
          <!-- Close button for error notifications -->
          <button v-if="isErrorNotification" 
                  class="error-close-button" 
                  @click="closeErrorNotification">
            <v-icon>mdi-close</v-icon>
          </button>
          
          {{ notificationMessage }}
        </div>
      </transition>
    </div>
  `
}