import getTropicalStyles from "../css/tropicalBookingStyles.js";

export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    // Calculate date ranges for filtering
    const nextWeek = new Date(today);
    nextWeek.setDate(today.getDate() + 7);
    const nextMonth = new Date(today);
    nextMonth.setMonth(today.getMonth() + 1);

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',      // Dark green
        secondary: '#DDA333',    // Golden
        accent: '#A3BA95',       // Light sage green
        lightBg: '#F8F5F0',      // Off-white
        creamBg: '#EADDCB',      // Cream
        darkText: '#1E1E1E',     // Almost black
        lightText: '#FFFFFF',    // White
        error: '#B71C1C',        // Red
        success: '#2E7D32',      // Green
        warning: '#FF8F00',      // Orange
        muted: '#78909C',        // Grayish blue
        brown: '#6D4C41',        // Brown
      },

      // Display options
      viewMode: 'grid',         // 'grid' or 'list'
      activeTab: 'upcoming',    // 'upcoming', 'featured', 'my-events'
      selectedEvent: null,      // Currently selected event
      searchQuery: '',          // Search query
      showFilters: false,       // Mobile filter visibility toggle
      loading: false,

      // Date related
      today,
      nextWeek,
      nextMonth,
      selectedDate: null,
      dateRangeFilter: 'all',    // 'today', 'week', 'month', 'all'

      // Filters
      filters: {
        categories: [],
        priceRange: [0, 1000],
        indoorOutdoor: 'all',    // 'indoor', 'outdoor', 'all'
        featuredOnly: false,
        familyFriendly: false,
        accessibility: false,
      },

      // Filter options (for selects and checkboxes)
      filterOptions: {
        categories: [
          { title: 'Cultural', value: 'cultural', icon: 'mdi-theater', color: '#9C27B0' },
          { title: 'Adventure', value: 'adventure', icon: 'mdi-kayaking', color: '#FF5722' },
          { title: 'Wellness', value: 'wellness', icon: 'mdi-yoga', color: '#8BC34A' },
          { title: 'Educational', value: 'educational', icon: 'mdi-school', color: '#03A9F4' },
          { title: 'Culinary', value: 'culinary', icon: 'mdi-food-fork-drink', color: '#E91E63' },
          { title: 'Entertainment', value: 'entertainment', icon: 'mdi-music', color: '#673AB7' },
          { title: 'Nature', value: 'nature', icon: 'mdi-pine-tree', color: '#4CAF50' }
        ],
        priceRanges: [
          { title: 'All Prices', value: [0, 1000] },
          { title: 'Free Events', value: [0, 0] },
          { title: 'Under RM 50', value: [1, 50] },
          { title: 'RM 50 - RM 200', value: [50, 200] },
          { title: 'RM 200+', value: [200, 1000] }
        ],
        indoorOutdoor: [
          { title: 'All Locations', value: 'all' },
          { title: 'Indoor Only', value: 'indoor' },
          { title: 'Outdoor Only', value: 'outdoor' }
        ],
        dateRanges: [
          { title: 'All Upcoming', value: 'all', icon: 'mdi-calendar-month' },
          { title: 'Today', value: 'today', icon: 'mdi-calendar-today' },
          { title: 'This Week', value: 'week', icon: 'mdi-calendar-week' },
          { title: 'This Month', value: 'month', icon: 'mdi-calendar-month-outline' }
        ]
      },

      // Events data
      events: [],
      featuredEvents: [],
      savedEvents: [],
      totalEvents: 0,

      // My events - bookmarked/favorited
      myEvents: [
        'evt001', 'evt003', 'evt008'
      ],

      // Event detail dialog
      showEventDetail: false,
      eventRegistrationForm: {
        attendees: 1,
        specialRequests: ''
      },
      showRegistrationSuccess: false,

      // Snackbar
      snackbar: {
        show: false,
        text: '',
        color: '',
        timeout: 3000
      }
    };
  },

  computed: {
    /**
     * Filtered events based on all active filters
     */
    filteredEvents() {
      return this.events;
    },

    /**
     * Check if there are any active filters
     */
    hasActiveFilters() {
      return (
        this.searchQuery.trim() !== '' ||
        this.filters.categories.length > 0 ||
        this.dateRangeFilter !== 'all' ||
        this.filters.priceRange[0] > 0 ||
        this.filters.priceRange[1] < 1000 ||
        this.filters.indoorOutdoor !== 'all' ||
        this.filters.featuredOnly ||
        this.filters.familyFriendly ||
        this.filters.accessibility
      );
    },

    /**
     * Get saved events
     */
    myEvents() {
      return this.savedEvents;
    },

    /**
     * Get a summary of active filters for display
     */
    activeFiltersSummary() {
      const filters = [];

      if (this.searchQuery.trim()) {
        filters.push(`"${this.searchQuery}"`);
      }

      if (this.filters.categories.length > 0) {
        const categoryNames = this.filters.categories.map(cat =>
          this.filterOptions.categories.find(c => c.value === cat)?.title || cat
        );
        filters.push(categoryNames.join(', '));
      }

      if (this.dateRangeFilter !== 'all') {
        const dateRange = this.filterOptions.dateRanges.find(d => d.value === this.dateRangeFilter);
        if (dateRange) {
          filters.push(dateRange.title);
        }
      }

      if (this.filters.priceRange[0] > 0 || this.filters.priceRange[1] < 1000) {
        filters.push(`RM ${this.filters.priceRange[0]} - RM ${this.filters.priceRange[1]}`);
      }

      if (this.filters.indoorOutdoor !== 'all') {
        filters.push(this.filters.indoorOutdoor === 'indoor' ? 'Indoor' : 'Outdoor');
      }

      if (this.filters.featuredOnly) {
        filters.push('Featured');
      }

      if (this.filters.familyFriendly) {
        filters.push('Family-Friendly');
      }

      if (this.filters.accessibility) {
        filters.push('Accessible');
      }

      if (filters.length === 0) {
        return 'All Events';
      }

      return filters.join(' • ');
    }
  },

  methods: {
    // Fetch events from API
    async fetchEvents() {
      this.loading = true;
      try {
        // Prepare filter parameters
        const params = {
          user_id: sessionStorage.getItem('id') || '',
          category: this.filters.categories.length > 0 ? this.filters.categories.join(',') : '',
          featured: this.filters.featuredOnly ? 'true' : '',
          date_filter: this.dateRangeFilter,
          indoor: this.filters.indoorOutdoor === 'indoor' ? 'true' :
            this.filters.indoorOutdoor === 'outdoor' ? 'false' : '',
          family_friendly: this.filters.familyFriendly ? 'true' : '',
          accessible: this.filters.accessibility ? 'true' : '',
          price_min: this.filters.priceRange[0],
          price_max: this.filters.priceRange[1],
          search: this.searchQuery,
          limit: 50,
          offset: 0
        };

        // Call the API
        const response = await this.$root.callApi('event', 'get_events', params);

        if (response && response.data && response.data.status === "YES") {
          // Process the response
          this.events = this.processEvents(response.data.events || []);
          this.featuredEvents = this.processEvents(response.data.featured_events || []);
          this.savedEvents = response.data.saved_events || [];
          this.totalEvents = response.data.total_records || 0;

          // Update filter options if provided by API
          if (response.data.filter_options && response.data.filter_options.categories) {
            this.filterOptions.categories = response.data.filter_options.categories;
          }
        } else {
          console.error("Error fetching events:", response?.data?.message || "Unknown error");
        }
      } catch (error) {
        console.error("Failed to fetch events:", error);
      } finally {
        this.loading = false;
      }
    },

    // Process events from API to match the expected format
    processEvents(apiEvents) {
      return apiEvents.map(event => {
        // Parse date strings into Date objects
        let eventDate = null;
        let eventEndDate = null;

        if (event.date) {
          const dateParts = event.date.split(', ');
          if (dateParts.length > 1) {
            const monthDay = dateParts[0].split(' ');
            const year = parseInt(dateParts[1]);
            const month = this.getMonthNumber(monthDay[0]);
            const day = parseInt(monthDay[1]);

            eventDate = new Date(year, month, day);

            // If time is provided, set the hours and minutes
            if (event.time) {
              const timeParts = event.time.split(' - ');
              if (timeParts.length > 0) {
                const startTime = this.parseTime(timeParts[0]);
                eventDate.setHours(startTime.hours, startTime.minutes);

                // Set end date/time if available
                if (timeParts.length > 1) {
                  eventEndDate = new Date(eventDate);
                  const endTime = this.parseTime(timeParts[1]);
                  eventEndDate.setHours(endTime.hours, endTime.minutes);
                }
              }
            }
          }
        }

        // Return processed event object
        return {
          id: event.id,
          title: event.title,
          description: event.description,
          longDescription: event.longDescription || event.description,
          category: event.category,
          image: event.image,
          date: eventDate,
          endDate: eventEndDate,
          location: event.location,
          price: parseFloat(event.price),
          featured: event.featured,
          rsvp: true, // Assuming all events require RSVP
          availableSpots: event.availability?.total || null,
          bookedSpots: event.availability?.total - event.availability?.available || null,
          indoor: event.indoor,
          familyFriendly: event.familyFriendly,
          accessibility: event.accessible,
          host: event.host || 'TBA',
          tags: event.tags || [event.category],
          saved: event.saved || false
        };
      });
    },

    // Helper method to convert month name to number (0-11)
    getMonthNumber(monthName) {
      const months = {
        'January': 0, 'February': 1, 'March': 2, 'April': 3, 'May': 4, 'June': 5,
        'July': 6, 'August': 7, 'September': 8, 'October': 9, 'November': 10, 'December': 11
      };
      return months[monthName] || 0;
    },

    // Helper method to parse time string (e.g., "7:30 PM")
    parseTime(timeStr) {
      const [timePart, ampm] = timeStr.split(' ');
      let [hours, minutes] = timePart.split(':').map(Number);

      if (ampm === 'PM' && hours < 12) {
        hours += 12;
      } else if (ampm === 'AM' && hours === 12) {
        hours = 0;
      }

      return { hours, minutes };
    },

    // Apply filters and refresh events
    applyFilters() {
      this.fetchEvents();
    },

    // Reset all filters to default values
    resetFilters() {
      this.filters = {
        categories: [],
        priceRange: [0, 1000],
        indoorOutdoor: 'all',
        featuredOnly: false,
        familyFriendly: false,
        accessibility: false,
      };
      this.dateRangeFilter = 'all';
      this.searchQuery = '';
      this.fetchEvents();
    },

    // Toggle saved status for an event
    async toggleSaveEvent(eventId) {
      try {
        const userId = sessionStorage.getItem('id');
        if (!userId) {
          console.error("User not logged in");
          return;
        }

        // Find if event is already saved
        const isSaved = this.savedEvents.includes(eventId);

        // Call API to save/unsave event
        const response = await this.$root.callApi('event', isSaved ? 'unsave_event' : 'save_event', {
          user_id: userId,
          event_id: eventId
        });

        if (response && response.data && response.data.status === "YES") {
          // Update local saved events list
          if (isSaved) {
            this.savedEvents = this.savedEvents.filter(id => id !== eventId);
          } else {
            this.savedEvents.push(eventId);
          }

          // Update saved status in events arrays
          this.events = this.events.map(event => {
            if (event.id === eventId) {
              return { ...event, saved: !isSaved };
            }
            return event;
          });

          this.featuredEvents = this.featuredEvents.map(event => {
            if (event.id === eventId) {
              return { ...event, saved: !isSaved };
            }
            return event;
          });
        }
      } catch (error) {
        console.error("Failed to toggle save event:", error);
      }
    },

    /**
     * Format date for display
     */
    formatDate(date) {
      if (!date) return '';

      const options = { weekday: 'short', month: 'short', day: 'numeric' };
      return date.toLocaleDateString('en-US', options);
    },

    /**
     * Format time for display
     */
    formatTime(date) {
      if (!date) return '';

      const options = { hour: 'numeric', minute: '2-digit', hour12: true };
      return date.toLocaleTimeString('en-US', options);
    },

    /**
     * Format date range for display
     */
    formatDateTimeRange(startDate, endDate) {
      const sameDay = startDate.getDate() === endDate.getDate() &&
        startDate.getMonth() === endDate.getMonth() &&
        startDate.getFullYear() === endDate.getFullYear();

      const startDateStr = this.formatDate(startDate);
      const startTimeStr = this.formatTime(startDate);
      const endTimeStr = this.formatTime(endDate);

      if (sameDay) {
        return `${startDateStr}, ${startTimeStr} - ${endTimeStr}`;
      } else {
        const endDateStr = this.formatDate(endDate);
        return `${startDateStr}, ${startTimeStr} - ${endDateStr}, ${endTimeStr}`;
      }
    },

    /**
     * Relative time (e.g. "Today", "Tomorrow", "In 3 days")
     */
    getRelativeTime(date) {
      const today = new Date();
      today.setHours(0, 0, 0, 0);

      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      const eventDate = new Date(date);
      eventDate.setHours(0, 0, 0, 0);

      const diffTime = eventDate - today;
      const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

      if (diffDays === 0) {
        return { text: 'Today', color: this.theme.success };
      } else if (diffDays === 1) {
        return { text: 'Tomorrow', color: this.theme.success };
      } else if (diffDays > 1 && diffDays <= 7) {
        return { text: `In ${diffDays} days`, color: this.theme.secondary };
      } else if (diffDays > 7) {
        return { text: this.formatDate(date), color: this.theme.primary };
      } else {
        return { text: 'Past Event', color: this.theme.muted };
      }
    },

    /**
     * Get availability status
     */
    getAvailabilityStatus(event) {
      if (!event.rsvp || !event.availableSpots) {
        return { text: 'No RSVP Required', color: this.theme.muted, icon: 'mdi-information-outline' };
      }

      const spotsLeft = event.availableSpots - event.bookedSpots;

      if (spotsLeft === 0) {
        return { text: 'Fully Booked', color: this.theme.error, icon: 'mdi-close-circle-outline' };
      } else if (spotsLeft <= 5) {
        return { text: `${spotsLeft} spots left`, color: this.theme.warning, icon: 'mdi-alert-outline' };
      } else {
        return { text: `${spotsLeft} spots available`, color: this.theme.success, icon: 'mdi-check-circle-outline' };
      }
    },

    /**
     * Get category details
     */
    getCategoryDetails(categoryId) {
      const category = this.filterOptions.categories.find(c => c.value === categoryId);
      if (!category) {
        // Default if not found
        return { title: categoryId, color: this.theme.primary, icon: 'mdi-tag' };
      }
      return category;
    },

    /**
     * View event details
     */
    viewEventDetails(event) {
      this.selectedEvent = event;
      this.showEventDetail = true;
      this.eventRegistrationForm = {
        attendees: 1,
        specialRequests: ''
      };
      this.showRegistrationSuccess = false;
    },

    /**
     * Toggle favorite/bookmark event
     */
    toggleFavorite(eventId) {
      this.toggleSaveEvent(eventId);

      const isSaved = this.savedEvents.includes(eventId);
      if (isSaved) {
        this.showSnackbarMessage('Event added to your list', 'success');
      } else {
        this.showSnackbarMessage('Event removed from your list', 'info');
      }
    },

    /**
     * Register for an event
     */
    registerForEvent() {
      this.loading = true;

      // Simulate API call
      setTimeout(() => {
        this.loading = false;
        this.showRegistrationSuccess = true;

        // Also add to my events
        if (!this.myEvents.includes(this.selectedEvent.id)) {
          this.myEvents.push(this.selectedEvent.id);
        }
      }, 1500);
    },

    /**
     * Clear all filters
     */
    clearFilters() {
      this.searchQuery = '';
      this.filters = {
        categories: [],
        priceRange: [0, 1000],
        indoorOutdoor: 'all',
        featuredOnly: false,
        familyFriendly: false,
        accessibility: false,
      };
      this.dateRangeFilter = 'all';

      this.showSnackbarMessage('All filters cleared', 'info');
      this.fetchEvents();
    },

    /**
     * Apply price range filter
     */
    applyPriceRangeFilter(range) {
      this.filters.priceRange = range;
      this.fetchEvents();
    },

    /**
     * Display snackbar message
     */
    showSnackbarMessage(text, color = 'success') {
      this.snackbar.text = text;
      this.snackbar.color = color;
      this.snackbar.show = true;
    },

    /**
     * Navigate to a specific tab
     */
    navigateToTab(tab) {
      this.activeTab = tab;
    }
  },

  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');

    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);

    this.fetchEvents();
  },

  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent :scrim="theme.primary" :opacity="0.8">
        <div class="text-center">
          <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
          <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Processing your request...</div>
        </div>
      </v-overlay>

      <!-- Navigation Tabs -->
      <v-tabs v-model="activeTab" :bg-color="theme.primary" grow slider-color="secondary" class="booking-tabs-nav" dark>
        <v-tab value="upcoming" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'upcoming'}">
          <v-icon start>mdi-calendar-month</v-icon> Upcoming Events
        </v-tab>
        <v-tab value="featured" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'featured'}">
          <v-icon start>mdi-star</v-icon> Featured
        </v-tab>
        <!--<v-tab value="my-events" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'my-events'}">
          <v-icon start>mdi-bookmark</v-icon> My Saved
        </v-tab>-->
      </v-tabs>

      <v-window v-model="activeTab" class="tropical-booking-main-content">
        <!-- UPCOMING EVENTS TAB -->
        <v-window-item value="upcoming" class="pa-sm-3 pa-2">
          <!-- Featured Events Carousel - Top Section -->
          <div class="mb-4">
            <h2 class="text-h5 mb-3 ml-2" :style="{color: theme.primary}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-star-circle</v-icon>
              Featured Events
            </h2>
            
            <v-carousel
              cycle
              :show-arrows="true"
              height="280"
              hide-delimiter-background
              :delimiter-icon="'mdi-circle-small'"
              :show-arrows="'hover'"
            >
              <v-carousel-item
                v-for="event in featuredEvents"
                :key="event.id"
                @click="viewEventDetails(event)"
                class="featured-event-slide"
              >
                <v-img
                  :src="event.image"
                  height="280"
                  cover
                  class="align-end"
                  gradient="to bottom, rgba(0,0,0,0.1), rgba(0,0,0,0.6)"
                >
                  <div class="pa-4">
                    <v-chip 
                      class="mb-2" 
                      size="small" 
                      label
                      variant="flat"
                      :text-color="theme.primary"
                      style="background-color: rgba(255,255,255,0.95) !important;"
                    >
                      <v-icon start size="small" :color="getCategoryDetails(event.category).color">{{ getCategoryDetails(event.category).icon }}</v-icon>
                      {{ getCategoryDetails(event.category).title }}
                    </v-chip>
                    
                    <h3 class="text-h5 font-weight-bold mb-1" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                      {{ event.title }}
                    </h3>
                    
                    <div class="d-flex align-center mb-2">
                      <v-icon :color="theme.lightText" size="small" class="mr-1">mdi-calendar</v-icon>
                      <span class="text-body-2" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                        {{ formatDateTimeRange(event.date, event.endDate) }}
                      </span>
                    </div>
                    
                    <v-btn
                      color="secondary"
                      size="small"
                      class="text-none"
                      variant="flat"
                    >
                      Learn More
                      <v-icon end>mdi-arrow-right</v-icon>
                    </v-btn>
                  </div>
                </v-img>
              </v-carousel-item>
            </v-carousel>
          </div>
          
          <!-- Search Bar & Filters -->
          <v-card class="content-card mb-4" elevation="6">
            <div class="d-flex flex-wrap align-center pa-2 gap-2">
              <!-- Search Field -->
              <v-text-field
                v-model="searchQuery"
                placeholder="Search events..."
                :color="theme.primary"
                prepend-inner-icon="mdi-magnify"
                variant="outlined"
                density="compact"
                hide-details
                class="mx-2 flex-grow-1 my-2"
                style="min-width: 200px;"
              ></v-text-field>
              
              <!-- Date Range Filter -->
              <v-btn-toggle
                v-model="dateRangeFilter"
                density="comfortable"
                color="primary"
                class="flex-grow-0 flex-shrink-0 my-2 mx-2 d-none d-md-flex"
              >
                <v-btn
                  v-for="option in filterOptions.dateRanges"
                  :key="option.value"
                  :value="option.value"
                  :text="option.title"
                >
                  <v-icon start>{{ option.icon }}</v-icon>
                  {{ option.title }}
                </v-btn>
              </v-btn-toggle>
              
              <!-- Mobile Date Filter -->
              <v-select
                v-model="dateRangeFilter"
                :items="filterOptions.dateRanges"
                item-title="title"
                item-value="value"
                label="Date"
                variant="outlined"
                density="compact"
                hide-details
                class="mx-2 my-2 d-md-none"
                style="max-width: 130px;"
              >
                <template v-slot:prepend-inner>
                  <v-icon size="small">mdi-calendar</v-icon>
                </template>
              </v-select>
              
              <!-- Filters Button -->
              <v-btn
                :color="hasActiveFilters ? theme.secondary : theme.muted"
                class="mx-2 my-2 d-flex d-md-none"
                @click="showFilters = !showFilters"
              >
                <v-badge
                  :content="Object.values(filters).flat().filter(v => Array.isArray(v) ? v[0] > 0 || v[1] < 1000 : Boolean(v)).length + (dateRangeFilter !== 'all' ? 1 : 0)"
                  :value="hasActiveFilters"
                  :color="theme.primary"
                  location="top start"
                >
                  <v-icon>mdi-filter-variant</v-icon>
                </v-badge>
                Filters
              </v-btn>
              
              <!-- View Toggle -->
              <div class="d-flex align-center mx-2 my-2">
                <v-btn-toggle
                  v-model="viewMode"
                  mandatory
                  color="primary"
                  density="comfortable"
                  rounded="lg"
                >
                  <v-btn value="grid" variant="text">
                    <v-icon>mdi-grid</v-icon>
                  </v-btn>
                  <v-btn value="list" variant="text">
                    <v-icon>mdi-format-list-bulleted</v-icon>
                  </v-btn>
                </v-btn-toggle>
              </div>
            </div>
            
            <!-- Mobile Filters Expansion Panel -->
            <v-expand-transition>
              <div v-if="showFilters" class="px-4 pb-4 pt-2">
                <v-divider class="mb-4"></v-divider>
                
                <v-row dense class="justify-center mobile-filter-row" style="width: 100%; margin: 0;">
                  <!-- Categories Filter -->
                  <v-col cols="12" class="mb-3">
                    <div class="text-subtitle-2 mb-3">Categories</div>
                    <v-chip-group
                      v-model="filters.categories"
                      column
                      multiple
                      selected-class="selected-chip"
                      class="filter-chip-group"
                    >
                      <v-chip
                        v-for="category in filterOptions.categories"
                        :key="category.value"
                        :value="category.value"
                        filter
                        :color="category.color"
                        text-color="white"
                        variant="flat"
                        class="ma-1"
                      >
                        <v-icon start size="small">{{ category.icon }}</v-icon>
                        {{ category.title }}
                      </v-chip>
                    </v-chip-group>
                  </v-col>
                  
                  <!-- Price Range Filter -->
                  <v-col cols="12" sm="6" class="mb-3">
                    <div class="text-subtitle-2 mb-3">Price Range</div>
                    <v-chip-group
                      column
                      selected-class="selected-chip"
                      class="filter-chip-group"
                    >
                      <v-chip
                        v-for="(range, index) in filterOptions.priceRanges"
                        :key="index"
                        filter
                        :color="JSON.stringify(filters.priceRange) === JSON.stringify(range.value) ? theme.primary : undefined"
                        :text-color="JSON.stringify(filters.priceRange) === JSON.stringify(range.value) ? 'white' : undefined"
                        @click="applyPriceRangeFilter(range.value)"
                        class="ma-1"
                      >
                        {{ range.title }}
                      </v-chip>
                    </v-chip-group>
                  </v-col>
                  
                  <!-- Indoor/Outdoor Filter -->
                  <v-col cols="12" sm="6" class="mb-3">
                    <div class="text-subtitle-2 mb-3">Location</div>
                    <v-chip-group
                      v-model="filters.indoorOutdoor"
                      mandatory
                      selected-class="selected-chip"
                      class="filter-chip-group"
                    >
                      <v-chip
                        v-for="option in filterOptions.indoorOutdoor"
                        :key="option.value"
                        :value="option.value"
                        filter
                        class="ma-1"
                      >
                        {{ option.title }}
                      </v-chip>
                    </v-chip-group>
                  </v-col>
                  
                  <!-- Additional Filters -->
                  <v-col cols="12" class="mb-2">
                    <v-checkbox
                      v-model="filters.featuredOnly"
                      label="Featured Events Only"
                      :color="theme.primary"
                      hide-details
                      class="mt-2 mb-2"
                      density="compact"
                    ></v-checkbox>
                    
                    <v-checkbox
                      v-model="filters.familyFriendly"
                      label="Family-Friendly"
                      :color="theme.primary"
                      hide-details
                      class="mt-2 mb-2"
                      density="compact"
                    ></v-checkbox>
                    
                    <v-checkbox
                      v-model="filters.accessibility"
                      label="Accessible"
                      :color="theme.primary"
                      hide-details
                      class="mt-2 mb-4"
                      density="compact"
                    ></v-checkbox>
                    
                    <v-btn
                      v-if="hasActiveFilters"
                      size="small"
                      :color="theme.muted"
                      variant="outlined"
                      prepend-icon="mdi-filter-remove"
                      @click="clearFilters"
                      class="mt-2"
                    >
                      Clear Filters
                    </v-btn>
                  </v-col>
                </v-row>
              </div>
            </v-expand-transition>
          </v-card>
          
          <!-- Desktop Side Filters -->
          <v-row>
            <v-col cols="12" md="3" class="d-none d-md-block">
              <v-card class="content-card" elevation="6">
                <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
                  <v-icon start :color="theme.secondary">mdi-filter-variant</v-icon>
                  Filters
                </v-card-title>
                
                <v-card-text class="pa-4">
                  <!-- Categories Filter -->
                  <div class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Categories</div>
                  <v-chip-group
                    v-model="filters.categories"
                    column
                    multiple
                    selected-class="selected-chip"
                  >
                    <v-chip
                      v-for="category in filterOptions.categories"
                      :key="category.value"
                      :value="category.value"
                      filter
                      :color="category.color"
                      text-color="white"
                      variant="flat"
                    >
                      <v-icon start size="small">{{ category.icon }}</v-icon>
                      {{ category.title }}
                    </v-chip>
                  </v-chip-group>
                  
                  <v-divider class="my-3"></v-divider>
                  
                  <!-- Price Range Filter -->
                  <div class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Price Range</div>
                  <v-range-slider
                    v-model="filters.priceRange"
                    :max="1000"
                    :min="0"
                    :step="10"
                    :color="theme.primary"
                    :thumb-color="theme.secondary"
                    :track-color="theme.accent"
                    thumb-label="always"
                  >
                    <template v-slot:prepend>
                      <div :style="{color: theme.primary}">RM {{ filters.priceRange[0] }}</div>
                    </template>
                    
                    <template v-slot:append>
                      <div :style="{color: theme.primary}">
                        {{ filters.priceRange[1] === 1000 ? 'RM 1000+' : 'RM ' + filters.priceRange[1] }}
                      </div>
                    </template>
                  </v-range-slider>
                  
                  <div class="mt-2 mb-3">
                    <v-chip-group>
                      <v-chip
                        v-for="(range, index) in filterOptions.priceRanges"
                        :key="index"
                        size="small"
                        filter
                        :color="JSON.stringify(filters.priceRange) === JSON.stringify(range.value) ? theme.primary : undefined"
                        :text-color="JSON.stringify(filters.priceRange) === JSON.stringify(range.value) ? 'white' : undefined"
                        @click="applyPriceRangeFilter(range.value)"
                      >
                        {{ range.title }}
                      </v-chip>
                    </v-chip-group>
                  </div>
                  
                  <v-divider class="my-3"></v-divider>
                  
                  <!-- Location Filter -->
                  <div class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Location</div>
                  <v-btn-toggle
                    v-model="filters.indoorOutdoor"
                    mandatory
                    divided
                    :color="theme.primary"
                    class="w-100"
                  >
                    <v-btn
                      v-for="option in filterOptions.indoorOutdoor"
                      :key="option.value"
                      :value="option.value"
                      class="flex-grow-1"
                    >
                      {{ option.title }}
                    </v-btn>
                  </v-btn-toggle>
                  
                  <v-divider class="my-3"></v-divider>
                  
                  <!-- Additional Filters -->
                  <div class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Additional Filters</div>
                  
                  <v-checkbox
                    v-model="filters.featuredOnly"
                    label="Featured Events Only"
                    :color="theme.primary"
                    hide-details
                    class="mt-1"
                  >
                    <template v-slot:prepend>
                      <v-icon :color="theme.secondary">mdi-star</v-icon>
                    </template>
                  </v-checkbox>
                  
                  <v-checkbox
                    v-model="filters.familyFriendly"
                    label="Family-Friendly"
                    :color="theme.primary"
                    hide-details
                    class="mt-1"
                  >
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-human-male-child</v-icon>
                    </template>
                  </v-checkbox>
                  
                  <v-checkbox
                    v-model="filters.accessibility"
                    label="Accessible"
                    :color="theme.primary"
                    hide-details
                    class="mt-1 mb-3"
                  >
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-wheelchair-accessibility</v-icon>
                    </template>
                  </v-checkbox>
                  
                  <v-btn
                    v-if="hasActiveFilters"
                    block
                    class="mt-4"
                    :color="theme.muted"
                    prepend-icon="mdi-filter-remove"
                    @click="clearFilters"
                  >
                    Clear All Filters
                  </v-btn>
                </v-card-text>
              </v-card>
            </v-col>
            
            <!-- Events Grid/List -->
            <v-col cols="12" md="9">
              <div v-if="hasActiveFilters" class="mb-4 px-2">
                <v-chip
                  closable
                  label
                  :color="theme.accent + '40'"
                  size="large"
                  @click:close="clearFilters"
                  class="filter-summary-chip"
                >
                  <template v-slot:prepend>
                    <v-icon :color="theme.primary">mdi-filter</v-icon>
                  </template>
                  <span class="filter-summary-text" :style="{color: theme.primary}">{{ activeFiltersSummary }}</span>
                </v-chip>
              </div>
              
              <div v-if="filteredEvents.length === 0" class="empty-events-state text-center pa-5">
                <v-icon size="64" :color="theme.accent" class="mb-3">mdi-calendar-search</v-icon>
                <h3 class="text-h5 mb-2" :style="{color: theme.primary}">No Events Found</h3>
                <p class="text-body-1 mb-4" :style="{color: theme.muted}">
                  Try adjusting your filters or search criteria
                </p>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  prepend-icon="mdi-filter-remove"
                  @click="clearFilters"
                >
                  Clear Filters
                </v-btn>
              </div>
              
              <!-- Grid View -->
              <v-row v-else-if="viewMode === 'grid'" class="justify-center" style="width: 100%; margin: 0;">
                <v-col
                  v-for="event in filteredEvents"
                  :key="event.id"
                  cols="12"
                  sm="6"
                  lg="4"
                  class=" pa-2"
                >
                  <v-card
                    class="event-card d-flex flex-column"
                    elevation="3"
                    @click="viewEventDetails(event)"
                    :ripple="true"
                  >
                    <div class="position-relative">
                      <v-img
                        :src="event.image"
                        height="180"
                        cover
                        :class="{'past-event-img': event.date < today}"
                        class="event-card-img"
                      >
                        <div class="event-card-overlay">
                          <div class="d-flex align-center justify-space-between pa-2">
                            <v-chip
                              size="small"
                              :text-color="theme.primary"
                              label
                              variant="flat"
                              class="mb-2"
                              style="background-color: rgba(255,255,255,0.95) !important;"
                            >
                              <v-icon start size="small" :color="getCategoryDetails(event.category).color">{{ getCategoryDetails(event.category).icon }}</v-icon>
                              {{ getCategoryDetails(event.category).title }}
                            </v-chip>
                          
                            <v-btn
                            icon="mdi-bookmark"
                            :icon="savedEvents.includes(event.id) ? 'mdi-bookmark' : 'mdi-bookmark-outline'"
                            :color="savedEvents.includes(event.id) ? theme.secondary : 'white'"
                            size="small"
                            variant="text"
                            @click.stop="toggleFavorite(event.id)"
                            class="mb-2"
                          ></v-btn>
                          </div>
                        </div>
                        
                        <template v-slot:placeholder>
                          <div class="d-flex align-center justify-center fill-height">
                            <v-progress-circular
                              :color="theme.primary"
                              indeterminate
                            ></v-progress-circular>
                          </div>
                        </template>
                      </v-img>
                      
                      <!-- Event timing chip -->
                      <v-chip
                        :color="getRelativeTime(event.date).color"
                        text-color="white"
                        class="event-timing-chip"
                        size="small"
                      >
                        <v-icon size="small" start>mdi-clock-outline</v-icon>
                        {{ getRelativeTime(event.date).text }}
                      </v-chip>
                    </div>
                    
                    <v-card-text class="flex-grow-1">
                      <div class="d-flex flex-column justify-space-between h-100">
                        <div>
                          <div class="d-flex align-center mb-1">
                            <v-icon :color="theme.primary" size="small" class="mr-1">mdi-calendar</v-icon>
                            <span class="text-caption text-truncate" :style="{color: theme.muted}">
                              {{ formatDateTimeRange(event.date, event.endDate) }}
                            </span>
                          </div>
                          
                          <h3 class="text-subtitle-1 font-weight-bold mb-1 event-title" :style="{color: theme.primary}">
                            {{ event.title }}
                          </h3>
                          
                          <div class="d-flex align-center mb-2">
                            <v-icon :color="theme.muted" size="small" class="mr-1">mdi-map-marker</v-icon>
                            <span class="text-caption text-truncate" :style="{color: theme.muted}">
                              {{ event.location }}
                            </span>
                          </div>
                          
                          <p class="text-caption event-description" :style="{color: theme.darkText}">
                            {{ event.description }}
                          </p>
                        </div>
                        
                        <div class="d-flex align-center justify-space-between mt-2">
                          <div class="d-flex align-center">
                            <v-icon
                              :color="getAvailabilityStatus(event).color"
                              size="small"
                              class="mr-1"
                            >
                              {{ getAvailabilityStatus(event).icon }}
                            </v-icon>
                            <span class="text-caption" :style="{color: getAvailabilityStatus(event).color}">
                              {{ getAvailabilityStatus(event).text }}
                            </span>
                          </div>
                          
                          <div>
                            <span class="text-subtitle-2 font-weight-bold" :style="{color: theme.primary}">
                              {{ event.price > 0 ? 'RM ' + event.price.toFixed(2) : 'Free' }}
                            </span>
                          </div>
                        </div>
                      </div>
                    </v-card-text>
                  </v-card>
                </v-col>
              </v-row>
              
              <!-- List View -->
              <v-card
                v-else
                class="content-card"
                elevation="6"
              >
                <v-list lines="two">
                  <v-list-item
                    v-for="event in filteredEvents"
                    :key="event.id"
                    @click="viewEventDetails(event)"
                    :ripple="true"
                    class="event-list-item"
                  >
                    <template v-slot:prepend>
                      <v-avatar
                        class="event-list-img"
                        rounded="lg"
                        size="80"
                      >
                        <v-img
                          :src="event.image"
                          cover
                          :class="{'past-event-img': event.date < today}"
                        >
                          <div class="d-flex justify-end pa-1">
                            <v-chip
                              size="x-small"
                              :text-color="theme.primary"
                              label
                              variant="flat"
                              style="background-color: rgba(255,255,255,0.95) !important;"
                            >
                              <v-icon size="x-small" class="mr-1" :color="getCategoryDetails(event.category).color">{{ getCategoryDetails(event.category).icon }}</v-icon>
                              {{ getCategoryDetails(event.category).title }}
                            </v-chip>
                          </div>
                        </v-img>
                      </v-avatar>
                    </template>
                    
                    <v-list-item-title class="font-weight-bold mb-1" :style="{color: theme.primary}">
                      {{ event.title }}
                      <v-chip
                        v-if="event.featured"
                        size="x-small"
                        :color="theme.secondary"
                        text-color="white"
                        label
                        class="ml-1"
                      >
                        <v-icon size="x-small" class="mr-1">mdi-star</v-icon>
                        Featured
                      </v-chip>
                    </v-list-item-title>
                    
                    <v-list-item-subtitle>
                      <div class="d-flex flex-column">
                        <div class="d-flex align-center">
                          <v-icon :color="theme.primary" size="x-small" class="mr-1">mdi-calendar-clock</v-icon>
                          <span class="text-caption" :style="{color: theme.muted}">
                            {{ formatDateTimeRange(event.date, event.endDate) }}
                          </span>
                          
                          <v-chip
                            size="x-small"
                            :color="getRelativeTime(event.date).color"
                            text-color="white"
                            class="ml-2"
                            label
                          >
                            {{ getRelativeTime(event.date).text }}
                          </v-chip>
                        </div>
                        
                        <div class="d-flex align-center mt-1">
                          <v-icon :color="theme.muted" size="x-small" class="mr-1">mdi-map-marker</v-icon>
                          <span class="text-caption" :style="{color: theme.muted}">
                            {{ event.location }}
                          </span>
                        </div>
                      </div>
                    </v-list-item-subtitle>
                    
                    <template v-slot:append>
                      <div class="d-flex flex-column align-end">
                        <span class="text-subtitle-2 font-weight-bold mb-1" :style="{color: theme.primary}">
                          {{ event.price > 0 ? 'RM ' + event.price.toFixed(2) : 'Free' }}
                        </span>
                        
                        <div class="d-flex align-center">
                          <span class="text-caption mr-2" :style="{color: getAvailabilityStatus(event).color}">
                            {{ getAvailabilityStatus(event).text }}
                          </span>
                          
                          <v-btn
                              :icon="savedEvents.includes(event.id) ? 'mdi-bookmark' : 'mdi-bookmark-outline'"
                              :color="savedEvents.includes(event.id) ? theme.secondary : theme.muted"
                              size="x-small"
                              variant="text"
                              @click.stop="toggleFavorite(event.id)"
                            ></v-btn>
                        </div>
                      </div>
                    </template>
                  </v-list-item>
                </v-list>
              </v-card>
            </v-col>
          </v-row>
        </v-window-item>
        
        <!-- FEATURED EVENTS TAB -->
        <v-window-item value="featured" class="pa-sm-4 pa-2">
          <h2 class="text-h5 mb-4" :style="{color: theme.primary}">
            <v-icon :color="theme.secondary" class="mr-2">mdi-star</v-icon>
            Featured Events
          </h2>
          
          <div v-if="featuredEvents.length === 0" class="empty-events-state text-center pa-5">
            <v-icon size="64" :color="theme.accent" class="mb-3">mdi-star-off</v-icon>
            <h3 class="text-h5 mb-2" :style="{color: theme.primary}">No Featured Events</h3>
            <p class="text-body-1" :style="{color: theme.muted}">
              Check back soon for exciting featured events!
            </p>
          </div>
          
          <v-row v-else>
            <!-- Large featured event -->
            <v-col v-if="featuredEvents.length > 0" cols="12">
              <v-card
                @click="viewEventDetails(featuredEvents[0])"
                class="featured-event-card"
                elevation="3"
                :ripple="true"
              >
                <v-img
                  :src="featuredEvents[0].image"
                  height="300"
                  cover
                  gradient="to bottom, rgba(0,0,0,0), rgba(0,0,0,0.7)"
                >
                  <div class="d-flex flex-column justify-end fill-height pa-4">
                    <div class="d-flex align-center mb-2">
                      <v-chip
                        :text-color="theme.primary"
                        label
                        variant="flat"
                        class="mr-2"
                        style="background-color: rgba(255,255,255,0.95) !important;"
                      >
                        <v-icon start size="small" :color="getCategoryDetails(featuredEvents[0].category).color">{{ getCategoryDetails(featuredEvents[0].category).icon }}</v-icon>
                        {{ getCategoryDetails(featuredEvents[0].category).title }}
                      </v-chip>
                      
                      <v-chip
                        :color="getRelativeTime(featuredEvents[0].date).color"
                        text-color="white"
                        class="mr-2"
                      >
                        <v-icon start size="small">mdi-clock-outline</v-icon>
                        {{ getRelativeTime(featuredEvents[0].date).text }}
                      </v-chip>
                      
                      <v-spacer></v-spacer>
                      
                      <v-btn
                        :icon="savedEvents.includes(featuredEvents[0].id) ? 'mdi-bookmark' : 'mdi-bookmark-outline'"
                        :color="savedEvents.includes(featuredEvents[0].id) ? theme.secondary : 'white'"
                        size="small"
                        variant="text"
                        @click.stop="toggleFavorite(featuredEvents[0].id)"
                      ></v-btn>
                    </div>
                    
                    <h2 class="text-h4 font-weight-bold mb-1" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                      {{ featuredEvents[0].title }}
                    </h2>
                    
                    <div class="d-flex align-center mb-2">
                      <v-icon :color="theme.lightText" size="small" class="mr-2">mdi-calendar</v-icon>
                      <span class="text-body-2" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                        {{ formatDateTimeRange(featuredEvents[0].date, featuredEvents[0].endDate) }}
                      </span>
                    </div>
                    
                    <div class="d-flex align-center mb-4">
                      <v-icon :color="theme.lightText" size="small" class="mr-2">mdi-map-marker</v-icon>
                      <span class="text-body-2" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                        {{ featuredEvents[0].location }}
                      </span>
                    </div>
                    
                    <div class="d-flex justify-space-between align-center">
                      <v-btn
                        color="secondary"
                        class="text-none"
                        variant="flat"
                      >
                        Learn More
                        <v-icon end>mdi-arrow-right</v-icon>
                      </v-btn>
                      
                      <span class="text-h6 font-weight-bold" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                        {{ featuredEvents[0].price > 0 ? 'RM ' + featuredEvents[0].price.toFixed(2) : 'Free' }}
                      </span>
                    </div>
                  </div>
                </v-img>
              </v-card>
            </v-col>
            
            <!-- Other featured events -->
            <v-col
              v-for="(event, index) in featuredEvents.slice(1)"
              :key="event.id"
              cols="12"
              sm="6"
              class=""
            >
              <v-card
                class="event-card d-flex flex-column"
                elevation="3"
                @click="viewEventDetails(event)"
                :ripple="true"
              >
                <div class="position-relative">
                  <v-img
                    :src="event.image"
                    height="180"
                    cover
                    :class="{'past-event-img': event.date < today}"
                    class="event-card-img"
                  >
                    <div class="event-card-overlay">
                      <div class="d-flex align-center justify-space-between pa-2">
                        <v-chip
                          size="small"
                          :text-color="theme.primary"
                          label
                          variant="flat"
                          class="mb-2"
                          style="background-color: rgba(255,255,255,0.95) !important;"
                        >
                          <v-icon start size="small" :color="getCategoryDetails(event.category).color">{{ getCategoryDetails(event.category).icon }}</v-icon>
                          {{ getCategoryDetails(event.category).title }}
                        </v-chip>
                      
                        <v-btn
                          icon="mdi-bookmark"
                          :icon="myEvents.includes(event.id) ? 'mdi-bookmark' : 'mdi-bookmark-outline'"
                          :color="myEvents.includes(event.id) ? theme.secondary : 'white'"
                          size="small"
                          variant="text"
                          @click.stop="toggleFavorite(event.id)"
                          class="mb-2"
                        ></v-btn>
                      </div>
                    </div>
                  </v-img>
                  
                  <!-- Event timing chip -->
                  <v-chip
                    :color="getRelativeTime(event.date).color"
                    text-color="white"
                    class="event-timing-chip"
                    size="small"
                  >
                    <v-icon size="small" start>mdi-clock-outline</v-icon>
                    {{ getRelativeTime(event.date).text }}
                  </v-chip>
                </div>
                
                <v-card-text class="flex-grow-1">
                  <div class="d-flex flex-column justify-space-between h-100">
                    <div>
                      <div class="d-flex align-center mb-1">
                        <v-icon :color="theme.primary" size="small" class="mr-1">mdi-calendar</v-icon>
                        <span class="text-caption text-truncate" :style="{color: theme.muted}">
                          {{ formatDateTimeRange(event.date, event.endDate) }}
                        </span>
                      </div>
                      
                      <h3 class="text-subtitle-1 font-weight-bold mb-1 event-title" :style="{color: theme.primary}">
                        {{ event.title }}
                      </h3>
                      
                      <div class="d-flex align-center mb-2">
                        <v-icon :color="theme.muted" size="small" class="mr-1">mdi-map-marker</v-icon>
                        <span class="text-caption text-truncate" :style="{color: theme.muted}">
                          {{ event.location }}
                        </span>
                      </div>
                      
                      <p class="text-caption event-description" :style="{color: theme.darkText}">
                        {{ event.description }}
                      </p>
                    </div>
                    
                    <div class="d-flex align-center justify-space-between mt-2">
                      <div class="d-flex align-center">
                        <v-icon
                          :color="getAvailabilityStatus(event).color"
                          size="small"
                          class="mr-1"
                        >
                          {{ getAvailabilityStatus(event).icon }}
                        </v-icon>
                        <span class="text-caption" :style="{color: getAvailabilityStatus(event).color}">
                          {{ getAvailabilityStatus(event).text }}
                        </span>
                      </div>
                      
                      <div>
                        <span class="text-subtitle-2 font-weight-bold" :style="{color: theme.primary}">
                          {{ event.price > 0 ? 'RM ' + event.price.toFixed(2) : 'Free' }}
                        </span>
                      </div>
                    </div>
                  </div>
                </v-card-text>
              </v-card>
            </v-col>
          </v-row>
        </v-window-item>
        
        <!-- MY EVENTS TAB -->
        <v-window-item value="my-events" class="pa-sm-4 pa-2">
          <h2 class="text-h5 mb-4" :style="{color: theme.primary}">
            <v-icon :color="theme.secondary" class="mr-2">mdi-bookmark-check</v-icon>
            My Saved Events
          </h2>
          
          <div v-if="myEvents.length === 0" class="empty-events-state text-center pa-5">
            <v-icon size="64" :color="theme.accent" class="mb-3">mdi-bookmark-off</v-icon>
            <h3 class="text-h5 mb-2" :style="{color: theme.primary}">No Saved Events</h3>
            <p class="text-body-1 mb-4" :style="{color: theme.muted}">
              Bookmark events you're interested in to find them here later
            </p>
            <v-btn
              :color="theme.primary"
              :text-color="theme.lightText"
              @click="navigateToTab('upcoming')"
            >
              <v-icon start>mdi-calendar-search</v-icon>
              Browse Events
            </v-btn>
          </div>
          
          <v-row v-else>
            <v-col
              v-for="eventId in myEvents"
              :key="eventId"
              cols="12"
              sm="6"
              md="4"
              class=""
            >
              <v-card
                class="event-card d-flex flex-column"
                elevation="3"
                @click="viewEventDetails(events.find(e => e.id === eventId))"
                :ripple="true"
                v-if="events.find(e => e.id === eventId)"
              >
                <div class="position-relative">
                  <v-img
                    :src="events.find(e => e.id === eventId).image"
                    height="180"
                    cover
                    :class="{'past-event-img': events.find(e => e.id === eventId).date < today}"
                    class="event-card-img"
                  >
                    <div class="event-card-overlay">
                      <div class="d-flex align-center justify-space-between pa-2">
                        <v-chip
                          size="small"
                          :text-color="theme.primary"
                          label
                          variant="flat"
                          class="mb-2"
                          style="background-color: rgba(255,255,255,0.95) !important;"
                        >
                          <v-icon start size="small" :color="getCategoryDetails(events.find(e => e.id === eventId).category).color">{{ getCategoryDetails(events.find(e => e.id === eventId).category).icon }}</v-icon>
                          {{ getCategoryDetails(events.find(e => e.id === eventId).category).title }}
                        </v-chip>
                      
                        <v-btn
                          icon="mdi-bookmark"
                          color="secondary"
                          size="small"
                          variant="text"
                          @click.stop="toggleFavorite(eventId)"
                          class="mb-2"
                        ></v-btn>
                      </div>
                    </div>
                  </v-img>
                  
                  <!-- Event timing chip -->
                  <v-chip
                    :color="getRelativeTime(events.find(e => e.id === eventId).date).color"
                    text-color="white"
                    class="event-timing-chip"
                    size="small"
                  >
                    <v-icon size="small" start>mdi-clock-outline</v-icon>
                    {{ getRelativeTime(events.find(e => e.id === eventId).date).text }}
                  </v-chip>
                </div>
                
                <v-card-text class="flex-grow-1">
                  <div class="d-flex flex-column justify-space-between h-100">
                    <div>
                      <div class="d-flex align-center mb-1">
                        <v-icon :color="theme.primary" size="small" class="mr-1">mdi-calendar</v-icon>
                        <span class="text-caption text-truncate" :style="{color: theme.muted}">
                          {{ formatDateTimeRange(events.find(e => e.id === eventId).date, events.find(e => e.id === eventId).endDate) }}
                        </span>
                      </div>
                      
                      <h3 class="text-subtitle-1 font-weight-bold mb-1 event-title" :style="{color: theme.primary}">
                        {{ events.find(e => e.id === eventId).title }}
                      </h3>
                      
                      <div class="d-flex align-center mb-2">
                        <v-icon :color="theme.muted" size="small" class="mr-1">mdi-map-marker</v-icon>
                        <span class="text-caption text-truncate" :style="{color: theme.muted}">
                          {{ events.find(e => e.id === eventId).location }}
                        </span>
                      </div>
                      
                      <p class="text-caption event-description" :style="{color: theme.darkText}">
                        {{ events.find(e => e.id === eventId).description }}
                      </p>
                    </div>
                    
                    <div class="d-flex align-center justify-space-between mt-2">
                      <div class="d-flex align-center">
                        <v-icon
                          :color="getAvailabilityStatus(events.find(e => e.id === eventId)).color"
                          size="small"
                          class="mr-1"
                        >
                          {{ getAvailabilityStatus(events.find(e => e.id === eventId)).icon }}
                        </v-icon>
                        <span class="text-caption" :style="{color: getAvailabilityStatus(events.find(e => e.id === eventId)).color}">
                          {{ getAvailabilityStatus(events.find(e => e.id === eventId)).text }}
                        </span>
                      </div>
                      
                      <div>
                        <span class="text-subtitle-2 font-weight-bold" :style="{color: theme.primary}">
                          {{ events.find(e => e.id === eventId).price > 0 ? 'RM ' + events.find(e => e.id === eventId).price.toFixed(2) : 'Free' }}
                        </span>
                      </div>
                    </div>
                  </div>
                </v-card-text>
              </v-card>
            </v-col>
          </v-row>
        </v-window-item>
      </v-window>
      
      <!-- Event Detail Dialog -->
      <v-dialog
        v-model="showEventDetail"
        :max-width="$vuetify.display.smAndDown ? undefined : '800px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        transition="dialog-bottom-transition"
        class="event-detail-dialog"
      >
        <v-card v-if="selectedEvent">
          <div v-if="!showRegistrationSuccess">
            <v-img
              :src="selectedEvent.image"
              height="250"
              cover
              class="position-relative"
            >
              <div class="event-detail-img-gradient"></div>
              
              <v-btn
                icon="mdi-close"
                color="black"
                variant="flat"
                size="small"
                @click="showEventDetail = false"
                class="position-absolute close-button-circle"
                style="top: 16px; right: 16px; z-index: 1; border-radius: 50%; width: 40px; height: 40px;"
              ></v-btn>
              
              <div class="position-absolute" style="bottom: 16px; left: 16px; right: 16px;">
                <div class="d-flex align-center mb-2">
                  <v-chip
                    :color="theme.lightText"
                    :text-color="theme.primary"
                    label
                    variant="flat"
                    class="mr-2"
                    style="background-color: rgba(255,255,255,0.95) !important;"
                  >
                    <v-icon start size="small" :color="getCategoryDetails(selectedEvent.category).color">{{ getCategoryDetails(selectedEvent.category).icon }}</v-icon>
                    {{ getCategoryDetails(selectedEvent.category).title }}
                  </v-chip>
                  
                  <v-chip
                    v-if="selectedEvent.featured"
                    :color="theme.secondary"
                    text-color="white"
                    label
                    class="mr-2"
                  >
                    <v-icon start size="small">mdi-star</v-icon>
                    Featured
                  </v-chip>
                  
                  <v-spacer></v-spacer>
                  
                  <v-btn
                        :icon="savedEvents.includes(selectedEvent.id) ? 'mdi-bookmark' : 'mdi-bookmark-outline'"
                        :color="savedEvents.includes(selectedEvent.id) ? theme.secondary : 'white'"
                        size="small"
                        variant="text"
                        @click.stop="toggleFavorite(selectedEvent.id)"
                      ></v-btn>
                </div>
                
                <h2 class="text-h4 font-weight-bold mb-0" :style="{color: theme.lightText, textShadow: '2px 2px 4px rgba(0,0,0,0.8)'}">
                  {{ selectedEvent.title }}
                </h2>
              </div>
            </v-img>
            
            <v-card-text class="pa-4">
              <v-row>
                <v-col cols="12" md="8">
                  <div class="d-flex align-center mb-4">
                    <v-chip
                      :color="getRelativeTime(selectedEvent.date).color"
                      :text-color="'white'"
                      class="mr-2"
                      variant="flat"
                      size="small"
                    >
                      <v-icon start>mdi-clock-outline</v-icon>
                      {{ getRelativeTime(selectedEvent.date).text }}
                    </v-chip>
                    
                    <v-chip
                      v-if="selectedEvent.price > 0"
                      :color="theme.primary"
                      text-color="white"
                      variant="flat"
                      class="mr-2"
                      size="small"
                    >
                      <v-icon start>mdi-cash</v-icon>
                      RM {{ selectedEvent.price.toFixed(2) }}
                    </v-chip>
                    
                    <v-chip
                      v-else
                      :color="theme.success"
                      text-color="white"
                      variant="flat"
                      class="mr-2"
                      size="small"
                    >
                      <v-icon start>mdi-cash-remove</v-icon>
                      Free Event
                    </v-chip>
                  </div>
                  
                  <v-card variant="outlined" class="mb-4 pa-3">
                    <div class="d-flex align-center mb-3">
                      <v-icon :color="theme.primary" class="mr-2">mdi-information-outline</v-icon>
                      <h3 class="text-h6" :style="{color: theme.primary}">Event Details</h3>
                    </div>
                    
                    <v-list density="compact" bg-color="transparent" class="pa-0">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary" size="small">mdi-calendar-range</v-icon>
                        </template>
                        <v-list-item-title>
                          {{ formatDateTimeRange(selectedEvent.date, selectedEvent.endDate) }}
                        </v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary" size="small">mdi-map-marker</v-icon>
                        </template>
                        <v-list-item-title>
                          {{ selectedEvent.location }}
                        </v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary" size="small">mdi-account</v-icon>
                        </template>
                        <v-list-item-title>
                          Hosted by {{ selectedEvent.host }}
                        </v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item v-if="selectedEvent.availableSpots">
                        <template v-slot:prepend>
                          <v-icon :color="getAvailabilityStatus(selectedEvent).color" size="small">
                            {{ getAvailabilityStatus(selectedEvent).icon }}
                          </v-icon>
                        </template>
                        <v-list-item-title :style="{color: getAvailabilityStatus(selectedEvent).color}">
                          {{ getAvailabilityStatus(selectedEvent).text }}
                        </v-list-item-title>
                      </v-list-item>
                    </v-list>
                    
                    <v-divider class="my-3"></v-divider>
                    
                    <div class="d-flex flex-wrap">
                      <v-chip
                        v-if="selectedEvent.indoor"
                        size="small"
                        class="mr-1 mb-1"
                        :color="theme.accent"
                        :text-color="theme.darkText"
                        variant="flat"
                      >
                        <v-icon start size="small">mdi-home</v-icon>
                        Indoor
                      </v-chip>
                      
                      <v-chip
                        v-else
                        size="small"
                        class="mr-1 mb-1"
                        :color="theme.accent"
                        :text-color="theme.darkText"
                        variant="flat"
                      >
                        <v-icon start size="small">mdi-tree</v-icon>
                        Outdoor
                      </v-chip>
                      
                      <v-chip
                        v-if="selectedEvent.familyFriendly"
                        size="small"
                        class="mr-1 mb-1"
                        :color="theme.secondary"
                        :text-color="theme.darkText"
                        variant="flat"
                      >
                        <v-icon start size="small">mdi-human-male-child</v-icon>
                        Family-Friendly
                      </v-chip>
                      
                      <v-chip
                        v-if="selectedEvent.accessibility"
                        size="small"
                        class="mr-1 mb-1"
                        :color="theme.primary"
                        :text-color="theme.lightText"
                        variant="flat"
                      >
                        <v-icon start size="small">mdi-wheelchair-accessibility</v-icon>
                        Accessible
                      </v-chip>
                    </div>
                  </v-card>
                  
                  <h3 class="text-h6 mb-2" :style="{color: theme.primary}">About This Event</h3>
                  <p class="text-body-1 mb-4" :style="{color: theme.darkText}">{{ selectedEvent.longDescription || selectedEvent.description }}</p>
                  
                  <div v-if="selectedEvent.tags && selectedEvent.tags.length">
                    <h4 class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Tags</h4>
                    <div class="d-flex flex-wrap">
                      <v-chip
                        v-for="tag in selectedEvent.tags"
                        :key="tag"
                        size="small"
                        class="mr-1 mb-1"
                        :color="theme.accent"
                        :text-color="theme.darkText"
                        variant="flat"
                      >
                        {{ tag }}
                      </v-chip>
                    </div>
                  </div>
                </v-col>
                
                <v-col cols="12" md="4" v-if="selectedEvent.rsvp && selectedEvent.date >= today">
                  <v-card
                    :color="theme.lightBg"
                    variant="flat"
                    class="registration-card pa-4"
                    :class="{'registration-card-disabled': getAvailabilityStatus(selectedEvent).text === 'Fully Booked'}"
                  >
                    <h3 class="text-h6 mb-3" :style="{color: theme.primary}">Register for this Event</h3>
                    
                    <div v-if="getAvailabilityStatus(selectedEvent).text === 'Fully Booked'">
                      <v-alert
                        type="error"
                        variant="outlined"
                        class="mb-3"
                      >
                        <div class="text-center">
                          <v-icon size="large" class="mb-2">mdi-close-circle</v-icon>
                          <div class="text-subtitle-1 font-weight-bold">Fully Booked</div>
                          <p class="mb-0">This event has reached capacity</p>
                        </div>
                      </v-alert>
                      
                      <v-btn
                        block
                        :color="theme.primary"
                        :text-color="theme.lightText"
                        prepend-icon="mdi-bell"
                        class="mb-2"
                      >
                        Join Waitlist
                      </v-btn>
                      
                      <v-btn
                        block
                        :color="theme.secondary"
                        variant="text"
                        @click="toggleFavorite(selectedEvent.id)"
                      >
                        <v-icon start>mdi-bookmark</v-icon>
                        Save for Later
                      </v-btn>
                    </div>
                    
                    <v-form v-else class="registration-form">
                      <v-select
                        v-model="eventRegistrationForm.attendees"
                        :items="[1, 2, 3, 4, 5, 6, 7, 8]"
                        label="Number of Attendees"
                        :rules="[v => !!v || 'Required', v => v > 0 || 'At least 1 attendee needed']"
                        :color="theme.primary"
                        variant="outlined"
                        density="comfortable"
                        class="mb-3"
                      ></v-select>
                      
                      <v-textarea
                        v-model="eventRegistrationForm.specialRequests"
                        label="Special Requests (Optional)"
                        :color="theme.primary"
                        variant="outlined"
                        density="comfortable"
                        rows="3"
                        class="mb-3"
                      ></v-textarea>
                      
                      <v-divider class="mb-3"></v-divider>
                      
                      <div class="d-flex justify-space-between align-center mb-4">
                        <div class="text-subtitle-1">Total:</div>
                        <div class="text-h6 font-weight-bold" :style="{color: theme.primary}">
                          {{ selectedEvent.price > 0 ? 'RM ' + (selectedEvent.price * eventRegistrationForm.attendees).toFixed(2) : 'Free' }}
                        </div>
                      </div>
                      
                      <v-btn
                        block
                        :color="theme.primary"
                        :text-color="theme.lightText"
                        size="large"
                        @click="registerForEvent"
                        :loading="loading"
                        class="mb-2"
                      >
                        <v-icon start>mdi-check-circle</v-icon>
                        Register Now
                      </v-btn>
                      
                      <p v-if="selectedEvent.price > 0" class="text-caption mb-0 text-center" :style="{color: theme.muted}">
                        Payment will be collected at the event
                      </p>
                    </v-form>
                  </v-card>
                </v-col>
              </v-row>
            </v-card-text>
          </div>
          
          <!-- Registration Success -->
          <v-card v-else flat class="success-registration">
            <div class="py-5 px-4 text-center" :style="{ backgroundColor: theme.success, color: theme.lightText }">
              <v-icon size="64" class="mb-2">mdi-check-circle</v-icon>
              <h2 class="text-h5 font-weight-bold">Registration Successful!</h2>
            </div>
            
            <v-card-text class="pa-4 text-center">
              <h3 class="text-h6 mb-3" :style="{ color: theme.primary }">You're all set for</h3>
              <p class="text-h5 mb-4 font-weight-bold" :style="{ color: theme.primary }">{{ selectedEvent.title }}</p>
              
              <v-card variant="outlined" class="mb-4">
                <v-card-text>
                  <v-list density="compact" bg-color="transparent" class="pa-0">
                    <v-list-item>
                      <template v-slot:prepend>
                        <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                      </template>
                      <v-list-item-title class="font-weight-medium">
                        {{ formatDateTimeRange(selectedEvent.date, selectedEvent.endDate) }}
                      </v-list-item-title>
                    </v-list-item>
                    
                    <v-list-item>
                      <template v-slot:prepend>
                        <v-icon :color="theme.primary">mdi-map-marker</v-icon>
                      </template>
                      <v-list-item-title class="font-weight-medium">
                        {{ selectedEvent.location }}
                      </v-list-item-title>
                    </v-list-item>
                    
                    <v-list-item>
                      <template v-slot:prepend>
                        <v-icon :color="theme.primary">mdi-account-multiple</v-icon>
                      </template>
                      <v-list-item-title class="font-weight-medium">
                        {{ eventRegistrationForm.attendees }} {{ eventRegistrationForm.attendees === 1 ? 'Person' : 'People' }}
                      </v-list-item-title>
                    </v-list-item>
                    
                    <v-list-item v-if="selectedEvent.price > 0">
                      <template v-slot:prepend>
                        <v-icon :color="theme.primary">mdi-cash</v-icon>
                      </template>
                      <v-list-item-title class="font-weight-medium">
                        RM {{ (selectedEvent.price * eventRegistrationForm.attendees).toFixed(2) }} (Pay at event)
                      </v-list-item-title>
                    </v-list-item>
                  </v-list>
                </v-card-text>
              </v-card>
              
              <p class="text-body-1 mb-4" :style="{color: theme.primary}">
                We've added this event to your saved events list.<br>
                An email confirmation has also been sent to your registered email address.
              </p>
              
              <div class="d-flex flex-wrap justify-center">
                <v-btn
                  :color="theme.secondary"
                  :text-color="theme.primary"
                  class="ma-2"
                  prepend-icon="mdi-calendar-export"
                >
                  Add to Calendar
                </v-btn>
                
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  class="ma-2"
                  @click="showEventDetail = false"
                >
                  Done
                </v-btn>
              </div>
            </v-card-text>
          </v-card>
        </v-card>
      </v-dialog>
      
      <v-snackbar
        v-model="snackbar.show"
        :color="snackbar.color"
        :timeout="snackbar.timeout"
      >
        {{ snackbar.text }}
        
        <template v-slot:actions>
          <v-btn
            variant="text"
            @click="snackbar.show = false"
          >
            Close
          </v-btn>
        </template>
      </v-snackbar>
    </v-container>


    <style>
    /* Events specific styles - to be appended to the tropicalBookingStyles.js */

/* Event cards */
.event-card {
  transition: transform 0.2s, box-shadow 0.2s;
  height: 100%;
  border-radius: 12px;
  overflow: hidden;
}

.event-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 8px 16px rgba(0,0,0,0.15) !important;
}

.event-card-img {
  position: relative;
}

.event-card-overlay {
  position: absolute;
  top: 0;
  left: 0;
  right: 0;
  z-index: 1;
  background: linear-gradient(to bottom, rgba(0,0,0,0.5) 0%, rgba(0,0,0,0) 100%);
}

.event-timing-chip {
  position: absolute;
  bottom: -12px;
  right: 12px;
  z-index: 2;
}

.event-title {
  display: -webkit-box;
  -webkit-line-clamp: 2;
  -webkit-box-orient: vertical;
  overflow: hidden;
  line-height: 1.3;
}

.event-description {
  display: -webkit-box;
  -webkit-line-clamp: 3;
  -webkit-box-orient: vertical;
  overflow: hidden;
  line-height: 1.4;
}

.past-event-img {
  filter: grayscale(0.8);
}

/* Featured events */
.featured-event-slide {
  cursor: pointer;
}

.featured-event-card {
  transition: transform 0.2s, box-shadow 0.2s;
  cursor: pointer;
  border-radius: 16px;
  overflow: hidden;
}

.featured-event-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 8px 20px rgba(0,0,0,0.2) !important;
}

/* Event list view */
.event-list-item {
  transition: background-color 0.2s;
  border-radius: 8px;
  margin-bottom: 4px;
}

.event-list-item:hover {
  background-color: rgba(0,0,0,0.03);
}

.event-list-img {
  overflow: hidden;
}

/* Event detail dialog */
.event-detail-img-gradient {
  position: absolute;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background: linear-gradient(to bottom, rgba(0,0,0,0.3) 0%, rgba(0,0,0,0.7) 100%);
}

.close-button-circle {
  backdrop-filter: blur(4px);
  box-shadow: 0 2px 8px rgba(0,0,0,0.3) !important;
  transition: all 0.2s ease;
}

.close-button-circle:hover {
  background-color: rgba(255,255,255,1) !important;
  transform: scale(1.1);
  box-shadow: 0 4px 12px rgba(0,0,0,0.4) !important;
}

.registration-card {
  border: 1px solid rgba(0,0,0,0.08);
  border-radius: 12px;
}

.registration-card-disabled {
  opacity: 0.85;
}

.success-registration {
  animation: fadeIn 0.5s ease-out;
}

.filter-summary-chip {
  border-radius: 8px;
  overflow: hidden;
}

.filter-summary-text {
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
  max-width: 90vw;
}

.filter-chip-group {
  display: flex;
  flex-wrap: wrap;
  gap: 8px;
}

.mobile-filter-row {
  row-gap: 8px;
}

@keyframes fadeIn {
  from { opacity: 0; }
  to { opacity: 1; }
}

/* Responsive adjustments */
@media (max-width: 600px) {
  .event-timing-chip {
    bottom: -10px;
    right: 8px;
  }
  
  .filter-summary-text {
    max-width: 60vw;
  }
  
  /* Add spacing between filter components on mobile */
  .filter-chip-group {
    margin-bottom: 8px;
  }
}

    </style>
  `
};
