export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C',
        success: '#2E7D32'
      },

      // QR Scanner States
      isScanning: false,
      scannerStream: null,
      scannerVideo: null,
      isInitializing: false,
      scanError: null,
      lastScanTime: 0,
      scanCooldown: 2000, // 2 second cooldown between scans

      // Access States
      isLoading: false,
      accessGranted: false,
      accessDenied: false,
      accessStep: 1, // 1: QR Scan, 2: Access Details, 3: Success/Denied
      accessDetails: null,
      scannedQrData: null,

      // User Data
      userData: {
        name: "Jason Tan",
        role: "platinum", 
        membershipId: "TAPANG-PLAT-2023",
        membershipStatus: "Active"
      },

      // Notification system
      showNotification: false,
      notificationMessage: '',
      isErrorNotification: false,

      // Animation states
      scannerActive: false,
      showSuccessAnimation: false,

      // Demo QR data for testing
      demoQrData: {
        accessType: "gym_entry",
        location: "Main Gym Door",
        timestamp: new Date().toISOString(),
        requiresAccess: true
      }
    }
  },

  computed: {
    // Check if user has valid access
    hasValidAccess() {
      return this.userData.membershipStatus === "Active"
    },

    // Format current time
    currentTime() {
      return new Date().toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
      })
    },

    // Format current date
    currentDate() {
      return new Date().toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
      })
    }
  },

  methods: {
    // Initialize QR Scanner
    async initQRScanner() {
      if (this.isInitializing) return
      
      this.isInitializing = true
      this.scanError = null

      try {
        // Check if getUserMedia is supported
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
          throw new Error('Camera access not supported')
        }

        // Request camera access
        const stream = await navigator.mediaDevices.getUserMedia({
          video: {
            facingMode: 'environment', // Use back camera
            width: { ideal: 1280 },
            height: { ideal: 720 }
          }
        })

        this.scannerStream = stream
        this.scannerVideo = this.$refs.scannerVideo
        this.scannerVideo.srcObject = stream
        
        await this.scannerVideo.play()
        this.isScanning = true
        this.scannerActive = true
        
        // Start scanning for QR codes
        this.startQRDetection()

      } catch (error) {
        console.error('QR Scanner initialization failed:', error)
        this.scanError = error.message.includes('Permission denied') 
          ? 'Camera permission denied. Please allow camera access and try again.'
          : 'Failed to access camera. Please check your device settings.'
        this.showErrorNotification(this.scanError)
      } finally {
        this.isInitializing = false
      }
    },

    // Start QR code detection
    startQRDetection() {
      if (!this.scannerVideo || !this.isScanning) return

      const canvas = document.createElement('canvas')
      const context = canvas.getContext('2d')
      
      const detectQR = () => {
        if (!this.isScanning || !this.scannerVideo) return

        const video = this.scannerVideo
        if (video.readyState === video.HAVE_ENOUGH_DATA) {
          canvas.width = video.videoWidth
          canvas.height = video.videoHeight
          context.drawImage(video, 0, 0, canvas.width, canvas.height)
          
          const imageData = context.getImageData(0, 0, canvas.width, canvas.height)
          
          // Use jsQR library for QR detection
          if (window.jsQR) {
            const code = window.jsQR(imageData.data, imageData.width, imageData.height)
            if (code && code.data) {
              this.handleQRCodeDetected(code.data)
              return
            }
          }
        }
        
        // Continue scanning
        if (this.isScanning) {
          requestAnimationFrame(detectQR)
        }
      }
      
      detectQR()
    },

    // Handle QR code detection
    async handleQRCodeDetected(qrData) {
      const now = Date.now()
      if (now - this.lastScanTime < this.scanCooldown) return
      
      this.lastScanTime = now
      
      try {
        // Parse QR code data
        let parsedData
        try {
          parsedData = JSON.parse(qrData)
        } catch {
          // If not JSON, treat as simple string
          parsedData = { accessType: "gym_entry", data: qrData.trim() }
        }

        this.scannedQrData = parsedData
        this.showRegularNotification('QR code scanned successfully!')
        
        // Process gym access
        await this.processGymAccess(parsedData)
        
      } catch (error) {
        console.error('QR processing error:', error)
        this.showErrorNotification('Failed to process QR code')
      }
    },

    // Process gym access request
    async processGymAccess(qrData) {
      this.isLoading = true
      this.stopQRScanner()

      try {
        // Simulate API call to validate access
        await new Promise(resolve => setTimeout(resolve, 1500))

        // Check if user has valid membership
        if (this.hasValidAccess) {
          this.accessGranted = true
          this.accessDenied = false
          this.accessDetails = {
            memberName: this.userData.name,
            membershipId: this.userData.membershipId,
            membershipType: this.userData.role,
            accessLocation: qrData.location || "Gym Access Point",
            accessTime: this.currentTime,
            accessDate: this.currentDate,
            status: "Access Granted"
          }
          this.accessStep = 2
          this.showSuccessAnimation = true
          this.showRegularNotification('Access granted! Welcome to the gym!')
          
          // Auto advance to success step after showing details
          setTimeout(() => {
            this.accessStep = 3
          }, 2000)
        } else {
          this.accessGranted = false
          this.accessDenied = true
          this.accessDetails = {
            memberName: this.userData.name,
            membershipId: this.userData.membershipId,
            membershipType: this.userData.role,
            accessLocation: qrData.location || "Gym Access Point",
            accessTime: this.currentTime,
            accessDate: this.currentDate,
            status: "Access Denied",
            reason: "Membership expired or invalid"
          }
          this.accessStep = 2
          this.showErrorNotification('Access denied. Please contact reception.')
          
          // Auto advance to denied step
          setTimeout(() => {
            this.accessStep = 3
          }, 2000)
        }
      } catch (error) {
        console.error('Error processing gym access:', error)
        this.showErrorNotification('Error processing access request')
      } finally {
        this.isLoading = false
      }
    },

    // Stop QR Scanner
    stopQRScanner() {
      this.isScanning = false
      this.scannerActive = false
      
      if (this.scannerStream) {
        this.scannerStream.getTracks().forEach(track => track.stop())
        this.scannerStream = null
      }
      
      if (this.scannerVideo) {
        this.scannerVideo.srcObject = null
      }
    },

    // Restart QR scanning
    restartQRScanning() {
      this.accessStep = 1
      this.accessGranted = false
      this.accessDenied = false
      this.accessDetails = null
      this.scannedQrData = null
      this.showSuccessAnimation = false
      this.initQRScanner()
    },

    // Demo scan simulation
    simulateQRScan() {
      this.handleQRCodeDetected(JSON.stringify(this.demoQrData))
    },

    // Show error notification (centered modal)
    showErrorNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = true

      setTimeout(() => {
        this.showNotification = false
        this.isErrorNotification = false
      }, 4000)
    },

    // Method to manually close error notification
    closeErrorNotification() {
      this.showNotification = false
      this.isErrorNotification = false
    },

    // Show regular notification
    showRegularNotification(message) {
      this.showNotification = true
      this.notificationMessage = message
      this.isErrorNotification = false
      setTimeout(() => {
        this.showNotification = false
      }, 3000)
    },

    // Go back to home
    goBack() {
      this.stopQRScanner()
      this.$root.loadComponent('modules/home.js')
    },

    // Reset and scan again
    scanAgain() {
      this.accessStep = 1
      this.accessGranted = false
      this.accessDenied = false
      this.accessDetails = null
      this.scannedQrData = null
      this.showSuccessAnimation = false
      this.initQRScanner()
    },

    // Navigation
    navigateTo(screen) {
      this.stopQRScanner()
      this.showNotification = true
      this.notificationMessage = 'Loading...'
      setTimeout(() => {
        this.$root.loadComponent('modules/' + screen)
      }, 600)
    }
  },

  async mounted() {
    // Start QR scanning immediately
    this.initQRScanner()

    // Inject CSS styles dynamically
    const styleElement = document.createElement('style')
    styleElement.textContent = `
      /* Base Styles - Gym Access */
      .gym-access-container {
        --primary: #1A3C28;
        --secondary: #DDA333;
        --accent: #A3BA95;
        --cream: #EADDCB;
        --muted: #889FAC;
        --text-dark: #1E1E1E;
        --text-light: #FFFFFF;
        --error: #B71C1C;
        --success: #2E7D32;
        
        font-family: 'Arial', sans-serif;
        background-color: var(--cream);
        min-height: 100vh;
        position: relative;
        overflow-x: hidden;
      }
      
      /* Header Section */
      .gym-access-header {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        padding: 16px;
        border-radius: 0 0 24px 24px;
        box-shadow: 0 6px 16px rgba(0,0,0,0.15);
        color: var(--text-light);
        position: relative;
        overflow: hidden;
      }
      
      .header-content {
        position: relative;
        z-index: 1;
      }
      
      .back-button {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        cursor: pointer;
        min-height: 44px;
        transition: opacity 0.2s ease;
      }
      
      .back-button:hover {
        opacity: 0.8;
      }
      
      .back-button-text {
        margin-left: 8px;
        font-weight: 500;
        font-size: 16px;
      }
      
      .page-title {
        font-size: 22px;
        font-weight: 700;
        margin-bottom: 12px;
      }
      
      /* Content Area */
      .content-area {
        padding: 0 16px 80px 16px;
        margin-top: 20px;
      }
      
      /* QR Scanner Styles */
      .qr-scanner-container {
        position: relative;
        background: black;
        border-radius: 20px;
        overflow: hidden;
        margin-bottom: 24px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.3);
      }
      
      .scanner-video {
        width: 100%;
        height: 300px;
        object-fit: cover;
        display: block;
      }
      
      .scanner-overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(0,0,0,0.3);
      }
      
      .scanner-frame {
        width: 200px;
        height: 200px;
        border: 3px solid var(--secondary);
        border-radius: 16px;
        position: relative;
        background: transparent;
      }
      
      .scanner-frame::before,
      .scanner-frame::after {
        content: '';
        position: absolute;
        width: 30px;
        height: 30px;
      }
      
      .scanner-frame::before {
        top: -3px;
        left: -3px;
        border-top: 3px solid var(--secondary);
        border-left: 3px solid var(--secondary);
        border-radius: 16px 0 0 0;
      }
      
      .scanner-frame::after {
        bottom: -3px;
        right: -3px;
        border-bottom: 3px solid var(--secondary);
        border-right: 3px solid var(--secondary);
        border-radius: 0 0 16px 0;
      }
      
      .scanner-instructions {
        position: absolute;
        bottom: 20px;
        left: 20px;
        right: 20px;
        text-align: center;
        color: white;
        background: rgba(0,0,0,0.7);
        padding: 12px 16px;
        border-radius: 12px;
        font-size: 14px;
      }
      
      .scanner-error {
        background: white;
        border-radius: 20px;
        padding: 24px;
        text-align: center;
        margin-bottom: 24px;
        border: 2px solid var(--error);
      }
      
      .scanner-error-icon {
        font-size: 48px;
        color: var(--error);
        margin-bottom: 16px;
      }
      
      .scanner-error-message {
        color: var(--error);
        font-weight: 600;
        margin-bottom: 16px;
      }
      
      /* Scanning Animation */
      .scanning-line {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 2px;
        background: var(--secondary);
        animation: scanningAnimation 2s infinite;
      }
      
      @keyframes scanningAnimation {
        0% { transform: translateY(0); opacity: 1; }
        50% { transform: translateY(198px); opacity: 0.7; }
        100% { transform: translateY(0); opacity: 1; }
      }
      
      /* Access Details Card */
      .access-details-card {
        background: white;
        border-radius: 20px;
        padding: 24px;
        margin-bottom: 24px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        border: 2px solid var(--accent);
      }
      
      .access-granted {
        border-color: var(--success);
        background: linear-gradient(135deg, rgba(46, 125, 50, 0.02) 0%, rgba(46, 125, 50, 0.05) 100%);
      }
      
      .access-denied {
        border-color: var(--error);
        background: linear-gradient(135deg, rgba(183, 28, 28, 0.02) 0%, rgba(183, 28, 28, 0.05) 100%);
      }
      
      .access-header {
        display: flex;
        align-items: center;
        margin-bottom: 20px;
      }
      
      .access-icon {
        width: 64px;
        height: 64px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 16px;
        font-size: 28px;
        color: white;
      }
      
      .access-granted .access-icon {
        background: var(--success);
      }
      
      .access-denied .access-icon {
        background: var(--error);
      }
      
      .access-info h3 {
        margin: 0 0 4px 0;
        color: var(--text-dark);
        font-weight: 700;
        font-size: 18px;
      }
      
      .access-info p {
        margin: 0;
        color: var(--muted);
        font-size: 14px;
      }
      
      .access-details {
        background: rgba(255, 255, 255, 0.5);
        border-radius: 12px;
        padding: 16px;
        margin-top: 16px;
      }
      
      .detail-row {
        display: flex;
        justify-content: space-between;
        padding: 8px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
      }
      
      .detail-row:last-child {
        border-bottom: none;
      }
      
      .detail-label {
        color: var(--muted);
        font-size: 14px;
        font-weight: 500;
      }
      
      .detail-value {
        font-weight: 600;
        font-size: 14px;
        color: var(--text-dark);
        text-align: right;
      }
      
      /* Button Styles */
      .btn-primary {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        border: none;
        border-radius: 16px;
        padding: 18px 28px;
        font-weight: 700;
        font-size: 16px;
        text-transform: capitalize;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
        box-shadow: 0 6px 20px rgba(26, 60, 40, 0.2);
        min-height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        text-decoration: none;
        cursor: pointer;
        width: 100%;
      }
      
      .btn-primary:hover:not(:disabled) {
        transform: translateY(-3px);
        box-shadow: 0 8px 28px rgba(26, 60, 40, 0.3);
        background: linear-gradient(135deg, #2d5a3d 0%, var(--secondary) 100%);
        color: white;
      }
      
      .btn-primary:disabled {
        background: linear-gradient(135deg, #e0e0e0 0%, #bdbdbd 100%);
        color: #9e9e9e;
        cursor: not-allowed;
        transform: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        opacity: 0.6;
      }
      
      .btn-secondary {
        background: linear-gradient(135deg, var(--secondary) 0%, #e6b84d 100%);
        border: none;
        border-radius: 16px;
        padding: 18px 28px;
        font-weight: 700;
        font-size: 16px;
        text-transform: capitalize;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
        box-shadow: 0 6px 20px rgba(221, 163, 51, 0.2);
        min-height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--primary);
        text-decoration: none;
        cursor: pointer;
        width: 100%;
      }
      
      .btn-secondary:hover {
        transform: translateY(-3px);
        box-shadow: 0 8px 28px rgba(221, 163, 51, 0.3);
      }
      
      /* Success Animation */
      .success-animation {
        animation: successPulse 1s ease-in-out;
      }
      
      @keyframes successPulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
      }
      
      /* Loading States */
      .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(26, 60, 40, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 1000;
      }
      
      .loading-spinner {
        display: flex;
        flex-direction: column;
        align-items: center;
      }
      
      .loading-text {
        margin-top: 16px;
        color: white;
        font-weight: 500;
      }
      
      /* Notification Styles */
      .notification-toast {
        position: fixed;
        bottom: 24px;
        left: 50%;
        transform: translateX(-50%);
        background-color: var(--secondary);
        color: var(--primary);
        padding: 14px 20px;
        border-radius: 20px;
        font-weight: 600;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 900;
        min-width: 80%;
        max-width: 90%;
        text-align: center;
      }
      
      .notification-toast.error-notification {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background-color: var(--error);
        color: white;
        padding: 20px 24px;
        border-radius: 16px;
        box-shadow: 0 8px 24px rgba(183, 28, 28, 0.3);
        z-index: 1001;
        min-width: 85%;
        max-width: 95%;
      }
      
      .error-notification-backdrop {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        backdrop-filter: blur(2px);
      }
      
      .error-close-button {
        position: absolute;
        top: 8px;
        right: 12px;
        background: none;
        border: none;
        color: white;
        font-size: 20px;
        cursor: pointer;
        padding: 4px;
        opacity: 0.8;
      }
    `
    document.head.appendChild(styleElement)
  },

  beforeUnmount() {
    // Clean up scanner when component is destroyed
    this.stopQRScanner()
  },

  template: `
    <div class="gym-access-container">
      <!-- Loading Overlay -->
      <div class="loading-overlay" v-if="isLoading">
        <div class="loading-spinner">
          <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
          <div class="loading-text">Processing access request...</div>
        </div>
      </div>
      
      <!-- Error Notification Backdrop -->
      <div class="error-notification-backdrop" 
           v-if="showNotification && isErrorNotification" 
           @click="closeErrorNotification">
      </div>
      
      <!-- Header -->
      <div class="gym-access-header">
        <div class="header-content">
          <div class="back-button" @click="goBack">
            <v-icon color="white">mdi-arrow-left</v-icon>
            <span class="back-button-text">Back</span>
          </div>
          
          <h1 class="page-title">Gym Access</h1>
          <p style="margin: 0; opacity: 0.9; font-size: 14px;">Scan QR code to unlock gym door</p>
        </div>
      </div>
      
      <!-- Content Area -->
      <div class="content-area">
        <!-- Step 1: QR Scanner -->
        <div v-if="accessStep === 1">
          <div v-if="scanError" class="scanner-error">
            <div class="scanner-error-icon">
              <v-icon size="48" color="error">mdi-camera-off</v-icon>
            </div>
            <div class="scanner-error-message">{{ scanError }}</div>
            <button class="btn-primary" @click="initQRScanner" :disabled="isInitializing">
              <v-icon start>mdi-camera</v-icon>
              {{ isInitializing ? 'Initializing...' : 'Try Again' }}
            </button>
          </div>
          
          <div v-else class="qr-scanner-container">
            <video ref="scannerVideo" class="scanner-video" playsinline></video>
            
            <div class="scanner-overlay" v-if="scannerActive">
              <div class="scanner-frame">
                <div class="scanning-line" v-if="isScanning"></div>
              </div>
            </div>
            
            <div class="scanner-instructions" v-if="isScanning">
              Point your camera at the gym access QR code
            </div>
            
            <div class="scanner-instructions" v-else-if="isInitializing">
              Initializing camera...
            </div>
          </div>
          
          <!-- Demo button for testing -->
          <div style="text-align: center; margin-top: 16px;">
            <button class="btn-secondary" @click="simulateQRScan" style="width: auto; padding: 12px 24px; min-height: auto;">
              <v-icon start>mdi-test-tube</v-icon>
              Demo Scan
            </button>
          </div>
        </div>
        
        <!-- Step 2: Access Details -->
        <div v-if="accessStep === 2 && accessDetails">
          <div class="access-details-card" 
               :class="{ 'access-granted': accessGranted, 'access-denied': accessDenied, 'success-animation': showSuccessAnimation }">
            <div class="access-header">
              <div class="access-icon">
                <v-icon v-if="accessGranted">mdi-check</v-icon>
                <v-icon v-else>mdi-close</v-icon>
              </div>
              <div class="access-info">
                <h3>{{ accessDetails.status }}</h3>
                <p>{{ accessDetails.accessLocation }}</p>
              </div>
            </div>
            
            <div class="access-details">
              <div class="detail-row">
                <div class="detail-label">Member Name</div>
                <div class="detail-value">{{ accessDetails.memberName }}</div>
              </div>
              
              <div class="detail-row">
                <div class="detail-label">Membership ID</div>
                <div class="detail-value">{{ accessDetails.membershipId }}</div>
              </div>
              
              <div class="detail-row">
                <div class="detail-label">Membership Type</div>
                <div class="detail-value">{{ accessDetails.membershipType.charAt(0).toUpperCase() + accessDetails.membershipType.slice(1) }}</div>
              </div>
              
              <div class="detail-row">
                <div class="detail-label">Access Time</div>
                <div class="detail-value">{{ accessDetails.accessTime }}</div>
              </div>
              
              <div class="detail-row">
                <div class="detail-label">Access Date</div>
                <div class="detail-value">{{ accessDetails.accessDate }}</div>
              </div>
              
              <div v-if="accessDenied && accessDetails.reason" class="detail-row">
                <div class="detail-label">Reason</div>
                <div class="detail-value" style="color: var(--error);">{{ accessDetails.reason }}</div>
              </div>
            </div>
          </div>
        </div>
        
        <!-- Step 3: Final Status -->
        <div v-if="accessStep === 3">
          <div class="access-details-card" :class="{ 'access-granted': accessGranted, 'access-denied': accessDenied }">
            <div style="text-align: center; padding: 20px;">
              <div class="access-icon" style="margin: 0 auto 20px auto;">
                <v-icon v-if="accessGranted" size="40">mdi-check</v-icon>
                <v-icon v-else size="40">mdi-close</v-icon>
              </div>
              
              <h2 style="margin: 0 0 8px 0; font-size: 24px; font-weight: 700;">
                {{ accessGranted ? 'Access Granted!' : 'Access Denied' }}
              </h2>
              
              <p style="margin: 0 0 24px 0; color: var(--muted); font-size: 16px;">
                {{ accessGranted ? 'Welcome to the gym! The door is now unlocked.' : 'Please contact reception for assistance.' }}
              </p>
              
              <div style="display: flex; gap: 12px; justify-content: center;">
                <button class="btn-secondary" @click="scanAgain" style="flex: 1;">
                  <v-icon start>mdi-qrcode-scan</v-icon>
                  Scan Again
                </button>
                
                <button class="btn-primary" @click="goBack" style="flex: 1;">
                  <v-icon start>mdi-home</v-icon>
                  Done
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Notification Toast -->
      <transition name="fade">
        <div class="notification-toast" 
             :class="{ 'error-notification': isErrorNotification }" 
             v-if="showNotification">
          
          <!-- Close button for error notifications -->
          <button v-if="isErrorNotification" 
                  class="error-close-button" 
                  @click="closeErrorNotification">
            <v-icon>mdi-close</v-icon>
          </button>
          
          {{ notificationMessage }}
        </div>
      </transition>
    </div>
  `
}