export default {
  props: ["jsonData"],
  data() {
    return {
      // Tropical Rainforest Premium Color Scheme
      theme: {
        primary: '#1A3C28',       // Rich rainforest green
        secondary: '#DDA333',     // Golden amber
        accent: '#A3BA95',        // Sage green
        lightBg: '#F8F5F0',       // Luxury paper
        darkText: '#1E1E1E',
        cream: '#EADDCB',         // Cream
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        info: '#1976D2'
      },

      // UI States
      isLoading: false,
      isInitialLoad: true,
      refreshing: false,

      // Notifications Data
      notifications: [],
      unreadCount: 0,

      // Filter and Search
      selectedCategory: 'all',
      showUnreadOnly: false,
      searchQuery: '',

      // Categories
      categories: [
        { key: 'all', label: 'All', icon: 'mdi-bell', count: 0 },
        { key: 'membership', label: 'Membership', icon: 'mdi-card-account-details', count: 0 },
        { key: 'booking', label: 'Bookings', icon: 'mdi-calendar', count: 0 },
        { key: 'payment', label: 'Payments', icon: 'mdi-credit-card', count: 0 },
        { key: 'promotion', label: 'Promotions', icon: 'mdi-tag', count: 0 },
        { key: 'system', label: 'System', icon: 'mdi-cog', count: 0 },
        { key: 'security', label: 'Security', icon: 'mdi-shield', count: 0 }
      ],

      // Selection and Actions
      selectedNotifications: new Set(),
      showActions: false,
      actionMode: false,

      // Notification Details
      selectedNotification: null,
      showNotificationDetail: false,

      // Notification system
      showSnackbar: false,
      snackbarText: '',
      snackbarColor: ''
    }
  },

  computed: {
    // Filter notifications based on current filters
    filteredNotifications() {
      let filtered = [...this.notifications]

      // Filter by category
      if (this.selectedCategory !== 'all') {
        filtered = filtered.filter(n => n.category === this.selectedCategory)
      }

      // Filter by read status
      if (this.showUnreadOnly) {
        filtered = filtered.filter(n => !n.is_read)
      }

      // Filter by search query
      if (this.searchQuery.trim()) {
        const query = this.searchQuery.toLowerCase()
        filtered = filtered.filter(n => 
          n.title.toLowerCase().includes(query) || 
          n.message.toLowerCase().includes(query)
        )
      }

      return filtered.sort((a, b) => {
        // Sort by importance first, then by date
        if (a.is_important !== b.is_important) {
          return b.is_important - a.is_important
        }
        return new Date(b.notification_time) - new Date(a.notification_time)
      })
    },

    // Get category with updated counts
    categoriesWithCounts() {
      return this.categories.map(cat => ({
        ...cat,
        count: cat.key === 'all' 
          ? this.notifications.length
          : this.notifications.filter(n => n.category === cat.key).length
      }))
    },

    // Check if any notifications are selected
    hasSelectedNotifications() {
      return this.selectedNotifications.size > 0
    },

    // Selected notifications array
    selectedNotificationsArray() {
      return this.notifications.filter(n => this.selectedNotifications.has(n.id))
    }
  },

  methods: {
    // Format notification time
    formatTime(timestamp) {
      const now = new Date()
      const time = new Date(timestamp)
      const diffMs = now - time
      const diffMins = Math.floor(diffMs / 60000)
      const diffHours = Math.floor(diffMs / 3600000)
      const diffDays = Math.floor(diffMs / 86400000)

      if (diffMins < 1) return 'Just now'
      if (diffMins < 60) return `${diffMins}m ago`
      if (diffHours < 24) return `${diffHours}h ago`
      if (diffDays < 7) return `${diffDays}d ago`
      
      return time.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        year: time.getFullYear() !== now.getFullYear() ? 'numeric' : undefined
      })
    },

    // Get notification icon based on category and custom icon
    getNotificationIcon(notification) {
      if (notification.icon) return notification.icon
      
      const iconMap = {
        membership: 'mdi-card-account-details',
        booking: 'mdi-calendar',
        payment: 'mdi-credit-card',
        promotion: 'mdi-tag',
        system: 'mdi-cog',
        security: 'mdi-shield'
      }
      
      return iconMap[notification.category] || 'mdi-bell'
    },

    // Get notification color
    getNotificationColor(notification) {
      if (notification.color) return notification.color
      
      const colorMap = {
        membership: this.theme.secondary,
        booking: this.theme.info,
        payment: this.theme.success,
        promotion: this.theme.warning,
        system: this.theme.accent,
        security: this.theme.error
      }
      
      return colorMap[notification.category] || this.theme.primary
    },

    // Fetch notifications from API
    async fetchNotifications() {
      this.isLoading = true
      
      try {
        const response = await this.$root.callApi('notifications', 'get_notifications', {
          user_id: sessionStorage.getItem('id')
        })

        if (response.data && response.data.status === "YES") {
          this.notifications = response.data.notifications || []
          this.unreadCount = this.notifications.filter(n => !n.is_read).length
        } else {
          // Load demo data if API fails
          this.loadDemoNotifications()
        }
      } catch (error) {
        console.error('Error fetching notifications:', error)
        this.loadDemoNotifications()
      } finally {
        this.isLoading = false
        this.isInitialLoad = false
      }
    },

    // Load demo notifications for testing
    loadDemoNotifications() {
      this.notifications = [
        {
          id: 1,
          title: 'Membership Renewal Required',
          message: 'Your premium membership expires in 7 days. Renew now to continue enjoying all benefits.',
          notification_time: new Date(Date.now() - 1000 * 60 * 30).toISOString(), // 30 mins ago
          is_read: false,
          category: 'membership',
          icon: 'mdi-crown',
          color: this.theme.warning,
          is_important: true,
          action_url: 'modules/screen_011_gym.js'
        },
        {
          id: 2,
          title: 'Class Booking Confirmed',
          message: 'Your HIIT Extreme class booking for tomorrow at 7:00 AM has been confirmed.',
          notification_time: new Date(Date.now() - 1000 * 60 * 60 * 2).toISOString(), // 2 hours ago
          is_read: false,
          category: 'booking',
          icon: 'mdi-calendar-check',
          color: this.theme.success,
          is_important: false,
          action_url: null
        },
        {
          id: 3,
          title: 'Payment Successful',
          message: 'Your wallet has been topped up with RM 200.00. New balance: RM 480.50',
          notification_time: new Date(Date.now() - 1000 * 60 * 60 * 5).toISOString(), // 5 hours ago
          is_read: true,
          category: 'payment',
          icon: 'mdi-check-circle',
          color: this.theme.success,
          is_important: false,
          action_url: 'modules/screen_008_wallet_detail.js'
        },
        {
          id: 4,
          title: 'Special Promotion: 30% Off',
          message: 'Limited time offer! Get 30% off on all personal training sessions this month.',
          notification_time: new Date(Date.now() - 1000 * 60 * 60 * 24).toISOString(), // 1 day ago
          is_read: false,
          category: 'promotion',
          icon: 'mdi-percent',
          color: this.theme.warning,
          is_important: false,
          action_url: null
        },
        {
          id: 5,
          title: 'Security Alert',
          message: 'New login detected from a different device. If this wasn\'t you, please contact support.',
          notification_time: new Date(Date.now() - 1000 * 60 * 60 * 24 * 2).toISOString(), // 2 days ago
          is_read: true,
          category: 'security',
          icon: 'mdi-shield-alert',
          color: this.theme.error,
          is_important: true,
          action_url: null
        },
        {
          id: 6,
          title: 'System Maintenance',
          message: 'Our systems will undergo maintenance on Sunday 2 AM - 4 AM. Some features may be unavailable.',
          notification_time: new Date(Date.now() - 1000 * 60 * 60 * 24 * 3).toISOString(), // 3 days ago
          is_read: true,
          category: 'system',
          icon: 'mdi-wrench',
          color: this.theme.accent,
          is_important: false,
          action_url: null
        }
      ]
      
      this.unreadCount = this.notifications.filter(n => !n.is_read).length
    },

    // Mark notification as read
    async markAsRead(notificationId) {
      try {
        const response = await this.$root.callApi('notifications', 'mark_as_read', {
          notification_id: notificationId,
          user_id: sessionStorage.getItem('id')
        })

        if (response.data && response.data.status === "YES") {
          // Update local data
          const notification = this.notifications.find(n => n.id === notificationId)
          if (notification && !notification.is_read) {
            notification.is_read = true
            this.unreadCount = Math.max(0, this.unreadCount - 1)
          }
        }
      } catch (error) {
        console.error('Error marking notification as read:', error)
        // Update locally anyway for demo
        const notification = this.notifications.find(n => n.id === notificationId)
        if (notification && !notification.is_read) {
          notification.is_read = true
          this.unreadCount = Math.max(0, this.unreadCount - 1)
        }
      }
    },

    // Mark multiple notifications as read
    async markSelectedAsRead() {
      this.isLoading = true
      
      try {
        const unreadSelected = this.selectedNotificationsArray.filter(n => !n.is_read)
        
        for (const notification of unreadSelected) {
          await this.markAsRead(notification.id)
        }
        
        this.showSnackbarMessage(`${unreadSelected.length} notifications marked as read`, 'success')
      } catch (error) {
        this.showSnackbarMessage('Error marking notifications as read', 'error')
      } finally {
        this.isLoading = false
        this.clearSelection()
      }
    },

    // Delete notification
    async deleteNotification(notificationId) {
      try {
        const response = await this.$root.callApi('notifications', 'delete_notification', {
          notification_id: notificationId,
          user_id: sessionStorage.getItem('id')
        })

        if (response.data && response.data.status === "YES") {
          // Remove from local data
          const index = this.notifications.findIndex(n => n.id === notificationId)
          if (index !== -1) {
            const notification = this.notifications[index]
            if (!notification.is_read) {
              this.unreadCount = Math.max(0, this.unreadCount - 1)
            }
            this.notifications.splice(index, 1)
          }
        }
      } catch (error) {
        console.error('Error deleting notification:', error)
        // Delete locally anyway for demo
        const index = this.notifications.findIndex(n => n.id === notificationId)
        if (index !== -1) {
          const notification = this.notifications[index]
          if (!notification.is_read) {
            this.unreadCount = Math.max(0, this.unreadCount - 1)
          }
          this.notifications.splice(index, 1)
        }
      }
    },

    // Delete selected notifications
    async deleteSelected() {
      this.isLoading = true
      
      try {
        for (const notification of this.selectedNotificationsArray) {
          await this.deleteNotification(notification.id)
        }
        
        this.showSnackbarMessage(`${this.selectedNotificationsArray.length} notifications deleted`, 'success')
      } catch (error) {
        this.showSnackbarMessage('Error deleting notifications', 'error')
      } finally {
        this.isLoading = false
        this.clearSelection()
      }
    },

    // Mark all as read
    async markAllAsRead() {
      this.isLoading = true
      
      try {
        const unreadNotifications = this.notifications.filter(n => !n.is_read)
        
        for (const notification of unreadNotifications) {
          await this.markAsRead(notification.id)
        }
        
        this.showSnackbarMessage('All notifications marked as read', 'success')
      } catch (error) {
        this.showSnackbarMessage('Error marking all as read', 'error')
      } finally {
        this.isLoading = false
      }
    },

    // Handle notification click
    handleNotificationClick(notification) {
      // Mark as read if unread
      if (!notification.is_read) {
        this.markAsRead(notification.id)
      }

      // Navigate to action URL if available
      if (notification.action_url) {
        this.$root.loadComponent(notification.action_url)
      } else {
        // Show notification detail
        this.selectedNotification = notification
        this.showNotificationDetail = true
      }
    },

    // Toggle notification selection
    toggleNotificationSelection(notificationId) {
      if (this.selectedNotifications.has(notificationId)) {
        this.selectedNotifications.delete(notificationId)
      } else {
        this.selectedNotifications.add(notificationId)
      }
    },

    // Select all filtered notifications
    selectAllFiltered() {
      this.filteredNotifications.forEach(n => {
        this.selectedNotifications.add(n.id)
      })
    },

    // Clear selection
    clearSelection() {
      this.selectedNotifications.clear()
      this.actionMode = false
    },

    // Toggle action mode
    toggleActionMode() {
      this.actionMode = !this.actionMode
      if (!this.actionMode) {
        this.clearSelection()
      }
    },

    // Refresh notifications
    async refreshNotifications() {
      this.refreshing = true
      await this.fetchNotifications()
      this.refreshing = false
      this.showSnackbarMessage('Notifications refreshed', 'info')
    },

    // Show snackbar message
    showSnackbarMessage(text, color) {
      this.snackbarText = text
      this.snackbarColor = color
      this.showSnackbar = true
    },

    // Navigation
    goBack() {
      this.$root.loadComponent('modules/home.js')
    }
  },

  async mounted() {
    // Fetch notifications on mount
    await this.fetchNotifications()

    // Inject CSS styles
    const styleElement = document.createElement('style')
    styleElement.textContent = `
      /* Base Styles - Notifications */
      .notifications-container {
        --primary: #1A3C28;
        --secondary: #DDA333;
        --accent: #A3BA95;
        --cream: #EADDCB;
        --muted: #889FAC;
        --text-dark: #1E1E1E;
        --text-light: #FFFFFF;
        --error: #B71C1C;
        --success: #2E7D32;
        --warning: #FF8F00;
        --info: #1976D2;
        
        font-family: 'Arial', sans-serif;
        background-color: var(--cream);
        min-height: 100vh;
        position: relative;
      }
      
      /* Header Section */
      .notifications-header {
        background: linear-gradient(135deg, var(--primary) 0%, #2d5a3d 100%);
        padding: 16px;
        border-radius: 0 0 24px 24px;
        box-shadow: 0 6px 16px rgba(0,0,0,0.15);
        color: var(--text-light);
        position: relative;
        overflow: hidden;
      }
      
      .header-content {
        position: relative;
        z-index: 1;
      }
      
      .back-button {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        cursor: pointer;
        min-height: 44px;
        transition: opacity 0.2s ease;
      }
      
      .back-button:hover {
        opacity: 0.8;
      }
      
      .back-button-text {
        margin-left: 8px;
        font-weight: 500;
        font-size: 16px;
      }
      
      .page-title {
        font-size: 22px;
        font-weight: 700;
        margin-bottom: 8px;
      }
      
      .unread-count {
        background: var(--secondary);
        color: var(--primary);
        border-radius: 12px;
        padding: 4px 12px;
        font-size: 14px;
        font-weight: 600;
        margin-left: 8px;
      }
      
      /* Search and Filter Section */
      .filter-section {
        background: white;
        margin: 16px;
        border-radius: 16px;
        padding: 16px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
      }
      
      .search-input {
        background: var(--cream);
        border: 1px solid rgba(26, 60, 40, 0.1);
        border-radius: 12px;
        padding: 12px 16px;
        font-size: 14px;
        width: 100%;
        margin-bottom: 16px;
      }
      
      .search-input:focus {
        outline: none;
        border-color: var(--accent);
        box-shadow: 0 0 0 3px rgba(163, 186, 149, 0.1);
      }
      
      .category-tabs {
        display: flex;
        gap: 8px;
        overflow-x: auto;
        padding-bottom: 8px;
      }
      
      .category-tab {
        display: flex;
        align-items: center;
        gap: 6px;
        padding: 8px 12px;
        border-radius: 20px;
        background: var(--cream);
        border: 1px solid rgba(26, 60, 40, 0.1);
        font-size: 12px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        white-space: nowrap;
        min-width: fit-content;
      }
      
      .category-tab.active {
        background: var(--primary);
        color: var(--text-light);
        border-color: var(--primary);
      }
      
      .category-tab:hover:not(.active) {
        background: var(--accent);
        color: white;
      }
      
      .category-count {
        background: rgba(255, 255, 255, 0.2);
        border-radius: 10px;
        padding: 2px 6px;
        font-size: 10px;
        font-weight: 600;
      }
      
      .filter-toggles {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-top: 16px;
        padding-top: 16px;
        border-top: 1px solid rgba(0,0,0,0.05);
      }
      
      .filter-toggle {
        display: flex;
        align-items: center;
        gap: 8px;
        font-size: 14px;
        cursor: pointer;
      }
      
      .toggle-checkbox {
        width: 16px;
        height: 16px;
        border: 2px solid var(--accent);
        border-radius: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s ease;
      }
      
      .toggle-checkbox.checked {
        background: var(--accent);
        color: white;
      }
      
      /* Action Bar */
      .action-bar {
        background: white;
        margin: 0 16px 16px 16px;
        border-radius: 16px;
        padding: 12px 16px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        display: flex;
        justify-content: space-between;
        align-items: center;
      }
      
      .action-buttons {
        display: flex;
        gap: 8px;
      }
      
      .action-btn {
        padding: 8px 16px;
        border-radius: 20px;
        border: none;
        font-size: 12px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 4px;
      }
      
      .action-btn.primary {
        background: var(--primary);
        color: white;
      }
      
      .action-btn.secondary {
        background: var(--secondary);
        color: var(--primary);
      }
      
      .action-btn.error {
        background: var(--error);
        color: white;
      }
      
      .action-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(0,0,0,0.15);
      }
      
      /* Notifications List */
      .notifications-list {
        padding: 0 16px 80px 16px;
      }
      
      .notification-item {
        background: white;
        border-radius: 16px;
        margin-bottom: 12px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.08);
        overflow: hidden;
        transition: all 0.2s ease;
        cursor: pointer;
        position: relative;
      }
      
      .notification-item:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(0,0,0,0.12);
      }
      
      .notification-item.unread {
        border-left: 4px solid var(--secondary);
      }
      
      .notification-item.important {
        border-left: 4px solid var(--error);
      }
      
      .notification-item.selected {
        background: rgba(26, 60, 40, 0.05);
        border: 2px solid var(--primary);
      }
      
      .notification-content {
        padding: 16px;
        display: flex;
        align-items: flex-start;
        gap: 12px;
      }
      
      .notification-icon {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 18px;
        flex-shrink: 0;
      }
      
      .notification-details {
        flex: 1;
        min-width: 0;
      }
      
      .notification-title {
        font-weight: 600;
        font-size: 14px;
        color: var(--text-dark);
        margin-bottom: 4px;
        line-height: 1.3;
      }
      
      .notification-message {
        font-size: 13px;
        color: var(--muted);
        line-height: 1.4;
        margin-bottom: 8px;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
      }
      
      .notification-meta {
        display: flex;
        align-items: center;
        justify-content: space-between;
        font-size: 12px;
      }
      
      .notification-time {
        color: var(--muted);
      }
      
      .notification-badges {
        display: flex;
        gap: 4px;
      }
      
      .notification-badge {
        padding: 2px 6px;
        border-radius: 8px;
        font-size: 10px;
        font-weight: 600;
        text-transform: uppercase;
      }
      
      .badge-unread {
        background: var(--secondary);
        color: var(--primary);
      }
      
      .badge-important {
        background: var(--error);
        color: white;
      }
      
      .notification-actions {
        position: absolute;
        top: 12px;
        right: 12px;
        display: flex;
        gap: 4px;
      }
      
      .notification-action {
        width: 24px;
        height: 24px;
        border-radius: 50%;
        border: none;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 12px;
        transition: all 0.2s ease;
      }
      
      .action-select {
        background: var(--accent);
        color: white;
      }
      
      .action-delete {
        background: var(--error);
        color: white;
      }
      
      /* Empty State */
      .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: var(--muted);
      }
      
      .empty-icon {
        font-size: 64px;
        margin-bottom: 16px;
        opacity: 0.5;
      }
      
      .empty-title {
        font-size: 18px;
        font-weight: 600;
        margin-bottom: 8px;
      }
      
      .empty-subtitle {
        font-size: 14px;
        opacity: 0.7;
      }
      
      /* Loading States */
      .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(26, 60, 40, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 1000;
      }
      
      .loading-spinner {
        display: flex;
        flex-direction: column;
        align-items: center;
      }
      
      .loading-text {
        margin-top: 16px;
        color: white;
        font-weight: 500;
      }
      
      /* Notification Detail Modal */
      .notification-detail-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 20px;
      }
      
      .notification-detail-modal {
        background: white;
        border-radius: 20px;
        max-width: 400px;
        width: 100%;
        max-height: 80vh;
        overflow-y: auto;
        position: relative;
      }
      
      .modal-header {
        padding: 20px;
        border-bottom: 1px solid rgba(0,0,0,0.05);
        display: flex;
        align-items: center;
        gap: 12px;
      }
      
      .modal-close {
        position: absolute;
        top: 16px;
        right: 16px;
        background: none;
        border: none;
        font-size: 20px;
        cursor: pointer;
        color: var(--muted);
      }
      
      .modal-body {
        padding: 20px;
      }
      
      .modal-footer {
        padding: 20px;
        border-top: 1px solid rgba(0,0,0,0.05);
        text-align: right;
      }
      
      /* Responsive */
      @media (max-width: 600px) {
        .category-tabs {
          padding-bottom: 4px;
        }
        
        .action-buttons {
          flex-wrap: wrap;
        }
        
        .notification-content {
          padding: 12px;
        }
        
        .filter-toggles {
          flex-direction: column;
          gap: 12px;
          align-items: flex-start;
        }
      }
    `
    document.head.appendChild(styleElement)
  },

  template: `
    <div class="notifications-container">
      <!-- Loading Overlay -->
      <div class="loading-overlay" v-if="isLoading && isInitialLoad">
        <div class="loading-spinner">
          <v-progress-circular indeterminate color="secondary" size="64"></v-progress-circular>
          <div class="loading-text">Loading notifications...</div>
        </div>
      </div>
      
      <!-- Header -->
      <div class="notifications-header">
        <div class="header-content">
          <div class="back-button" @click="goBack">
            <v-icon color="white">mdi-arrow-left</v-icon>
            <span class="back-button-text">Back</span>
          </div>
          
          <div style="display: flex; align-items: center;">
            <h1 class="page-title">Notifications</h1>
            <span v-if="unreadCount > 0" class="unread-count">{{ unreadCount }} new</span>
          </div>
          <p style="margin: 0; opacity: 0.9; font-size: 14px;">Stay updated with your latest activities</p>
        </div>
      </div>
      
      <!-- Search and Filter Section -->
      <div class="filter-section">
        <!-- Search Input -->
        <input 
          v-model="searchQuery"
          type="text" 
          class="search-input" 
          placeholder="Search notifications..."
        />
        
        <!-- Category Tabs -->
        <div class="category-tabs">
          <div 
            v-for="category in categoriesWithCounts"
            :key="category.key"
            class="category-tab"
            :class="{ active: selectedCategory === category.key }"
            @click="selectedCategory = category.key"
          >
            <v-icon size="14">{{ category.icon }}</v-icon>
            <span>{{ category.label }}</span>
            <span v-if="category.count > 0" class="category-count">{{ category.count }}</span>
          </div>
        </div>
        
        <!-- Filter Toggles -->
        <div class="filter-toggles">
          <div class="filter-toggle" @click="showUnreadOnly = !showUnreadOnly">
            <div class="toggle-checkbox" :class="{ checked: showUnreadOnly }">
              <v-icon v-if="showUnreadOnly" size="12">mdi-check</v-icon>
            </div>
            <span>Show unread only</span>
          </div>
          
          <div class="filter-toggle" @click="toggleActionMode">
            <v-icon size="16">{{ actionMode ? 'mdi-close' : 'mdi-checkbox-multiple-marked' }}</v-icon>
            <span>{{ actionMode ? 'Cancel' : 'Select' }}</span>
          </div>
        </div>
      </div>
      
      <!-- Action Bar -->
      <div v-if="actionMode" class="action-bar">
        <div>
          <span style="font-size: 14px; font-weight: 600;">
            {{ selectedNotifications.size }} selected
          </span>
        </div>
        <div class="action-buttons">
          <button v-if="filteredNotifications.length > 0" class="action-btn secondary" @click="selectAllFiltered">
            <v-icon size="14">mdi-select-all</v-icon>
            All
          </button>
          <button v-if="hasSelectedNotifications" class="action-btn primary" @click="markSelectedAsRead">
            <v-icon size="14">mdi-email-open</v-icon>
            Read
          </button>
          <button v-if="hasSelectedNotifications" class="action-btn error" @click="deleteSelected">
            <v-icon size="14">mdi-delete</v-icon>
            Delete
          </button>
        </div>
      </div>
      
      <!-- Quick Actions -->
      <div v-if="!actionMode && notifications.length > 0" class="action-bar">
        <div>
          <button class="action-btn secondary" @click="refreshNotifications" :disabled="refreshing">
            <v-icon size="14" :class="{ 'fa-spin': refreshing }">mdi-refresh</v-icon>
            Refresh
          </button>
        </div>
        <div class="action-buttons">
          <button v-if="unreadCount > 0" class="action-btn primary" @click="markAllAsRead">
            <v-icon size="14">mdi-email-open-multiple</v-icon>
            Mark All Read
          </button>
        </div>
      </div>
      
      <!-- Notifications List -->
      <div class="notifications-list">
        <!-- Empty State -->
        <div v-if="filteredNotifications.length === 0 && !isLoading" class="empty-state">
          <div class="empty-icon">
            <v-icon size="64">{{ searchQuery || selectedCategory !== 'all' || showUnreadOnly ? 'mdi-magnify-close' : 'mdi-bell-off' }}</v-icon>
          </div>
          <div class="empty-title">
            {{ searchQuery || selectedCategory !== 'all' || showUnreadOnly ? 'No matching notifications' : 'No notifications yet' }}
          </div>
          <div class="empty-subtitle">
            {{ searchQuery || selectedCategory !== 'all' || showUnreadOnly ? 'Try adjusting your filters' : "We\'ll notify you when there\'s something new" }}
          </div>
        </div>
        
        <!-- Notification Items -->
        <div 
          v-for="notification in filteredNotifications"
          :key="notification.id"
          class="notification-item"
          :class="{ 
            unread: !notification.is_read && !notification.is_important,
            important: notification.is_important,
            selected: selectedNotifications.has(notification.id)
          }"
          @click="actionMode ? toggleNotificationSelection(notification.id) : handleNotificationClick(notification)"
        >
          <div class="notification-content">
            <!-- Selection Checkbox (in action mode) -->
            <div v-if="actionMode" class="toggle-checkbox" :class="{ checked: selectedNotifications.has(notification.id) }">
              <v-icon v-if="selectedNotifications.has(notification.id)" size="12">mdi-check</v-icon>
            </div>
            
            <!-- Notification Icon -->
            <div class="notification-icon" :style="{ backgroundColor: getNotificationColor(notification) }">
              <v-icon size="18">{{ getNotificationIcon(notification) }}</v-icon>
            </div>
            
            <!-- Notification Details -->
            <div class="notification-details">
              <div class="notification-title">{{ notification.title }}</div>
              <div class="notification-message">{{ notification.message }}</div>
              
              <div class="notification-meta">
                <div class="notification-time">{{ formatTime(notification.notification_time) }}</div>
                <div class="notification-badges">
                  <span v-if="!notification.is_read" class="notification-badge badge-unread">New</span>
                  <span v-if="notification.is_important" class="notification-badge badge-important">Important</span>
                </div>
              </div>
            </div>
            
            <!-- Quick Actions (not in action mode) -->
            <div v-if="!actionMode" class="notification-actions" @click.stop="">
              <button v-if="!notification.is_read" class="notification-action action-select" @click="markAsRead(notification.id)">
                <v-icon size="12">mdi-email-open</v-icon>
              </button>
              <button class="notification-action action-delete" @click="deleteNotification(notification.id)">
                <v-icon size="12">mdi-delete</v-icon>
              </button>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Notification Detail Modal -->
      <div v-if="showNotificationDetail && selectedNotification" class="notification-detail-overlay" @click="showNotificationDetail = false">
        <div class="notification-detail-modal" @click.stop="">
          <button class="modal-close" @click="showNotificationDetail = false">
            <v-icon>mdi-close</v-icon>
          </button>
          
          <div class="modal-header">
            <div class="notification-icon" :style="{ backgroundColor: getNotificationColor(selectedNotification) }">
              <v-icon size="20">{{ getNotificationIcon(selectedNotification) }}</v-icon>
            </div>
            <div>
              <div class="notification-title" style="font-size: 16px;">{{ selectedNotification.title }}</div>
              <div class="notification-time">{{ formatTime(selectedNotification.notification_time) }}</div>
            </div>
          </div>
          
          <div class="modal-body">
            <p style="line-height: 1.6; margin: 0;">{{ selectedNotification.message }}</p>
          </div>
          
          <div class="modal-footer">
            <button class="action-btn primary" @click="showNotificationDetail = false">
              Close
            </button>
          </div>
        </div>
      </div>
      
      <!-- Snackbar -->
      <v-snackbar
        v-model="showSnackbar"
        :color="snackbarColor"
        :timeout="3000"
        location="bottom"
      >
        {{ snackbarText }}
        <template v-slot:actions>
          <v-btn variant="text" @click="showSnackbar = false">Close</v-btn>
        </template>
      </v-snackbar>
    </div>
  `
}