import getTropicalStyles from "../css/tropicalBookingStyles.js"; 

export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',
        secondary: '#DDA333',
        accent: '#A3BA95',
        lightBg: '#F8F5F0',
        creamBg: '#EADDCB',
        darkText: '#1E1E1E',
        lightText: '#FFFFFF',
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        muted: '#78909C',
        brown: '#6D4C41',
      },

      // Current step in the parking registration process
      currentStep: 1,
      
      // Car information
      carForm: {
        plateNumber: '',
        carMake: '',
        carModel: '',
        carColor: '',
      },
      carFormValid: false,
      
      // Parking details
      parkingPlanOptions: [
        { id: 'hourly', name: 'Hourly Parking', icon: 'mdi-clock-outline', color: '#1A3C28', rate: 5, unit: 'hour' },
        { id: 'daily', name: 'Daily Rate', icon: 'mdi-calendar-today', color: '#1A3C28', rate: 25, unit: 'day' },
        { id: 'weekly', name: 'Weekly Rate', icon: 'mdi-calendar-week', color: '#1A3C28', rate: 120, unit: 'week' },
        { id: 'monthly', name: 'Monthly Pass', icon: 'mdi-calendar-month', color: '#1A3C28', rate: 300, unit: 'month' }
      ],
      selectedParkingPlan: null,
      parkingDuration: 1,
      parkingStartDate: today,
      minDateForPicker: today,
      startDateMenu: false,
      parkingEndDate: null,
      
      // Payment information
      paymentMethods: [
        { id: 'credit', name: 'Credit Card', icon: 'mdi-credit-card', description: 'Pay with Visa, Mastercard, or Amex' },
        { id: 'debit', name: 'Debit Card', icon: 'mdi-credit-card-outline', description: 'Pay with your debit card' },
        { id: 'ewallet', name: 'E-Wallet', icon: 'mdi-wallet', description: 'Pay with Touch n Go, GrabPay, etc.' },
        { id: 'cash', name: 'Cash at Counter', icon: 'mdi-cash', description: 'Pay at the hotel reception' }
      ],
      selectedPaymentMethod: null,
      cardDetails: {
        cardNumber: '',
        cardholderName: '',
        expiryDate: '',
        cvv: '',
      },
      cardFormValid: false,
      
      // E-wallet options
      eWallets: [
        { id: 'tng', name: 'Touch n Go eWallet', icon: 'mdi-contactless-payment-circle', logo: 'https://via.placeholder.com/60x30?text=TnG' },
        { id: 'grab', name: 'GrabPay', icon: 'mdi-alpha-g-box', logo: 'https://via.placeholder.com/60x30?text=Grab' },
        { id: 'boost', name: 'Boost', icon: 'mdi-rocket', logo: 'https://via.placeholder.com/60x30?text=Boost' },
        { id: 'maybank', name: 'MAE by Maybank', icon: 'mdi-bank', logo: 'https://via.placeholder.com/60x30?text=MAE' }
      ],
      selectedEWallet: null,
      
      // Parking history
      parkingHistory: [
        { 
          id: 'pk1001', 
          plateNumber: 'ABC1234', 
          startDate: new Date(new Date().setDate(today.getDate() - 2)).toISOString().substring(0,10), 
          endDate: new Date(new Date().setDate(today.getDate() - 1)).toISOString().substring(0,10),
          duration: 1,
          durationUnit: 'day',
          amount: 25,
          status: 'completed',
          paymentMethod: 'Credit Card'
        },
        { 
          id: 'pk1002', 
          plateNumber: 'DEF5678', 
          startDate: new Date(new Date().setDate(today.getDate() - 7)).toISOString().substring(0,10), 
          endDate: new Date().toISOString().substring(0,10),
          duration: 1,
          durationUnit: 'week',
          amount: 120,
          status: 'active',
          paymentMethod: 'E-Wallet'
        }
      ],
      
      // UI control
      activeTab: 'register',
      loading: false,
      showSuccessDialog: false,
      showRefundDialog: false,
      showExtendDialog: false,
      selectedParkingToCancel: null,
      selectedParkingToExtend: null,
      extensionDuration: 1,
      extensionRate: 25, // Default rate for extensions
      receiptDetails: null,
    };
  },
  
  computed: {
    formattedStartDate() {
      if (!this.parkingStartDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return this.parkingStartDate.toLocaleDateString('en-US', options);
    },
    
    formattedEndDate() {
      if (!this.parkingEndDate) return "";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return this.parkingEndDate.toLocaleDateString('en-US', options);
    },
    
    calculatedEndDate() {
      if (!this.parkingStartDate || !this.selectedParkingPlan || !this.parkingDuration) {
        return null;
      }
      
      const endDate = new Date(this.parkingStartDate);
      
      switch(this.selectedParkingPlan.unit) {
        case 'hour':
          endDate.setHours(endDate.getHours() + this.parkingDuration);
          break;
        case 'day':
          endDate.setDate(endDate.getDate() + this.parkingDuration);
          break;
        case 'week':
          endDate.setDate(endDate.getDate() + (this.parkingDuration * 7));
          break;
        case 'month':
          endDate.setMonth(endDate.getMonth() + this.parkingDuration);
          break;
      }
      
      return endDate;
    },
    
    parkingCost() {
      if (!this.selectedParkingPlan || !this.parkingDuration) {
        return 0;
      }
      return this.selectedParkingPlan.rate * this.parkingDuration;
    },
    
    activeParkings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.parkingHistory.filter(parking => {
        const endDate = new Date(parking.endDate);
        endDate.setHours(0,0,0,0);
        return endDate >= todayNormalized && parking.status !== 'cancelled';
      }).sort((a, b) => new Date(a.endDate) - new Date(b.endDate));
    },
    
    pastParkings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.parkingHistory.filter(parking => {
        const endDate = new Date(parking.endDate);
        endDate.setHours(0,0,0,0);
        return endDate < todayNormalized || parking.status === 'cancelled';
      }).sort((a, b) => new Date(b.endDate) - new Date(a.endDate));
    },
    
    canMakePayment() {
      if (this.selectedPaymentMethod === 'credit' || this.selectedPaymentMethod === 'debit') {
        return this.cardFormValid;
      } else if (this.selectedPaymentMethod === 'ewallet') {
        return !!this.selectedEWallet;
      } else if (this.selectedPaymentMethod === 'cash') {
        return true;
      }
      return false;
    },
  },
  
  methods: {
    // Navigation methods
    nextStep() {
      if (this.currentStep < 4) {
        this.currentStep++;
        window.scrollTo(0, 0);
      }
    },
    
    prevStep() {
      if (this.currentStep > 1) {
        this.currentStep--;
        window.scrollTo(0, 0);
      }
    },
    
    resetForm() {
      this.currentStep = 1;
      this.carForm = {
        plateNumber: '',
        carMake: '',
        carModel: '',
        carColor: '',
      };
      this.selectedParkingPlan = null;
      this.parkingDuration = 1;
      this.parkingStartDate = new Date();
      this.parkingStartDate.setHours(0,0,0,0);
      this.selectedPaymentMethod = null;
      this.cardDetails = {
        cardNumber: '',
        cardholderName: '',
        expiryDate: '',
        cvv: '',
      };
      this.selectedEWallet = null;
    },
    
    // Parking methods
    selectParkingPlan(plan) {
      this.selectedParkingPlan = plan;
      this.parkingDuration = 1;
      
      // Calculate end date
      this.updateEndDate();
    },
    
    updateEndDate() {
      if (this.selectedParkingPlan && this.parkingStartDate && this.parkingDuration) {
        this.parkingEndDate = this.calculatedEndDate;
      }
    },
    
    // Payment methods
    selectPaymentMethod(method) {
      this.selectedPaymentMethod = method.id;
    },
    
    selectEWallet(wallet) {
      this.selectedEWallet = wallet.id;
    },
    
    processPayment() {
      this.loading = true;
      setTimeout(() => {
        // Create a new parking record
        const newParking = {
          id: 'pk' + Math.floor(1000 + Math.random() * 9000),
          plateNumber: this.carForm.plateNumber.toUpperCase(),
          startDate: this.parkingStartDate.toISOString().substring(0,10),
          endDate: this.parkingEndDate.toISOString().substring(0,10),
          duration: this.parkingDuration,
          durationUnit: this.selectedParkingPlan.unit,
          amount: this.parkingCost,
          status: 'active',
          paymentMethod: this.getPaymentMethodName()
        };
        
        // Add car details
        if (this.carForm.carMake) newParking.carMake = this.carForm.carMake;
        if (this.carForm.carModel) newParking.carModel = this.carForm.carModel;
        if (this.carForm.carColor) newParking.carColor = this.carForm.carColor;
        
        this.parkingHistory.push(newParking);
        
        // Set receipt details
        this.receiptDetails = {
          ...newParking,
          transactionId: 'TXN' + Date.now().toString().substring(6),
          paymentDate: new Date().toISOString(),
        };
        
        this.loading = false;
        this.currentStep = 4; // Move to confirmation step
        this.showSuccessDialog = true;
      }, 1500);
    },
    
    getPaymentMethodName() {
      if (this.selectedPaymentMethod === 'credit') {
        return 'Credit Card';
      } else if (this.selectedPaymentMethod === 'debit') {
        return 'Debit Card';
      } else if (this.selectedPaymentMethod === 'ewallet') {
        const wallet = this.eWallets.find(w => w.id === this.selectedEWallet);
        return wallet ? wallet.name : 'E-Wallet';
      } else if (this.selectedPaymentMethod === 'cash') {
        return 'Cash';
      }
      return 'Unknown';
    },
    
    formatDate(dateStr) {
      if (!dateStr) return '';
      const parts = dateStr.split('-');
      const dateObj = new Date(parts[0], parts[1] - 1, parts[2]);
      const options = { weekday: 'short', month: 'short', day: 'numeric' };
      return dateObj.toLocaleDateString('en-US', options);
    },
    
    formatDateRange(startDateStr, endDateStr) {
      return `${this.formatDate(startDateStr)} - ${this.formatDate(endDateStr)}`;
    },
    
    formatMoney(amount) {
      return amount.toFixed(2);
    },
    
    getStatusDetails(status) {
      switch(status) {
        case 'active': return { text: 'Active', color: this.theme.success, icon: 'mdi-check-circle' };
        case 'pending': return { text: 'Pending', color: this.theme.warning, icon: 'mdi-clock-alert-outline' };
        case 'cancelled': return { text: 'Cancelled', color: this.theme.error, icon: 'mdi-cancel' };
        case 'completed': return { text: 'Completed', color: this.theme.muted, icon: 'mdi-history' };
        default: return { text: status, color: this.theme.muted, icon: 'mdi-information-outline' };
      }
    },
    
    // Card methods
    formatCardNumber(value) {
      if (!value) return '';
      const v = value.replace(/\s+/g, '').replace(/[^0-9]/gi, '');
      const matches = v.match(/\d{4,16}/g);
      const match = matches && matches[0] || '';
      const parts = [];
      
      for (let i = 0, len = match.length; i < len; i += 4) {
        parts.push(match.substring(i, i + 4));
      }
      
      if (parts.length) {
        return parts.join(' ');
      } else {
        return value;
      }
    },
    
    formatExpiryDate(value) {
      if (!value) return '';
      const v = value.replace(/\s+/g, '').replace(/[^0-9]/gi, '');
      
      if (v.length >= 2) {
        return v.substring(0, 2) + (v.length > 2 ? '/' + v.substring(2, 4) : '');
      }
      
      return v;
    },
    
    initiateRefund(parking) {
      this.selectedParkingToCancel = parking;
      this.showRefundDialog = true;
    },
    
    confirmRefund() {
      this.loading = true;
      setTimeout(() => {
        const index = this.parkingHistory.findIndex(p => p.id === this.selectedParkingToCancel.id);
        if (index !== -1) {
          this.parkingHistory[index].status = 'cancelled';
        }
        this.showRefundDialog = false;
        this.selectedParkingToCancel = null;
        
        // Show success snackbar
        if (this.$root && this.$root.showSnackbar) {
          this.$root.showSnackbar('Refund has been processed!', 'success');
        } else {
          console.info('Snackbar: Refund has been processed!');
        }
        
        this.loading = false;
      }, 1000);
    },
    
    initiateExtension(parking) {
      this.selectedParkingToExtend = parking;
      this.extensionDuration = 1;
      
      // Set extension rate based on original parking unit
      const rateMapping = {
        'hour': 5,
        'day': 25,
        'week': 120,
        'month': 300
      };
      this.extensionRate = rateMapping[parking.durationUnit] || 25;
      
      this.showExtendDialog = true;
    },
    
    confirmExtension() {
      this.loading = true;
      setTimeout(() => {
        const index = this.parkingHistory.findIndex(p => p.id === this.selectedParkingToExtend.id);
        
        if (index !== -1) {
          const parking = this.parkingHistory[index];
          const endDate = new Date(parking.endDate);
          
          switch(parking.durationUnit) {
            case 'hour':
              endDate.setHours(endDate.getHours() + this.extensionDuration);
              break;
            case 'day':
              endDate.setDate(endDate.getDate() + this.extensionDuration);
              break;
            case 'week':
              endDate.setDate(endDate.getDate() + (this.extensionDuration * 7));
              break;
            case 'month':
              endDate.setMonth(endDate.getMonth() + this.extensionDuration);
              break;
          }
          
          parking.endDate = endDate.toISOString().substring(0,10);
          parking.duration += this.extensionDuration;
          parking.amount += this.extensionRate * this.extensionDuration;
        }
        
        this.showExtendDialog = false;
        this.selectedParkingToExtend = null;
        
        // Show success snackbar
        if (this.$root && this.$root.showSnackbar) {
          this.$root.showSnackbar('Parking has been extended!', 'success');
        } else {
          console.info('Snackbar: Parking has been extended!');
        }
        
        this.loading = false;
      }, 1000);
    }
  },
  
  watch: {
    selectedParkingPlan() {
      this.updateEndDate();
    },
    
    parkingDuration() {
      this.updateEndDate();
    },
    
    parkingStartDate() {
      this.updateEndDate();
    }
  },
  
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
    
    // Initialize the end date
    if (this.parkingStartDate) {
      const endDate = new Date(this.parkingStartDate);
      endDate.setDate(endDate.getDate() + 1); // Default to 1 day
      this.parkingEndDate = endDate;
    }
  },
  
  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent scrim="#1A3C28" :opacity="0.8">
        <div class="text-center">
            <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
            <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Processing your request...</div>
        </div>
      </v-overlay>

      <v-tabs v-model="activeTab" :bg-color="theme.primary" grow slider-color="secondary" class="booking-tabs-nav" dark>
        <v-tab value="register" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'register'}">
          <v-icon start>mdi-car-outline</v-icon> Register Parking
        </v-tab>
        <v-tab value="history" class="booking-tab-item" :class="{'booking-tab-item-active': activeTab === 'history'}">
          <v-icon start>mdi-history</v-icon> Parking History
        </v-tab>
      </v-tabs>

      <v-window v-model="activeTab" class="tropical-booking-main-content">
        <!-- REGISTER PARKING TAB -->
        <v-window-item value="register" class="pa-sm-4 pa-2">
          <v-card class="content-card mb-4" elevation="6">
            <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon start :color="theme.secondary">mdi-car-outline</v-icon>
              <span v-if="currentStep === 1">Car Registration</span>
              <span v-else-if="currentStep === 2">Parking Duration</span>
              <span v-else-if="currentStep === 3">Payment Details</span>
              <span v-else>Confirmation</span>
            </v-card-title>
            
            <!-- Progress Indicator -->
            <div class="px-4 pt-4">
              <div class="d-flex justify-space-between align-center mb-2">
                <div v-for="step in 4" :key="step" class="text-center" style="flex: 1;">
                  <v-avatar
                    :color="step === currentStep ? theme.primary : (step < currentStep ? theme.accent : theme.creamBg)"
                    :size="step === currentStep ? 36 : 28"
                    class="mb-1"
                    :class="{'elevation-2': step === currentStep}"
                  >
                    <v-icon :color="step <= currentStep ? theme.lightText : theme.primary" :size="step === currentStep ? 'default' : 'small'">
                      {{ step === 1 ? 'mdi-car-info' : 
                         step === 2 ? 'mdi-calendar-clock' : 
                         step === 3 ? 'mdi-credit-card-outline' : 
                         'mdi-check-circle-outline' }}
                    </v-icon>
                  </v-avatar>
                  <div class="text-caption d-none d-sm-block" :style="{color: step === currentStep ? theme.primary : theme.muted, fontWeight: step === currentStep ? 'bold' : 'normal'}">
                    {{ step === 1 ? 'Car Details' : 
                       step === 2 ? 'Parking Plan' : 
                       step === 3 ? 'Payment' : 
                       'Confirmation' }}
                  </div>
                </div>
              </div>
              <v-progress-linear
                :model-value="((currentStep - 1) / 3) * 100"
                :color="theme.primary"
                height="6"
                rounded
                class="mb-4"
              ></v-progress-linear>
            </div>

            <!-- Step Content -->
            <v-card-text>
              <!-- Step 1: Car Details -->
              <div v-if="currentStep === 1">
                <v-form v-model="carFormValid" class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Car Information</h3>
                  
                  <v-row>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.plateNumber"
                        :rules="[v => !!v || 'Car plate number is required']"
                        label="Car Plate Number*"
                        placeholder="e.g., ABC1234"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-card-account-details-outline"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.carMake"
                        label="Car Make"
                        placeholder="e.g., Toyota"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-car-estate"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.carModel"
                        label="Car Model"
                        placeholder="e.g., Camry"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-car-side"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.carColor"
                        label="Car Color"
                        placeholder="e.g., Black"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-palette"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                  </v-row>
                  
                  <p class="text-caption" :style="{color: theme.muted}">* Required fields</p>
                </v-form>
              </div>
              
              <!-- Step 2: Parking Plan -->
              <div v-else-if="currentStep === 2">
                <v-form class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Select Parking Plan</h3>
                  
                  <v-row>
                    <v-col v-for="plan in parkingPlanOptions" :key="plan.id" cols="12" sm="6">
                      <v-card
                        :elevation="selectedParkingPlan && selectedParkingPlan.id === plan.id ? 4 : 1"
                        :class="{ 'border': selectedParkingPlan && selectedParkingPlan.id === plan.id }"
                        :style="{ borderColor: selectedParkingPlan && selectedParkingPlan.id === plan.id ? theme.primary : 'transparent', borderWidth: '2px', borderStyle: 'solid' }"
                        hover
                        class="mb-2 facility-item"
                        @click="selectParkingPlan(plan)"
                      >
                        <v-card-text class="d-flex align-center">
                          <v-avatar :color="plan.color" size="40" class="mr-3 elevation-1">
                            <v-icon color="white">{{ plan.icon }}</v-icon>
                          </v-avatar>
                          <div class="flex-grow-1">
                            <div class="text-subtitle-1 font-weight-bold" :style="{ color: theme.primary }">{{ plan.name }}</div>
                            <div class="text-caption" :style="{ color: theme.muted }">RM {{ plan.rate }} per {{ plan.unit }}</div>
                          </div>
                          <v-radio-group v-model="selectedParkingPlan" :mandatory="false" hide-details class="ma-0 pa-0">
                            <v-radio :value="plan" :color="theme.primary"></v-radio>
                          </v-radio-group>
                        </v-card-text>
                      </v-card>
                    </v-col>
                  </v-row>
                  
                  <v-row v-if="selectedParkingPlan" class="mt-4">
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model.number="parkingDuration"
                        type="number"
                        min="1"
                        :rules="[v => v > 0 || 'Duration must be greater than 0']"
                        :label="'Duration (' + selectedParkingPlan.unit + 's)'"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-timer-outline"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-menu
                        v-model="startDateMenu"
                        :close-on-content-click="false"
                        transition="scale-transition"
                        offset-y
                        min-width="auto"
                      >
                        <template v-slot:activator="{ props }">
                          <v-text-field
                            v-model="formattedStartDate"
                            label="Start Date"
                            prepend-inner-icon="mdi-calendar"
                            readonly
                            variant="outlined"
                            :color="theme.primary"
                            density="comfortable"
                            v-bind="props"
                          ></v-text-field>
                        </template>
                        <v-date-picker
                          v-model="parkingStartDate"
                          :min="minDateForPicker"
                          :color="theme.primary"
                          @update:model-value="startDateMenu = false"
                        ></v-date-picker>
                      </v-menu>
                    </v-col>
                    <v-col cols="12">
                      <v-text-field
                        v-model="formattedEndDate"
                        label="End Date"
                        prepend-inner-icon="mdi-calendar-end"
                        readonly
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                  </v-row>
                  
                  <v-alert
                    v-if="selectedParkingPlan"
                    color="info"
                    class="mt-4"
                    variant="tonal"
                    icon="mdi-information"
                  >
                    <div class="font-weight-bold">Parking Cost: RM {{ formatMoney(parkingCost) }}</div>
                  </v-alert>
                </v-form>
              </div>
              
              <!-- Step 3: Payment -->
              <div v-else-if="currentStep === 3">
                <div class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Payment Method</h3>
                  
                  <v-row>
                    <v-col cols="12" md="6">
                      <v-card class="mb-4 summary-card" variant="outlined">
                        <v-card-title class="text-subtitle-1 pt-3 pb-2" :style="{color: theme.primary}">
                          Parking Summary
                        </v-card-title>
                        <v-card-text>
                          <div class="d-flex justify-space-between mb-1">
                            <span class="text-body-2">Car Plate:</span>
                            <span class="text-body-2 font-weight-medium">{{ carForm.plateNumber }}</span>
                          </div>
                          <div class="d-flex justify-space-between mb-1">
                            <span class="text-body-2">Start Date:</span>
                            <span class="text-body-2">{{ formatDate(parkingStartDate.toISOString().substring(0,10)) }}</span>
                          </div>
                          <div class="d-flex justify-space-between mb-1">
                            <span class="text-body-2">End Date:</span>
                            <span class="text-body-2">{{ formatDate(parkingEndDate.toISOString().substring(0,10)) }}</span>
                          </div>
                          <div class="d-flex justify-space-between mb-1">
                            <span class="text-body-2">Duration:</span>
                            <span class="text-body-2">{{ parkingDuration }} {{ selectedParkingPlan.unit }}(s)</span>
                          </div>
                          <div class="d-flex justify-space-between mb-1">
                            <span class="text-body-2">Rate:</span>
                            <span class="text-body-2">RM {{ formatMoney(selectedParkingPlan.rate) }} per {{ selectedParkingPlan.unit }}</span>
                          </div>
                          <v-divider class="my-2"></v-divider>
                          <div class="d-flex justify-space-between">
                            <span class="text-subtitle-2 font-weight-bold">Total:</span>
                            <span class="text-subtitle-2 font-weight-bold">RM {{ formatMoney(parkingCost) }}</span>
                          </div>
                        </v-card-text>
                      </v-card>
                    </v-col>
                    
                    <v-col cols="12" md="6">
                      <h4 class="text-subtitle-2 mb-2" :style="{color: theme.primary}">Choose Payment Method</h4>
                      
                      <v-list class="payment-methods-list mb-4 facility-selection-list" select-strategy="single">
                        <v-list-item
                          v-for="method in paymentMethods"
                          :key="method.id"
                          :title="method.name"
                          :subtitle="method.description"
                          :prepend-icon="method.icon"
                          :active="selectedPaymentMethod === method.id"
                          :active-color="theme.primary"
                          rounded="lg"
                          class="facility-item mb-2"
                          @click="selectPaymentMethod(method)"
                        ></v-list-item>
                      </v-list>
                    </v-col>
                    
                    <v-col cols="12" v-if="selectedPaymentMethod === 'credit' || selectedPaymentMethod === 'debit'">
                      <v-form v-model="cardFormValid" class="mb-4">
                        <v-row>
                          <v-col cols="12">
                            <v-text-field
                              v-model="cardDetails.cardNumber"
                              label="Card Number"
                              :rules="[v => !!v || 'Card number is required', v => v.replace(/\\s/g, '').length >= 13 || 'Invalid card number']"
                              placeholder="0000 0000 0000 0000"
                              variant="outlined"
                              :color="theme.primary"
                              prepend-inner-icon="mdi-credit-card"
                              density="comfortable"
                            ></v-text-field>
                          </v-col>
                          <v-col cols="12">
                            <v-text-field
                              v-model="cardDetails.cardholderName"
                              label="Cardholder Name"
                              :rules="[v => !!v || 'Cardholder name is required']"
                              placeholder="John Smith"
                              variant="outlined"
                              :color="theme.primary"
                              prepend-inner-icon="mdi-account"
                              density="comfortable"
                            ></v-text-field>
                          </v-col>
                          <v-col cols="6">
                            <v-text-field
                              v-model="cardDetails.expiryDate"
                              label="Expiry Date (MM/YY)"
                              :rules="[v => !!v || 'Expiry date is required', v => /^\\d{2}\\/\\d{2}$/.test(v) || 'Format should be MM/YY']"
                              placeholder="MM/YY"
                              variant="outlined"
                              :color="theme.primary"
                              prepend-inner-icon="mdi-calendar"
                              density="comfortable"
                              maxlength="5"
                            ></v-text-field>
                          </v-col>
                          <v-col cols="6">
                            <v-text-field
                              v-model="cardDetails.cvv"
                              label="CVV"
                              :rules="[v => !!v || 'CVV is required', v => v.length >= 3 || 'Invalid CVV']"
                              type="password"
                              placeholder="123"
                              variant="outlined"
                              :color="theme.primary"
                              prepend-inner-icon="mdi-shield"
                              density="comfortable"
                              maxlength="4"
                            ></v-text-field>
                          </v-col>
                        </v-row>
                      </v-form>
                    </v-col>
                    
                    <v-col cols="12" v-if="selectedPaymentMethod === 'ewallet'">
                      <h4 class="text-subtitle-2 mb-2" :style="{color: theme.primary}">Select E-Wallet</h4>
                      
                      <v-row>
                        <v-col 
                          v-for="wallet in eWallets" 
                          :key="wallet.id"
                          cols="6" sm="3"
                        >
                          <v-card
                            :elevation="selectedEWallet === wallet.id ? 4 : 1"
                            :class="{ 'border': selectedEWallet === wallet.id }"
                            :style="{ borderColor: selectedEWallet === wallet.id ? theme.primary : 'transparent', borderWidth: '2px', borderStyle: 'solid' }"
                            hover
                            @click="selectEWallet(wallet)"
                            height="100"
                            class="d-flex flex-column align-center justify-center"
                          >
                            <v-img
                              :src="wallet.logo"
                              width="60"
                              height="30"
                              contain
                              class="mb-2"
                            ></v-img>
                            <div class="text-caption font-weight-medium text-center">{{ wallet.name }}</div>
                          </v-card>
                        </v-col>
                      </v-row>
                    </v-col>
                    
                    <v-col cols="12" v-if="selectedPaymentMethod === 'cash'">
                      <v-alert
                        :color="theme.warning"
                        icon="mdi-cash-multiple"
                        variant="tonal"
                      >
                        Please proceed to the hotel reception to make your cash payment. Your parking slot will be reserved for the next 30 minutes.
                      </v-alert>
                    </v-col>
                  </v-row>
                </div>
              </div>
              
              <!-- Step 4: Confirmation -->
              <div v-else>
                <div class="py-3">
                  <v-alert
                    :color="theme.success"
                    icon="mdi-check-circle"
                    variant="tonal"
                    class="mb-4"
                  >
                    <p class="text-h6 font-weight-bold mb-1">Parking Registration Successful!</p>
                    <p class="mb-0">Your vehicle has been successfully registered for parking.</p>
                  </v-alert>
                  
                  <v-row v-if="receiptDetails">
                    <v-col cols="12" md="10" class="mx-auto">
                      <v-card class="receipt-card" variant="outlined">
                        <div class="receipt-header pa-4 text-center" :style="{ backgroundColor: theme.primary, color: theme.lightText }">
                          <h3 class="text-h6 font-weight-bold">PARKING RECEIPT</h3>
                          <div class="text-caption">Rainforest Resort</div>
                        </div>
                        
                        <v-card-text class="pa-4">
                          <v-row>
                            <v-col cols="12" md="6">
                              <div class="text-subtitle-2 font-weight-bold mb-1" :style="{ color: theme.primary }">Parking Details</div>
                              <v-list density="compact" class="bg-transparent pa-0">
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-car</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">License Plate: {{ receiptDetails.plateNumber }}</v-list-item-title>
                                </v-list-item>
                                
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-calendar-range</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">Period: {{ formatDateRange(receiptDetails.startDate, receiptDetails.endDate) }}</v-list-item-title>
                                </v-list-item>
                                
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-timer-outline</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">Duration: {{ receiptDetails.duration }} {{ receiptDetails.durationUnit }}(s)</v-list-item-title>
                                </v-list-item>
                              </v-list>
                            </v-col>
                            
                            <v-col cols="12" md="6">
                              <div class="text-subtitle-2 font-weight-bold mb-1" :style="{ color: theme.primary }">Payment Information</div>
                              <v-list density="compact" class="bg-transparent pa-0">
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-receipt-text-outline</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">Receipt #: {{ receiptDetails.transactionId }}</v-list-item-title>
                                </v-list-item>
                                
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-credit-card-outline</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">Paid via: {{ receiptDetails.paymentMethod }}</v-list-item-title>
                                </v-list-item>
                                
                                <v-list-item density="compact">
                                  <template v-slot:prepend>
                                    <v-icon :color="theme.primary" size="small">mdi-cash</v-icon>
                                  </template>
                                  <v-list-item-title class="text-body-2">Amount: RM {{ formatMoney(receiptDetails.amount) }}</v-list-item-title>
                                </v-list-item>
                              </v-list>
                            </v-col>
                          </v-row>
                          
                          <v-divider class="my-3"></v-divider>
                          
                          <div class="text-center">
                            <div class="text-caption" :style="{ color: theme.muted }">Date of Issue: {{ new Date(receiptDetails.paymentDate).toLocaleDateString() }} {{ new Date(receiptDetails.paymentDate).toLocaleTimeString() }}</div>
                            <v-icon :color="theme.success" size="large" class="mt-2">mdi-check-decagram</v-icon>
                          </div>
                        </v-card-text>
                        
                        <v-card-actions class="pa-4 justify-center">
                          <v-btn
                            :color="theme.primary"
                            :text-color="theme.lightText"
                            prepend-icon="mdi-printer"
                          >
                            Print Receipt
                          </v-btn>
                          <v-btn
                            :color="theme.secondary"
                            :text-color="theme.primary"
                            class="ml-2"
                            prepend-icon="mdi-email-outline"
                          >
                            Email Receipt
                          </v-btn>
                        </v-card-actions>
                      </v-card>
                    </v-col>
                  </v-row>
                </div>
              </div>
            </v-card-text>
            
            <!-- Navigation Buttons -->
            <v-card-actions class="pa-4">
              <v-btn 
                v-if="currentStep > 1"
                :color="theme.muted" 
                variant="text"
                @click="prevStep"
              >
                <v-icon start>mdi-arrow-left</v-icon>
                Back
              </v-btn>
              <v-spacer></v-spacer>
              <v-btn v-if="currentStep === 4" 
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="resetForm"
                elevation="2"
              >
                <v-icon start>mdi-plus-circle</v-icon>
                Register New Parking
              </v-btn>
              <v-btn v-else-if="currentStep === 3"
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="processPayment"
                :disabled="!selectedPaymentMethod || !canMakePayment"
                size="large"
                elevation="2"
                :loading="loading"
              >
                <v-icon start>mdi-check-circle</v-icon>
                Pay RM {{ formatMoney(parkingCost) }}
              </v-btn>
              <v-btn v-else-if="currentStep === 2"
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="nextStep"
                :disabled="!selectedParkingPlan || !parkingDuration || parkingDuration <= 0"
                size="large"
                elevation="2"
              >
                Continue
                <v-icon end>mdi-arrow-right</v-icon>
              </v-btn>
              <v-btn v-else-if="currentStep === 1"
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="nextStep"
                :disabled="!carFormValid"
                size="large"
                elevation="2"
              >
                Continue
                <v-icon end>mdi-arrow-right</v-icon>
              </v-btn>
            </v-card-actions>
          </v-card>
        </v-window-item>

        <!-- PARKING HISTORY TAB -->
        <v-window-item value="history" class="pa-sm-4 pa-2">
          <v-card class="content-card" elevation="6">
            <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon start :color="theme.secondary">mdi-history</v-icon> Parking History
            </v-card-title>
            <v-card-text class="pa-sm-4 pa-2">
              <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3 d-flex align-center" :style="{color: theme.primary}">
                <v-icon :color="theme.success" class="mr-2">mdi-car-clock</v-icon>
                Active Parking <v-chip class="ml-2" size="small" :color="theme.success" :text-color="theme.lightText" label>{{ activeParkings.length }}</v-chip>
              </h3>
              <div v-if="activeParkings.length === 0" class="empty-bookings-state text-center pa-5 mb-6">
                <v-icon size="60" :color="theme.accent" class="mb-2">mdi-car-off</v-icon>
                <p :class="$vuetify.display.xs ? 'text-body-1' : 'text-h6'" :style="{color: theme.darkText}">No active parking!</p>
                <p :style="{color: theme.muted, fontSize: $vuetify.display.xs ? '0.8rem' : 'inherit'}">Your vehicle is not currently parked with us.</p>
                <v-btn :color="theme.primary" :text-color="theme.lightText" class="mt-4 elevation-2" @click="activeTab = 'register'" :size="$vuetify.display.xs ? 'small' : 'default'">
                  <v-icon start>mdi-car-outline</v-icon> Register New Parking
                </v-btn>
              </div>
              <v-expansion-panels v-else accordion class="mb-6 themed-expansion-panels" variant="accordion">
                <v-expansion-panel
                  v-for="parking in activeParkings" :key="parking.id"
                  class="booking-panel-item elevation-2" :style="{'border-left-color': getStatusDetails(parking.status).color + ' !important'}"
                  rounded="lg"
                >
                  <v-expansion-panel-title class="booking-panel-title" :class="{'booking-panel-title-mobile': $vuetify.display.xs}">
                    <v-avatar :color="theme.primary" :size="$vuetify.display.xs ? 36 : 40" class="mr-sm-4 mr-2 elevation-1">
                      <v-icon :color="theme.lightText">mdi-car</v-icon>
                    </v-avatar>
                    <div class="flex-grow-1">
                      <div class="font-weight-bold" :class="$vuetify.display.xs ? 'text-body-2' : 'text-subtitle-1'" :style="{color: theme.primary}">{{ parking.plateNumber }}</div>
                      <div :class="$vuetify.display.xs ? 'text-caption' : 'text-body-2'" :style="{color: theme.muted}">{{ formatDateRange(parking.startDate, parking.endDate) }}</div>
                    </div>
                    <v-chip label :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="getStatusDetails(parking.status).color" :text-color="theme.lightText" class="status-chip-panel ml-1">
                      <v-icon start :size="$vuetify.display.xs ? 'x-small' : 'small'">{{ getStatusDetails(parking.status).icon }}</v-icon>
                      {{ getStatusDetails(parking.status).text }}
                    </v-chip>
                  </v-expansion-panel-title>
                  <v-expansion-panel-text class="booking-panel-content">
                    <v-list dense class="py-0 bg-transparent" :class="{'list-mobile-dense': $vuetify.display.xs}">
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.primary">mdi-calendar-range</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Period:</strong> {{ formatDateRange(parking.startDate, parking.endDate) }}
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.primary">mdi-timer</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Duration:</strong> {{ parking.duration }} {{ parking.durationUnit }}(s)
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.primary">mdi-cash</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Amount Paid:</strong> RM {{ formatMoney(parking.amount) }}
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item density="compact" v-if="parking.carMake || parking.carModel">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.primary">mdi-car</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Vehicle:</strong> {{ [parking.carMake, parking.carModel, parking.carColor].filter(Boolean).join(' ') }}
                        </v-list-item-title>
                      </v-list-item>
                    </v-list>
                    
                    <v-card-actions class="pa-1 justify-end">
                      <v-btn :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="theme.warning" variant="text" @click.stop="initiateExtension(parking)">
                        <v-icon start>mdi-calendar-plus</v-icon> Extend
                      </v-btn>
                      <v-btn :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="theme.error" variant="text" @click.stop="initiateRefund(parking)">
                        <v-icon start>mdi-cash-refund</v-icon> Request Refund
                      </v-btn>
                    </v-card-actions>
                  </v-expansion-panel-text>
                </v-expansion-panel>
              </v-expansion-panels>

              <v-divider class="my-6"></v-divider>

              <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3 d-flex align-center" :style="{color: theme.primary}">
                <v-icon :color="theme.muted" class="mr-2">mdi-history</v-icon>
                Past Parking <v-chip class="ml-2" size="small" :color="theme.muted" :text-color="theme.lightText" label>{{ pastParkings.length }}</v-chip>
              </h3>
              <div v-if="pastParkings.length === 0" class="empty-bookings-state text-center pa-5">
                <v-icon size="60" :color="theme.accent" class="mb-2">mdi-archive-outline</v-icon>
                <p :style="{color: theme.darkText, fontSize: $vuetify.display.xs ? '0.9rem' : 'inherit'}">No past parking history.</p>
              </div>
              <v-expansion-panels v-else accordion class="themed-expansion-panels" variant="accordion">
                <v-expansion-panel
                  v-for="parking in pastParkings" :key="parking.id"
                  class="booking-panel-item past-booking elevation-1" :style="{'border-left-color': getStatusDetails(parking.status).color + ' !important'}"
                  rounded="lg"
                >
                  <v-expansion-panel-title class="booking-panel-title" :class="{'booking-panel-title-mobile': $vuetify.display.xs}">
                    <v-avatar :color="getStatusDetails(parking.status).color" :size="$vuetify.display.xs ? 36 : 40" class="mr-sm-4 mr-2 elevation-1" style="opacity: 0.7;">
                      <v-icon :color="theme.lightText">mdi-car</v-icon>
                    </v-avatar>
                    <div class="flex-grow-1" style="opacity: 0.7;">
                      <div class="font-weight-medium" :class="$vuetify.display.xs ? 'text-body-2' : 'text-subtitle-1'" :style="{color: theme.darkText}">{{ parking.plateNumber }}</div>
                      <div :class="$vuetify.display.xs ? 'text-caption' : 'text-body-2'" :style="{color: theme.muted}">{{ formatDateRange(parking.startDate, parking.endDate) }}</div>
                    </div>
                    <v-chip label :size="$vuetify.display.xs ? 'x-small' : 'small'" :color="getStatusDetails(parking.status).color" :text-color="theme.lightText" class="status-chip-panel ml-1">
                      <v-icon start :size="$vuetify.display.xs ? 'x-small' : 'small'">{{ getStatusDetails(parking.status).icon }}</v-icon>
                      {{ getStatusDetails(parking.status).text }}
                    </v-chip>
                  </v-expansion-panel-title>
                  <v-expansion-panel-text class="booking-panel-content">
                    <v-list dense class="py-0 bg-transparent" :class="{'list-mobile-dense': $vuetify.display.xs}">
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.muted">mdi-calendar-range</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Period:</strong> {{ formatDateRange(parking.startDate, parking.endDate) }}
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.muted">mdi-timer</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Duration:</strong> {{ parking.duration }} {{ parking.durationUnit }}(s)
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item density="compact">
                        <template v-slot:prepend>
                          <v-icon small start :color="theme.muted">mdi-cash</v-icon>
                        </template>
                        <v-list-item-title>
                          <strong>Amount Paid:</strong> RM {{ formatMoney(parking.amount) }}
                        </v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-expansion-panel-text>
                </v-expansion-panel>
              </v-expansion-panels>
            </v-card-text>
          </v-card>
        </v-window-item>
      </v-window>

      <!-- Success Dialog -->
      <v-dialog
        v-model="showSuccessDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '500px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        persistent
        transition="dialog-bottom-transition"
      >
        <v-card>
          <v-toolbar :color="theme.success" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2">mdi-check-circle</v-icon>
            <v-toolbar-title class="dialog-title">Payment Successful</v-toolbar-title>
          </v-toolbar>
          <v-card-text class="pa-4 text-center">
            <v-icon :color="theme.success" size="64" class="my-4">mdi-check-circle</v-icon>
            <h3 class="text-h5 mb-2" :style="{color: theme.primary}">Thank You!</h3>
            <p class="mb-4">Your parking has been successfully registered.</p>
            
            <v-alert
              :color="theme.primary"
              :text-color="theme.lightText"
              icon="mdi-information"
              density="comfortable"
              class="text-left mb-4"
            >
              <div class="font-weight-bold">Please remember:</div>
              <ul class="mt-2 ps-3">
                <li>Park only in designated areas</li>
                <li>Keep your receipt for validation</li>
                <li>Display your e-permit if provided</li>
              </ul>
            </v-alert>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 justify-center">
            <v-btn
              :color="theme.primary"
              :text-color="theme.lightText"
              @click="showSuccessDialog = false"
              size="large"
              elevation="2"
            >
              Close
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>

      <!-- Refund Dialog -->
      <v-dialog
        v-model="showRefundDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '450px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        persistent
        transition="dialog-bottom-transition"
      >
        <v-card v-if="selectedParkingToCancel">
          <v-toolbar :color="theme.error" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2">mdi-cash-refund</v-icon>
            <v-toolbar-title class="dialog-title">Request Refund</v-toolbar-title>
            <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showRefundDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4 text-center">
            <p :class="$vuetify.display.xs ? 'text-body-2' : 'text-body-1'" class="mt-3">Request refund for parking:</p>
            <p :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="font-weight-medium my-2" :style="{color: theme.primary}">{{ selectedParkingToCancel.plateNumber }}</p>
            <p class="text-body-2" :style="{color: theme.muted}">{{ formatDateRange(selectedParkingToCancel.startDate, selectedParkingToCancel.endDate) }}</p>
            <v-alert type="warning" density="compact" outlined class="mt-4 text-left text-caption">
              <div class="font-weight-medium">Refund Policy:</div>
              <ul class="mt-1 ps-3 text-caption">
                <li>Full refund if cancelled more than 24 hours before start time</li>
                <li>50% refund if cancelled within 24 hours of start time</li>
                <li>No refund for cancellations after parking has started</li>
              </ul>
            </v-alert>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 dialog-actions-sticky">
            <v-btn :color="theme.muted" variant="text" @click="showRefundDialog = false" :size="$vuetify.display.xs ? 'small' : 'default'">Cancel</v-btn>
            <v-spacer></v-spacer>
            <v-btn :color="theme.error" :text-color="theme.lightText" :size="$vuetify.display.xs ? 'small' : 'default'" elevation="2" @click="confirmRefund" :loading="loading">
              <v-icon start>mdi-cash-refund</v-icon> Confirm Refund
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
      
      <!-- Extend Dialog -->
      <v-dialog
        v-model="showExtendDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '450px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        persistent
        transition="dialog-bottom-transition"
      >
        <v-card v-if="selectedParkingToExtend">
          <v-toolbar :color="theme.primary" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2" :color="theme.secondary">mdi-calendar-plus</v-icon>
            <v-toolbar-title class="dialog-title">Extend Parking</v-toolbar-title>
            <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showExtendDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4">
            <v-alert border="start" :border-color="theme.accent" colored-border elevation="2" class="mb-4">
              <div class="mb-1">Extending parking for:</div>
              <div class="font-weight-bold" :style="{color: theme.primary}">{{ selectedParkingToExtend.plateNumber }}</div>
              <div class="text-caption">Current End Date: {{ formatDate(selectedParkingToExtend.endDate) }}</div>
            </v-alert>
            
            <v-text-field
              v-model.number="extensionDuration"
              label="Additional Duration"
              type="number"
              min="1"
              :hint="'Add ' + extensionDuration + ' more ' + selectedParkingToExtend.durationUnit + (extensionDuration > 1 ? 's' : '')"
              :rules="[v => v > 0 || 'Extension must be greater than 0']"
              variant="outlined"
              :color="theme.primary"
              density="comfortable"
              persistent-hint
            ></v-text-field>
            
            <v-alert
              :color="theme.accent"
              icon="mdi-information"
              variant="tonal"
              class="mt-4 mb-2"
              density="comfortable"
            >
              <div class="font-weight-medium">Extension Cost:</div>
              <div class="font-weight-bold" :style="{color: theme.primary}">RM {{ formatMoney(extensionRate * extensionDuration) }}</div>
            </v-alert>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 dialog-actions-sticky">
            <v-btn :color="theme.muted" variant="text" @click="showExtendDialog = false" :size="$vuetify.display.xs ? 'small' : 'default'">Cancel</v-btn>
            <v-spacer></v-spacer>
            <v-btn :color="theme.primary" :text-color="theme.lightText" :size="$vuetify.display.xs ? 'small' : 'default'" elevation="2" @click="confirmExtension" :loading="loading" :disabled="extensionDuration <= 0">
              <v-icon start>mdi-check-circle</v-icon> Extend & Pay
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
    </v-container>
  `
};
