import getTropicalStyles from "../css/tropicalBookingStyles.js"; 

export default {
  props: ["jsonData"],
  data() {
    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',
        secondary: '#DDA333',
        accent: '#A3BA95',
        lightBg: '#F8F5F0',
        creamBg: '#EADDCB',
        darkText: '#1E1E1E',
        lightText: '#FFFFFF',
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        muted: '#78909C',
        brown: '#6D4C41',
      },

      // Current step in the car registration process
      currentStep: 1,
      
      // Car information
      carForm: {
        plateNumber: '',
        carMake: '',
        carModel: '',
        carColor: '',
        ownerName: '',
        ownerPhone: '',
        relationship: 'self', // self, family, friend
        purpose: '',
        startDate: new Date().toISOString().substring(0,10), // Default to today
        endDate: '', // Will be set by user
      },
      carFormValid: false,
      
      // Relationship options
      relationshipOptions: [
        { id: 'self', name: 'Self (Member)' },
        { id: 'family', name: 'Family Member' },
        { id: 'friend', name: 'Friend/Guest' },
      ],
      
      // Request history - now empty arrays as we'll fetch from API
      vehicles: [],
      registrations: [],
      
      // UI control
      activeTab: 'register',
      loading: false,
      showSuccessDialog: false,
      showInfoDialog: false,
      infoDialogContent: {
        title: '',
        message: '',
        icon: ''
      },
      
      // Snackbar for notifications
      showSnackbar: false,
      snackbar: {
        text: '',
        color: 'success',
        timeout: 3000
      },
      
      // Define backgroundStyle directly instead of expecting it from getTropicalStyles
      backgroundStyle: {
        position: 'fixed',
        top: 0,
        left: 0,
        width: '100%',
        height: '100%',
        background: 'linear-gradient(135deg, #EADDCB 0%, #F8F5F0 100%)',
        zIndex: -1,
        opacity: 1
      }
    };
  },
  
  computed: {
    activeRequests() {
      return this.registrations.filter(reg => reg.status === 'PENDING');
    },
    
    approvedRequests() {
      return this.registrations.filter(reg => reg.status === 'APPROVED' || reg.status === 'ACTIVE');
    },
    
    rejectedRequests() {
      return this.registrations.filter(reg => reg.status === 'REJECTED');
    },
  },
  
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme) + `
      /* Custom styles for car registration */
      .step-indicator {
        display: flex;
        flex-direction: column;
        align-items: center;
        position: relative;
        z-index: 1;
      }
      
      .step-number {
        width: 36px;
        height: 36px;
        border-radius: 50%;
        background-color: #EADDCB;
        color: #1A3C28;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        margin-bottom: 8px;
        border: 2px solid #DDA333;
        transition: all 0.3s ease;
      }
      
      .step-label {
        font-size: 0.85rem;
        color: #1A3C28;
        font-weight: 500;
        text-align: center;
      }
      
      .step-line {
        flex: 1;
        height: 2px;
        background-color: #EADDCB;
        position: relative;
        margin: 0 8px;
        top: -26px;
        transition: background-color 0.3s ease;
      }
      
      .step-indicator.active .step-number {
        background-color: #1A3C28;
        color: white;
        transform: scale(1.1);
        box-shadow: 0 4px 8px rgba(0,0,0,0.15);
      }
      
      .step-indicator.completed .step-number {
        background-color: #2E7D32;
        color: white;
      }
      
      .step-line.active {
        background-color: #DDA333;
        height: 3px;
      }
      
      .content-card {
        border-radius: 12px !important;
        overflow: hidden;
        box-shadow: 0 6px 16px rgba(0,0,0,0.08) !important;
        border: none !important;
      }
      
      .card-header {
        padding: 16px 20px !important;
        font-size: 1.2rem !important;
        letter-spacing: 0.5px;
      }
      
      .v-tabs {
        border-radius: 8px 8px 0 0;
        overflow: hidden;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
      }
      
      .v-tab {
        font-weight: 600 !important;
        letter-spacing: 0.5px;
        text-transform: none !important;
        min-height: 48px;
      }
      
      .v-text-field {
        margin-bottom: 8px;
      }
      
      /*.v-btn {
        text-transform: none !important;
        font-weight: 600;
        letter-spacing: 0.5px;
        border-radius: 8px !important;
      }*/
      
      .v-card-text {
        padding: 24px !important;
      }
      
      .v-chip {
        font-weight: 600;
      }
    `;
    document.head.appendChild(styleElement);
    
    // Fetch registration data when component is mounted
    this.fetchRegistrations();
  },
  
  methods: {
    // Fetch registrations from API
    fetchRegistrations() {
      this.loading = true;
      
      const requestData = {
        user_id: sessionStorage.getItem('id')
      };
      
      this.$root.callApi('parking', 'get_registrations', requestData)
        .then(response => {
          this.loading = false;
          
          if (response.data && response.data.status === "OK") {
            // Store the raw vehicles data
            this.vehicles = response.data.data.vehicles || [];
            
            // Process registrations to match UI format
            this.registrations = [];
            
            // Process each vehicle and its registrations
            this.vehicles.forEach(vehicle => {
              if (vehicle.registrations && vehicle.registrations.length > 0) {
                vehicle.registrations.forEach(reg => {
                  this.registrations.push({
                    id: reg.id,
                    plateNumber: vehicle.plate_number,
                    carMake: vehicle.make,
                    carModel: vehicle.model,
                    carColor: vehicle.color,
                    ownerName: vehicle.owner_name,
                    ownerPhone: vehicle.owner_phone,
                    relationship: vehicle.relationship,
                    requestDate: reg.created_at,
                    status: reg.status,
                    approvalDate: reg.updated_at,
                    startDate: reg.start_date,
                    endDate: reg.end_date,
                    paymentStatus: reg.payment_status
                  });
                });
              }
            });
            
            console.log('Processed registrations:', this.registrations);
          } else {
            // Show error message
            this.showInfo('Error', response.data?.message || 'Failed to fetch car registration data', 'mdi-alert-circle');
          }
        })
        .catch(error => {
          this.loading = false;
          console.error('Error:', error);
          this.showInfo('Error', 'An unexpected error occurred while fetching registration data.', 'mdi-alert-circle');
        });
    },
    
    // Navigation methods
    nextStep() {
      if (this.currentStep < 3) {
        this.currentStep++;
        window.scrollTo(0, 0);
      }
    },
    
    prevStep() {
      if (this.currentStep > 1) {
        this.currentStep--;
        window.scrollTo(0, 0);
      }
    },
    
    resetForm() {
      this.currentStep = 1;
      this.carForm = {
        plateNumber: '',
        carMake: '',
        carModel: '',
        carColor: '',
        ownerName: '',
        ownerPhone: '',
        relationship: 'self',
        purpose: '',
        startDate: new Date().toISOString().substring(0,10),
        endDate: ''
      };
    },
    
    // Submit request
    submitRequest() {
      this.loading = true;
      
      // Prepare data for API call
      const requestData = {
        user_id: sessionStorage.getItem('id'), // Get user_id from sessionStorage instead of jsonData
        plate_number: this.carForm.plateNumber.toUpperCase(),
        car_make: this.carForm.carMake,
        car_model: this.carForm.carModel,
        car_color: this.carForm.carColor,
        owner_name: this.carForm.ownerName,
        owner_phone: this.carForm.ownerPhone,
        relationship: this.carForm.relationship,
        purpose: this.carForm.purpose,
        start_date: this.carForm.startDate,
        end_date: this.carForm.endDate
      };
      
      // Call the API using the callApi function
      this.$root.callApi('parking', 'submit_registration', requestData)
        .then(response => {
          this.loading = false;
          
          if (response.data && response.data.status === "OK") {
            // After successful submission, refresh the registrations list
            this.fetchRegistrations();
            this.currentStep = 3; // Move to confirmation step
            this.showSuccessDialog = true;
          } else {
            // Show error message
            this.showInfo('Error', response.data?.message || 'Failed to submit car registration request', 'mdi-alert-circle');
          }
        })
        .catch(error => {
          this.loading = false;
          console.error('Error:', error);
          this.showInfo('Error', 'An unexpected error occurred. Please try again later.', 'mdi-alert-circle');
        });
    },
    
    // Format date for display
    formatDate(dateString) {
      if (!dateString) return "N/A";
      
      const date = new Date(dateString);
      return date.toLocaleDateString('en-MY', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
      });
    },
    
    // Get status details for display
    getStatusDetails(status) {
      const statusMap = {
        "PENDING": {
          text: "Pending",
          color: this.theme.warning,
          icon: "mdi-clock-outline"
        },
        "PROCESSING": {
          text: "Processing",
          color: this.theme.accent,
          icon: "mdi-progress-clock"
        },
        "APPROVED": {
          text: "Approved",
          color: this.theme.success,
          icon: "mdi-check-circle"
        },
        "REJECTED": {
          text: "Rejected",
          color: this.theme.error,
          icon: "mdi-close-circle"
        }
      };
      
      return statusMap[status] || {
        text: status,
        color: this.theme.muted,
        icon: "mdi-help-circle"
      };
    },
    
    // Show info dialog
    showInfo(title, message, icon = "mdi-information") {
      this.infoDialogContent = {
        title,
        message,
        icon
      };
      
      this.showInfoDialog = true;
    }
  },
  
  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent :scrim="theme.primary" :opacity="0.8">
        <div class="text-center">
          <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
          <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Processing your request...</div>
        </div>
      </v-overlay>
      
      <!-- Tab Navigation -->
      <v-tabs
        v-model="activeTab"
        :color="theme.secondary"
        :bg-color="theme.primary"
        :text-color="theme.lightText"
        align-tabs="center"
        class="mb-4 mt-2 elevation-2 rounded-t-lg"
        grow
        slider-color="theme.secondary"
      >
        <v-tab value="register" class="text-body-1 py-3 px-4">
          <v-icon start class="mr-2">mdi-car-key</v-icon>
          Register Car
        </v-tab>
        <v-tab value="history" class="text-body-1 py-3 px-4">
          <v-icon start class="mr-2">mdi-history</v-icon>
          Request History
        </v-tab>
      </v-tabs>
      
      <v-window v-model="activeTab">
        <!-- REGISTER CAR TAB -->
        <v-window-item value="register" class="pa-sm-4 pa-2">
          <v-card class="content-card mb-4" elevation="6">
            <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon start :color="theme.secondary">mdi-car-key</v-icon>
              <span>Car Registration</span>
            </v-card-title>
            
            <v-card-text class="pa-4">
              <!-- Step 1: Progress Indicator -->
              <div class="mb-5 mt-2 px-4">
                <div class="d-flex justify-space-between align-center">
                  <div class="step-indicator" :class="{active: currentStep >= 1, completed: currentStep > 1}">
                    <div class="step-number">1</div>
                    <div class="step-label">Car Details</div>
                  </div>
                  <div class="step-line" :class="{active: currentStep > 1}"></div>
                  <div class="step-indicator" :class="{active: currentStep >= 2, completed: currentStep > 2}">
                    <div class="step-number">2</div>
                    <div class="step-label">Owner Info</div>
                  </div>
                  <div class="step-line" :class="{active: currentStep > 2}"></div>
                  <div class="step-indicator" :class="{active: currentStep >= 3}">
                    <div class="step-number">3</div>
                    <div class="step-label">Confirmation</div>
                  </div>
                </div>
              </div>
              
              <!-- Step 1: Car Details -->
              <div v-if="currentStep === 1" class="py-2">
                <v-form v-model="carFormValid" @submit.prevent="nextStep">
                  <v-row class="mb-2">
                    <v-col cols="12">
                      <v-text-field
                        v-model="carForm.plateNumber"
                        label="License Plate Number*"
                        :rules="[v => !!v || 'License plate number is required']"
                        placeholder="e.g., ABC1234"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-car"
                        density="comfortable"
                        class="rounded-lg"
                        bg-color="white"
                      ></v-text-field>
                    </v-col>
                  </v-row>
                  <v-row>
                    <v-col cols="12" sm="4">
                      <v-text-field
                        v-model="carForm.carMake"
                        label="Car Make*"
                        :rules="[v => !!v || 'Car make is required']"
                        placeholder="e.g., Toyota"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-car-side"
                        density="comfortable"
                        class="rounded-lg"
                        bg-color="white"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="4">
                      <v-text-field
                        v-model="carForm.carModel"
                        label="Car Model*"
                        :rules="[v => !!v || 'Car model is required']"
                        placeholder="e.g., Camry"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-car-info"
                        density="comfortable"
                        class="rounded-lg"
                        bg-color="white"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="4">
                      <v-text-field
                        v-model="carForm.carColor"
                        label="Car Color*"
                        :rules="[v => !!v || 'Car color is required']"
                        placeholder="e.g., Silver"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-palette"
                        density="comfortable"
                        class="rounded-lg"
                        bg-color="white"
                      ></v-text-field>
                    </v-col>
                  </v-row>
                </v-form>
              </div>
              
              <!-- Step 2: Owner Information -->
              <div v-else-if="currentStep === 2">
                <v-form v-model="carFormValid" @submit.prevent="submitRequest">
                  <v-row>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.ownerName"
                        label="Owner Name*"
                        :rules="[v => !!v || 'Owner name is required']"
                        placeholder="e.g., John Smith"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-account"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.ownerPhone"
                        label="Contact Number*"
                        :rules="[v => !!v || 'Contact number is required']"
                        placeholder="e.g., 012-3456789"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-phone"
                        density="comfortable"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.startDate"
                        label="Start Date*"
                        :rules="[v => !!v || 'Start date is required']"
                        type="date"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-calendar-start"
                        density="comfortable"
                        class="themed-date-picker"
                        style="min-height: 56px;"
                        :style="{'--v-input-control-height': '56px'}"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="6">
                      <v-text-field
                        v-model="carForm.endDate"
                        label="End Date*"
                        :rules="[v => !!v || 'End date is required', v => !carForm.startDate || !v || new Date(v) >= new Date(carForm.startDate) || 'End date must be after start date']"
                        type="date"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-calendar-end"
                        density="comfortable"
                        class="themed-date-picker"
                        style="min-height: 56px;"
                        :style="{'--v-input-control-height': '56px'}"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12">
                      <v-select
                        v-model="carForm.relationship"
                        :items="relationshipOptions"
                        item-title="name"
                        item-value="id"
                        label="Relationship to Member*"
                        :rules="[v => !!v || 'Relationship is required']"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-account-group"
                        density="comfortable"
                      ></v-select>
                    </v-col>
                    <v-col cols="12">
                      <v-textarea
                        v-model="carForm.purpose"
                        label="Purpose of Registration*"
                        :rules="[v => !!v || 'Purpose is required']"
                        placeholder="Please explain why you are registering this car (e.g., personal use, family member's car, etc.)"
                        variant="outlined"
                        :color="theme.primary"
                        prepend-inner-icon="mdi-text-box"
                        rows="3"
                        auto-grow
                        density="comfortable"
                      ></v-textarea>
                    </v-col>
                  </v-row>
                  
                  <v-alert
                    color="info"
                    class="mt-4"
                    variant="tonal"
                    icon="mdi-information"
                    density="comfortable"
                  >
                    <div class="font-weight-bold">What happens next?</div>
                    <p class="mb-0 mt-1">After submission, your request will be reviewed by the club management. You will be notified once your car has been approved for boom gate access.</p>
                  </v-alert>
                </v-form>
              </div>
              
              <!-- Step 3: Confirmation -->
              <div v-else>
                <div class="py-3 text-center">
                  <v-icon :color="theme.success" size="64" class="my-4">mdi-check-circle</v-icon>
                  <h3 class="text-h5 mb-2" :style="{color: theme.primary}">Request Submitted!</h3>
                  <p class="mb-4">Your car registration request has been successfully submitted for approval.</p>
                  
                  <v-alert
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    icon="mdi-information"
                    density="comfortable"
                    class="text-left mb-4"
                  >
                    <div class="font-weight-bold">Please note:</div>
                    <ul class="mt-2 ps-3">
                      <li>Your request is pending approval</li>
                      <li>You will be notified once approved</li>
                      <li>Approval typically takes 1-2 business days</li>
                      <li>The boom gate will only recognize your car after approval</li>
                    </ul>
                  </v-alert>
                  
                  <v-btn
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    @click="resetForm"
                    size="large"
                    class="mt-4"
                    elevation="2"
                  >
                    <v-icon start>mdi-plus</v-icon> Register Another Car
                  </v-btn>
                </div>
              </div>
            </v-card-text>
            
            <!-- Step Navigation -->
            <v-divider v-if="currentStep < 3"></v-divider>
            <v-card-actions v-if="currentStep < 3" class="pa-4 d-flex justify-space-between">
              <v-btn
                v-if="currentStep > 1"
                :color="theme.muted"
                variant="outlined"
                @click="prevStep"
                class="px-4 py-2"
                elevation="0"
              >
                <v-icon start class="mr-1">mdi-arrow-left</v-icon> Back
              </v-btn>
              <v-spacer v-if="currentStep === 1"></v-spacer>
              <v-btn
                v-if="currentStep < 2"
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="nextStep"
                :disabled="!carFormValid"
                elevation="2"
                class="px-5 py-2"
                size="large"
              >
                Next <v-icon end class="ml-1">mdi-arrow-right</v-icon>
              </v-btn>
              <v-btn
                v-else
                :color="theme.primary"
                :text-color="theme.lightText"
                @click="submitRequest"
                elevation="2"
                class="px-5 py-2"
                size="large"
              >
                <v-icon start class="mr-1">mdi-send</v-icon> Submit Request
              </v-btn>
            </v-card-actions>
          </v-card>
        </v-window-item>
        
        <!-- REQUEST HISTORY TAB -->
        <v-window-item value="history" class="pa-sm-4 pa-2">
          <v-card class="content-card mb-4" elevation="6">
            <v-card-title class="card-header" :class="{'card-header-mobile': $vuetify.display.xs}" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon start :color="theme.secondary">mdi-history</v-icon>
              <span>Car Registration History</span>
            </v-card-title>
            
            <v-card-text>
              <v-progress-circular
                v-if="loading"
                :size="50"
                :width="5"
                :color="theme.primary"
                indeterminate
                class="ma-4 d-flex mx-auto"
              ></v-progress-circular>
              
              <div v-else>
                <!-- Pending Requests -->
                <div class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Pending Requests</h3>
                  
                  <div v-if="activeRequests.length > 0">
                    <v-card
                      v-for="request in activeRequests"
                      :key="request.id"
                      class="mb-3 request-card"
                      variant="outlined"
                      :style="{borderColor: theme.warning + '80', borderRadius: '10px'}"
                      elevation="1"
                    >
                      <v-card-text class="pa-4">
                        <div class="d-flex align-center mb-3">
                          <v-avatar :color="theme.warning + '20'" size="48" class="mr-3">
                            <v-icon :color="theme.warning" size="24">mdi-clock-alert-outline</v-icon>
                          </v-avatar>
                          <div class="flex-grow-1">
                            <div class="text-h6 font-weight-bold" :style="{color: theme.primary}">{{ request.plateNumber }}</div>
                            <div class="text-body-2" :style="{color: theme.muted}">{{ request.carMake }} {{ request.carModel }} ({{ request.carColor }})</div>
                          </div>
                          <v-chip
                            :color="getStatusDetails(request.status).color"
                            size="small"
                            class="text-caption font-weight-medium"
                            variant="tonal"
                            elevation="0"
                          >
                            <v-icon start size="x-small">{{ getStatusDetails(request.status).icon }}</v-icon>
                            {{ getStatusDetails(request.status).text }}
                          </v-chip>
                        </div>
                        <v-divider class="my-2"></v-divider>
                        <div class="d-flex align-center justify-space-between mt-3">
                          <div class="text-body-2" :style="{color: theme.muted}">
                            <div><v-icon size="small" class="mr-1">mdi-calendar-clock</v-icon> Requested on {{ formatDate(request.requestDate) }}</div>
                            <div class="mt-1"><v-icon size="small" class="mr-1">mdi-calendar-range</v-icon> {{ formatDate(request.startDate) }} to {{ formatDate(request.endDate) }}</div>
                          </div>
                          <v-chip
                            size="small"
                            :color="request.paymentStatus === 'PAID' ? theme.success : theme.warning"
                            class="ml-2"
                            variant="tonal"
                          >
                            <v-icon start size="x-small">{{ request.paymentStatus === 'PAID' ? 'mdi-cash-check' : 'mdi-cash-clock' }}</v-icon>
                            {{ request.paymentStatus }}
                          </v-chip>
                        </div>
                      </v-card-text>
                    </v-card>
                  </div>
                  <v-alert
                    v-else
                    type="info"
                    variant="tonal"
                    class="mb-3"
                    icon="mdi-information"
                    density="comfortable"
                  >
                    You have no pending car registration requests.
                  </v-alert>
                </div>
                
                <!-- Approved Requests -->
                <div class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Approved Cars</h3>
                  
                  <div v-if="approvedRequests.length > 0">
                    <v-card
                      v-for="request in approvedRequests"
                      :key="request.id"
                      class="mb-3"
                      variant="outlined"
                      :style="{borderColor: theme.success + '80', borderRadius: '10px'}"
                      elevation="1"
                    >
                      <v-card-text class="pa-4">
                        <div class="d-flex align-center mb-2">
                          <v-avatar :color="theme.success + '20'" size="40" class="mr-3">
                            <v-icon :color="theme.success">mdi-check-circle</v-icon>
                          </v-avatar>
                          <div>
                            <div class="text-subtitle-1 font-weight-bold" :style="{color: theme.primary}">{{ request.plateNumber }}</div>
                            <div class="text-caption" :style="{color: theme.muted}">{{ request.carMake }} {{ request.carModel }} ({{ request.carColor }})</div>
                          </div>
                          <v-spacer></v-spacer>
                          <v-chip
                            :color="getStatusDetails(request.status).color"
                            size="small"
                            class="text-caption"
                            variant="tonal"
                          >
                            <v-icon start size="x-small">{{ getStatusDetails(request.status).icon }}</v-icon>
                            {{ getStatusDetails(request.status).text }}
                          </v-chip>
                        </div>
                        <div class="text-caption mt-2" :style="{color: theme.muted}">
                          Approved on {{ formatDate(request.approvalDate) }}
                          <br>
                          Valid from {{ formatDate(request.startDate) }} to {{ formatDate(request.endDate) }}
                          <br>
                          <v-chip
                            size="x-small"
                            :color="request.paymentStatus === 'PAID' ? theme.success : theme.warning"
                            class="mt-1"
                            variant="tonal"
                          >
                            <v-icon start size="x-small">{{ request.paymentStatus === 'PAID' ? 'mdi-cash-check' : 'mdi-cash-clock' }}</v-icon>
                            {{ request.paymentStatus }}
                          </v-chip>
                        </div>
                      </v-card-text>
                    </v-card>
                  </div>
                  <v-alert
                    v-else
                    type="info"
                    variant="tonal"
                    class="mb-3"
                    icon="mdi-information"
                    density="comfortable"
                  >
                    You have no approved cars registered for boom gate access.
                  </v-alert>
                </div>
                
                <!-- Rejected Requests -->
                <div v-if="rejectedRequests.length > 0" class="py-3">
                  <h3 :class="$vuetify.display.xs ? 'text-subtitle-1' : 'text-h6'" class="mb-3" :style="{color: theme.primary}">Rejected Requests</h3>
                  
                  <v-card
                    v-for="request in rejectedRequests"
                    :key="request.id"
                    class="mb-3"
                    variant="outlined"
                    :style="{borderColor: theme.error + '80', borderRadius: '10px'}"
                    elevation="1"
                  >
                    <v-card-text class="pa-4">
                      <div class="d-flex align-center mb-2">
                        <v-avatar :color="theme.error + '20'" size="40" class="mr-3">
                          <v-icon :color="theme.error">mdi-cancel</v-icon>
                        </v-avatar>
                        <div>
                          <div class="text-subtitle-1 font-weight-bold" :style="{color: theme.primary}">{{ request.plateNumber }}</div>
                          <div class="text-caption" :style="{color: theme.muted}">{{ request.carMake }} {{ request.carModel }} ({{ request.carColor }})</div>
                        </div>
                        <v-spacer></v-spacer>
                        <v-chip
                          :color="getStatusDetails(request.status).color"
                          size="small"
                          class="text-caption"
                          variant="tonal"
                        >
                          <v-icon start size="x-small">{{ getStatusDetails(request.status).icon }}</v-icon>
                          {{ getStatusDetails(request.status).text }}
                        </v-chip>
                      </div>
                      <div class="text-caption mt-2" :style="{color: theme.muted}">
                        Rejected on {{ formatDate(request.approvalDate) }}
                        <br>
                        <v-chip
                          size="x-small"
                          :color="request.paymentStatus === 'PAID' ? theme.success : theme.warning"
                          class="mt-1"
                          variant="tonal"
                        >
                          <v-icon start size="x-small">{{ request.paymentStatus === 'PAID' ? 'mdi-cash-check' : 'mdi-cash-clock' }}</v-icon>
                          {{ request.paymentStatus }}
                        </v-chip>
                      </div>
                    </v-card-text>
                  </v-card>
                </div>
              </div>
            </v-card-text>
          </v-card>
        </v-window-item>
      </v-window>

      <!-- Success Dialog -->
      <v-dialog
        v-model="showSuccessDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '450px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        persistent
        transition="dialog-bottom-transition"
      >
        <v-card>
          <v-toolbar :color="theme.success" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2">mdi-check-circle</v-icon>
            <v-toolbar-title class="dialog-title">Request Submitted</v-toolbar-title>
            <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showSuccessDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4 text-center">
            <v-icon :color="theme.success" size="64" class="my-4">mdi-check-circle</v-icon>
            <h3 class="text-h5 mb-2" :style="{color: theme.primary}">Thank You!</h3>
            <p class="mb-4">Your car registration request has been successfully submitted.</p>
            
            <v-alert
              :color="theme.primary"
              :text-color="theme.lightText"
              icon="mdi-information"
              density="comfortable"
              class="text-left mb-4"
            >
              <div class="font-weight-bold">What happens next?</div>
              <ul class="mt-2 ps-3">
                <li>Your request will be reviewed by club management</li>
                <li>Approval typically takes 1-2 business days</li>
                <li>You will be notified once your request is processed</li>
                <li>The boom gate will recognize your car after approval</li>
              </ul>
            </v-alert>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 justify-center">
            <v-btn
              :color="theme.primary"
              :text-color="theme.lightText"
              @click="showSuccessDialog = false"
              size="large"
              elevation="2"
            >
              Close
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>

      <!-- Info Dialog -->
      <v-dialog
        v-model="showInfoDialog"
        :max-width="$vuetify.display.smAndDown ? undefined : '450px'"
        :fullscreen="$vuetify.display.smAndDown"
        :scrim="!$vuetify.display.smAndDown"
        transition="dialog-bottom-transition"
      >
        <v-card>
          <v-toolbar :color="theme.primary" :text-color="theme.lightText" density="compact" flat>
            <v-icon class="ml-3 mr-2" :color="theme.secondary">{{ infoDialogContent.icon }}</v-icon>
            <v-toolbar-title class="dialog-title">{{ infoDialogContent.title }}</v-toolbar-title>
            <v-spacer></v-spacer>
            <v-btn icon="mdi-close" @click="showInfoDialog = false" :color="theme.lightText"></v-btn>
          </v-toolbar>
          <v-card-text class="pa-4">
            <p v-html="infoDialogContent.message"></p>
          </v-card-text>
          <v-divider></v-divider>
          <v-card-actions class="pa-4 justify-center">
            <v-btn
              :color="theme.primary"
              :text-color="theme.lightText"
              @click="showInfoDialog = false"
              size="large"
              elevation="2"
            >
              Close
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
      
      <!-- Snackbar for notifications -->
      <v-snackbar
        v-model="showSnackbar"
        :color="snackbar.color"
        :timeout="snackbar.timeout"
        location="top"
      >
        {{ snackbar.text }}
        <template v-slot:actions>
          <v-btn
            variant="text"
            @click="showSnackbar = false"
          >
            Close
          </v-btn>
        </template>
      </v-snackbar>
    </v-container>
  `
};