import getTropicalStyles from "../css/tropicalBookingStyles.js";
import RoomImageCarousel from "../components/RoomImageCarousel.js";

export default {
  props: ["jsonData"],
  components: {
    RoomImageCarousel
  },
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    // Calculate tomorrow and max date (1 year from today)
    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);

    const maxDate = new Date(today);
    maxDate.setFullYear(maxDate.getFullYear() + 1);

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',      // Dark green
        secondary: '#DDA333',    // Golden
        accent: '#A3BA95',       // Light sage green
        lightBg: '#F8F5F0',      // Off-white
        creamBg: '#EADDCB',      // Cream
        darkText: '#1E1E1E',     // Almost black
        lightText: '#FFFFFF',    // White
        error: '#B71C1C',        // Red
        success: '#2E7D32',      // Green
        warning: '#FF8F00',      // Orange
        muted: '#78909C',        // Grayish blue
        brown: '#6D4C41',        // Brown
      },

      // Booking process control
      currentStep: 1,
      totalSteps: 5,

      // Date selection
      dateRange: {
        checkIn: tomorrow.toISOString().substr(0, 10),
        checkOut: null,
      },
      minDate: today.toISOString().substr(0, 10),
      maxDate: maxDate.toISOString().substr(0, 10),
      datePickerOpen: false,

      // Room selection
      availableRooms: [],  // Will be populated from API
      roomsLoading: false, // New loading state for rooms
      selectedRoom: null,

      // Guest information
      guestForm: {
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        address: '',
        city: '',
        country: '',
        specialRequests: '',
        adults: 2,
        children: 0,
      },
      guestFormValid: false,

      // User data and wallet
      userData: null,
      walletBalance: {
        balance: '0.00'
      },
      walletLoading: false,

      // Room preferences
      roomPreferences: {
        floorLevel: 'any',      // 'low', 'middle', 'high', 'any'
        bedType: 'king',        // 'king', 'queen', 'twin'
        pillowType: 'standard', // 'standard', 'firm', 'soft', 'hypoallergenic'
        view: 'forest',         // 'garden', 'pool', 'forest'
        smoking: false,
        accessibility: false,
        quietZone: true,
        turndown: true,
      },

      // Add-on services
      additionalServices: [
        { id: 'breakfast', name: 'Rainforest Breakfast Buffet', price: 25, selected: true, icon: 'mdi-food-variant', perPerson: true },
        { id: 'parking', name: 'Valet Parking (Per Day)', price: 15, selected: false, icon: 'mdi-car', perPerson: false },
        { id: 'wifi', name: 'Premium High-Speed WiFi', price: 10, selected: false, icon: 'mdi-wifi', perPerson: false },
        { id: 'spa', name: 'Welcome Spa Treatment', price: 85, selected: false, icon: 'mdi-spa', perPerson: true },
        { id: 'airport', name: 'Airport Transfer', price: 60, selected: false, icon: 'mdi-airplane', perPerson: false },
        { id: 'earlyCheckin', name: 'Early Check-in (Before 2PM)', price: 40, selected: false, icon: 'mdi-clock-fast', perPerson: false },
        { id: 'lateCheckout', name: 'Late Check-out (Until 4PM)', price: 40, selected: false, icon: 'mdi-clock-time-four', perPerson: false },
        { id: 'minibar', name: 'Premium Minibar Package', price: 30, selected: false, icon: 'mdi-bottle-wine', perPerson: false },
      ],

      // Payment information
      useWalletPayment: true, // Default to wallet payment

      // NEW: Booking history properties
      viewMode: 'booking', // 'booking' or 'history'
      bookingHistory: [],
      historyLoading: false,
      historyFilters: {
        status: 'all', // 'all', 'upcoming', 'completed', 'cancelled'
        dateFrom: null,
        dateTo: null,
        searchQuery: ''
      },
      selectedBookingDetails: null,
      showBookingDetailsModal: false,
      showCancelBookingModal: false,
      bookingToCancel: null,
      filterPanelOpen: null, // For expansion panel control
      searchTimeout: null, // For debouncing search
      detailsTab: 'booking', // For booking details modal tabs
      cancellationReason: '', // For cancellation modal
      confirmCancellation: false, // For cancellation confirmation
      showRebookModal: false, // For rebook confirmation modal
      bookingToRebook: null, // Booking selected for rebooking
      historyError: null, // Error message for booking history

      // Terms acceptance
      termsAccepted: false,
      dataConfirmed: false,

      // Booking completion data
      bookingComplete: false,
      bookingReference: null,
      bookingDetails: null,

      // UI control
      loading: false,
      showSnackbar: false,
      snackbar: {
        text: '',
        color: 'success',
        timeout: 3000
      },
    };
  },

  computed: {
    // Date related computations
    formattedCheckInDate() {
      if (!this.dateRange.checkIn) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return new Date(this.dateRange.checkIn).toLocaleDateString('en-US', options);
    },

    formattedCheckOutDate() {
      if (!this.dateRange.checkOut) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return new Date(this.dateRange.checkOut).toLocaleDateString('en-US', options);
    },

    nightsCount() {
      if (!this.dateRange.checkIn || !this.dateRange.checkOut) return 0;
      const checkIn = new Date(this.dateRange.checkIn);
      const checkOut = new Date(this.dateRange.checkOut);
      const diffTime = Math.abs(checkOut - checkIn);
      return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    },

    formattedDateRange() {
      if (!this.dateRange.checkIn || !this.dateRange.checkOut) return '';
      const checkIn = new Date(this.dateRange.checkIn);
      const checkOut = new Date(this.dateRange.checkOut);
      const options = { month: 'short', day: 'numeric' };
      return `${checkIn.toLocaleDateString('en-US', options)} - ${checkOut.toLocaleDateString('en-US', options)} (${this.nightsCount} nights)`;
    },

    // Price calculations
    roomTotal() {
      if (!this.selectedRoom || !this.nightsCount) return 0;
      return this.selectedRoom.price * this.nightsCount;
    },

    selectedServicesTotal() {
      if (!this.nightsCount) return 0;

      return this.additionalServices
        .filter(service => service.selected)
        .reduce((total, service) => {
          if (service.perPerson) {
            // Per person services are multiplied by the number of adults + children
            return total + (service.price * (this.guestForm.adults + this.guestForm.children));
          } else if (service.id === 'parking' || service.id === 'wifi') {
            // Per day services are multiplied by the number of nights
            return total + (service.price * this.nightsCount);
          } else {
            // One-time services
            return total + service.price;
          }
        }, 0);
    },

    totalAmount() {
      return this.roomTotal + this.selectedServicesTotal;
    },

    // Validation
    isDateRangeValid() {
      return this.dateRange.checkIn && this.dateRange.checkOut && this.nightsCount > 0;
    },

    isRoomSelected() {
      return this.selectedRoom !== null;
    },

    canProceedToPayment() {
      return this.guestFormValid && this.isDateRangeValid && this.isRoomSelected;
    },

    // NEW WALLET COMPUTATIONS
    // Check if wallet has sufficient balance
    hasWalletSufficientBalance() {
      if (!this.walletBalance || !this.walletBalance.balance) return false;
      return parseFloat(this.walletBalance.balance) >= this.totalAmount;
    },

    // Format wallet balance for display
    formattedWalletBalance() {
      return this.formatCurrency(parseFloat(this.walletBalance.balance || 0));
    },

    // Format available credit for display
    formattedAvailableCredit() {
      return this.formatCurrency(parseFloat(this.walletBalance.available_credit || 0));
    },

    // Format pending amount for display
    formattedPendingAmount() {
      return this.formatCurrency(parseFloat(this.walletBalance.pending_amount || 0));
    },

    // Check if user can complete booking with wallet
    canCompleteBookingWithWallet() {
      return this.hasWalletSufficientBalance && this.termsAccepted && this.dataConfirmed;
    },

    // Override the original canCompleteBooking method
    canCompleteBooking() {
      return this.canCompleteBookingWithWallet;
    },

    // Format user's full name
    userFullName() {
      if (!this.userData || !this.userData.fullname) return '';
      return this.userData.fullname;
    },

    // Get first name from full name
    userFirstName() {
      if (!this.userFullName) return '';
      const nameParts = this.userFullName.split(' ');
      return nameParts[0] || '';
    },

    // Get last name from full name
    userLastName() {
      if (!this.userFullName) return '';
      const nameParts = this.userFullName.split(' ');
      return nameParts.slice(1).join(' ') || '';
    },

    // NEW BOOKING HISTORY COMPUTATIONS
    // Filter booking history based on current filters
    filteredBookingHistory() {
      if (!this.bookingHistory || this.bookingHistory.length === 0) return [];

      return this.bookingHistory.filter(booking => {
        // Status filter
        if (this.historyFilters.status !== 'all') {
          const today = new Date().toISOString().substr(0, 10);
          switch (this.historyFilters.status) {
            case 'upcoming':
              if (booking.check_in_date < today || booking.booking_status !== 'Confirmed') return false;
              break;
            case 'completed':
              if (booking.check_out_date >= today || booking.booking_status !== 'Confirmed') return false;
              break;
            case 'cancelled':
              if (booking.booking_status !== 'Cancelled') return false;
              break;
          }
        }

        // Date range filter
        if (this.historyFilters.dateFrom && booking.check_in_date < this.historyFilters.dateFrom) return false;
        if (this.historyFilters.dateTo && booking.check_in_date > this.historyFilters.dateTo) return false;

        // Search query filter
        if (this.historyFilters.searchQuery) {
          const query = this.historyFilters.searchQuery.toLowerCase();
          const searchableText = `${booking.booking_reference} ${booking.room_name}`.toLowerCase();
          if (!searchableText.includes(query)) return false;
        }

        return true;
      });
    },

    // Count of upcoming bookings
    upcomingBookingsCount() {
      if (!this.bookingHistory) return 0;
      const today = new Date().toISOString().substr(0, 10);
      return this.bookingHistory.filter(booking =>
        booking.check_in_date >= today && booking.booking_status === 'Confirmed'
      ).length;
    },

    // Check if a booking can be cancelled
    canCancelBooking() {
      return (booking) => {
        if (!booking || booking.booking_status !== 'Confirmed') return false;
        return booking.actions && booking.actions.can_cancel;
      };
    }
  },

  methods: {
    // Navigation methods
    nextStep() {
      if (this.currentStep < this.totalSteps) {
        this.currentStep++;
      }
      // Scroll to top for better UX when navigating
      window.scrollTo(0, 0);
    },

    prevStep() {
      if (this.currentStep > 1) {
        this.currentStep--;
      }
      // Scroll to top for better UX when navigating
      window.scrollTo(0, 0);
    },

    // Date selection methods
    setDateRange(range) {
      this.dateRange = range;
      this.datePickerOpen = false;
    },

    // Room selection methods
    selectRoom(room) {
      this.availableRooms.forEach(r => r.selected = false);
      room.selected = true;
      this.selectedRoom = room;
    },

    // Refresh room images if needed
    refreshRoomImages() {
      console.log('Refreshing room images...');
      this.availableRooms.forEach(room => {
        if (room.images && room.images.length > 0) {
          // Validate first image URL
          const img = new Image();
          img.onload = () => {
            console.log(`Image loaded successfully for room ${room.name}`);
          };
          img.onerror = () => {
            console.warn(`Image failed to load for room ${room.name}, setting fallback`);
            room.images = ['https://via.placeholder.com/300x200?text=Room+Image'];
          };
          img.src = room.images[0];
        }
      });
    },

    // Handle image loading errors
    handleImageError(event, item, context = 'room') {
      console.warn(`Image failed to load for ${context}:`, event.target.src);

      // Set fallback image
      const fallbackImage = 'https://via.placeholder.com/300x200?text=Room+Image';

      if (context === 'room') {
        // For room selection
        const roomIndex = this.availableRooms.findIndex(r => r.id === item.id);
        if (roomIndex !== -1) {
          this.availableRooms[roomIndex].images = [fallbackImage];
        }
      } else if (context === 'history') {
        // For booking history
        const bookingIndex = this.bookingHistory.findIndex(b => b.id === item.id);
        if (bookingIndex !== -1) {
          this.bookingHistory[bookingIndex].room_image = fallbackImage;
        }
      } else if (context === 'details') {
        // For booking details modal
        if (this.selectedBookingDetails) {
          this.selectedBookingDetails.room_image = fallbackImage;
        }
      } else if (context === 'cancel') {
        // For cancellation modal
        if (this.bookingToCancel) {
          this.bookingToCancel.room_image = fallbackImage;
        }
      } else if (context === 'rebook') {
        // For rebook modal
        if (this.bookingToRebook) {
          this.bookingToRebook.room_image = fallbackImage;
        }
      }

      // Update the image source directly
      event.target.src = fallbackImage;
    },

    // Process booking room image from various possible API response formats
    processBookingRoomImage(booking) {
      console.log('Processing booking room image for:', booking.booking_reference || booking.id, booking);

      // Try different possible field names for room image
      const possibleImageFields = [
        'room_image',
        'room_images',
        'image',
        'images',
        'room_photo',
        'room_photos'
      ];

      for (const field of possibleImageFields) {
        if (booking[field]) {
          console.log(`Found image in field '${field}':`, booking[field]);
          if (typeof booking[field] === 'string') {
            return booking[field];
          } else if (Array.isArray(booking[field]) && booking[field].length > 0) {
            return booking[field][0];
          } else if (typeof booking[field] === 'object' && booking[field].url) {
            return booking[field].url;
          }
        }
      }

      // Check nested room object
      if (booking.room) {
        console.log('Checking nested room object:', booking.room);
        for (const field of possibleImageFields) {
          if (booking.room[field]) {
            console.log(`Found image in nested room field '${field}':`, booking.room[field]);
            if (typeof booking.room[field] === 'string') {
              return booking.room[field];
            } else if (Array.isArray(booking.room[field]) && booking.room[field].length > 0) {
              return booking.room[field][0];
            }
          }
        }
      }

      // Fallback to placeholder
      console.warn('No room image found for booking, using placeholder');
      return 'https://via.placeholder.com/300x200?text=Room+Image';
    },

    // Process room amenities from various possible API response formats
    processRoomAmenities(amenities) {
      if (!amenities) {
        // Return basic amenities as fallback when API returns null
        return ['Air Conditioning', 'Free WiFi', 'Private Bathroom', 'Room Service'];
      }

      if (Array.isArray(amenities)) {
        return amenities.filter(amenity => amenity && typeof amenity === 'string');
      }

      if (typeof amenities === 'string') {
        try {
          const parsed = JSON.parse(amenities);
          if (Array.isArray(parsed)) {
            return parsed.filter(amenity => amenity && typeof amenity === 'string');
          }
        } catch (e) {
          // If parsing fails, treat as single amenity
          return [amenities];
        }
      }

      return ['Air Conditioning', 'Free WiFi', 'Private Bathroom', 'Room Service'];
    },

    // Process room images from various possible API response formats
    processRoomImages(images, image, room_image) {
      console.log('Processing room images:', { images, image, room_image });

      // Try to get images from different possible fields
      let imageArray = [];

      if (images) {
        if (Array.isArray(images)) {
          imageArray = images.filter(img => img && typeof img === 'string');
        } else if (typeof images === 'string') {
          // If images is a JSON string, try to parse it
          try {
            const parsed = JSON.parse(images);
            if (Array.isArray(parsed)) {
              imageArray = parsed.filter(img => img && typeof img === 'string');
            } else if (typeof parsed === 'string') {
              imageArray = [parsed];
            }
          } catch (e) {
            // If parsing fails, treat as single image URL
            imageArray = [images];
          }
        }
      }

      // Fallback to single image fields
      if (imageArray.length === 0) {
        if (image && typeof image === 'string') {
          imageArray = [image];
        } else if (room_image && typeof room_image === 'string') {
          imageArray = [room_image];
        }
      }

      // If still no images, use placeholder
      if (imageArray.length === 0) {
        console.warn('No valid images found, using placeholder');
        imageArray = ['https://via.placeholder.com/300x200?text=Room+Image'];
      }

      console.log('Processed images result:', imageArray);
      return imageArray;
    },

    // Fetch available rooms from API
    async fetchAvailableRooms() {
      this.roomsLoading = true;
      try {
        const response = await this.$root.callApi('rooms', 'get_rooms', {});

        if (response && response.data && response.data.status === "YES" && Array.isArray(response.data.rooms)) {
          // Transform API response to match our component's data structure
          this.availableRooms = response.data.rooms.map(room => ({
            id: room.id, // Changed from room.room_id to room.id
            name: room.name,
            description: room.description,
            price: parseFloat(room.price),
            capacity: {
              adults: parseInt(room.capacity_adults), // Changed from room.capacity
              children: parseInt(room.capacity_children) // Changed from calculation
            },
            amenities: this.processRoomAmenities(room.amenities), // Enhanced amenities processing
            images: this.processRoomImages(room.images, room.image, room.room_image), // Enhanced image processing
            available: room.available === "1",
            selected: false
          }));

          console.log('Rooms loaded successfully', this.availableRooms);
        } else {
          console.error('Failed to load rooms', response);
          this.showSnackbarMessage('Failed to load available rooms. Using default data.', 'warning');
          this.loadHardcodedRooms();
        }
      } catch (error) {
        console.error('Error fetching rooms:', error);
        this.showSnackbarMessage('Error loading available rooms. Using default data.', 'warning');
        this.loadHardcodedRooms();
      } finally {
        this.roomsLoading = false;
      }
    },

    // Fallback to load hardcoded rooms if API fails
    loadHardcodedRooms() {
      this.availableRooms = [
        {
          id: 'jungle-suite',
          name: 'Jungle View Suite',
          description: 'Immerse yourself in the lush rainforest with panoramic views from your private balcony.',
          price: 250.00,
          capacity: { adults: 2, children: 2 },
          amenities: ['King Bed', 'Balcony', 'Rain Shower', 'Mini Bar', 'Free WiFi'],
          images: ['https://via.placeholder.com/300x200?text=Jungle+Suite'],
          available: true,
          selected: false
        },
        // ... other hardcoded rooms ...
      ];
      console.log('Loaded hardcoded rooms as fallback');
    },

    // NEW BOOKING HISTORY METHODS
    // Navigation methods for view switching
    switchToHistory() {
      this.viewMode = 'history';
      // Always fetch if we don't have data or if there was an error
      if (this.bookingHistory.length === 0 || this.historyError) {
        this.fetchBookingHistory();
      }
    },

    switchToBooking() {
      this.viewMode = 'booking';
    },

    // Fetch booking history from API with comprehensive error handling
    async fetchBookingHistory(retryCount = 0) {
      this.historyLoading = true;
      const maxRetries = 2;

      try {
        const userId = sessionStorage.getItem('id');

        // Validate user session
        if (!userId) {
          throw new Error('User session expired. Please log in again.');
        }

        const requestData = {
          user_id: userId,
          status: this.historyFilters.status,
          date_from: this.historyFilters.dateFrom,
          date_to: this.historyFilters.dateTo,
          search_query: this.historyFilters.searchQuery
        };

        console.log('Fetching booking history with filters:', requestData);

        const response = await this.$root.callApi('rooms', 'get_booking_history', requestData);

        // Handle different response scenarios
        if (!response) {
          throw new Error('No response received from server');
        }

        if (response.error) {
          throw new Error(response.error);
        }

        if (response.data && response.data.status === "YES") {
          const bookings = response.data.data?.bookings || [];
          console.log('Raw booking history data:', bookings);

          // Process booking data to ensure room images are properly handled
          this.bookingHistory = bookings.map(booking => ({
            ...booking,
            room_image: this.processBookingRoomImage(booking)
          }));
          this.historyError = null; // Clear any previous errors

          console.log(`Booking history loaded successfully: ${bookings.length} bookings`);

          // Show success message only if this was a retry or recovering from error
          if (retryCount > 0 || this.historyError) {
            this.showSnackbarMessage('Booking history loaded successfully!', 'success');
          }

        } else if (response.data && response.data.status === "NO") {
          const errorMessage = response.data.message || 'Failed to load booking history';
          console.error('API returned error:', errorMessage);

          // Handle specific error cases
          if (errorMessage.includes('User ID is required')) {
            this.showSnackbarMessage('Session expired. Please refresh the page and try again.', 'error');
          } else {
            this.showSnackbarMessage(errorMessage, 'error');
          }

        } else {
          throw new Error('Invalid response format from server');
        }

      } catch (error) {
        console.error('Error fetching booking history:', error);

        // Handle different types of errors
        if (error.name === 'TypeError' && error.message.includes('fetch')) {
          // Network error
          if (retryCount < maxRetries) {
            console.log(`Network error, retrying... (${retryCount + 1}/${maxRetries})`);
            setTimeout(() => {
              this.fetchBookingHistory(retryCount + 1);
            }, 1000 * (retryCount + 1)); // Exponential backoff
            return;
          } else {
            this.showSnackbarMessage('Network error. Please check your connection and try again.', 'error');
          }
        } else if (error.message.includes('session') || error.message.includes('User ID')) {
          // Session/authentication error
          this.showSnackbarMessage('Your session has expired. Please refresh the page and log in again.', 'error');
        } else {
          // Generic error
          const errorMessage = error.message || 'An unexpected error occurred while loading your booking history.';
          this.showSnackbarMessage(errorMessage, 'error');
        }

        // Set error state and empty array
        this.historyError = errorMessage;
        this.bookingHistory = [];

      } finally {
        this.historyLoading = false;
      }
    },

    // Retry loading booking history
    retryLoadHistory() {
      this.historyError = null;
      this.fetchBookingHistory();
    },

    // Filter and search methods
    applyHistoryFilters() {
      this.fetchBookingHistory();
    },

    clearHistoryFilters() {
      this.historyFilters = {
        status: 'all',
        dateFrom: null,
        dateTo: null,
        searchQuery: ''
      };
      this.fetchBookingHistory();
    },

    searchBookings() {
      // Debounce search to avoid too many API calls
      clearTimeout(this.searchTimeout);
      this.searchTimeout = setTimeout(() => {
        this.fetchBookingHistory();
      }, 500);
    },

    quickFilter(status) {
      this.historyFilters.status = status;
      // Clear other filters when using quick filter
      this.historyFilters.dateFrom = null;
      this.historyFilters.dateTo = null;
      this.historyFilters.searchQuery = '';
      this.applyHistoryFilters();
    },

    // Booking management methods
    viewBookingDetails(booking) {
      this.selectedBookingDetails = booking;
      this.detailsTab = 'booking'; // Reset to first tab
      this.showBookingDetailsModal = true;

      // On mobile, ensure modal is properly sized
      if (this.isMobile()) {
        this.$nextTick(() => {
          // Scroll to top of modal content
          const modalContent = document.querySelector('.v-dialog .v-card');
          if (modalContent) {
            modalContent.scrollTop = 0;
          }
        });
      }
    },

    cancelBooking(booking) {
      this.bookingToCancel = booking;
      this.cancellationReason = '';
      this.confirmCancellation = false;
      this.showCancelBookingModal = true;
    },

    async confirmCancelBooking(reason = '') {
      if (!this.bookingToCancel) {
        this.showSnackbarMessage('No booking selected for cancellation.', 'error');
        return;
      }

      this.loading = true;

      try {
        const userId = sessionStorage.getItem('id');

        // Validate session
        if (!userId) {
          throw new Error('User session expired. Please log in again.');
        }

        // Validate booking can still be cancelled
        if (!this.bookingToCancel.actions?.can_cancel) {
          throw new Error('This booking can no longer be cancelled.');
        }

        const requestData = {
          user_id: userId,
          booking_id: this.bookingToCancel.id,
          cancellation_reason: reason || 'No reason provided'
        };

        console.log('Cancelling booking:', requestData);

        const response = await this.$root.callApi('rooms', 'cancel_booking', requestData);

        if (!response) {
          throw new Error('No response received from server');
        }

        if (response.data && response.data.status === "YES") {
          const refundData = response.data.data;

          // Show detailed success message
          let successMessage = 'Booking cancelled successfully!';
          if (refundData?.refund_amount > 0) {
            successMessage += ` Refund of ${this.formatCurrency(refundData.refund_amount)} has been processed to your wallet.`;
          }

          this.showSnackbarMessage(successMessage, 'success', 6000);

          // Reset modal state
          this.showCancelBookingModal = false;
          this.bookingToCancel = null;
          this.cancellationReason = '';
          this.confirmCancellation = false;

          // Refresh data
          await Promise.all([
            this.fetchBookingHistory(),
            this.fetchUserDataAndWallet()
          ]);

        } else if (response.data && response.data.status === "NO") {
          const errorMessage = response.data.message || 'Failed to cancel booking';
          console.error('Cancellation failed:', errorMessage);

          // Handle specific error cases
          if (errorMessage.includes('deadline')) {
            this.showSnackbarMessage('Cancellation deadline has passed. Please contact support for assistance.', 'error');
          } else if (errorMessage.includes('already been cancelled')) {
            this.showSnackbarMessage('This booking has already been cancelled.', 'warning');
            // Refresh booking history to show updated status
            await this.fetchBookingHistory();
          } else {
            this.showSnackbarMessage(errorMessage, 'error');
          }

        } else {
          throw new Error('Invalid response format from server');
        }

      } catch (error) {
        console.error('Error cancelling booking:', error);

        // Handle different error types
        if (error.message.includes('session') || error.message.includes('User ID')) {
          this.showSnackbarMessage('Your session has expired. Please refresh the page and try again.', 'error');
        } else if (error.message.includes('network') || error.name === 'TypeError') {
          this.showSnackbarMessage('Network error. Please check your connection and try again.', 'error');
        } else {
          const errorMessage = error.message || 'An unexpected error occurred while cancelling your booking.';
          this.showSnackbarMessage(errorMessage, 'error');
        }

      } finally {
        this.loading = false;
      }
    },

    rebookFromHistory(booking) {
      // Pre-populate form with booking data
      this.guestForm.firstName = booking.guest_info.first_name;
      this.guestForm.lastName = booking.guest_info.last_name;
      this.guestForm.email = booking.guest_info.email;
      this.guestForm.phone = booking.guest_info.phone;
      this.guestForm.adults = booking.adults;
      this.guestForm.children = booking.children;
      this.guestForm.specialRequests = booking.special_requests;

      // Set room preferences
      if (booking.room_preferences) {
        this.roomPreferences = { ...this.roomPreferences, ...booking.room_preferences };
      }

      // Switch to booking view
      this.switchToBooking();
      this.currentStep = 1;

      this.showSnackbarMessage('Booking details pre-filled. Please select new dates and room.', 'info');
    },

    // Helper methods for booking details modal
    formatPreferenceKey(key) {
      const keyMap = {
        'bedType': 'Bed Type',
        'pillowType': 'Pillow Type',
        'view': 'View',
        'floorLevel': 'Floor Level',
        'smoking': 'Smoking',
        'accessibility': 'Accessibility',
        'quietZone': 'Quiet Zone',
        'turndown': 'Turndown Service'
      };
      return keyMap[key] || key.charAt(0).toUpperCase() + key.slice(1);
    },

    formatPreferenceValue(value) {
      if (typeof value === 'boolean') {
        return value ? 'Yes' : 'No';
      }
      if (typeof value === 'string' && value.length > 0) {
        return value.charAt(0).toUpperCase() + value.slice(1);
      }
      return value || '';
    },

    formatServiceName(serviceId) {
      const serviceMap = {
        'breakfast': 'Rainforest Breakfast Buffet',
        'parking': 'Valet Parking',
        'wifi': 'Premium High-Speed WiFi',
        'spa': 'Welcome Spa Treatment',
        'airport': 'Airport Transfer',
        'earlyCheckin': 'Early Check-in',
        'lateCheckout': 'Late Check-out',
        'minibar': 'Premium Minibar Package'
      };
      return serviceMap[serviceId] || serviceId.charAt(0).toUpperCase() + serviceId.slice(1);
    },

    // Calculate cancellation fee based on days until check-in
    calculateCancellationFee(booking) {
      if (!booking) return { fee: 0, refund: 0 };

      const checkInDate = new Date(booking.check_in_date);
      const today = new Date();
      const daysUntilCheckIn = Math.ceil((checkInDate - today) / (1000 * 60 * 60 * 24));

      let feePercentage = 0;
      if (daysUntilCheckIn <= 2) {
        feePercentage = 1; // 100% fee (no refund)
      } else if (daysUntilCheckIn <= 7) {
        feePercentage = 0.1; // 10% fee
      }

      const fee = booking.pricing.total_amount * feePercentage;
      const refund = booking.pricing.total_amount - fee;

      return { fee, refund, feePercentage };
    },

    // Enhanced "Book Again" functionality
    rebookFromHistory(booking) {
      this.bookingToRebook = booking;
      this.showRebookModal = true;
    },

    processRebooking(booking) {
      // Validate booking data first
      if (!this.validateRebookingData(booking)) {
        return;
      }

      try {
        // Pre-populate guest form with booking data
        this.guestForm.firstName = booking.guest_info.first_name || '';
        this.guestForm.lastName = booking.guest_info.last_name || '';
        this.guestForm.email = booking.guest_info.email || '';
        this.guestForm.phone = booking.guest_info.phone || '';
        this.guestForm.address = booking.guest_info.address || '';
        this.guestForm.city = booking.guest_info.city || '';
        this.guestForm.country = booking.guest_info.country || '';
        this.guestForm.adults = booking.adults || 2;
        this.guestForm.children = booking.children || 0;
        this.guestForm.specialRequests = booking.special_requests || '';

        // Set room preferences if available
        if (booking.room_preferences && typeof booking.room_preferences === 'object') {
          this.roomPreferences = {
            ...this.roomPreferences,
            ...booking.room_preferences
          };
        }

        // Pre-select additional services if available
        if (booking.additional_services && Array.isArray(booking.additional_services)) {
          // Reset all services first
          this.additionalServices.forEach(service => {
            service.selected = false;
          });

          // Select services from previous booking
          booking.additional_services.forEach(serviceId => {
            const service = this.additionalServices.find(s => s.id === serviceId);
            if (service) {
              service.selected = true;
            }
          });
        }

        // Clear date selection to force user to select new dates
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        this.dateRange.checkIn = tomorrow.toISOString().substr(0, 10);
        this.dateRange.checkOut = null;

        // Clear room selection to force user to select available room
        this.selectedRoom = null;
        this.availableRooms.forEach(room => {
          room.selected = false;
        });

        // Switch to booking view and start from step 1
        this.switchToBooking();
        this.currentStep = 1;

        // Show success message with details
        const servicesCount = booking.additional_services ? booking.additional_services.length : 0;
        const message = `Booking details pre-filled! Guest info, preferences${servicesCount > 0 ? ', and ' + servicesCount + ' services' : ''} have been restored. Please select new dates and room.`;

        this.showSnackbarMessage(message, 'success', 5000);

        // Scroll to top for better UX
        window.scrollTo({ top: 0, behavior: 'smooth' });

        // Track rebooking event (for analytics)
        this.trackRebookingEvent(booking);

      } catch (error) {
        console.error('Error processing rebooking:', error);
        this.showSnackbarMessage('Error preparing booking details. Please try again.', 'error');
      }
    },

    // Validate rebooking data
    validateRebookingData(booking) {
      const issues = [];

      if (!booking.guest_info || !booking.guest_info.first_name) {
        issues.push('Guest information is incomplete');
      }

      if (!booking.room_name) {
        issues.push('Room information is missing');
      }

      if (issues.length > 0) {
        this.showSnackbarMessage(`Cannot rebook: ${issues.join(', ')}`, 'warning');
        return false;
      }

      return true;
    },

    // Track rebooking event for analytics
    trackRebookingEvent(booking) {
      try {
        // This could be enhanced to send analytics data to a tracking service
        console.log('Rebooking event:', {
          originalBookingRef: booking.booking_reference,
          roomName: booking.room_name,
          timestamp: new Date().toISOString(),
          servicesCount: booking.additional_services ? booking.additional_services.length : 0
        });
      } catch (error) {
        console.error('Error tracking rebooking event:', error);
      }
    },

    // Global error handler for the component
    handleComponentError(error, context = 'Unknown operation') {
      console.error(`Error in ${context}:`, error);

      // Determine error type and show appropriate message
      let errorMessage = 'An unexpected error occurred.';
      let errorType = 'error';

      if (error.name === 'TypeError' && error.message.includes('fetch')) {
        errorMessage = 'Network connection error. Please check your internet connection.';
      } else if (error.message.includes('session') || error.message.includes('unauthorized')) {
        errorMessage = 'Your session has expired. Please refresh the page and log in again.';
      } else if (error.message.includes('timeout')) {
        errorMessage = 'Request timed out. Please try again.';
      } else if (error.message) {
        errorMessage = error.message;
      }

      this.showSnackbarMessage(`${context}: ${errorMessage}`, errorType);

      // Log error for debugging
      this.logError(error, context);
    },

    // Error logging method
    logError(error, context) {
      const errorLog = {
        timestamp: new Date().toISOString(),
        context: context,
        error: {
          name: error.name,
          message: error.message,
          stack: error.stack
        },
        userAgent: navigator.userAgent,
        url: window.location.href,
        userId: sessionStorage.getItem('id')
      };

      console.error('Component Error Log:', errorLog);

      // In a production environment, you might want to send this to an error tracking service
      // this.sendErrorToTrackingService(errorLog);
    },

    // Enhanced loading state management
    setLoadingState(operation, isLoading) {
      switch (operation) {
        case 'history':
          this.historyLoading = isLoading;
          break;
        case 'booking':
          this.loading = isLoading;
          break;
        case 'wallet':
          this.walletLoading = isLoading;
          break;
        case 'rooms':
          this.roomsLoading = isLoading;
          break;
        default:
          this.loading = isLoading;
      }
    },

    // Mobile-specific helper methods
    isMobile() {
      return window.innerWidth <= 600;
    },

    isTablet() {
      return window.innerWidth <= 960 && window.innerWidth > 600;
    },

    // Handle mobile-specific interactions
    handleMobileCardClick(booking, event) {
      // On mobile, prevent card hover effects and handle touch better
      if (this.isMobile()) {
        event.preventDefault();
        // Add a small delay to prevent accidental double-taps
        setTimeout(() => {
          this.viewBookingDetails(booking);
        }, 100);
      }
    },

    // NEW WALLET METHODS
    // Fetch user data and wallet balance
    async fetchUserDataAndWallet() {
      this.walletLoading = true;
      try {
        // Get user ID from session storage
        const userId = sessionStorage.getItem('id');

        // Include user_id in the request data
        const response = await this.$root.callApi('wallet', 'get_balance', {
          user_id: userId
        });

        if (response && response.session && response.session.status === "YES" &&
          response.session.user_data && response.data && response.data.status === "YES") {

          // Store user data
          this.userData = response.session.user_data;

          // Store wallet balance
          this.walletBalance = {
            balance: response.data.balance || '0.00',
            available_credit: response.data.available_credit || '0.00',
            pending_amount: response.data.pending_amount || '0.00'
          };

          // Pre-fill guest form with user data
          this.prefillGuestForm();

          console.log('User data and wallet loaded successfully');
        } else {
          console.error('Failed to load user data or wallet balance', response);
          this.showSnackbarMessage('Failed to load wallet balance. Please try again.', 'error');
        }
      } catch (error) {
        console.error('Error fetching user data and wallet:', error);
        this.showSnackbarMessage('Error loading wallet balance. Please try again.', 'error');
      } finally {
        this.walletLoading = false;
      }
    },

    // Pre-fill guest form with user data
    prefillGuestForm() {
      if (!this.userData) return;

      this.guestForm.firstName = this.userFirstName;
      this.guestForm.lastName = this.userLastName;
      this.guestForm.email = this.userData.email || '';
      this.guestForm.phone = this.userData.contact || '';
    },

    // Toggle payment method
    togglePaymentMethod() {
      this.useWalletPayment = !this.useWalletPayment;
    },

    // Booking completion with wallet payment
    async completeBooking() {
      this.loading = true;

      try {
        // Get user ID from session storage
        const userId = sessionStorage.getItem('id');

        // Prepare booking data
        const bookingData = {
          user_id: userId,
          room_id: this.selectedRoom.id,
          check_in_date: this.dateRange.checkIn,
          check_out_date: this.dateRange.checkOut,
          adults: this.guestForm.adults,
          children: this.guestForm.children,
          guest_first_name: this.guestForm.firstName,
          guest_last_name: this.guestForm.lastName,
          guest_email: this.guestForm.email,
          guest_phone: this.guestForm.phone,
          special_requests: this.guestForm.specialRequests,
          room_preferences: JSON.stringify(this.roomPreferences),
          additional_services: JSON.stringify(
            this.additionalServices
              .filter(service => service.selected)
              .map(service => service.id)
          ),
          total_amount: this.totalAmount.toString(),
          payment_method: 'wallet'
        };

        // Call the booking API
        const response = await this.$root.callApi('rooms', 'book_rooms', bookingData);

        if (response && response.data && response.data.status === "YES") {
          // Booking successful
          this.bookingComplete = true;
          this.bookingReference = response.data.booking_reference;
          this.bookingDetails = {
            room: this.selectedRoom,
            checkIn: this.dateRange.checkIn,
            checkOut: this.dateRange.checkOut,
            nights: this.nightsCount,
            guests: {
              adults: this.guestForm.adults,
              children: this.guestForm.children
            },
            amount: this.totalAmount,
            paymentStatus: 'Confirmed',
            paymentMethod: 'Wallet',
            bookedDate: new Date().toISOString().substr(0, 10)
          };

          // Update wallet balance after successful booking
          await this.fetchUserDataAndWallet();

          this.showSnackbarMessage('Booking completed successfully!', 'success');
        } else {
          console.error('Booking failed', response);
          this.showSnackbarMessage(response?.data?.message || 'Booking failed. Please try again.', 'error');
        }
      } catch (error) {
        console.error('Error completing booking:', error);
        this.showSnackbarMessage('Error processing your booking. Please try again.', 'error');
      } finally {
        this.loading = false;
      }
    },


    // UI helpers
    showSnackbarMessage(text, color = 'success', timeout = 3000) {
      this.snackbar.text = text;
      this.snackbar.color = color;
      this.snackbar.timeout = timeout;
      this.showSnackbar = true;
    },

    formatDate(dateString) {
      if (!dateString) return '';
      const options = { weekday: 'short', year: 'numeric', month: 'short', day: 'numeric' };
      return new Date(dateString).toLocaleDateString('en-US', options);
    },

    formatCurrency(amount) {
      return 'RM ' + amount.toFixed(2);
    },

    // Enhanced mobile-specific helper methods
    isMobile() {
      return window.innerWidth <= 768;
    },

    isTablet() {
      return window.innerWidth <= 960 && window.innerWidth > 768;
    },

    // Handle mobile card interactions
    handleMobileCardClick(booking, event) {
      if (this.isMobile()) {
        event.preventDefault();
        // Add haptic feedback if available
        if (navigator.vibrate) {
          navigator.vibrate(50);
        }
        // Small delay to prevent accidental double-taps
        setTimeout(() => {
          this.viewBookingDetails(booking);
        }, 100);
      }
    },

    // Enhanced date formatting for mobile
    formatDate(dateString) {
      if (!dateString) return '';
      const date = new Date(dateString);
      const options = this.isMobile() ?
        { month: 'short', day: 'numeric', year: '2-digit' } :
        { weekday: 'short', year: 'numeric', month: 'short', day: 'numeric' };
      return date.toLocaleDateString('en-US', options);
    },

    // Mobile-optimized currency formatting
    formatCurrencyMobile(amount) {
      if (this.isMobile() && amount >= 1000) {
        return 'RM ' + (amount / 1000).toFixed(1) + 'k';
      }
      return this.formatCurrency(amount);
    }
  },

  async mounted() {

    const styles = getTropicalStyles(this.theme);
    document.head.insertAdjacentHTML('beforeend', `<style>${styles}</style>`);

    // Enhanced mobile-friendly booking history styles

    // Enhanced mobile-friendly booking history styles
    const historyStyles = `
      .booking-history-card {
        transition: all 0.3s ease;
        border: 1px solid ${this.theme.creamBg};
        border-radius: 12px;
        overflow: hidden;
        cursor: pointer;
      }
      .booking-history-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 24px rgba(0,0,0,0.12);
        border-color: ${this.theme.accent};
      }
      .booking-history-card .room-image-carousel {
        border-radius: 0;
      }
      .room-card .room-image-carousel {
        border-radius: 8px 0 0 8px;
        width: 100%;
        height: 200px;
      }
      .room-card .room-image-carousel .carousel-container {
        height: 200px;
        width: 100%;
      }
      .room-card .room-image-carousel .image-container {
        height: 200px;
        width: 100%;
      }
      .room-card .room-image-carousel .carousel-image {
        width: 100% !important;
        height: 200px !important;
      }
      .cursor-pointer {
        cursor: pointer;
        transition: all 0.2s ease;
      }
      .cursor-pointer:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
      }
      
      /* Enhanced Mobile Responsive Styles */
      @media (max-width: 960px) {
        .booking-history-card .v-row {
          flex-direction: column !important;
        }
        .booking-history-card .room-image-carousel {
          height: 200px !important;
          width: 100% !important;
        }
        .booking-history-card .room-image-carousel .carousel-container {
          height: 200px !important;
          width: 100% !important;
        }
        .booking-history-card .room-image-carousel .image-container {
          height: 200px !important;
          width: 100% !important;
        }
        .booking-history-card .room-image-carousel .carousel-image {
          width: 100% !important;
          height: 200px !important;
        }
        .booking-history-card .v-card-text {
          padding: 20px !important;
        }
      }
      
      @media (max-width: 768px) {
        .content-card {
          margin: 0 4px 16px 4px !important;
          border-radius: 12px !important;
        }
        .booking-history-card {
          margin-bottom: 20px !important;
          border-radius: 16px !important;
          box-shadow: 0 4px 16px rgba(0,0,0,0.08) !important;
        }
        .booking-history-card .v-card-text {
          padding: 16px !important;
        }
        .booking-history-card .text-h6 {
          font-size: 1.1rem !important;
          line-height: 1.3 !important;
          margin-bottom: 8px !important;
        }
        .booking-history-card .text-body-2 {
          font-size: 0.85rem !important;
          line-height: 1.4 !important;
        }
        .booking-history-card .v-btn {
          font-size: 0.8rem !important;
          padding: 8px 12px !important;
          min-height: 36px !important;
          margin: 2px !important;
        }
        
        /* Better mobile stats cards */
        .v-col[cols="4"] .v-card {
          padding: 16px 8px !important;
          min-height: 90px !important;
          display: flex !important;
          flex-direction: column !important;
          justify-content: center !important;
          border-radius: 12px !important;
        }
        .text-h5.text-sm-h4 {
          font-size: 1.8rem !important;
          line-height: 1.2 !important;
          font-weight: 700 !important;
        }
        .text-caption.text-sm-body-2 {
          font-size: 0.8rem !important;
          margin-top: 4px !important;
          font-weight: 500 !important;
        }
        
        /* Enhanced mobile filter chips */
        .quick-filter-chips {
          gap: 8px !important;
          justify-content: center !important;
          margin-bottom: 20px !important;
        }
        .quick-filter-chips .v-chip {
          font-size: 0.85rem !important;
          height: 36px !important;
          padding: 0 16px !important;
          font-weight: 500 !important;
          border-radius: 18px !important;
        }
        
        /* Better mobile expansion panel */
        .mobile-filter-panel {
          border-radius: 12px !important;
          overflow: hidden;
        }
        .mobile-filter-panel .v-expansion-panel-title {
          font-size: 0.95rem !important;
          padding: 16px !important;
          font-weight: 500 !important;
        }
        .mobile-filter-panel .v-expansion-panel-text {
          padding: 0 16px 20px 16px !important;
        }
        .mobile-filter-panel .v-expansion-panel-text .v-row .v-col {
          padding: 4px 8px !important;
        }
        .mobile-filter-panel .v-text-field {
          margin-bottom: 8px !important;
        }
      }
      
      @media (max-width: 480px) {
        .booking-history-card .d-flex.gap-2.flex-wrap {
          flex-direction: column !important;
          gap: 8px !important;
        }
        .booking-history-card .v-btn {
          width: 100% !important;
          justify-content: center !important;
          min-height: 44px !important;
          font-weight: 500 !important;
        }
        
        /* Stack filter chips on very small screens */
        .quick-filter-chips {
          flex-direction: column !important;
          align-items: center !important;
        }
        .quick-filter-chips .v-chip {
          width: 200px !important;
          justify-content: center !important;
        }
        
        /* Better mobile view toggle */
        .v-btn-toggle .v-btn {
          font-size: 0.9rem !important;
          padding: 12px 16px !important;
          min-height: 48px !important;
          font-weight: 500 !important;
        }
        
        /* Enhanced mobile modals */
        .v-dialog .v-card {
          margin: 0 !important;
          border-radius: 20px 20px 0 0 !important;
          max-height: 90vh !important;
        }
        .v-dialog .v-card-title {
          padding: 20px 16px 16px 16px !important;
          font-size: 1.1rem !important;
          font-weight: 600 !important;
        }
        .v-dialog .v-card-text {
          padding: 16px !important;
        }
        .v-dialog .v-card-actions {
          padding: 16px !important;
          gap: 12px !important;
        }
        
        /* Better mobile alerts */
        .v-alert {
          font-size: 0.85rem !important;
          padding: 16px !important;
          border-radius: 12px !important;
          margin-bottom: 16px !important;
        }
        .v-alert .v-icon {
          font-size: 1.2rem !important;
        }
        
        /* Enhanced mobile empty state */
        .text-center.pa-8 {
          padding: 32px 16px !important;
          border-radius: 16px !important;
        }
        .text-center.pa-8 .v-icon {
          font-size: 4rem !important;
        }
        .text-center.pa-8 .text-h6 {
          font-size: 1.2rem !important;
          margin-bottom: 12px !important;
        }
        .text-center.pa-8 .text-body-2 {
          font-size: 0.9rem !important;
          margin-bottom: 20px !important;
        }
      }
      
      /* Enhanced touch interactions */
      @media (hover: none) and (pointer: coarse) {
        .booking-history-card {
          transition: transform 0.2s ease !important;
        }
        .booking-history-card:hover {
          transform: none !important;
          box-shadow: 0 4px 16px rgba(0,0,0,0.1) !important;
        }
        .booking-history-card:active {
          transform: scale(0.98) !important;
        }
        .cursor-pointer:hover {
          transform: none !important;
        }
        .cursor-pointer:active {
          transform: scale(0.95) !important;
        }
        .v-btn {
          min-height: 48px !important;
          font-weight: 500 !important;
          transition: transform 0.1s ease !important;
        }
        .v-btn:active {
          transform: scale(0.95) !important;
        }
        .v-chip.cursor-pointer {
          min-height: 44px !important;
          font-weight: 500 !important;
          transition: transform 0.1s ease !important;
        }
        .v-chip.cursor-pointer:active {
          transform: scale(0.95) !important;
        }
      }
      
      /* Loading and error states */
      .text-center.py-8 {
        padding: 48px 16px !important;
      }
      .text-center.py-8 .v-progress-circular {
        margin-bottom: 16px !important;
      }
      .text-center.py-8 .text-body-1 {
        font-size: 0.95rem !important;
        font-weight: 500 !important;
      }
      
      /* Better mobile spacing */
      @media (max-width: 600px) {
        .pa-sm-4.pa-2 {
          padding: 16px !important;
        }
        .mb-4 {
          margin-bottom: 20px !important;
        }
        .gap-2 {
          gap: 8px !important;
        }
      }
    `;

    document.head.insertAdjacentHTML('beforeend', `<style>${styles}${historyStyles}</style>`);

    // Set a default checkout date (checkout = checkin + 1 day)
    const checkoutDate = new Date(this.dateRange.checkIn);
    checkoutDate.setDate(checkoutDate.getDate() + 1);
    this.dateRange.checkOut = checkoutDate.toISOString().substr(0, 10);

    // Fetch user data, wallet balance, and available rooms
    await this.fetchUserDataAndWallet();
    await this.fetchAvailableRooms();
  },

  template: `
    <v-container class="tropical-bookings-page pa-0" fluid>
      <!-- Animated Background Decoration -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>

      <v-overlay v-model="loading" class="align-center justify-center" app persistent :scrim="theme.primary" :opacity="0.8">
        <div class="text-center">
          <v-progress-circular indeterminate :color="theme.secondary" size="64" width="6"></v-progress-circular>
          <div class="mt-3" :style="{color: theme.lightText, fontSize: '16px', fontWeight: '500'}">Processing your booking...</div>
        </div>
      </v-overlay>

      <!-- Hotel Logo & Header -->
      <v-row>
        <v-col cols="12" class="text-center pt-6 pb-2">
          <v-img src="https://via.placeholder.com/180x120?text=Rainforest+Resort" height="120" contain class="mx-auto mb-3"></v-img>
          <h1 :style="{color: theme.primary, fontFamily: 'Playfair Display, serif', fontSize: '2.5rem', fontWeight: 700, marginBottom: 0, letterSpacing: '-0.5px'}">Tapang Club</h1>
          <h2 :style="{color: theme.secondary, fontFamily: 'Poppins, sans-serif', fontSize: '1.5rem', fontWeight: 300, marginTop: 0, letterSpacing: '1px'}">Accommodation Booking</h2>
        </v-col>
      </v-row>

      <!-- Main Content -->
      <v-row>
        <v-col cols="12" sm="10" md="8" lg="6" class="mx-auto px-sm-4 px-2">
          <!-- Enhanced Mobile-Friendly View Toggle -->
          <v-card class="content-card mb-4" flat elevation="2" :style="{borderRadius: '16px', overflow: 'hidden'}">
            <v-card-text class="pa-3">
              <v-btn-toggle
                v-model="viewMode"
                :color="theme.primary"
                variant="outlined"
                divided
                mandatory
                class="w-100"
                :style="{borderRadius: '12px', overflow: 'hidden'}"
              >
                <v-btn
                  value="booking"
                  @click="switchToBooking"
                  :color="viewMode === 'booking' ? theme.primary : theme.muted"
                  :text-color="viewMode === 'booking' ? theme.lightText : theme.darkText"
                  class="flex-grow-1"
                  :style="{
                    minHeight: isMobile() ? '52px' : '44px',
                    fontSize: isMobile() ? '0.9rem' : '0.875rem',
                    fontWeight: '500',
                    borderRadius: '12px 0 0 12px'
                  }"
                >
                  <v-icon start :size="isMobile() ? 'default' : 'small'">mdi-plus-circle</v-icon>
                  <span class="d-none d-sm-inline">New Booking</span>
                  <span class="d-sm-none">Book</span>
                </v-btn>
                <v-btn
                  value="history"
                  @click="switchToHistory"
                  :color="viewMode === 'history' ? theme.primary : theme.muted"
                  :text-color="viewMode === 'history' ? theme.lightText : theme.darkText"
                  class="flex-grow-1 position-relative"
                  :style="{
                    minHeight: isMobile() ? '52px' : '44px',
                    fontSize: isMobile() ? '0.9rem' : '0.875rem',
                    fontWeight: '500',
                    borderRadius: '0 12px 12px 0'
                  }"
                >
                  <v-icon start :size="isMobile() ? 'default' : 'small'">mdi-history</v-icon>
                  <span class="d-none d-sm-inline">Booking History</span>
                  <span class="d-sm-none">History</span>
                  <v-badge
                    v-if="upcomingBookingsCount > 0"
                    :content="upcomingBookingsCount"
                    :color="theme.secondary"
                    :text-color="theme.primary"
                    :style="{
                      position: 'absolute',
                      top: isMobile() ? '8px' : '6px',
                      right: isMobile() ? '8px' : '6px'
                    }"
                  ></v-badge>
                </v-btn>
              </v-btn-toggle>
            </v-card-text>
          </v-card>

          <!-- Steps Indicator (only show for booking view) -->
          <v-card v-if="viewMode === 'booking'" class="content-card mb-4" flat>
            <v-card-text class="pa-2">
              <div class="d-flex justify-space-between align-center">
                <div v-for="step in totalSteps" :key="step" class="text-center" style="flex: 1;">
                  <v-avatar
                    :color="step === currentStep ? theme.primary : (step < currentStep ? theme.accent : theme.creamBg)"
                    :size="step === currentStep ? 40 : 32"
                    class="mb-1"
                    :class="{'elevation-2': step === currentStep}"
                  >
                    <v-icon :color="step <= currentStep ? theme.lightText : theme.primary">
                      {{ step === 1 ? 'mdi-calendar-range' : 
                         step === 2 ? 'mdi-bed' : 
                         step === 3 ? 'mdi-account-check' : 
                         step === 4 ? 'mdi-plus-circle' : 
                         'mdi-check-circle' }}
                    </v-icon>
                  </v-avatar>
                  <div class="text-caption d-none d-sm-block" :style="{color: step === currentStep ? theme.primary : theme.muted, fontWeight: step === currentStep ? 'bold' : 'normal'}">
                    {{ step === 1 ? 'Dates' : 
                       step === 2 ? 'Room' : 
                       step === 3 ? 'Details' : 
                       step === 4 ? 'Add-ons' : 
                       'Payment' }}
                  </div>
                </div>
              </div>
              <v-progress-linear
                :model-value="((currentStep - 1) / (totalSteps - 1)) * 100"
                :color="theme.primary"
                height="6"
                rounded
                class="mt-2"
              ></v-progress-linear>
            </v-card-text>
          </v-card>

          

          <!-- Content Card with Steps (only show for booking view) -->
          <v-card v-if="viewMode === 'booking'" class="content-card" elevation="6">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">
                {{ currentStep === 1 ? 'mdi-calendar-range' : 
                   currentStep === 2 ? 'mdi-bed' : 
                   currentStep === 3 ? 'mdi-account-check' : 
                   currentStep === 4 ? 'mdi-plus-circle' : 
                   'mdi-check-circle' }}
              </v-icon>
              {{ currentStep === 1 ? 'Select Your Dates' : 
                 currentStep === 2 ? 'Choose Your Room' : 
                 currentStep === 3 ? 'Guest Information' : 
                 currentStep === 4 ? 'Additional Services' : 
                 'Payment & Confirmation' }}
            </v-card-title>

            <!-- Step 1: Date Selection -->
            <v-card-text v-if="currentStep === 1" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">When would you like to stay with us?</p>
              
              <v-row>
                <v-col cols="12" sm="6">
                  <v-text-field
                    v-model="dateRange.checkIn"
                    label="Check-in Date"
                    type="date"
                    :min="minDate"
                    :max="maxDate"
                    variant="outlined"
                    :color="theme.primary"
                    prepend-inner-icon="mdi-calendar-arrow-right"
                    density="comfortable"
                    @update:model-value="dateRange.checkOut = null"
                  ></v-text-field>
                </v-col>
                <v-col cols="12" sm="6">
                  <v-text-field
                    v-model="dateRange.checkOut"
                    label="Check-out Date"
                    type="date"
                    :min="dateRange.checkIn"
                    :max="maxDate"
                    variant="outlined"
                    :color="theme.primary"
                    prepend-inner-icon="mdi-calendar-arrow-left"
                    density="comfortable"
                    :disabled="!dateRange.checkIn"
                  ></v-text-field>
                </v-col>
              </v-row>
              
              <v-alert
                v-if="isDateRangeValid"
                :color="theme.accent"
                icon="mdi-information"
                variant="tonal"
                class="mb-4"
                density="comfortable"
              >
                Your stay: <strong>{{ formattedDateRange }}</strong>
              </v-alert>
              
              <v-alert
                v-if="!isDateRangeValid && (dateRange.checkIn || dateRange.checkOut)"
                type="warning"
                variant="tonal"
                icon="mdi-alert-circle"
                class="mt-4 mb-4"
                title="Invalid Date Range"
              >
                Please select both check-in and check-out dates. Check-out must be after check-in.
              </v-alert>
              
              <div class="d-flex mt-6">
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  :disabled="!isDateRangeValid"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 2: Room Selection -->
            <v-card-text v-if="currentStep === 2" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Select your perfect rainforest retreat</p>
              
              <v-alert
                :color="theme.accent"
                icon="mdi-calendar-check"
                variant="tonal"
                class="mb-4"
                density="comfortable"
              >
                Your stay: <strong>{{ formattedDateRange }}</strong>
              </v-alert>
              
              <div v-for="room in availableRooms" :key="room.id" class="mb-4">
                <v-card 
                  :class="{ 'selected-room': room.selected }"
                  :color="room.selected ? theme.accent + '15' : ''"
                  variant="outlined"
                  @click="selectRoom(room)"
                  class="room-card"
                >
                  <v-row no-gutters>
                    <v-col cols="12" sm="4">
                      <RoomImageCarousel
                        :images="room.images"
                        :room-name="room.name"
                        :height="200"
                        :show-fullscreen="true"
                        @image-change="(index) => console.log('Image changed to:', index)"
                      />
                    </v-col>
                    <v-col cols="12" sm="8">
                      <v-card-text>
                        <div class="d-flex justify-space-between align-center mb-2">
                          <h3 class="text-h6" :style="{color: theme.primary}">{{ room.name }}</h3>
                          <div class="text-h6 font-weight-bold" :style="{color: theme.secondary}">
                            {{ formatCurrency(room.price) }} <span class="text-caption">/ night</span>
                          </div>
                        </div>
                        
                        <p class="text-body-2 mb-3">{{ room.description }}</p>
                        
                        <div class="d-flex align-center mb-2">
                          <v-icon size="small" :color="theme.primary" class="mr-1">mdi-account-multiple</v-icon>
                          <span class="text-body-2">{{ room.capacity.adults }} Adults, {{ room.capacity.children }} Children max</span>
                        </div>
                        
                        <div class="d-flex flex-wrap">
                          <v-chip
                            v-for="(amenity, index) in room.amenities"
                            :key="index"
                            size="small"
                            class="mr-1 mb-1"
                            :color="theme.accent"
                            variant="outlined"
                          >
                            {{ amenity }}
                          </v-chip>
                        </div>
                        
                        <v-divider class="my-3"></v-divider>
                        
                        <div class="d-flex justify-space-between align-center">
                          <div class="text-body-1 font-weight-medium">
                            {{ formatCurrency(room.price * nightsCount) }} total
                          </div>
                          <v-btn
                            :color="room.selected ? theme.primary : theme.secondary"
                            :text-color="room.selected ? theme.lightText : theme.primary"
                            variant="elevated"
                            size="small"
                          >
                            {{ room.selected ? 'Selected' : 'Select Room' }}
                          </v-btn>
                        </div>
                      </v-card-text>
                    </v-col>
                  </v-row>
                </v-card>
              </div>
              
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  :disabled="!isRoomSelected"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 3: Guest Information - MODIFIED -->
            <v-card-text v-if="currentStep === 3" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Please confirm your details</p>
              
              <v-alert
                :color="theme.accent"
                icon="mdi-information"
                variant="tonal"
                class="mb-4"
                density="comfortable"
              >
                <div><strong>{{ selectedRoom.name }}</strong></div>
                <div>{{ formattedDateRange }}</div>
              </v-alert>
              
              <v-form v-model="guestFormValid">
                <h4 class="text-subtitle-1 mb-2" :style="{color: theme.primary}">Guest Information</h4>
                
                <v-row>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.firstName"
                      label="First Name"
                      :rules="[v => !!v || 'First name is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-account"
                      :readonly="userData !== null"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.lastName"
                      label="Last Name"
                      :rules="[v => !!v || 'Last name is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      :readonly="userData !== null"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.email"
                      label="Email"
                      :rules="[
                        v => !!v || 'Email is required',
                        v => /.+@.+\..+/.test(v) || 'Email must be valid'
                      ]"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-email"
                      :readonly="userData !== null"
                    ></v-text-field>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-text-field
                      v-model="guestForm.phone"
                      label="Phone"
                      :rules="[v => !!v || 'Phone is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-phone"
                      :readonly="userData !== null"
                    ></v-text-field>
                  </v-col>
                  
                  <v-col cols="12">
                    <v-divider class="my-2"></v-divider>
                    <h4 class="text-subtitle-2 mb-2 mt-2" :style="{color: theme.primary}">Number of Guests</h4>
                  </v-col>
                  
                  <v-col cols="12" sm="6">
                    <v-select
                      v-model="guestForm.adults"
                      :items="[1, 2, 3, 4]"
                      label="Adults"
                      :rules="[v => v > 0 || 'At least 1 adult is required']"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-account-multiple"
                    ></v-select>
                  </v-col>
                  <v-col cols="12" sm="6">
                    <v-select
                      v-model="guestForm.children"
                      :items="[0, 1, 2, 3, 4]"
                      label="Children"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-account-child"
                    ></v-select>
                  </v-col>
                  
                  <v-col cols="12">
                    <v-textarea
                      v-model="guestForm.specialRequests"
                      label="Special Requests"
                      hint="Please let us know if you have any special requests"
                      variant="outlined"
                      :color="theme.primary"
                      density="comfortable"
                      prepend-inner-icon="mdi-text-box"
                      rows="3"
                    ></v-textarea>
                  </v-col>
                </v-row>
              </v-form>
                
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  :disabled="!guestFormValid"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 4: Additional Services -->
            <v-card-text v-if="currentStep === 4" class="pa-sm-4 pa-2">
              <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Enhance your rainforest experience</p>
              
              <v-row>
                <v-col cols="12">
                  <v-alert
                    :color="theme.accent"
                    icon="mdi-leaf-circle"
                    variant="tonal"
                    class="mb-4"
                    density="comfortable"
                  >
                    Select any extra services to enhance your stay at Tapang Club
                  </v-alert>
                  
                  <div class="services-grid">
                    <v-card
                      v-for="service in additionalServices"
                      :key="service.id"
                      :class="{ 'selected-service': service.selected }"
                      class="service-card mb-3"
                      @click="service.selected = !service.selected"
                      elevation="1"
                      :color="service.selected ? theme.accent + '15' : ''"
                      variant="outlined"
                    >
                      <div class="d-flex align-center pa-3">
                        <v-avatar :color="service.selected ? theme.primary : theme.accent" size="42" class="mr-3">
                          <v-icon :color="service.selected ? theme.lightText : theme.primary">{{ service.icon }}</v-icon>
                        </v-avatar>
                        
                        <div class="flex-grow-1">
                          <div class="font-weight-medium" :style="{color: theme.primary}">{{ service.name }}</div>
                          <div class="text-caption" :style="{color: theme.muted}">
                            {{ formatCurrency(service.price) }}
                            <span v-if="service.perPerson"> per person</span>
                            <span v-else-if="service.id === 'parking' || service.id === 'wifi'"> per day</span>
                          </div>
                        </div>
                        
                        <v-checkbox
                          v-model="service.selected"
                          :color="theme.primary"
                          hide-details
                          density="compact"
                        ></v-checkbox>
                      </div>
                    </v-card>
                  </div>
                  
                  <v-divider class="my-4"></v-divider>
                  
                  <v-card class="summary-card pa-4" variant="outlined">
                    <div class="d-flex justify-space-between mb-2">
                      <span class="text-body-1">{{ selectedRoom.name }} ({{ nightsCount }} nights):</span>
                      <span class="text-body-1">{{ formatCurrency(roomTotal) }}</span>
                    </div>
                    <div class="d-flex justify-space-between mb-2">
                      <span class="text-body-1">Additional Services:</span>
                      <span class="text-body-1">{{ formatCurrency(selectedServicesTotal) }}</span>
                    </div>
                    <v-divider class="my-2"></v-divider>
                    <div class="d-flex justify-space-between font-weight-bold" :style="{color: theme.primary}">
                      <span class="text-h6">Total Amount:</span>
                      <span class="text-h6">{{ formatCurrency(totalAmount) }}</span>
                    </div>
                  </v-card>
                </v-col>
              </v-row>
              
              <div class="d-flex mt-6">
                <v-btn 
                  :color="theme.muted" 
                  variant="text"
                  @click="prevStep"
                >
                  <v-icon start>mdi-arrow-left</v-icon>
                  Back
                </v-btn>
                <v-spacer></v-spacer>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="nextStep"
                  size="large"
                  elevation="2"
                >
                  Continue
                  <v-icon end>mdi-arrow-right</v-icon>
                </v-btn>
              </div>
            </v-card-text>
            
            <!-- Step 5: Payment & Confirmation - MODIFIED WITH WALLET -->
            <v-card-text v-if="currentStep === 5" class="pa-sm-4 pa-2">
              <div v-if="!bookingComplete">
                <p class="text-subtitle-2 mb-4" :style="{color: theme.muted}">Complete your booking</p>
                
                <v-alert
                  :color="theme.success"
                  icon="mdi-check-circle"
                  variant="tonal"
                  class="mb-4"
                >
                  You're almost done! Review your details and complete your booking.
                </v-alert>
                
                <v-card class="mb-4" variant="outlined">
                  <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">Booking Summary</v-card-title>
                  <v-card-text class="py-2">
                    <v-list density="compact" bg-color="transparent">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-account</v-icon>
                        </template>
                        <v-list-item-title>{{ guestForm.firstName }} {{ guestForm.lastName }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-bed</v-icon>
                        </template>
                        <v-list-item-title>{{ selectedRoom.name }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                        </template>
                        <v-list-item-title>{{ formattedDateRange }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-account-multiple</v-icon>
                        </template>
                        <v-list-item-title>
                          {{ guestForm.adults }} Adult{{ guestForm.adults > 1 ? 's' : '' }}
                          <span v-if="guestForm.children > 0">, {{ guestForm.children }} Child{{ guestForm.children > 1 ? 'ren' : '' }}</span>
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item v-if="selectedServicesTotal > 0">
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-plus-circle</v-icon>
                        </template>
                        <v-list-item-title>
                          Additional Services: {{ formatCurrency(selectedServicesTotal) }}
                        </v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-cash</v-icon>
                        </template>
                        <v-list-item-title class="font-weight-bold" :style="{color: theme.primary}">
                          Total: {{ formatCurrency(totalAmount) }}
                        </v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-card-text>
                </v-card>
                
                <!-- NEW: Wallet Balance Card -->
                <v-card class="mb-4" variant="outlined">
                  <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">
                    <v-icon :color="theme.primary" class="mr-2">mdi-wallet</v-icon>
                    Wallet Balance
                    <v-spacer></v-spacer>
                    <v-chip
                      :color="hasWalletSufficientBalance ? theme.success : theme.error"
                      text-color="white"
                      size="small"
                    >
                      {{ hasWalletSufficientBalance ? 'Sufficient Balance' : 'Insufficient Balance' }}
                    </v-chip>
                  </v-card-title>
                  <v-card-text class="py-2">
                    <v-list density="compact" bg-color="transparent">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon :color="theme.primary">mdi-cash-multiple</v-icon>
                        </template>
                        <v-list-item-title>Balance: {{ formattedWalletBalance }}</v-list-item-title>
                      </v-list-item>
                    </v-list>
                    <v-alert
                      v-if="!hasWalletSufficientBalance"
                      type="warning"
                      variant="tonal"
                      icon="mdi-alert-circle"
                      class="mt-2"
                      title="Insufficient Wallet Balance"
                    >
                      Your wallet balance is insufficient for this booking. Please add funds to your wallet to continue.
                    </v-alert>
                  </v-card-text>
                </v-card>
                
                <!-- NEW: Payment Method Selection -->
                <v-card class="mb-4" variant="outlined">
                  <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">Payment Method</v-card-title>
                  <v-card-text class="py-2">
                    <v-radio-group v-model="useWalletPayment" inline>
                      <v-radio
                        :value="true"
                        :label="'Wallet Payment'"
                        :color="theme.primary"
                        :disabled="!hasWalletSufficientBalance"
                      ></v-radio>
                    </v-radio-group>
                    
                    <v-alert
                      v-if="!hasWalletSufficientBalance && useWalletPayment"
                      type="warning"
                      variant="tonal"
                      icon="mdi-alert-circle"
                      class="mt-2"
                      title="Insufficient Wallet Balance"
                    >
                      Your wallet balance is insufficient for this booking. Please choose credit card payment or add funds to your wallet.
                    </v-alert>
                  </v-card-text>
                </v-card>
                
                <!-- Credit Card Form (shown only when credit card payment is selected) -->
                <v-form v-model="paymentFormValid" v-if="!useWalletPayment">
                  <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Payment Information</h4>
                  
                  <v-row>
                    <v-col cols="12">
                      <v-text-field
                        v-model="paymentForm.cardholderName"
                        label="Cardholder Name"
                        :rules="[v => !!v || 'Cardholder name is required']"
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                        prepend-inner-icon="mdi-account"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12">
                      <v-text-field
                        v-model="paymentForm.cardNumber"
                        label="Card Number"
                        :rules="[v => !!v || 'Card number is required']"
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                        prepend-inner-icon="mdi-credit-card"
                      ></v-text-field>
                    </v-col>
                    <v-col cols="12" sm="4">
                      <v-select
                        v-model="paymentForm.expiryMonth"
                        :items="['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']"
                        label="Month"
                        :rules="[v => !!v || 'Month is required']"
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                      ></v-select>
                    </v-col>
                    <v-col cols="12" sm="4">
                      <v-select
                        v-model="paymentForm.expiryYear"
                        :items="['2023', '2024', '2025', '2026', '2027', '2028', '2029', '2030']"
                        label="Year"
                        :rules="[v => !!v || 'Year is required']"
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                      ></v-select>
                    </v-col>
                    <v-col cols="12" sm="4">
                      <v-text-field
                        v-model="paymentForm.cvv"
                        label="CVV"
                        :rules="[v => !!v || 'CVV is required']"
                        variant="outlined"
                        :color="theme.primary"
                        density="comfortable"
                        type="password"
                      ></v-text-field>
                    </v-col>
                  </v-row>
                </v-form>
                
                <v-divider class="my-4"></v-divider>
                
                <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Terms & Conditions</h4>
                
                <v-checkbox
                  v-model="termsAccepted"
                  :color="theme.primary"
                  label="I accept the hotel's terms and policies"
                  :rules="[v => !!v || 'You must accept the terms to continue']"
                ></v-checkbox>
                
                <v-checkbox
                  v-model="dataConfirmed"
                  :color="theme.primary"
                  label="I confirm that all the information provided is correct"
                  :rules="[v => !!v || 'You must confirm your information']"
                ></v-checkbox>
                
                <div class="d-flex mt-6">
                  <v-btn 
                    :color="theme.muted" 
                    variant="text"
                    @click="prevStep"
                  >
                    <v-icon start>mdi-arrow-left</v-icon>
                    Back
                  </v-btn>
                  <v-spacer></v-spacer>
                  <v-btn
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    @click="completeBooking"
                    size="large"
                    elevation="2"
                    :loading="loading"
                    :disabled="!canCompleteBooking"
                  >
                    <v-icon start>mdi-check-circle</v-icon>
                    Complete Booking
                  </v-btn>
                </div>
              </div>
              
              <!-- Booking Success Card - MODIFIED -->
              <div v-else>
                <div class="text-center py-4" :style="{ backgroundColor: theme.success, color: theme.lightText, borderRadius: '8px 8px 0 0' }">
                  <v-icon size="64" class="mb-2">mdi-check-circle</v-icon>
                  <h2 class="text-h5 font-weight-bold">Booking Confirmed!</h2>
                </div>
                
                <div class="pa-4">
                  <div class="text-center mb-4">
                    <h3 class="text-h6" :style="{ color: theme.primary }">Thank you for choosing Tapang Club</h3>
                    <p class="text-body-1">Your booking has been confirmed and we look forward to welcoming you!</p>
                  </div>
                  
                  <v-card class="mb-4" variant="outlined">
                    <v-card-text class="text-center py-4">
                      <div class="text-h6 mb-2" :style="{ color: theme.primary }">Booking Reference</div>
                      <div class="text-h4 font-weight-bold mb-3" :style="{ color: theme.secondary }">{{ bookingReference }}</div>
                      <p class="text-body-2">Please save this reference number for your records</p>
                    </v-card-text>
                  </v-card>
                  
                  <v-card class="mb-4" variant="outlined">
                    <v-card-title class="text-subtitle-1 pb-1 pt-3" :style="{color: theme.primary}">Booking Details</v-card-title>
                    <v-card-text class="py-2">
                      <v-list density="compact" bg-color="transparent">
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-account</v-icon>
                          </template>
                          <v-list-item-title>{{ guestForm.firstName }} {{ guestForm.lastName }}</v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-bed</v-icon>
                          </template>
                          <v-list-item-title>{{ selectedRoom.name }}</v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                          </template>
                          <v-list-item-title>{{ formattedDateRange }}</v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">{{ bookingDetails.paymentMethod === 'Wallet' ? 'mdi-wallet' : 'mdi-credit-card' }}</v-icon>
                          </template>
                          <v-list-item-title>
                            Payment Method: {{ bookingDetails.paymentMethod }}
                          </v-list-item-title>
                        </v-list-item>
                        <v-list-item>
                          <template v-slot:prepend>
                            <v-icon :color="theme.primary">mdi-cash</v-icon>
                          </template>
                          <v-list-item-title>
                            Total: {{ formatCurrency(totalAmount) }} (Paid)
                          </v-list-item-title>
                        </v-list-item>
                      </v-list>
                    </v-card-text>
                  </v-card>
                  
                  <v-divider class="my-4"></v-divider>
                  
                  <div class="important-info">
                    <h4 class="text-subtitle-1 mb-2" :style="{ color: theme.primary }">Important Information:</h4>
                    <v-list density="compact" bg-color="transparent">
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-clock</v-icon>
                        </template>
                        <v-list-item-title>Check-in time: 2:00 PM / Check-out time: 12:00 PM</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-email-fast</v-icon>
                        </template>
                        <v-list-item-title>A confirmation email has been sent to {{ guestForm.email }}</v-list-item-title>
                      </v-list-item>
                      <v-list-item>
                        <template v-slot:prepend>
                          <v-icon small :color="theme.primary">mdi-information</v-icon>
                        </template>
                        <v-list-item-title>Please present your booking reference upon arrival</v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </div>
                
                  <div class="d-flex mt-6 justify-center">
                    <v-btn
                      :color="theme.secondary"
                      :text-color="theme.primary"
                      elevation="2"
                      prepend-icon="mdi-file-document"
                    >
                      Download Receipt
                    </v-btn>
                    <v-btn
                      :color="theme.primary"
                      :text-color="theme.lightText"
                      elevation="2"
                      class="ml-4"
                      prepend-icon="mdi-email"
                    >
                      Email Details
                    </v-btn>
                  </div>
                </div>
              </div>
            </v-card-text>
          </v-card>

          <!-- Booking History View -->
          <v-card v-if="viewMode === 'history'" class="content-card" elevation="6">
            <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
              <v-icon :color="theme.secondary" class="mr-2">mdi-history</v-icon>
              Your Booking History
            </v-card-title>

            <!-- Filter Controls -->
            <v-card-text class="pa-sm-4 pa-2 pb-0">
              <v-expansion-panels v-model="filterPanelOpen" variant="accordion" class="mb-4 mobile-filter-panel">
                <v-expansion-panel>
                  <v-expansion-panel-title>
                    <v-icon :color="theme.primary" class="mr-2">mdi-filter-variant</v-icon>
                    Filter & Search Options
                  </v-expansion-panel-title>
                  <v-expansion-panel-text>
                    <v-row>
                      <!-- Status Filter -->
                      <v-col cols="12" sm="6" md="3">
                        <v-select
                          v-model="historyFilters.status"
                          :items="[
                            { title: 'All Bookings', value: 'all' },
                            { title: 'Upcoming', value: 'upcoming' },
                            { title: 'Completed', value: 'completed' },
                            { title: 'Cancelled', value: 'cancelled' }
                          ]"
                          label="Status"
                          variant="outlined"
                          :color="theme.primary"
                          density="comfortable"
                          prepend-inner-icon="mdi-filter"
                          @update:model-value="applyHistoryFilters"
                        ></v-select>
                      </v-col>

                      <!-- Date From Filter -->
                      <v-col cols="12" sm="6" md="3">
                        <v-text-field
                          v-model="historyFilters.dateFrom"
                          label="From Date"
                          type="date"
                          variant="outlined"
                          :color="theme.primary"
                          density="comfortable"
                          prepend-inner-icon="mdi-calendar-arrow-right"
                          @update:model-value="applyHistoryFilters"
                        ></v-text-field>
                      </v-col>

                      <!-- Date To Filter -->
                      <v-col cols="12" sm="6" md="3">
                        <v-text-field
                          v-model="historyFilters.dateTo"
                          label="To Date"
                          type="date"
                          variant="outlined"
                          :color="theme.primary"
                          density="comfortable"
                          prepend-inner-icon="mdi-calendar-arrow-left"
                          :min="historyFilters.dateFrom"
                          @update:model-value="applyHistoryFilters"
                        ></v-text-field>
                      </v-col>

                      <!-- Search Query -->
                      <v-col cols="12" sm="6" md="3">
                        <v-text-field
                          v-model="historyFilters.searchQuery"
                          label="Search bookings..."
                          variant="outlined"
                          :color="theme.primary"
                          density="comfortable"
                          prepend-inner-icon="mdi-magnify"
                          clearable
                          @update:model-value="searchBookings"
                          placeholder="Reference or room name"
                        ></v-text-field>
                      </v-col>
                    </v-row>

                    <!-- Filter Actions -->
                    <div class="d-flex justify-end mt-3">
                      <v-btn
                        variant="outlined"
                        :color="theme.muted"
                        @click="clearHistoryFilters"
                        class="mr-2"
                      >
                        <v-icon start>mdi-filter-remove</v-icon>
                        Clear Filters
                      </v-btn>
                      <v-btn
                        :color="theme.primary"
                        :text-color="theme.lightText"
                        @click="applyHistoryFilters"
                        :loading="historyLoading"
                      >
                        <v-icon start>mdi-magnify</v-icon>
                        Apply Filters
                      </v-btn>
                    </div>
                  </v-expansion-panel-text>
                </v-expansion-panel>
              </v-expansion-panels>

              <!-- Quick Filter Buttons -->
              <div class="d-flex flex-wrap gap-2 mb-4 quick-filter-chips">
                <v-chip
                  :color="historyFilters.status === 'all' ? theme.primary : theme.muted"
                  :variant="historyFilters.status === 'all' ? 'flat' : 'outlined'"
                  @click="quickFilter('all')"
                  class="cursor-pointer"
                >
                  <v-icon start size="small">mdi-view-list</v-icon>
                  All Bookings
                </v-chip>
                <v-chip
                  :color="historyFilters.status === 'upcoming' ? theme.secondary : theme.muted"
                  :variant="historyFilters.status === 'upcoming' ? 'flat' : 'outlined'"
                  @click="quickFilter('upcoming')"
                  class="cursor-pointer"
                >
                  <v-icon start size="small">mdi-calendar-clock</v-icon>
                  Upcoming
                </v-chip>
                <v-chip
                  :color="historyFilters.status === 'completed' ? theme.success : theme.muted"
                  :variant="historyFilters.status === 'completed' ? 'flat' : 'outlined'"
                  @click="quickFilter('completed')"
                  class="cursor-pointer"
                >
                  <v-icon start size="small">mdi-check-circle</v-icon>
                  Completed
                </v-chip>
                <v-chip
                  :color="historyFilters.status === 'cancelled' ? theme.error : theme.muted"
                  :variant="historyFilters.status === 'cancelled' ? 'flat' : 'outlined'"
                  @click="quickFilter('cancelled')"
                  class="cursor-pointer"
                >
                  <v-icon start size="small">mdi-cancel</v-icon>
                  Cancelled
                </v-chip>
              </div>
            </v-card-text>

            <v-card-text class="pa-sm-4 pa-2 pt-0">
              <!-- Filter Results Summary -->
              <div v-if="!historyLoading && bookingHistory.length > 0" class="mb-4">
                <v-alert
                  :color="theme.accent"
                  variant="tonal"
                  density="compact"
                  class="text-center"
                >
                  <div class="d-flex align-center justify-center">
                    <v-icon class="mr-2">mdi-information</v-icon>
                    <span>
                      Showing {{ filteredBookingHistory.length }} of {{ bookingHistory.length }} bookings
                      <span v-if="historyFilters.status !== 'all' || historyFilters.dateFrom || historyFilters.dateTo || historyFilters.searchQuery">
                        (filtered)
                      </span>
                    </span>
                  </div>
                </v-alert>
              </div>

              <!-- Loading State -->
              <div v-if="historyLoading" class="text-center py-8">
                <v-progress-circular indeterminate :color="theme.primary" size="64" width="6"></v-progress-circular>
                <div class="mt-3" :style="{color: theme.muted, fontSize: '16px'}">Loading your booking history...</div>
                <div class="mt-2">
                  <v-btn
                    variant="text"
                    size="small"
                    :color="theme.muted"
                    @click="historyLoading = false"
                  >
                    Cancel
                  </v-btn>
                </div>
              </div>

              <!-- Error State -->
              <div v-else-if="historyError" class="text-center py-8">
                <v-icon size="80" :color="theme.error" class="mb-4">mdi-alert-circle</v-icon>
                <h3 class="text-h6 mb-2" :style="{color: theme.error}">Unable to Load Booking History</h3>
                <p class="text-body-1 mb-4" :style="{color: theme.muted}">{{ historyError }}</p>
                <div class="d-flex justify-center gap-2">
                  <v-btn
                    :color="theme.primary"
                    :text-color="theme.lightText"
                    @click="retryLoadHistory"
                    :loading="historyLoading"
                  >
                    <v-icon start>mdi-refresh</v-icon>
                    Try Again
                  </v-btn>
                  <v-btn
                    variant="outlined"
                    :color="theme.muted"
                    @click="switchToBooking"
                  >
                    <v-icon start>mdi-plus-circle</v-icon>
                    Make New Booking
                  </v-btn>
                </div>
              </div>

              <!-- Empty State -->
              <div v-else-if="!historyLoading && !historyError && bookingHistory.length === 0" class="text-center py-8">
                <v-icon size="80" :color="theme.muted" class="mb-4">mdi-calendar-blank</v-icon>
                <h3 class="text-h6 mb-2" :style="{color: theme.primary}">No Bookings Yet</h3>
                <p class="text-body-1 mb-4" :style="{color: theme.muted}">You haven't made any room bookings yet. Start planning your perfect getaway!</p>
                <v-btn
                  :color="theme.primary"
                  :text-color="theme.lightText"
                  @click="switchToBooking"
                  elevation="2"
                >
                  <v-icon start>mdi-plus-circle</v-icon>
                  Make Your First Booking
                </v-btn>
              </div>

              <!-- No Results State (when filters return no results) -->
              <div v-else-if="!historyLoading && !historyError && bookingHistory.length > 0 && filteredBookingHistory.length === 0" class="text-center py-8">
                <v-icon size="80" :color="theme.muted" class="mb-4">mdi-filter-remove</v-icon>
                <h3 class="text-h6 mb-2" :style="{color: theme.primary}">No Bookings Match Your Filters</h3>
                <p class="text-body-1 mb-4" :style="{color: theme.muted}">Try adjusting your search criteria or clearing the filters.</p>
                <v-btn
                  variant="outlined"
                  :color="theme.primary"
                  @click="clearHistoryFilters"
                >
                  <v-icon start>mdi-filter-remove</v-icon>
                  Clear All Filters
                </v-btn>
              </div>

              <!-- Booking History Content -->
              <div v-else-if="!historyLoading && !historyError && filteredBookingHistory.length > 0">
                <!-- Summary Stats -->
                <v-row class="mb-4">
                  <v-col cols="4" sm="4">
                    <v-card variant="outlined" class="text-center pa-2 pa-sm-3">
                      <div class="text-h5 text-sm-h4 font-weight-bold" :style="{color: theme.primary}">{{ bookingHistory.length }}</div>
                      <div class="text-caption text-sm-body-2" :style="{color: theme.muted}">Total</div>
                    </v-card>
                  </v-col>
                  <v-col cols="4" sm="4">
                    <v-card variant="outlined" class="text-center pa-2 pa-sm-3">
                      <div class="text-h5 text-sm-h4 font-weight-bold" :style="{color: theme.secondary}">{{ upcomingBookingsCount }}</div>
                      <div class="text-caption text-sm-body-2" :style="{color: theme.muted}">Upcoming</div>
                    </v-card>
                  </v-col>
                  <v-col cols="4" sm="4">
                    <v-card variant="outlined" class="text-center pa-2 pa-sm-3">
                      <div class="text-h5 text-sm-h4 font-weight-bold" :style="{color: theme.success}">{{ bookingHistory.filter(b => b.booking_status === 'Confirmed' && b.check_out_date < new Date().toISOString().substr(0, 10)).length }}</div>
                      <div class="text-caption text-sm-body-2" :style="{color: theme.muted}">Completed</div>
                    </v-card>
                  </v-col>
                </v-row>

                <!-- Booking List -->
                <div v-for="booking in filteredBookingHistory" :key="booking.id" class="mb-4">
                  <v-card variant="outlined" class="booking-history-card">
                    <v-row no-gutters>
                      <v-col cols="12" sm="3">
                        <RoomImageCarousel
                          :images="booking.room_image ? [booking.room_image] : []"
                          :room-name="booking.room_name"
                          :height="200"
                          :show-fullscreen="true"
                          class="rounded-s"
                        />
                      </v-col>
                      <v-col cols="12" sm="9">
                        <v-card-text class="pa-4">
                          <div class="d-flex justify-space-between align-start mb-2">
                            <div>
                              <h3 class="text-h6 mb-1" :style="{color: theme.primary}">{{ booking.room_name }}</h3>
                              <div class="text-body-2 mb-1" :style="{color: theme.muted}">
                                Booking Reference: <strong>{{ booking.booking_reference }}</strong>
                              </div>
                            </div>
                            <v-chip
                              :color="booking.status_color"
                              variant="flat"
                              size="small"
                              class="text-white"
                            >
                              {{ booking.status_display }}
                            </v-chip>
                          </div>

                          <v-row class="mb-3">
                            <v-col cols="12" sm="6">
                              <div class="d-flex align-center mb-2">
                                <v-icon size="small" :color="theme.primary" class="mr-2">mdi-calendar-range</v-icon>
                                <span class="text-body-2">{{ booking.formatted_check_in }} - {{ booking.formatted_check_out }}</span>
                              </div>
                              <div class="d-flex align-center mb-2">
                                <v-icon size="small" :color="theme.primary" class="mr-2">mdi-account-multiple</v-icon>
                                <span class="text-body-2">{{ booking.adults }} Adults, {{ booking.children }} Children</span>
                              </div>
                            </v-col>
                            <v-col cols="12" sm="6">
                              <div class="d-flex align-center mb-2">
                                <v-icon size="small" :color="theme.primary" class="mr-2">mdi-calendar-clock</v-icon>
                                <span class="text-body-2">{{ booking.nights }} {{ booking.nights === 1 ? 'Night' : 'Nights' }}</span>
                              </div>
                              <div class="d-flex align-center mb-2">
                                <v-icon size="small" :color="theme.primary" class="mr-2">mdi-cash</v-icon>
                                <span class="text-body-2 font-weight-medium">{{ formatCurrency(booking.pricing.total_amount) }}</span>
                              </div>
                            </v-col>
                          </v-row>

                          <v-divider class="mb-3"></v-divider>

                          <div class="d-flex justify-space-between align-center">
                            <div class="text-caption" :style="{color: theme.muted}">
                              Booked on {{ booking.formatted_created_at }}
                            </div>
                            <div class="d-flex gap-2 flex-wrap">
                              <v-btn
                                size="small"
                                variant="outlined"
                                :color="theme.primary"
                                @click="viewBookingDetails(booking)"
                              >
                                <v-icon start size="small">mdi-eye</v-icon>
                                View Details
                              </v-btn>
                              
                              <v-btn
                                v-if="booking.actions.can_cancel"
                                size="small"
                                variant="outlined"
                                color="error"
                                @click="cancelBooking(booking)"
                              >
                                <v-icon start size="small">mdi-cancel</v-icon>
                                Cancel
                              </v-btn>
                              
                              <!--<v-btn
                                size="small"
                                variant="outlined"
                                :color="theme.secondary"
                                @click="rebookFromHistory(booking)"
                              >
                                <v-icon start size="small">mdi-repeat</v-icon>
                                Book Again
                              </v-btn>-->
                            </div>
                          </div>
                        </v-card-text>
                      </v-col>
                    </v-row>
                  </v-card>
                </div>

                <!-- Load More Button (if needed for pagination) -->
                <div v-if="bookingHistory.length > 0" class="text-center mt-4">
                  <v-btn
                    variant="outlined"
                    :color="theme.primary"
                    @click="fetchBookingHistory"
                    :loading="historyLoading"
                  >
                    <v-icon start>mdi-refresh</v-icon>
                    Refresh History
                  </v-btn>
                </div>
              </div>
            </v-card-text>
          </v-card>
        </v-col>
      </v-row>

      <!-- Booking Details Modal -->
      <v-dialog 
        v-model="showBookingDetailsModal" 
        :max-width="isMobile() ? '95vw' : '800'" 
        :fullscreen="isMobile()"
        scrollable
        transition="dialog-bottom-transition"
      >
        <v-card v-if="selectedBookingDetails">
          <v-card-title class="card-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
            <v-icon :color="theme.secondary" class="mr-2">mdi-file-document-outline</v-icon>
            Booking Details
            <v-spacer></v-spacer>
            <v-btn
              icon
              variant="text"
              :color="theme.lightText"
              @click="showBookingDetailsModal = false"
            >
              <v-icon>mdi-close</v-icon>
            </v-btn>
          </v-card-title>

          <v-card-text class="pa-0">
            <!-- Booking Header -->
            <div class="pa-4 text-center" :style="{backgroundColor: theme.lightBg}">
              <h2 class="text-h5 mb-2" :style="{color: theme.primary}">{{ selectedBookingDetails.room_name }}</h2>
              <v-chip
                :color="selectedBookingDetails.status_color"
                variant="flat"
                size="large"
                class="text-white mb-2"
              >
                {{ selectedBookingDetails.status_display }}
              </v-chip>
              <div class="text-h6 font-weight-bold" :style="{color: theme.secondary}">
                {{ selectedBookingDetails.booking_reference }}
              </div>
            </div>

            <v-divider></v-divider>

            <!-- Room Image -->
            <RoomImageCarousel
              :images="selectedBookingDetails.room_image ? [selectedBookingDetails.room_image] : []"
              :room-name="selectedBookingDetails.room_name"
              :height="200"
              :show-fullscreen="true"
            />

            <v-divider></v-divider>

            <!-- Booking Information Tabs -->
            <v-tabs 
              v-model="detailsTab" 
              :color="theme.primary" 
              class="px-4"
              show-arrows
              mobile-breakpoint="sm"
            >
              <v-tab value="booking">Booking Info</v-tab>
              <v-tab value="guest">Guest Details</v-tab>
              <v-tab value="services">Services</v-tab>
              <v-tab value="payment">Payment</v-tab>
            </v-tabs>

            <v-tabs-window v-model="detailsTab" class="pa-4">
              <!-- Booking Information Tab -->
              <v-tabs-window-item value="booking">
                <v-list density="comfortable" bg-color="transparent">
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-calendar-range</v-icon>
                    </template>
                    <v-list-item-title>Check-in Date</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.formatted_check_in }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-calendar-arrow-left</v-icon>
                    </template>
                    <v-list-item-title>Check-out Date</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.formatted_check_out }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-calendar-clock</v-icon>
                    </template>
                    <v-list-item-title>Duration</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.nights }} {{ selectedBookingDetails.nights === 1 ? 'Night' : 'Nights' }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-account-multiple</v-icon>
                    </template>
                    <v-list-item-title>Guests</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.adults }} Adults, {{ selectedBookingDetails.children }} Children</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-calendar-plus</v-icon>
                    </template>
                    <v-list-item-title>Booked On</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.formatted_created_at }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item v-if="selectedBookingDetails.special_requests">
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-message-text</v-icon>
                    </template>
                    <v-list-item-title>Special Requests</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.special_requests }}</v-list-item-subtitle>
                  </v-list-item>
                </v-list>
              </v-tabs-window-item>

              <!-- Guest Details Tab -->
              <v-tabs-window-item value="guest">
                <v-list density="comfortable" bg-color="transparent">
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-account</v-icon>
                    </template>
                    <v-list-item-title>Guest Name</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.guest_info.first_name }} {{ selectedBookingDetails.guest_info.last_name }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-email</v-icon>
                    </template>
                    <v-list-item-title>Email</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.guest_info.email }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-phone</v-icon>
                    </template>
                    <v-list-item-title>Phone</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.guest_info.phone }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item v-if="selectedBookingDetails.guest_info.address">
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-map-marker</v-icon>
                    </template>
                    <v-list-item-title>Address</v-list-item-title>
                    <v-list-item-subtitle>
                      {{ selectedBookingDetails.guest_info.address }}
                      <span v-if="selectedBookingDetails.guest_info.city">, {{ selectedBookingDetails.guest_info.city }}</span>
                      <span v-if="selectedBookingDetails.guest_info.country">, {{ selectedBookingDetails.guest_info.country }}</span>
                    </v-list-item-subtitle>
                  </v-list-item>
                </v-list>

                <!-- Room Preferences -->
                <v-divider class="my-4"></v-divider>
                <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Room Preferences</h4>
                <div v-if="selectedBookingDetails.room_preferences && Object.keys(selectedBookingDetails.room_preferences).length > 0">
                  <v-chip
                    v-for="(value, key) in selectedBookingDetails.room_preferences"
                    :key="key"
                    size="small"
                    variant="outlined"
                    :color="theme.accent"
                    class="mr-2 mb-2"
                  >
                    {{ formatPreferenceKey(key) }}: {{ formatPreferenceValue(value) }}
                  </v-chip>
                </div>
                <div v-else class="text-body-2" :style="{color: theme.muted}">
                  No specific preferences recorded
                </div>
              </v-tabs-window-item>

              <!-- Services Tab -->
              <v-tabs-window-item value="services">
                <div v-if="selectedBookingDetails.additional_services && selectedBookingDetails.additional_services.length > 0">
                  <v-list density="comfortable" bg-color="transparent">
                    <v-list-item
                      v-for="service in selectedBookingDetails.additional_services"
                      :key="service"
                    >
                      <template v-slot:prepend>
                        <v-icon :color="theme.secondary">mdi-check-circle</v-icon>
                      </template>
                      <v-list-item-title>{{ formatServiceName(service) }}</v-list-item-title>
                    </v-list-item>
                  </v-list>
                </div>
                <div v-else class="text-center py-4">
                  <v-icon size="48" :color="theme.muted" class="mb-2">mdi-room-service-outline</v-icon>
                  <div class="text-body-1" :style="{color: theme.muted}">No additional services selected</div>
                </div>
              </v-tabs-window-item>

              <!-- Payment Tab -->
              <v-tabs-window-item value="payment">
                <v-list density="comfortable" bg-color="transparent">
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">mdi-cash</v-icon>
                    </template>
                    <v-list-item-title>Total Amount</v-list-item-title>
                    <v-list-item-subtitle class="text-h6 font-weight-bold" :style="{color: theme.secondary}">
                      {{ formatCurrency(selectedBookingDetails.pricing.total_amount) }}
                    </v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="theme.primary">{{ selectedBookingDetails.payment.method === 'wallet' ? 'mdi-wallet' : 'mdi-credit-card' }}</v-icon>
                    </template>
                    <v-list-item-title>Payment Method</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.payment.method === 'wallet' ? 'Wallet Payment' : 'Credit Card' }}</v-list-item-subtitle>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon :color="selectedBookingDetails.payment.status === 'Confirmed' ? theme.success : theme.warning">
                        {{ selectedBookingDetails.payment.status === 'Confirmed' ? 'mdi-check-circle' : 'mdi-clock' }}
                      </v-icon>
                    </template>
                    <v-list-item-title>Payment Status</v-list-item-title>
                    <v-list-item-subtitle>{{ selectedBookingDetails.payment.status }}</v-list-item-subtitle>
                  </v-list-item>
                </v-list>

                <!-- Pricing Breakdown -->
                <v-divider class="my-4"></v-divider>
                <h4 class="text-subtitle-1 mb-3" :style="{color: theme.primary}">Pricing Breakdown</h4>
                <v-list density="compact" bg-color="transparent">
                  <v-list-item v-if="selectedBookingDetails.pricing.room_total > 0">
                    <v-list-item-title>Room ({{ selectedBookingDetails.nights }} nights)</v-list-item-title>
                    <template v-slot:append>
                      <span>{{ formatCurrency(selectedBookingDetails.pricing.room_total) }}</span>
                    </template>
                  </v-list-item>
                  
                  <v-list-item v-if="selectedBookingDetails.pricing.services_total > 0">
                    <v-list-item-title>Additional Services</v-list-item-title>
                    <template v-slot:append>
                      <span>{{ formatCurrency(selectedBookingDetails.pricing.services_total) }}</span>
                    </template>
                  </v-list-item>
                  
                  <v-divider></v-divider>
                  
                  <v-list-item class="font-weight-bold">
                    <v-list-item-title>Total</v-list-item-title>
                    <template v-slot:append>
                      <span class="text-h6" :style="{color: theme.secondary}">
                        {{ formatCurrency(selectedBookingDetails.pricing.total_amount) }}
                      </span>
                    </template>
                  </v-list-item>
                </v-list>
              </v-tabs-window-item>
            </v-tabs-window>
          </v-card-text>

          <v-divider></v-divider>

          <v-card-actions class="pa-4">
            <v-btn
              variant="outlined"
              :color="theme.muted"
              @click="showBookingDetailsModal = false"
            >
              Close
            </v-btn>
            <v-spacer></v-spacer>
            
            <v-btn
              v-if="selectedBookingDetails.actions.can_cancel"
              variant="outlined"
              color="error"
              @click="cancelBooking(selectedBookingDetails); showBookingDetailsModal = false"
            >
              <v-icon start>mdi-cancel</v-icon>
              Cancel Booking
            </v-btn>
            
            <!--<v-btn
              :color="theme.secondary"
              :text-color="theme.primary"
              @click="rebookFromHistory(selectedBookingDetails); showBookingDetailsModal = false"
              class="ml-2"
            >
              <v-icon start>mdi-repeat</v-icon>
              Book Again
            </v-btn>-->
          </v-card-actions>
        </v-card>
      </v-dialog>

      <!-- Cancel Booking Modal -->
      <v-dialog 
        v-model="showCancelBookingModal" 
        :max-width="isMobile() ? '95vw' : '600'" 
        :fullscreen="isMobile()"
        persistent
        scrollable
      >
        <v-card v-if="bookingToCancel">
          <v-card-title class="card-header" :style="{backgroundColor: theme.error, color: theme.lightText}">
            <v-icon :color="theme.lightText" class="mr-2">mdi-cancel</v-icon>
            Cancel Booking
          </v-card-title>

          <v-card-text class="pa-4">
            <!-- Warning Alert -->
            <v-alert
              type="warning"
              variant="tonal"
              icon="mdi-alert-triangle"
              class="mb-4"
              title="Booking Cancellation"
            >
              You are about to cancel your booking. Please review the details below before proceeding.
            </v-alert>

            <!-- Booking Summary -->
            <v-card variant="outlined" class="mb-4">
              <v-card-text class="pa-3">
                <div class="d-flex align-center mb-3">
                  <v-img
                    :src="bookingToCancel.room_image"
                    width="80"
                    height="60"
                    cover
                    class="rounded mr-3"
                    @error="handleImageError($event, bookingToCancel, 'cancel')"
                  ></v-img>
                  <div>
                    <h4 class="text-subtitle-1" :style="{color: theme.primary}">{{ bookingToCancel.room_name }}</h4>
                    <div class="text-body-2" :style="{color: theme.muted}">{{ bookingToCancel.booking_reference }}</div>
                  </div>
                </div>
                
                <v-list density="compact" bg-color="transparent">
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon size="small" :color="theme.primary">mdi-calendar-range</v-icon>
                    </template>
                    <v-list-item-title class="text-body-2">{{ bookingToCancel.formatted_check_in }} - {{ bookingToCancel.formatted_check_out }}</v-list-item-title>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon size="small" :color="theme.primary">mdi-cash</v-icon>
                    </template>
                    <v-list-item-title class="text-body-2">Total Paid: {{ formatCurrency(bookingToCancel.pricing.total_amount) }}</v-list-item-title>
                  </v-list-item>
                </v-list>
              </v-card-text>
            </v-card>

            <!-- Cancellation Policy -->
            <v-card variant="outlined" class="mb-4">
              <v-card-title class="text-subtitle-1 pb-2" :style="{color: theme.primary}">
                <v-icon class="mr-2">mdi-information</v-icon>
                Cancellation Policy
              </v-card-title>
              <v-card-text class="pt-0">
                <v-list density="compact" bg-color="transparent">
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon size="small" color="success">mdi-check-circle</v-icon>
                    </template>
                    <v-list-item-title class="text-body-2">Free cancellation up to 7 days before check-in</v-list-item-title>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon size="small" :color="theme.warning">mdi-alert-circle</v-icon>
                    </template>
                    <v-list-item-title class="text-body-2">10% cancellation fee for cancellations within 7 days</v-list-item-title>
                  </v-list-item>
                  
                  <v-list-item>
                    <template v-slot:prepend>
                      <v-icon size="small" color="error">mdi-close-circle</v-icon>
                    </template>
                    <v-list-item-title class="text-body-2">No refund for cancellations within 2 days of check-in</v-list-item-title>
                  </v-list-item>
                </v-list>
              </v-card-text>
            </v-card>

            <!-- Refund Calculation -->
            <v-card variant="outlined" class="mb-4" :color="theme.lightBg">
              <v-card-title class="text-subtitle-1 pb-2" :style="{color: theme.primary}">
                <v-icon class="mr-2">mdi-calculator</v-icon>
                Refund Calculation
              </v-card-title>
              <v-card-text class="pt-0">
                <v-list density="compact" bg-color="transparent">
                  <v-list-item>
                    <v-list-item-title class="text-body-2">Original Amount</v-list-item-title>
                    <template v-slot:append>
                      <span class="text-body-2">{{ formatCurrency(bookingToCancel.pricing.total_amount) }}</span>
                    </template>
                  </v-list-item>
                  
                  <v-list-item v-if="calculateCancellationFee(bookingToCancel).fee > 0">
                    <v-list-item-title class="text-body-2">
                      Cancellation Fee ({{ Math.round(calculateCancellationFee(bookingToCancel).feePercentage * 100) }}%)
                    </v-list-item-title>
                    <template v-slot:append>
                      <span class="text-body-2 text-error">-{{ formatCurrency(calculateCancellationFee(bookingToCancel).fee) }}</span>
                    </template>
                  </v-list-item>
                  
                  <v-divider class="my-2"></v-divider>
                  
                  <v-list-item class="font-weight-bold">
                    <v-list-item-title class="text-body-1">Refund Amount</v-list-item-title>
                    <template v-slot:append>
                      <span class="text-body-1" :style="{color: calculateCancellationFee(bookingToCancel).refund > 0 ? theme.success : theme.error}">
                        {{ formatCurrency(calculateCancellationFee(bookingToCancel).refund) }}
                      </span>
                    </template>
                  </v-list-item>
                </v-list>
              </v-card-text>
            </v-card>

            <!-- Cancellation Reason -->
            <v-textarea
              v-model="cancellationReason"
              label="Reason for Cancellation (Optional)"
              variant="outlined"
              :color="theme.primary"
              rows="3"
              placeholder="Please let us know why you're cancelling..."
            ></v-textarea>

            <!-- Confirmation Checkbox -->
            <v-checkbox
              v-model="confirmCancellation"
              :color="theme.primary"
              class="mt-2"
            >
              <template v-slot:label>
                <span class="text-body-2">
                  I understand the cancellation policy and confirm that I want to cancel this booking
                </span>
              </template>
            </v-checkbox>
          </v-card-text>

          <v-divider></v-divider>

          <v-card-actions class="pa-4">
            <v-btn
              variant="outlined"
              :color="theme.muted"
              @click="showCancelBookingModal = false; bookingToCancel = null; cancellationReason = ''; confirmCancellation = false"
            >
              Keep Booking
            </v-btn>
            <v-spacer></v-spacer>
            
            <v-btn
              color="error"
              :text-color="theme.lightText"
              @click="confirmCancelBooking(cancellationReason)"
              :disabled="!confirmCancellation"
              :loading="loading"
            >
              <v-icon start>mdi-cancel</v-icon>
              Cancel Booking
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>

      <!-- Book Again Confirmation Modal -->
      <v-dialog 
        v-model="showRebookModal" 
        :max-width="isMobile() ? '95vw' : '500'" 
        persistent
        scrollable
      >
        <v-card v-if="bookingToRebook">
          <v-card-title class="card-header" :style="{backgroundColor: theme.secondary, color: theme.primary}">
            <v-icon :color="theme.primary" class="mr-2">mdi-repeat</v-icon>
            Book Again
          </v-card-title>

          <v-card-text class="pa-4">
            <div class="text-center mb-4">
              <div class="mx-auto mb-3" style="max-width: 200px;">
                <RoomImageCarousel
                  :images="bookingToRebook.room_image ? [bookingToRebook.room_image] : []"
                  :room-name="bookingToRebook.room_name"
                  :height="120"
                  :show-fullscreen="true"
                />
              </div>
              <h3 class="text-h6 mb-2" :style="{color: theme.primary}">{{ bookingToRebook.room_name }}</h3>
              <div class="text-body-2" :style="{color: theme.muted}">{{ bookingToRebook.booking_reference }}</div>
            </div>

            <v-alert
              :color="theme.accent"
              variant="tonal"
              icon="mdi-information"
              class="mb-4"
              title="What will be pre-filled?"
            >
              <ul class="text-body-2 mt-2">
                <li>Guest information and contact details</li>
                <li>Room preferences (bed type, view, etc.)</li>
                <li v-if="bookingToRebook.additional_services && bookingToRebook.additional_services.length > 0">
                  Additional services ({{ bookingToRebook.additional_services.length }} selected)
                </li>
                <li>Number of guests</li>
              </ul>
            </v-alert>

            <v-alert
              type="info"
              variant="tonal"
              icon="mdi-calendar-clock"
              class="mb-4"
              title="Please note"
            >
              You'll need to select new dates and confirm room availability. Current pricing will apply.
            </v-alert>
          </v-card-text>

          <v-divider></v-divider>

          <v-card-actions class="pa-4">
            <v-btn
              variant="outlined"
              :color="theme.muted"
              @click="showRebookModal = false; bookingToRebook = null"
            >
              Cancel
            </v-btn>
            <v-spacer></v-spacer>
            
            <v-btn
              :color="theme.secondary"
              :text-color="theme.primary"
              @click="processRebooking(bookingToRebook); showRebookModal = false; bookingToRebook = null"
            >
              <v-icon start>mdi-repeat</v-icon>
              Yes, Book Again
            </v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
      
      <v-snackbar
        v-model="showSnackbar"
        :color="snackbar.color"
        :timeout="snackbar.timeout"
        location="top"
      >
        {{ snackbar.text }}
        <template v-slot:actions>
          <v-btn
            variant="text"
            @click="showSnackbar = false"
          >
            Close
          </v-btn>
        </template>
      </v-snackbar>
    </v-container>
  `
};