import getTropicalStyles from "../css/tropicalBookingStyles.js"; 

export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day
    
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);
    
    const lastWeek = new Date(today);
    lastWeek.setDate(lastWeek.getDate() - 7);

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',
        secondary: '#DDA333',
        accent: '#A3BA95',
        lightBg: '#F8F5F0',
        creamBg: '#EADDCB',
        darkText: '#1E1E1E',
        lightText: '#FFFFFF',
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        muted: '#78909C',
        brown: '#6D4C41',
      },

      // Filter state
      activeFilter: 'all', // all, unread, important
      
      // UI state
      loading: false,
      showDeleteDialog: false,
      selectedNotification: null,
      showFilterMenu: false,
      selectionMode: false,
      selectedNotifications: [],
      
      // Notification data
      notifications: [
        {
          id: 'n001',
          title: 'Welcome to Rainforest Resort!',
          message: 'Thank you for choosing Rainforest Resort for your stay. We hope you enjoy the lush tropical ambiance and our world-class amenities.',
          time: new Date(today.getTime() - 1 * 60 * 60 * 1000), // 1 hour ago
          read: false,
          category: 'welcome',
          icon: 'mdi-palm-tree',
          color: '#2E7D32',
          important: true
        },
        {
          id: 'n002',
          title: 'Your Room is Ready',
          message: 'Your Jungle View Suite is now ready for check-in. Please visit the front desk to collect your room key.',
          time: new Date(today.getTime() - 2 * 60 * 60 * 1000), // 2 hours ago
          read: false,
          category: 'room',
          icon: 'mdi-door-open',
          color: '#1A3C28',
          important: true
        },
        {
          id: 'n003',
          title: 'Spa Appointment Reminder',
          message: 'This is a reminder of your spa appointment today at 3:00 PM. Please arrive 15 minutes early to complete your wellness consultation.',
          time: new Date(today.getTime() - 5 * 60 * 60 * 1000), // 5 hours ago
          read: true,
          category: 'appointment',
          icon: 'mdi-spa',
          color: '#A3BA95',
          important: false
        },
        {
          id: 'n004',
          title: 'Restaurant Reservation Confirmed',
          message: 'Your reservation at The Canopy Restaurant has been confirmed for tonight at 7:30 PM for 2 guests.',
          time: new Date(yesterday.getTime() + 18 * 60 * 60 * 1000), // Yesterday evening
          read: true,
          category: 'dining',
          icon: 'mdi-silverware-fork-knife',
          color: '#DDA333',
          important: false
        },
        {
          id: 'n005',
          title: 'Resort Activity: Jungle Trek',
          message: 'Join our guided jungle trek tomorrow morning at 9:00 AM. Meet at the resort lobby. Comfortable shoes and insect repellent recommended.',
          time: new Date(yesterday.getTime() + 12 * 60 * 60 * 1000), // Yesterday noon
          read: false,
          category: 'activity',
          icon: 'mdi-hiking',
          color: '#6D4C41',
          important: false
        },
        {
          id: 'n006',
          title: 'Special Offer: 20% Off Spa Treatments',
          message: 'Enjoy 20% off all spa treatments this week when you book through our resort app. Limited time offer!',
          time: new Date(lastWeek.getTime() + 2 * 24 * 60 * 60 * 1000), // 5 days ago
          read: true,
          category: 'promotion',
          icon: 'mdi-ticket-percent',
          color: '#DDA333',
          important: true
        },
        {
          id: 'n007',
          title: 'Facility Maintenance Notice',
          message: 'The swimming pool will be closed for maintenance on Friday from 2:00 PM to 5:00 PM. We apologize for any inconvenience.',
          time: new Date(lastWeek.getTime() + 3 * 24 * 60 * 60 * 1000), // 4 days ago
          read: true,
          category: 'maintenance',
          icon: 'mdi-tools',
          color: '#78909C',
          important: false
        }
      ]
    };
  },
  
  computed: {
    // Notifications sorted by time (newest first)
    sortedNotifications() {
      return [...this.notifications].sort((a, b) => b.time - a.time);
    },
    
    // Filtered notifications based on active filter
    filteredNotifications() {
      if (this.activeFilter === 'all') {
        return this.sortedNotifications;
      } else if (this.activeFilter === 'unread') {
        return this.sortedNotifications.filter(n => !n.read);
      } else if (this.activeFilter === 'important') {
        return this.sortedNotifications.filter(n => n.important);
      }
      return this.sortedNotifications;
    },
    
    // Notifications grouped by date
    groupedNotifications() {
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      
      const yesterday = new Date(today);
      yesterday.setDate(yesterday.getDate() - 1);
      
      const todayNotifications = this.filteredNotifications.filter(n => {
        const noteDate = new Date(n.time);
        noteDate.setHours(0, 0, 0, 0);
        return noteDate.getTime() === today.getTime();
      });
      
      const yesterdayNotifications = this.filteredNotifications.filter(n => {
        const noteDate = new Date(n.time);
        noteDate.setHours(0, 0, 0, 0);
        return noteDate.getTime() === yesterday.getTime();
      });
      
      const earlierNotifications = this.filteredNotifications.filter(n => {
        const noteDate = new Date(n.time);
        noteDate.setHours(0, 0, 0, 0);
        return noteDate.getTime() < yesterday.getTime();
      });
      
      return {
        today: todayNotifications,
        yesterday: yesterdayNotifications,
        earlier: earlierNotifications
      };
    },
    
    // Count of unread notifications
    unreadCount() {
      return this.notifications.filter(n => !n.read).length;
    },
    
    // Count of important notifications
    importantCount() {
      return this.notifications.filter(n => n.important).length;
    },
    
    // Check if all notifications are selected
    allSelected() {
      return this.filteredNotifications.length > 0 && this.selectedNotifications.length === this.filteredNotifications.length;
    }
  },
  
  methods: {
    // Go back to previous screen
    goBack() {
      if (this.selectionMode) {
        this.selectionMode = false;
        this.selectedNotifications = [];
      } else {
        // Refresh the current page instead of going back
        window.location.reload();
      }
    },
    
    // Mark notification as read/unread
    toggleReadStatus(notification) {
      if (this.selectionMode) {
        this.toggleSelection(notification);
        return;
      }
      
      const index = this.notifications.findIndex(n => n.id === notification.id);
      if (index !== -1) {
        this.notifications[index].read = !this.notifications[index].read;
      }
    },
    
    // Mark notification as important/not important
    toggleImportant(notification) {
      const index = this.notifications.findIndex(n => n.id === notification.id);
      if (index !== -1) {
        this.notifications[index].important = !this.notifications[index].important;
      }
    },
    
    // Delete a notification
    deleteNotification(notification) {
      this.selectedNotification = notification;
      this.showDeleteDialog = true;
    },
    
    // Confirm deletion of notification
    confirmDelete() {
      if (this.selectedNotification) {
        const index = this.notifications.findIndex(n => n.id === this.selectedNotification.id);
        if (index !== -1) {
          this.notifications.splice(index, 1);
        }
      }
      this.showDeleteDialog = false;
      this.selectedNotification = null;
    },
    
    // Mark all notifications as read
    markAllAsRead() {
      this.notifications.forEach(notification => {
        notification.read = true;
      });
    },
    
    // Format time relative to now (just now, X mins ago, etc.)
    formatRelativeTime(dateTime) {
      const now = new Date();
      const diff = Math.floor((now - dateTime) / 1000); // seconds
      
      if (diff < 60) {
        return 'Just now';
      } else if (diff < 3600) {
        const mins = Math.floor(diff / 60);
        return `${mins} min${mins > 1 ? 's' : ''} ago`;
      } else if (diff < 86400) {
        const hours = Math.floor(diff / 3600);
        return `${hours} hour${hours > 1 ? 's' : ''} ago`;
      } else {
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        
        const yesterday = new Date(today);
        yesterday.setDate(yesterday.getDate() - 1);
        
        const notificationDate = new Date(dateTime);
        notificationDate.setHours(0, 0, 0, 0);
        
        if (notificationDate.getTime() === today.getTime()) {
          return this.formatTimeOnly(dateTime);
        } else if (notificationDate.getTime() === yesterday.getTime()) {
          return `Yesterday, ${this.formatTimeOnly(dateTime)}`;
        } else {
          return this.formatDateAndTime(dateTime);
        }
      }
    },
    
    // Format time only (HH:MM AM/PM)
    formatTimeOnly(dateTime) {
      const date = new Date(dateTime);
      return date.toLocaleTimeString('en-US', { hour: 'numeric', minute: 'numeric', hour12: true });
    },
    
    // Format date and time
    formatDateAndTime(dateTime) {
      const date = new Date(dateTime);
      return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }) + ', ' + 
        date.toLocaleTimeString('en-US', { hour: 'numeric', minute: 'numeric', hour12: true });
    },
    
    // Toggle selection mode
    toggleSelectionMode() {
      this.selectionMode = !this.selectionMode;
      if (!this.selectionMode) {
        this.selectedNotifications = [];
      }
    },
    
    // Toggle selection of a notification
    toggleSelection(notification) {
      const index = this.selectedNotifications.findIndex(n => n.id === notification.id);
      if (index === -1) {
        this.selectedNotifications.push(notification);
      } else {
        this.selectedNotifications.splice(index, 1);
      }
    },
    
    // Check if notification is selected
    isSelected(notification) {
      return this.selectedNotifications.some(n => n.id === notification.id);
    },
    
    // Select or deselect all notifications
    toggleSelectAll() {
      if (this.allSelected) {
        this.selectedNotifications = [];
      } else {
        this.selectedNotifications = [...this.filteredNotifications];
      }
    },
    
    // Delete selected notifications
    deleteSelected() {
      if (this.selectedNotifications.length > 0) {
        this.selectedNotifications.forEach(notification => {
          const index = this.notifications.findIndex(n => n.id === notification.id);
          if (index !== -1) {
            this.notifications.splice(index, 1);
          }
        });
        this.selectedNotifications = [];
        this.selectionMode = false;
      }
    },
    
    // Mark selected notifications as read
    markSelectedAsRead() {
      if (this.selectedNotifications.length > 0) {
        this.selectedNotifications.forEach(notification => {
          const index = this.notifications.findIndex(n => n.id === notification.id);
          if (index !== -1) {
            this.notifications[index].read = true;
          }
        });
        this.selectedNotifications = [];
        this.selectionMode = false;
      }
    }
  },
  
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
  },
  
  template: `
    <div class="tropical-bookings-page">
      <!-- Animated Background -->
      <div class="rainforest-bg">
        <div class="bg-leaf-animated leaf-1"></div>
        <div class="bg-leaf-animated leaf-2"></div>
        <div class="bg-leaf-animated leaf-3"></div>
        <div class="bg-sunlight-flare"></div>
      </div>
      
      <!-- Sticky Header -->
      <div class="sticky-header" :style="{backgroundColor: theme.primary, color: theme.lightText}">
        <div class="d-flex align-center px-4 py-3">
          <v-btn 
            icon="mdi-arrow-left" 
            variant="text" 
            :color="theme.lightText" 
            size="small" 
            @click="goBack"
          ></v-btn>
          <h2 class="text-h6 ms-3 my-0">
            {{ selectionMode ? selectedNotifications.length + ' selected' : 'Notifications' }}
          </h2>
          <v-spacer></v-spacer>
          
          <v-menu v-if="!selectionMode" v-model="showFilterMenu" :close-on-content-click="true">
            <template v-slot:activator="{ props }">
              <v-btn icon v-bind="props" :color="theme.lightText" size="small">
                <v-icon>mdi-filter-variant</v-icon>
              </v-btn>
            </template>
            <v-list density="compact">
              <v-list-item @click="activeFilter = 'all'">
                <template v-slot:prepend>
                  <v-icon :color="activeFilter === 'all' ? theme.primary : undefined" size="small">
                    {{ activeFilter === 'all' ? 'mdi-check-circle' : 'mdi-circle-outline' }}
                  </v-icon>
                </template>
                <v-list-item-title>All Notifications</v-list-item-title>
              </v-list-item>
              <v-list-item @click="activeFilter = 'unread'">
                <template v-slot:prepend>
                  <v-icon :color="activeFilter === 'unread' ? theme.primary : undefined" size="small">
                    {{ activeFilter === 'unread' ? 'mdi-check-circle' : 'mdi-circle-outline' }}
                  </v-icon>
                </template>
                <v-list-item-title>Unread</v-list-item-title>
                <template v-slot:append>
                  <v-chip size="x-small" :color="theme.primary" :text-color="theme.lightText">{{ unreadCount }}</v-chip>
                </template>
              </v-list-item>
              <v-list-item @click="activeFilter = 'important'">
                <template v-slot:prepend>
                  <v-icon :color="activeFilter === 'important' ? theme.primary : undefined" size="small">
                    {{ activeFilter === 'important' ? 'mdi-check-circle' : 'mdi-circle-outline' }}
                  </v-icon>
                </template>
                <v-list-item-title>Important</v-list-item-title>
                <template v-slot:append>
                  <v-chip size="x-small" :color="theme.primary" :text-color="theme.lightText">{{ importantCount }}</v-chip>
                </template>
              </v-list-item>
            </v-list>
          </v-menu>
          
          <template v-if="!selectionMode">
            <v-btn icon @click="toggleSelectionMode" :color="theme.lightText" size="small">
              <v-icon>mdi-checkbox-multiple-marked-outline</v-icon>
            </v-btn>
            <v-btn icon @click="markAllAsRead" :color="theme.lightText" size="small">
              <v-icon>mdi-email-check-outline</v-icon>
            </v-btn>
          </template>
          
          <template v-else>
            <v-btn icon @click="toggleSelectAll" :color="theme.lightText" size="small">
              <v-icon>{{ allSelected ? 'mdi-checkbox-marked-outline' : 'mdi-checkbox-blank-outline' }}</v-icon>
            </v-btn>
            <v-btn icon @click="markSelectedAsRead" :color="theme.lightText" size="small" :disabled="selectedNotifications.length === 0">
              <v-icon>mdi-email-check-outline</v-icon>
            </v-btn>
            <v-btn icon @click="deleteSelected" :color="theme.lightText" size="small" :disabled="selectedNotifications.length === 0">
              <v-icon>mdi-delete-outline</v-icon>
            </v-btn>
          </template>
        </div>
        
        <!-- Filter Chips -->
        <v-chip-group
          v-model="activeFilter"
          :selected-class="'selected-' + activeFilter"
          class="filter-chips px-4 pb-2"
          mandatory
        >
          <v-chip
            value="all"
            :class="{'selected-chip': activeFilter === 'all'}"
            :style="{
              backgroundColor: activeFilter === 'all' ? theme.primary : theme.lightBg,
              color: activeFilter === 'all' ? theme.lightText : theme.darkText
            }"
            variant="elevated"
            size="small"
          >
            <v-icon start size="small">mdi-bell-outline</v-icon>
            All
          </v-chip>
          
          <v-chip
            value="unread"
            :class="{'selected-chip': activeFilter === 'unread'}"
            :style="{
              backgroundColor: activeFilter === 'unread' ? theme.primary : theme.lightBg,
              color: activeFilter === 'unread' ? theme.lightText : theme.darkText
            }"
            variant="elevated"
            size="small"
          >
            <v-icon start size="small">mdi-email-unread-outline</v-icon>
            Unread
            <span v-if="unreadCount > 0" class="ms-1">({{ unreadCount }})</span>
          </v-chip>
          
          <v-chip
            value="important"
            :class="{'selected-chip': activeFilter === 'important'}"
            :style="{
              backgroundColor: activeFilter === 'important' ? theme.primary : theme.lightBg,
              color: activeFilter === 'important' ? theme.lightText : theme.darkText
            }"
            variant="elevated"
            size="small"
          >
            <v-icon start size="small">mdi-star-outline</v-icon>
            Important
            <span v-if="importantCount > 0" class="ms-1">({{ importantCount }})</span>
          </v-chip>
        </v-chip-group>
      </div>
      
      <!-- Main Content -->
      <div class="notifications-container px-sm-4 px-2 pb-8 pt-2">
        <!-- Empty State -->
        <div v-if="filteredNotifications.length === 0" class="empty-notifications text-center py-8">
          <v-icon size="64" :color="theme.accent" class="mb-2">mdi-bell-off-outline</v-icon>
          <h3 :class="$vuetify.display.xs ? 'text-h6' : 'text-h5'" class="mb-2" :style="{color: theme.primary}">No notifications</h3>
          <p :style="{color: theme.muted}" class="text-body-2">There are no notifications to display at this time.</p>
        </div>
        
        <!-- Notifications List -->
        <v-card v-else class="content-card" elevation="1">
          <!-- Today's Notifications -->
          <div v-if="groupedNotifications.today.length > 0">
            <v-card-subtitle class="pt-4 pb-2" :style="{color: theme.primary, backgroundColor: theme.lightBg}">
              <v-icon start size="small" :color="theme.primary">mdi-calendar-today</v-icon>
              Today
            </v-card-subtitle>
            
            <v-list class="pa-0">
              <v-list-item
                v-for="notification in groupedNotifications.today"
                :key="notification.id"
                :active="isSelected(notification)"
                :active-color="theme.accent"
                :style="{ backgroundColor: notification.read ? '' : theme.lightBg + '80' }"
                class="notification-item"
                @click="toggleReadStatus(notification)"
              >
                <template v-slot:prepend>
                  <div class="d-flex flex-column align-center me-3">
                    <v-checkbox
                      v-if="selectionMode"
                      v-model="selectedNotifications"
                      :value="notification"
                      hide-details
                      :color="theme.primary"
                      class="mt-0 pt-0"
                      @click.stop
                    ></v-checkbox>
                    <v-avatar
                      v-else
                      :color="notification.color"
                      size="36"
                      class="elevation-1"
                    >
                      <v-icon :color="theme.lightText">{{ notification.icon }}</v-icon>
                    </v-avatar>
                  </div>
                </template>
                
                <v-list-item-title 
                  :class="{'font-weight-bold': !notification.read}"
                  :style="{color: theme.primary}"
                >
                  {{ notification.title }}
                  <v-icon 
                    v-if="notification.important" 
                    size="small" 
                    :color="theme.secondary"
                    class="ms-1"
                  >
                    mdi-star
                  </v-icon>
                </v-list-item-title>
                
                <v-list-item-subtitle class="text-wrap mt-1" :style="{color: theme.darkText}">
                  {{ notification.message }}
                </v-list-item-subtitle>
                
                <v-list-item-subtitle class="mt-2 text-caption" :style="{color: theme.muted}">
                  {{ formatRelativeTime(notification.time) }}
                </v-list-item-subtitle>
                
                <template v-slot:append>
                  <div v-if="!notification.read && !selectionMode" class="unread-indicator" :style="{backgroundColor: theme.secondary}"></div>
                  <v-menu v-if="!selectionMode">
                    <template v-slot:activator="{ props }">
                      <v-btn 
                        icon="mdi-dots-vertical" 
                        variant="text" 
                        size="small"
                        v-bind="props"
                        :color="theme.muted"
                        @click.stop
                      ></v-btn>
                    </template>
                    <v-list density="compact">
                      <v-list-item @click.stop="toggleReadStatus(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.read ? 'mdi-email-mark-as-unread' : 'mdi-email-check-outline' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.read ? 'Mark as unread' : 'Mark as read' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item @click.stop="toggleImportant(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.important ? 'mdi-star-off' : 'mdi-star' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.important ? 'Remove importance' : 'Mark as important' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-divider></v-divider>
                      
                      <v-list-item @click.stop="deleteNotification(notification)" :color="theme.error">
                        <template v-slot:prepend>
                          <v-icon size="small" :color="theme.error">mdi-delete-outline</v-icon>
                        </template>
                        <v-list-item-title>Delete</v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-menu>
                </template>
              </v-list-item>
            </v-list>
          </div>
          
          <!-- Yesterday's Notifications -->
          <div v-if="groupedNotifications.yesterday.length > 0">
            <v-card-subtitle class="pt-4 pb-2" :style="{color: theme.primary, backgroundColor: theme.lightBg}">
              <v-icon start size="small" :color="theme.primary">mdi-calendar-blank-outline</v-icon>
              Yesterday
            </v-card-subtitle>
            
            <v-list class="pa-0">
              <v-list-item
                v-for="notification in groupedNotifications.yesterday"
                :key="notification.id"
                :active="isSelected(notification)"
                :active-color="theme.accent"
                :style="{ backgroundColor: notification.read ? '' : theme.lightBg + '80' }"
                class="notification-item"
                @click="toggleReadStatus(notification)"
              >
                <template v-slot:prepend>
                  <div class="d-flex flex-column align-center me-3">
                    <v-checkbox
                      v-if="selectionMode"
                      v-model="selectedNotifications"
                      :value="notification"
                      hide-details
                      :color="theme.primary"
                      class="mt-0 pt-0"
                      @click.stop
                    ></v-checkbox>
                    <v-avatar
                      v-else
                      :color="notification.color"
                      size="36"
                      class="elevation-1"
                    >
                      <v-icon :color="theme.lightText">{{ notification.icon }}</v-icon>
                    </v-avatar>
                  </div>
                </template>
                
                <v-list-item-title 
                  :class="{'font-weight-bold': !notification.read}"
                  :style="{color: theme.primary}"
                >
                  {{ notification.title }}
                  <v-icon 
                    v-if="notification.important" 
                    size="small" 
                    :color="theme.secondary"
                    class="ms-1"
                  >
                    mdi-star
                  </v-icon>
                </v-list-item-title>
                
                <v-list-item-subtitle class="text-wrap mt-1" :style="{color: theme.darkText}">
                  {{ notification.message }}
                </v-list-item-subtitle>
                
                <v-list-item-subtitle class="mt-2 text-caption" :style="{color: theme.muted}">
                  {{ formatRelativeTime(notification.time) }}
                </v-list-item-subtitle>
                
                <template v-slot:append>
                  <div v-if="!notification.read && !selectionMode" class="unread-indicator" :style="{backgroundColor: theme.secondary}"></div>
                  <v-menu v-if="!selectionMode">
                    <template v-slot:activator="{ props }">
                      <v-btn 
                        icon="mdi-dots-vertical" 
                        variant="text" 
                        size="small"
                        v-bind="props"
                        :color="theme.muted"
                        @click.stop
                      ></v-btn>
                    </template>
                    <v-list density="compact">
                      <v-list-item @click.stop="toggleReadStatus(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.read ? 'mdi-email-mark-as-unread' : 'mdi-email-check-outline' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.read ? 'Mark as unread' : 'Mark as read' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item @click.stop="toggleImportant(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.important ? 'mdi-star-off' : 'mdi-star' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.important ? 'Remove importance' : 'Mark as important' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-divider></v-divider>
                      
                      <v-list-item @click.stop="deleteNotification(notification)" :color="theme.error">
                        <template v-slot:prepend>
                          <v-icon size="small" :color="theme.error">mdi-delete-outline</v-icon>
                        </template>
                        <v-list-item-title>Delete</v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-menu>
                </template>
              </v-list-item>
            </v-list>
          </div>
          
          <!-- Earlier Notifications -->
          <div v-if="groupedNotifications.earlier.length > 0">
            <v-card-subtitle class="pt-4 pb-2" :style="{color: theme.primary, backgroundColor: theme.lightBg}">
              <v-icon start size="small" :color="theme.primary">mdi-calendar-range</v-icon>
              Earlier
            </v-card-subtitle>
            
            <v-list class="pa-0">
              <v-list-item
                v-for="notification in groupedNotifications.earlier"
                :key="notification.id"
                :active="isSelected(notification)"
                :active-color="theme.accent"
                :style="{ backgroundColor: notification.read ? '' : theme.lightBg + '80' }"
                class="notification-item"
                @click="toggleReadStatus(notification)"
              >
                <template v-slot:prepend>
                  <div class="d-flex flex-column align-center me-3">
                    <v-checkbox
                      v-if="selectionMode"
                      v-model="selectedNotifications"
                      :value="notification"
                      hide-details
                      :color="theme.primary"
                      class="mt-0 pt-0"
                      @click.stop
                    ></v-checkbox>
                    <v-avatar
                      v-else
                      :color="notification.color"
                      size="36"
                      class="elevation-1"
                    >
                      <v-icon :color="theme.lightText">{{ notification.icon }}</v-icon>
                    </v-avatar>
                  </div>
                </template>
                
                <v-list-item-title 
                  :class="{'font-weight-bold': !notification.read}"
                  :style="{color: theme.primary}"
                >
                  {{ notification.title }}
                  <v-icon 
                    v-if="notification.important" 
                    size="small" 
                    :color="theme.secondary"
                    class="ms-1"
                  >
                    mdi-star
                  </v-icon>
                </v-list-item-title>
                
                <v-list-item-subtitle class="text-wrap mt-1" :style="{color: theme.darkText}">
                  {{ notification.message }}
                </v-list-item-subtitle>
                
                <v-list-item-subtitle class="mt-2 text-caption" :style="{color: theme.muted}">
                  {{ formatRelativeTime(notification.time) }}
                </v-list-item-subtitle>
                
                <template v-slot:append>
                  <div v-if="!notification.read && !selectionMode" class="unread-indicator" :style="{backgroundColor: theme.secondary}"></div>
                  <v-menu v-if="!selectionMode">
                    <template v-slot:activator="{ props }">
                      <v-btn 
                        icon="mdi-dots-vertical" 
                        variant="text" 
                        size="small"
                        v-bind="props"
                        :color="theme.muted"
                        @click.stop
                      ></v-btn>
                    </template>
                    <v-list density="compact">
                      <v-list-item @click.stop="toggleReadStatus(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.read ? 'mdi-email-mark-as-unread' : 'mdi-email-check-outline' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.read ? 'Mark as unread' : 'Mark as read' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-list-item @click.stop="toggleImportant(notification)">
                        <template v-slot:prepend>
                          <v-icon size="small">{{ notification.important ? 'mdi-star-off' : 'mdi-star' }}</v-icon>
                        </template>
                        <v-list-item-title>{{ notification.important ? 'Remove importance' : 'Mark as important' }}</v-list-item-title>
                      </v-list-item>
                      
                      <v-divider></v-divider>
                      
                      <v-list-item @click.stop="deleteNotification(notification)" :color="theme.error">
                        <template v-slot:prepend>
                          <v-icon size="small" :color="theme.error">mdi-delete-outline</v-icon>
                        </template>
                        <v-list-item-title>Delete</v-list-item-title>
                      </v-list-item>
                    </v-list>
                  </v-menu>
                </template>
              </v-list-item>
            </v-list>
          </div>
        </v-card>
      </div>
      
      <!-- Delete Confirmation Dialog -->
      <v-dialog v-model="showDeleteDialog" max-width="400px">
        <v-card>
          <v-card-title class="text-subtitle-1 pt-3 pb-2" :style="{color: theme.error}">
            <v-icon :color="theme.error" start>mdi-alert-circle-outline</v-icon>
            Delete Notification
          </v-card-title>
          <v-card-text>
            Are you sure you want to delete this notification? This action cannot be undone.
          </v-card-text>
          <v-card-actions>
            <v-spacer></v-spacer>
            <v-btn variant="text" :color="theme.muted" @click="showDeleteDialog = false">Cancel</v-btn>
            <v-btn variant="text" :color="theme.error" @click="confirmDelete">Delete</v-btn>
          </v-card-actions>
        </v-card>
      </v-dialog>
    </div>
  `,
  
  style: `
    .sticky-header {
      position: sticky;
      top: 0;
      z-index: 100;
      box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .notifications-container {
      margin-top: 0;
      min-height: calc(100vh - 122px);
    }
    
    .notification-item {
      border-bottom: 1px solid rgba(0,0,0,0.05);
      padding: 12px 16px;
    }
    
    .notification-item:last-child {
      border-bottom: none;
    }
    
    .unread-indicator {
      width: 10px;
      height: 10px;
      border-radius: 50%;
      margin-right: 8px;
    }
    
    .empty-notifications {
      max-width: 300px;
      margin: 0 auto;
    }
    
    .filter-chips .v-chip {
      margin-right: 8px;
    }
    
    .selected-chip {
      font-weight: bold;
    }
    
    @media (max-width: 600px) {
      .sticky-header h2 {
        font-size: 1.1rem;
      }
      
      .notification-item {
        padding: 10px 12px;
      }
    }
  `
};
