import getTropicalStyles from "../css/tropicalBookingStyles.js"; 


export default {
  props: ["jsonData"],
  data() {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize to start of day

    return {
      // Theme colors
      theme: {
        primary: '#1A3C28',
        secondary: '#DDA333',
        accent: '#A3BA95',
        lightBg: '#F8F5F0',
        creamBg: '#EADDCB',
        darkText: '#1E1E1E',
        lightText: '#FFFFFF',
        error: '#B71C1C',
        success: '#2E7D32',
        warning: '#FF8F00',
        muted: '#78909C',
        brown: '#6D4C41',
      },

      selectedDate: today,
      minDateForPicker: today,

      bookingTypes: [],
      selectedType: null,

      facilities: { // Data remains the same
        sports: [
          { id: 'tennis1', name: 'Tennis Court 1', icon: 'mdi-tennis', availability: 'high', type: 'sports' },
          { id: 'tennis2', name: 'Tennis Court 2', icon: 'mdi-tennis', availability: 'medium', type: 'sports' },
          { id: 'swim', name: 'Swimming Pool', icon: 'mdi-pool', availability: 'low', type: 'sports' },
          { id: 'gym', name: 'Gymnasium', icon: 'mdi-dumbbell', availability: 'high', type: 'sports' },
          { id: 'badminton', name: 'Badminton Court', icon: 'mdi-badminton', availability: 'medium', type: 'sports' }
        ],
        events: [
          { id: 'banquet', name: 'Banquet Hall', icon: 'mdi-table-chair', availability: 'high', type: 'events' },
          { id: 'meeting', name: 'Meeting Room', icon: 'mdi-presentation', availability: 'medium', type: 'events' },
          { id: 'lounge', name: 'Rainforest Lounge', icon: 'mdi-sofa-single', availability: 'low', type: 'events' }
        ],
        services: [
          { id: 'spa', name: 'Jungle Spa Retreat', icon: 'mdi-spa', availability: 'medium', type: 'services' },
          { id: 'salon', name: 'Canopy Beauty Salon', icon: 'mdi-content-cut', availability: 'high', type: 'services' },
          { id: 'trainer', name: 'Personal Trainer', icon: 'mdi-human-run', availability: 'low', type: 'services' }
        ],
        dining: [
          { id: 'restaurant', name: 'The Canopy Restaurant', icon: 'mdi-food-variant', availability: 'medium', type: 'dining' },
          { id: 'cafe', name: 'Orchid Café', icon: 'mdi-coffee', availability: 'high', type: 'dining' },
          { id: 'bar', name: 'Treetop Bar', icon: 'mdi-glass-cocktail', availability: 'low', type: 'dining' }
        ]
      },

      selectedFacility: null,
      timeSlots: [],
      selectedTimeSlot: null,

      myBookings: [ // Data remains the same
        { id: 'bk1001', facilityId: 'tennis1', facilityName: 'Tennis Court 1', date: new Date(new Date().setDate(today.getDate() + 2)).toISOString().substring(0,10), timeSlot: '10:00 - 11:00', status: 'confirmed' },
        { id: 'bk1002', facilityId: 'restaurant', facilityName: 'The Canopy Restaurant', date: new Date(new Date().setDate(today.getDate() + 1)).toISOString().substring(0,10), timeSlot: '19:00 - 21:00', guests: 4, status: 'confirmed' },
        { id: 'bk1003', facilityId: 'spa', facilityName: 'Jungle Spa Retreat', date: new Date(new Date().setDate(today.getDate() + 5)).toISOString().substring(0,10), timeSlot: '14:00 - 15:00', status: 'pending' },
        { id: 'bk1004', facilityId: 'swim', facilityName: 'Swimming Pool', date: new Date(new Date().setDate(today.getDate() - 2)).toISOString().substring(0,10), timeSlot: '11:00 - 12:00', status: 'completed' },
      ],

      bookingForm: { guests: 1, notes: '', duration: 1 },
      loading: false,
      showBookingDialog: false,
      showCancelDialog: false,
      selectedBookingToCancel: null,
      activeTab: 'find',
      formValid: false,
    };
  },
  computed: { // All computed properties remain the same
    availableFacilities() {
      return this.facilities[this.selectedType] || [];
    },
    formattedSelectedDate() {
      if (!this.selectedDate) return "No date selected";
      const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
      return this.selectedDate.toLocaleDateString('en-US', options);
    },
    upcomingBookings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.myBookings.filter(booking => {
        const bookingDate = new Date(booking.date);
        bookingDate.setHours(0,0,0,0);
        return bookingDate >= todayNormalized && booking.status !== 'cancelled' && booking.status !== 'completed';
      }).sort((a, b) => new Date(a.date) - new Date(b.date) || a.timeSlot.localeCompare(b.timeSlot));
    },
    pastBookings() {
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.myBookings.filter(booking => {
        const bookingDate = new Date(booking.date);
        bookingDate.setHours(0,0,0,0);
        return bookingDate < todayNormalized || booking.status === 'cancelled' || booking.status === 'completed';
      }).sort((a, b) => new Date(b.date) - new Date(a.date) || b.timeSlot.localeCompare(a.timeSlot));
    },
    isToday() {
      if (!this.selectedDate) return false;
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.selectedDate.getTime() === todayNormalized.getTime();
    },
    isPastDate() {
      if (!this.selectedDate) return false;
      const todayNormalized = new Date();
      todayNormalized.setHours(0,0,0,0);
      return this.selectedDate < todayNormalized;
    },
    selectedTypeDetails() {
        return this.bookingTypes.find(bt => bt.id === this.selectedType);
    },
    selectedDateAsString() {
        return this.selectedDate ? this.selectedDate.toISOString().substring(0,10) : null;
    }
  },
  methods: { // All methods remain largely the same, just minor console.log fallbacks for snackbar
    initializeBookingTypes() {
        this.bookingTypes = [
          { id: 'sports', name: 'Sports & Recreation', icon: 'mdi-tennis', color: this.theme.primary, textColor: this.theme.lightText },
          { id: 'events', name: 'Event Venues', icon: 'mdi-party-popper', color: this.theme.secondary, textColor: this.theme.primary },
          { id: 'services', name: 'Wellness & Spa', icon: 'mdi-spa', color: this.theme.accent, textColor: this.theme.primary },
          { id: 'dining', name: 'Dining Experiences', icon: 'mdi-silverware-fork-knife', color: this.theme.brown, textColor: this.theme.lightText }
        ];
        if (this.bookingTypes.length > 0 && !this.selectedType) {
            this.selectedType = this.bookingTypes[0].id;
        }
    },
    getFacilityTypeDetails(facilityId) {
        for (const typeKey in this.facilities) {
            const facility = this.facilities[typeKey].find(f => f.id === facilityId);
            if (facility) {
                return this.bookingTypes.find(bt => bt.id === facility.type) || this.bookingTypes[0];
            }
        }
        return this.bookingTypes[0];
    },
    updateMyBookingsColors() {
        this.myBookings = this.myBookings.map(booking => {
            const typeDetails = this.getFacilityTypeDetails(booking.facilityId);
            return { ...booking, color: typeDetails.color, icon: typeDetails.icon };
        });
    },
    updateDate(newDateVal) {
      if (this.selectedDate) {
        this.selectedDate.setHours(0,0,0,0);
      }
      if (this.selectedFacility) {
        this.generateTimeSlots();
      }
    },
    selectFacility(facility) {
      this.selectedFacility = facility;
      this.generateTimeSlots();
    },
    clearFacilitySelection() {
      this.selectedFacility = null;
      this.timeSlots = [];
      this.selectedTimeSlot = null;
    },
    generateTimeSlots() {
      if (!this.selectedDate) {
          this.timeSlots = [];
          return;
      }
      this.loading = true;
      this.timeSlots = [];
      setTimeout(() => {
        const slots = [];
        const facilityTypeDetails = this.getFacilityTypeDetails(this.selectedFacility.id);
        const startHour = facilityTypeDetails.id === 'dining' ? 11 : 8;
        const endHour = facilityTypeDetails.id === 'dining' ? 22 : 20;
        const slotDuration = facilityTypeDetails.id === 'services' ? 60 : (facilityTypeDetails.id === 'dining' ? 120 : 60);

        const bookedSlots = new Set();
        const availability = this.selectedFacility.availability;
        const bookedCount = availability === 'low' ? Math.floor((endHour - startHour) * (60 / slotDuration) * 0.7) :
                           availability === 'medium' ? Math.floor((endHour - startHour) * (60 / slotDuration) * 0.4) :
                           Math.floor((endHour - startHour) * (60 / slotDuration) * 0.2);

        for (let i = 0; i < bookedCount; i++) {
          const randomHour = Math.floor(Math.random() * (endHour - startHour)) + startHour;
          const slotsInHour = 60 / slotDuration;
          const randomMinuteBlock = Math.floor(Math.random() * slotsInHour);
          const randomMinute = randomMinuteBlock * slotDuration;
          bookedSlots.add(`${String(randomHour).padStart(2, '0')}:${String(randomMinute).padStart(2, '0')}`);
        }

        for (let hour = startHour; hour < endHour; hour++) {
          for (let minute = 0; minute < 60; minute += slotDuration) {
            const startTime = `${String(hour).padStart(2, '0')}:${String(minute).padStart(2, '0')}`;
            let endTimeHour = hour;
            let endTimeMinute = minute + slotDuration;
            if (endTimeMinute >= 60) {
              endTimeHour += Math.floor(endTimeMinute / 60);
              endTimeMinute %= 60;
            }
            const endTime = `${String(endTimeHour).padStart(2, '0')}:${String(endTimeMinute).padStart(2, '0')}`;
            const timeSlot = `${startTime} - ${endTime}`;
            const isBooked = bookedSlots.has(startTime);

            if (this.isToday) {
              const now = new Date();
              const slotHour = parseInt(startTime.split(':')[0]);
              const slotMinute = parseInt(startTime.split(':')[1]);
              if (slotHour < now.getHours() || (slotHour === now.getHours() && slotMinute < now.getMinutes())) {
                continue;
              }
            }
            if (this.isPastDate) continue;

            slots.push({
              time: timeSlot,
              booked: isBooked,
              price: this.getSlotPrice(hour, facilityTypeDetails.id)
            });
          }
        }
        this.timeSlots = slots;
        this.loading = false;
      }, 800);
    },
    getSlotPrice(hour, typeId) {
      let basePrice = 40;
      if (typeId === 'services') basePrice = 80;
      else if (typeId === 'events') basePrice = 200;
      else if (typeId === 'dining') basePrice = 50;
      return (hour >= 17 && hour < 21) ? basePrice * 1.5 : basePrice;
    },
    selectTimeSlot(slot) {
      if (!slot.booked && !this.isPastDate && this.selectedDate) {
        this.selectedTimeSlot = slot;
        this.bookingForm = { guests: (this.selectedType === 'dining' ? 2 : 1) , notes: '', duration: 1 };
        this.showBookingDialog = true;
      }
    },
    submitBooking() {
      this.loading = true;
      setTimeout(() => {
        const facilityTypeDetails = this.getFacilityTypeDetails(this.selectedFacility.id);
        const newBooking = {
          id: 'bk' + Math.floor(1000 + Math.random() * 9000),
          facilityId: this.selectedFacility.id,
          facilityName: this.selectedFacility.name,
          date: this.selectedDateAsString,
          timeSlot: this.selectedTimeSlot.time,
          status: 'confirmed',
          icon: facilityTypeDetails.icon,
          color: facilityTypeDetails.color,
          guests: this.bookingForm.guests,
          notes: this.bookingForm.notes
        };
        this.myBookings.unshift(newBooking);
        this.bookingForm = { guests: 1, notes: '', duration: 1 };
        this.showBookingDialog = false;
        this.selectedTimeSlot = null;
        if (this.$root.showSnackbar) { this.$root.showSnackbar('Booking confirmed in the rainforest!', 'success'); }
        else { console.info('Snackbar: Booking confirmed in the rainforest!'); }
        this.activeTab = 'bookings';
        this.loading = false;
      }, 1200);
    },
    confirmCancelBooking() {
      this.loading = true;
      setTimeout(() => {
        const index = this.myBookings.findIndex(b => b.id === this.selectedBookingToCancel.id);
        if (index !== -1) {
          this.myBookings[index].status = 'cancelled';
        }
        this.showCancelDialog = false;
        this.selectedBookingToCancel = null;
        if (this.$root.showSnackbar) { this.$root.showSnackbar('Booking successfully cancelled.', 'info'); }
        else { console.info('Snackbar: Booking successfully cancelled.'); }
        this.loading = false;
      }, 800);
    },
    formatDateForDisplay(dateStr_YYYY_MM_DD) {
      if (!dateStr_YYYY_MM_DD) return '';
      const parts = dateStr_YYYY_MM_DD.split('-');
      const dateObj = new Date(parts[0], parts[1] - 1, parts[2]);
      const options = { weekday: 'short', month: 'short', day: 'numeric' };
      return dateObj.toLocaleDateString('en-US', options);
    },
    getStatusDetails(status) {
      switch(status) {
        case 'confirmed': return { text: 'Confirmed', color: this.theme.success, icon: 'mdi-check-circle' };
        case 'pending': return { text: 'Pending', color: this.theme.warning, icon: 'mdi-clock-alert-outline' };
        case 'cancelled': return { text: 'Cancelled', color: this.theme.error, icon: 'mdi-cancel' };
        case 'completed': return { text: 'Completed', color: this.theme.muted, icon: 'mdi-history' };
        default: return { text: status, color: this.theme.muted, icon: 'mdi-information-outline' };
      }
    },
    getAvailabilityDetails(availability) {
        switch(availability) {
            case 'high': return { text: 'Many Slots', color: this.theme.success, icon: 'mdi-leaf-circle' };
            case 'medium': return { text: 'Some Slots', color: this.theme.warning, icon: 'mdi-leaf-circle-outline' };
            case 'low': return { text: 'Few Slots Left', color: this.theme.error, icon: 'mdi-alert-circle-outline' };
            default: return { text: 'Unknown', color: this.theme.muted, icon: 'mdi-help-circle-outline'};
        }
    }
  },
  created() {
    this.initializeBookingTypes();
    this.updateMyBookingsColors();
  },
  watch: {
    selectedDate: {
        handler(newDate, oldDate) {
            if (this.selectedFacility) {
                 this.generateTimeSlots();
            }
        },
    },
    selectedType() {
        this.clearFacilitySelection();
    }
  },
  template: `

  `,
  mounted() {
    // Create a style element and apply the CSS from the external file
    const styleElement = document.createElement('style');
    
    // Pass the theme object to the getTropicalStyles function
    styleElement.textContent = getTropicalStyles(this.theme);
    document.head.appendChild(styleElement);
    
    // Rest of your mounted code
    if (this.bookingTypes && this.bookingTypes.length > 0 && !this.selectedType) {
      this.selectedType = this.bookingTypes[0].id;
    }
  }
};